using Microsoft.VisualStudio.TestTools.UnitTesting;
using AADJoin.Messages;
using System.Security.Cryptography;
using System;
using System.Text;
using AADJoin.Utils;
using System.Globalization;
using System.Numerics;
using System.Collections.Generic;
using System.Linq;
using AADJoin;
using System.Threading.Tasks;
using Newtonsoft.Json.Bson;
using Microsoft.IdentityModel.Tokens;
using System.IO;
using System.Security.Cryptography.X509Certificates;

namespace UnitTests
{
    [TestClass]
    public class UnitTest1
    {
        [TestMethod]
        public void TestTenantIdParsing()
        {
            var authResponse = AuthResponse.fromString("{\"token_type\":\"Bearer\",\"scope\":\"user_impersonation\",\"expires_in\":\"3958\",\"ext_expires_in\":\"3958\",\"expires_on\":\"1660206842\",\"not_before\":\"1660202583\",\"resource\":\"https://graph.windows.net\",\"access_token\":\"eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsIng1dCI6IjJaUXBKM1VwYmpBWVhZR2FYRUpsOGxWMFRPSSIsImtpZCI6IjJaUXBKM1VwYmpBWVhZR2FYRUpsOGxWMFRPSSJ9.eyJhdWQiOiJodHRwczovL2dyYXBoLndpbmRvd3MubmV0IiwiaXNzIjoiaHR0cHM6Ly9zdHMud2luZG93cy5uZXQvYTkyNTI0NDgtOWFiNy00OWIwLWJiNWMtZjJmOTIzYzg0NjcyLyIsImlhdCI6MTY2MDIwMjU4MywibmJmIjoxNjYwMjAyNTgzLCJleHAiOjE2NjAyMDY4NDIsImFjciI6IjEiLCJhaW8iOiJBU1FBMi84VEFBQUF4dHBFbTMvWnEwQlZPcnQ3M1BXazNYTjJzRHlrUGdRUjgvWENqLzA2OEpzPSIsImFtciI6WyJwd2QiXSwiYXBwaWQiOiIxYjczMDk1NC0xNjg1LTRiNzQtOWJmZC1kYWMyMjRhN2I4OTQiLCJhcHBpZGFjciI6IjAiLCJpcGFkZHIiOiIxOTQuNDQuNDUuMTIxIiwibmFtZSI6InM0Iiwib2lkIjoiZWE4YjhlODctNzM0OC00MzJhLWFlNTUtNGIyMjQ2MWE3NGJkIiwicHVpZCI6IjEwMDMyMDAyMUFFRDgzNjciLCJyaCI6IjAuQVhrQVNDUWxxYmVhc0VtN1hQTDVJOGhHY2dJQUFBQUFBQUFBd0FBQUFBQUFBQUNVQUkwLiIsInNjcCI6InVzZXJfaW1wZXJzb25hdGlvbiIsInN1YiI6ImRzeU5ZNUp3SzYxbU5PbTAwakdNOFgtak5VdS00X1BuMmRIQmxoMnc4QkkiLCJ0ZW5hbnRfcmVnaW9uX3Njb3BlIjoiRVUiLCJ0aWQiOiJhOTI1MjQ0OC05YWI3LTQ5YjAtYmI1Yy1mMmY5MjNjODQ2NzIiLCJ1bmlxdWVfbmFtZSI6InM0QGRhdGFhbnMuY29tIiwidXBuIjoiczRAZGF0YWFucy5jb20iLCJ1dGkiOiJyRTl4QVdjWnlrSzJVT0VQeEdGWUFBIiwidmVyIjoiMS4wIn0.Ma5odOE1Lnm0C_5aCPPLdyTISO-1wNhTjLcvgXl4D_1LyxeXSYnMv5H0D2_HNjntDWPqBWzpXbujCE3rwZYeoSlrBSC2N3Ua0T38ceOCxS7fwA70Th4dgpFITPHokintL26eLXnlwr7erOOO_1T8RGcj9RRolF5gI1ArnL4nXmBsKisQBjZuFqKUYZgMzqBbBtcxbRGy_NduaOx_NKwxE_c7Pmo4mS6YX22n4z4p1hakCXqk3UmGin3basWHRoiD32Z7vJyXX1KWkvv8MIVupVkurW8KAvRfx4asSAsQn7Ii2_JsItxLCDoFXY--6VFaB4mwdp6xKQWchb1BVZi0dA\",\"refresh_token\":\"0.AXkASCQlqbeasEm7XPL5I8hGclQJcxuFFnRLm_3awiSnuJSUAI0.AgABAAEAAAD--DLA3VO7QrddgJg7WevrAgDs_wQA9P-v1WHQotebkoyXd9fjaE78Ld2Esf6sLTg0AeeyRqOL7XekLSSdFiAncPZTwZYzG_E60e6DZYgrAhRxBt5QRgRLNtHDAJZ2h0tiXGdpSQsLvlXKWQEre1FeosL6KeAGPCCfphBX_1khW8tk-YIxXi_5kwUxZedbwPS3qA5VnQKvYvPMF9bQyHYUdk8AVfPsorcln4KTQdTaRpoGZCsdwTtat13jnOnfs6NSr7JV_l6U2ysCyZZDtS2Q327lGH8_MTwhf-rtxJe60T8dDFsPQhGPGXwYSKJoToLE3UXMIGcjMtkgsiPkku-XGDLvR4c2WlDYm4TwZUKQ5JUZvBk9jeFPLS2WCrdguK5pWKjZw02vLBu3eBxE7AesZXNew-bNvjyG65qBJRI_0du7z0NH5CaiqCeWmgc7bHejl3pkM6_WXhhBpj_OvnzwJFbPbmSW1_N24FaJiqralvPIyQqqBjkX1GTzHDLHONaGD5en2w2Z4m_4glTkko4z3IP3YGP_AN_WjdJ5NPrgBEYzAmlaMNJDEFejXPe2ifWjxucwqhVbPBu5_TcwJhuErx1adl4YQ_Pzq56BaP3Ybi6gNxdnnsqtcOPiDlMC5lv8BCIOT4kLW8bpksM1tqpxUrxHNk65C4qiHEVWABH8cpabmg\",\"id_token\":\"eyJ0eXAiOiJKV1QiLCJhbGciOiJub25lIn0.eyJhdWQiOiIxYjczMDk1NC0xNjg1LTRiNzQtOWJmZC1kYWMyMjRhN2I4OTQiLCJpc3MiOiJodHRwczovL3N0cy53aW5kb3dzLm5ldC9hOTI1MjQ0OC05YWI3LTQ5YjAtYmI1Yy1mMmY5MjNjODQ2NzIvIiwiaWF0IjoxNjYwMjAyNTgzLCJuYmYiOjE2NjAyMDI1ODMsImV4cCI6MTY2MDIwNjQ4MywiYW1yIjpbInB3ZCJdLCJpcGFkZHIiOiIxOTQuNDQuNDUuMTIxIiwibmFtZSI6InM0Iiwib2lkIjoiZWE4YjhlODctNzM0OC00MzJhLWFlNTUtNGIyMjQ2MWE3NGJkIiwicHVpZCI6IjEwMDMyMDAyMUFFRDgzNjciLCJyaCI6IjAuQVhrQVNDUWxxYmVhc0VtN1hQTDVJOGhHY2xRSmN4dUZGblJMbV8zYXdpU251SlNVQUkwLiIsInN1YiI6ImVzQzkwWWdfN3MyTVBXNTA0cE04UVp5Njl4ZGZPUVQxbExoQnZqYjdJRjgiLCJ0ZW5hbnRfcmVnaW9uX3Njb3BlIjoiRVUiLCJ0aWQiOiJhOTI1MjQ0OC05YWI3LTQ5YjAtYmI1Yy1mMmY5MjNjODQ2NzIiLCJ1bmlxdWVfbmFtZSI6InM0QGRhdGFhbnMuY29tIiwidXBuIjoiczRAZGF0YWFucy5jb20iLCJ2ZXIiOiIxLjAifQ.\"}");

            Assert.AreEqual("a9252448-9ab7-49b0-bb5c-f2f923c84672", authResponse.GetTenantId());
        }

        [TestMethod]
        public void TestP2PCertificatesResponseParsing()
        {
            var rawResponse = "{\"token_type\":\"x509\",\"expires_in\":\"86399\",\"ext_expires_in\":\"0\",\"expires_on\":\"1660480130\",\"x5c\":\"MIIDgDCCAmigAwIBAgIQYNQVBIdbLvF59KM1CaK9UzANBgkqhkiG9w0BAQsFADBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wHhcNMjIwODEzMTIyMzUwWhcNMjIwODE0MTIyODUwWjBlMTQwMgYKCZImiZPyLGQBGRYkYTkyNTI0NDgtOWFiNy00OWIwLWJiNWMtZjJmOTIzYzg0NjcyMS0wKwYDVQQDDCRkNzgyNTIyNS02YTAzLTQ1Y2YtOTAzOC01MGM2ODM0MWRmMWYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDiw1YUnKRxOMMOdBOc2DXgqR6qUp+4+43AURHRrez+lHu3/vi5LtQJZlh1SVLJxIRSzPo7LAvbD3lFga9f5TX5lzVn+fNUR6YmQxSGhNwt6XM6O+aiW/gS0QnS8CpLVI1rBkqA1HomHGAu6IWsIWBv+7BKLP50XaHP9HZLHeisJPWQqUkeZD319io20MM/prPCFCtvA70SFaZrFSGCtTPUbDeU57YgX/68Nwx34djD40CEein3q1cyGYRcPdJpS0W38hPLl9w5l6NVyXid5zRm4DFnICf+bqgsCOryx/LLLmoecI2Km51uELP1xlCl8A1rzyyv6O9H1oGP6lHsB9INAgMBAAGjRDBCMA4GA1UdDwEB/wQEAwIFoDATBgNVHSUEDDAKBggrBgEFBQcDATAbBgkrBgEEAYI3FQoEDjAMMAoGCCsGAQUFBwMBMA0GCSqGSIb3DQEBCwUAA4IBAQBcRy7zmvoWE8SvRQh1QJJf3sPvnmP6FnWkd5rU0oQrqeYx7Y++l7C0Hpl600FcsxMaEDi65uiW80vUpCTOjTlgCZv57Z7Ra8nTm0jI4Y+EiszoA8FZxPTjFv0hQ/f92sbjCLgi7TN6EcO8i+eGeez9uCAuvk2dITZBobOgEaqschpt0gnEeGHqHq3rIOuNqsFoBSmS3niIPULgtGElbhCNtsuGlAAdMch0gaYw6U29I+ZAQ0XRPSA+8IuC6hZcv4+ij3ItAkay544NCN4rEzAXiuEtee6YlBPjCqG3s4Gb0ebVFdEvZQ7k4Bh0h8O+hqDgde63Ki8o2Yu0bTMH3hbP\",\"cert_token_use\":\"device_cert\",\"x5c_ca\":\"MIIDRTCCAi2gAwIBAgIQFjJo2vHWmoNDMmy1tdDXjjANBgkqhkiG9w0BAQsFADBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wHhcNMjIwODA4MDAwMDAwWhcNMjMwODA4MDAwMDAwWjBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDL5cI1sb5q4MzqePzFPJkD+cZILh1+pbTriwbSY7pwPNdEEVQ6Z1aBcdVxYUHXYX7eEDulwH+0CQNLw+RHtrajIBEEf3n1DgHPno54rBaCK6jGeDo8SdM03B6HzFnH2Z8zDbYog42jtMyLsGgo6eGpLpL2kHHTwPnkqcVKT5zDodqlHu6Cl7ZSYJlJMV0nqYsIEocEidw3QBwPM3Eb2ZUIU+aLg64GGoCfoXFnoXqMVRymZLyMXZaDfT4J3pFQl8f6yutqmBu6j7XDxIAEamkMxD1FWicOVwOZMK9BYTLgaXXGawlx8Nf//3QhCaCDSliWDw7OeONBp1BodWymqamRAgMBAAGjITAfMB0GA1UdDgQWBBQRWuiJpM68E5FnjQ1sbUpkoa9EJTANBgkqhkiG9w0BAQsFAAOCAQEAWKFD5lu244+UPfSTPpUOaePzs0ofuDjDhYzrx9dNPiOU1SOKfmwNXz2qgeaWaggc2/zEhpgJ5S7593HBHNT8LbKRmwlT9+4qO78B3JsN0hKzqChXV8rczS4rFylmoUJIpVNrip+CHDyP4VVynillduVYEQDt5bkkBovQjonlWPq+nMxZz+vvl7mGCSYeN2FhwiUWO4AaDv0NGSjR3YJ1mpeZa9uNw/pyr5iF1g0BfjmUIDgCBpof3ZEIxFMkzDhmscIOOtTRKPV8H2MP907boXn1oNeJhb7WeoEaW2iwYr8uEef82f8NFKvDIVnr2nhcfCSoo/7bF2hvLEXw432ouQ==\",\"resource\":\"urn:p2p_cert\"}";

            var p2pCertificatesResponse = P2PCertificatesResponse.FromString(rawResponse);

            Assert.AreEqual("x509", p2pCertificatesResponse.token_type);
            Assert.AreEqual("86399", p2pCertificatesResponse.expires_in);
            Assert.AreEqual("0", p2pCertificatesResponse.ext_expires_in);
            Assert.AreEqual(
                "MIIDgDCCAmigAwIBAgIQYNQVBIdbLvF59KM1CaK9UzANBgkqhkiG9w0BAQsFADBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wHhcNMjIwODEzMTIyMzUwWhcNMjIwODE0MTIyODUwWjBlMTQwMgYKCZImiZPyLGQBGRYkYTkyNTI0NDgtOWFiNy00OWIwLWJiNWMtZjJmOTIzYzg0NjcyMS0wKwYDVQQDDCRkNzgyNTIyNS02YTAzLTQ1Y2YtOTAzOC01MGM2ODM0MWRmMWYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDiw1YUnKRxOMMOdBOc2DXgqR6qUp+4+43AURHRrez+lHu3/vi5LtQJZlh1SVLJxIRSzPo7LAvbD3lFga9f5TX5lzVn+fNUR6YmQxSGhNwt6XM6O+aiW/gS0QnS8CpLVI1rBkqA1HomHGAu6IWsIWBv+7BKLP50XaHP9HZLHeisJPWQqUkeZD319io20MM/prPCFCtvA70SFaZrFSGCtTPUbDeU57YgX/68Nwx34djD40CEein3q1cyGYRcPdJpS0W38hPLl9w5l6NVyXid5zRm4DFnICf+bqgsCOryx/LLLmoecI2Km51uELP1xlCl8A1rzyyv6O9H1oGP6lHsB9INAgMBAAGjRDBCMA4GA1UdDwEB/wQEAwIFoDATBgNVHSUEDDAKBggrBgEFBQcDATAbBgkrBgEEAYI3FQoEDjAMMAoGCCsGAQUFBwMBMA0GCSqGSIb3DQEBCwUAA4IBAQBcRy7zmvoWE8SvRQh1QJJf3sPvnmP6FnWkd5rU0oQrqeYx7Y++l7C0Hpl600FcsxMaEDi65uiW80vUpCTOjTlgCZv57Z7Ra8nTm0jI4Y+EiszoA8FZxPTjFv0hQ/f92sbjCLgi7TN6EcO8i+eGeez9uCAuvk2dITZBobOgEaqschpt0gnEeGHqHq3rIOuNqsFoBSmS3niIPULgtGElbhCNtsuGlAAdMch0gaYw6U29I+ZAQ0XRPSA+8IuC6hZcv4+ij3ItAkay544NCN4rEzAXiuEtee6YlBPjCqG3s4Gb0ebVFdEvZQ7k4Bh0h8O+hqDgde63Ki8o2Yu0bTMH3hbP",
                p2pCertificatesResponse.x5c
            );
            Assert.AreEqual("device_cert", p2pCertificatesResponse.cert_token_use);
            Assert.AreEqual(
                "MIIDRTCCAi2gAwIBAgIQFjJo2vHWmoNDMmy1tdDXjjANBgkqhkiG9w0BAQsFADBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wHhcNMjIwODA4MDAwMDAwWhcNMjMwODA4MDAwMDAwWjBNMUswSQYDVQQDHkIATQBTAC0ATwByAGcAYQBuAGkAegBhAHQAaQBvAG4ALQBQADIAUAAtAEEAYwBjAGUAcwBzACAAWwAyADAAMgAyAF0wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDL5cI1sb5q4MzqePzFPJkD+cZILh1+pbTriwbSY7pwPNdEEVQ6Z1aBcdVxYUHXYX7eEDulwH+0CQNLw+RHtrajIBEEf3n1DgHPno54rBaCK6jGeDo8SdM03B6HzFnH2Z8zDbYog42jtMyLsGgo6eGpLpL2kHHTwPnkqcVKT5zDodqlHu6Cl7ZSYJlJMV0nqYsIEocEidw3QBwPM3Eb2ZUIU+aLg64GGoCfoXFnoXqMVRymZLyMXZaDfT4J3pFQl8f6yutqmBu6j7XDxIAEamkMxD1FWicOVwOZMK9BYTLgaXXGawlx8Nf//3QhCaCDSliWDw7OeONBp1BodWymqamRAgMBAAGjITAfMB0GA1UdDgQWBBQRWuiJpM68E5FnjQ1sbUpkoa9EJTANBgkqhkiG9w0BAQsFAAOCAQEAWKFD5lu244+UPfSTPpUOaePzs0ofuDjDhYzrx9dNPiOU1SOKfmwNXz2qgeaWaggc2/zEhpgJ5S7593HBHNT8LbKRmwlT9+4qO78B3JsN0hKzqChXV8rczS4rFylmoUJIpVNrip+CHDyP4VVynillduVYEQDt5bkkBovQjonlWPq+nMxZz+vvl7mGCSYeN2FhwiUWO4AaDv0NGSjR3YJ1mpeZa9uNw/pyr5iF1g0BfjmUIDgCBpof3ZEIxFMkzDhmscIOOtTRKPV8H2MP907boXn1oNeJhb7WeoEaW2iwYr8uEef82f8NFKvDIVnr2nhcfCSoo/7bF2hvLEXw432ouQ==",
                p2pCertificatesResponse.x5c_ca
            );
            Assert.AreEqual(
                "urn:p2p_cert",
                p2pCertificatesResponse.resource
            );
        }

        [TestMethod]
        public void TestJoinDeviceResponseParsing()
        {
            var rawJoinDeviceResponse = "{\"Certificate\":{\"Thumbprint\":\"63663A9293FD6CB5F108B1435C78136C36B0407B\",\"RawBody\":\"MIID8jCCAtqgAwIBAgIQFaOoakv7w5ZOUX3Jgk2P8TANBgkqhkiG9w0BAQsFADB4MXYwEQYKCZImiZPyLGQBGRYDbmV0MBUGCgmSJomT8ixkARkWB3dpbmRvd3MwHQYDVQQDExZNUy1Pcmdhbml6YXRpb24tQWNjZXNzMCsGA1UECxMkODJkYmFjYTQtM2U4MS00NmNhLTljNzMtMDk1MGMxZWFjYTk3MB4XDTIyMDgxMjIxMDQ1M1oXDTMyMDgxMjIxMzQ1M1owLzEtMCsGA1UEAxMkYTgwZWM0YjctODA4Ny00N2VmLWJjODYtZmNmY2FiOWJmNWQ5MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAwwiKdRD944dG340IpqnREL+DXVbLeiYSATsOFEegVAOB6UXVDTZ3QEEM6So6BqT0wCpDav7RxCmFUhkB6MH7PVv3FwqnYwxXSLBfHH9fVZXUISqqu3s2Vi5PbG0whw+OAchFATb1+/ttwHP+nnhhLOo9SJmNoznJ/ho3MJVLzn3/NJdZL2UIEVNPL+lPI01Wja6miDTbdHtmuByqqOYJJlxh2PYfW7LKSoV/5ZX7OYczowrsfxjuJBBj6NKctUB92LaTxtcwRoGKVZ8ARBO2mBmMHacRbqm2tVAVp4sK9c2nmKnSsVyVPkKdY0xNDIBy+JutPs/yA+RBWZrb6DK7/QIDAQABo4HAMIG9MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwIwIgYLKoZIhvcUAQWCHAIEEwSBELfEDqiHgO9HvIb8/Kub9dkwIgYLKoZIhvcUAQWCHAMEEwSBEIeOi+pIcypDrlVLIkYadL0wIgYLKoZIhvcUAQWCHAUEEwSBEEgkJam3mrBJu1zy+SPIRnIwFAYLKoZIhvcUAQWCHAgEBQSBAkVVMBMGCyqGSIb3FAEFghwHBAQEgQExMA0GCSqGSIb3DQEBCwUAA4IBAQCBiJ4r1FKrdn/iqfDsYLdr7TbIulLccwXZJz3pNlUN/9EeTY6sVrW7H2iLCLk9bVk2S9b8WGuC+gO+E6CRpOPwWHHXAhrGCoEdlRXaWSY0kDArhvdMR9xGJOTPO0m6q8nn1Z3G8EtwD8qIbgwPtE08zJ5ls6UKmDLk9jiKXH6vCgwtgI9WCxNTJYG9npHkK0C6chEa1QZlo3aRKhRPqAJ8B1kh1XRZigXKT8Rqt2YjD+SWjWOYWupAAylysQaJwW+gAIXRQJW3OVpeaP4ajQqpYV11VlVR8t/mIWL1PE+ONJqZc/YFBrT6E9wPYBJzjCMurTj8kHr8yXfZzjNQaGN8\"},\"User\":{\"Upn\":\"s4@dataans.com\"},\"MembershipChanges\":[{\"LocalSID\":\"S-1-5-32-544\",\"AddSIDs\":[\"S-1-12-1-161258645-1267993664-2068636298-1054913958\",\"S-1-12-1-1201544774-1151677024-232476811-374148655\",\"S-1-12-1-3935014535-1126855496-575362478-3178502726\"]}]}";

            var joinResponse = JoinDeviceResponse.FromString(rawJoinDeviceResponse);

            Assert.AreEqual("63663A9293FD6CB5F108B1435C78136C36B0407B", joinResponse.Certificate.Thumbprint);
            Assert.AreEqual(
                "MIID8jCCAtqgAwIBAgIQFaOoakv7w5ZOUX3Jgk2P8TANBgkqhkiG9w0BAQsFADB4MXYwEQYKCZImiZPyLGQBGRYDbmV0MBUGCgmSJomT8ixkARkWB3dpbmRvd3MwHQYDVQQDExZNUy1Pcmdhbml6YXRpb24tQWNjZXNzMCsGA1UECxMkODJkYmFjYTQtM2U4MS00NmNhLTljNzMtMDk1MGMxZWFjYTk3MB4XDTIyMDgxMjIxMDQ1M1oXDTMyMDgxMjIxMzQ1M1owLzEtMCsGA1UEAxMkYTgwZWM0YjctODA4Ny00N2VmLWJjODYtZmNmY2FiOWJmNWQ5MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAwwiKdRD944dG340IpqnREL+DXVbLeiYSATsOFEegVAOB6UXVDTZ3QEEM6So6BqT0wCpDav7RxCmFUhkB6MH7PVv3FwqnYwxXSLBfHH9fVZXUISqqu3s2Vi5PbG0whw+OAchFATb1+/ttwHP+nnhhLOo9SJmNoznJ/ho3MJVLzn3/NJdZL2UIEVNPL+lPI01Wja6miDTbdHtmuByqqOYJJlxh2PYfW7LKSoV/5ZX7OYczowrsfxjuJBBj6NKctUB92LaTxtcwRoGKVZ8ARBO2mBmMHacRbqm2tVAVp4sK9c2nmKnSsVyVPkKdY0xNDIBy+JutPs/yA+RBWZrb6DK7/QIDAQABo4HAMIG9MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwIwIgYLKoZIhvcUAQWCHAIEEwSBELfEDqiHgO9HvIb8/Kub9dkwIgYLKoZIhvcUAQWCHAMEEwSBEIeOi+pIcypDrlVLIkYadL0wIgYLKoZIhvcUAQWCHAUEEwSBEEgkJam3mrBJu1zy+SPIRnIwFAYLKoZIhvcUAQWCHAgEBQSBAkVVMBMGCyqGSIb3FAEFghwHBAQEgQExMA0GCSqGSIb3DQEBCwUAA4IBAQCBiJ4r1FKrdn/iqfDsYLdr7TbIulLccwXZJz3pNlUN/9EeTY6sVrW7H2iLCLk9bVk2S9b8WGuC+gO+E6CRpOPwWHHXAhrGCoEdlRXaWSY0kDArhvdMR9xGJOTPO0m6q8nn1Z3G8EtwD8qIbgwPtE08zJ5ls6UKmDLk9jiKXH6vCgwtgI9WCxNTJYG9npHkK0C6chEa1QZlo3aRKhRPqAJ8B1kh1XRZigXKT8Rqt2YjD+SWjWOYWupAAylysQaJwW+gAIXRQJW3OVpeaP4ajQqpYV11VlVR8t/mIWL1PE+ONJqZc/YFBrT6E9wPYBJzjCMurTj8kHr8yXfZzjNQaGN8",
                joinResponse.Certificate.RawBody
            );
            Assert.AreEqual("s4@dataans.com", joinResponse.User.Upn);
            Assert.AreEqual("S-1-5-32-544", joinResponse.MembershipChanges[0].LocalSID);
            Assert.AreEqual("S-1-12-1-161258645-1267993664-2068636298-1054913958", joinResponse.MembershipChanges[0].AddSIDs[0]);
            Assert.AreEqual("S-1-12-1-1201544774-1151677024-232476811-374148655", joinResponse.MembershipChanges[0].AddSIDs[1]);
            Assert.AreEqual("S-1-12-1-3935014535-1126855496-575362478-3178502726", joinResponse.MembershipChanges[0].AddSIDs[2]);
        }

        [TestMethod]
        public void TestJwtSignature()
        {
            byte[] key = {37, 150, 246, 91, 244, 248, 131, 90, 221, 132, 103, 95, 89, 45, 99, 167, 229, 126, 64, 98, 54, 106, 164, 199, 58, 196, 255, 251, 3, 198, 189, 231, 255, 37, 235, 58, 145, 190, 222, 58, 16, 117, 188, 67, 38, 97, 255, 119, 125, 2, 200, 124, 15, 119, 242, 48, 145, 86, 2, 101, 131, 60, 14, 60, 21, 112, 93, 234, 148, 196, 107, 51, 36, 135, 255, 65, 128, 43, 11, 99, 66, 248, 147, 27, 140, 222, 27, 43, 24, 50, 204, 253, 161, 77, 211, 105, 220, 142, 171, 131, 101, 251, 206, 53, 197, 163, 84, 80, 180, 87, 64, 235, 155, 72, 52, 247, 151, 25, 14, 36, 140, 203, 186, 104, 21, 74, 40, 178, 176, 36, 215, 244, 106, 196, 120, 225, 107, 197, 34, 170, 202, 227, 6, 163, 27, 140, 124, 76, 246, 174, 179, 100, 164, 197, 64, 254, 63, 157, 220, 17, 112, 188, 145, 55, 131, 106, 147, 43, 100, 170, 192, 36, 142, 40, 196, 141, 144, 23, 183, 29, 22, 130, 239, 56, 53, 34, 210, 186, 61, 142, 156, 72, 144, 67, 228, 247, 38, 78, 127, 51, 225, 11, 79, 62, 116, 43, 129, 69, 30, 176, 138, 189, 127, 135, 109, 105, 198, 200, 92, 254, 242, 33, 166, 251, 38, 81, 100, 252, 220, 49, 29, 13, 208, 125, 229, 143, 165, 127, 141, 89, 76, 215, 58, 114, 101, 79, 81, 107, 174, 8, 103, 238, 228, 207, 208, 142};
            byte[] data = Encoding.ASCII.GetBytes("eyJhbGciOiJIUzI1NiIsICJrZGZfdmVyIjoyLCAiY3R4IjoiYVlKdEQ1NHZPN0h0R0dQMkp1Q0ZScnBQTk1kYUQ4SnYifQ.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");

            HMACSHA256 hmac = new HMACSHA256(key);
            byte[] hash = hmac.ComputeHash(data);

            foreach (var b in hash)
            {
                Console.Write("{0} ", b);
            }
            Console.WriteLine();

            var base64Hash = Convert.ToBase64String(hash);
            Console.WriteLine(base64Hash);
            Console.WriteLine("AtxpdY3mOqjsWGTCsDNBKbKDTjQk-jAP7HHZZerX57E");

            Assert.AreEqual("AtxpdY3mOqjsWGTCsDNBKbKDTjQk-jAP7HHZZerX57E", base64Hash);
        }

    }
}
