//! Macros for defining field element types.

mod fiat;

/// Creates a ZST representing the Montgomery parameters for a given field modulus.
///
/// Accepts the following parameters:
///
/// - name of the ZST representing the field modulus
/// - hex serialization of the modulus
/// - `crypto-bigint` unsigned integer type (e.g. U256)
/// - number of bytes in an encoded field element
/// - byte order to use when encoding/decoding field elements
/// - documentation string for the field modulus type
///
/// ```
/// use primefield::{ByteOrder, bigint::U256, consts::U32};
///
/// primefield::monty_field_params!(
///     name: FieldParams,
///     modulus: "ffffffff00000001000000000000000000000000ffffffffffffffffffffffff",
///     uint: U256,
///     byte_order: ByteOrder::BigEndian,
///     multiplicative_generator: 6,
///     fe_name: "FieldElement",
///     doc: "P-256 field modulus"
/// );
/// ```
#[macro_export]
macro_rules! monty_field_params {
    (
        name: $name:ident,
        modulus: $modulus_hex:expr,
        uint: $uint_type:ty,
        byte_order: $byte_order:expr,
        multiplicative_generator: $multiplicative_generator:expr,
        fe_name: $fe_name:expr,
        doc: $doc:expr
    ) => {
        use $crate::bigint::modular::ConstMontyParams;

        $crate::bigint::const_monty_params!($name, $uint_type, $modulus_hex, $doc);

        impl $crate::MontyFieldParams<{ <$uint_type>::LIMBS }> for $name {
            type ByteSize = $crate::bigint::hybrid_array::typenum::U<
                { $name::PARAMS.modulus().as_ref().bits().div_ceil(8) as usize },
            >;
            const BYTE_ORDER: $crate::ByteOrder = $byte_order;
            const FIELD_ELEMENT_NAME: &'static str = $fe_name;
            const MODULUS_HEX: &'static str = $modulus_hex;
            const MULTIPLICATIVE_GENERATOR: u64 = $multiplicative_generator;
            #[cfg(target_pointer_width = "32")]
            const T: &'static [u64] = &$crate::compute_t::<
                { <$uint_type>::LIMBS.div_ceil(2) },
                { <$uint_type>::LIMBS },
            >($name::PARAMS.modulus().as_ref());
            #[cfg(target_pointer_width = "64")]
            const T: &'static [u64] = &$crate::compute_t::<
                { <$uint_type>::LIMBS },
                { <$uint_type>::LIMBS },
            >($name::PARAMS.modulus().as_ref());
        }
    };
}

/// Implements a field element type whose internal representation is in
/// Montgomery form, providing a combination of trait impls and inherent impls
/// which are `const fn` where possible.
///
/// Accepts a set of `const fn` arithmetic operation functions as arguments.
///
/// # Inherent impls
/// - `const ZERO: Self`
/// - `const ONE: Self` (multiplicative identity)
/// - `pub fn from_bytes`
/// - `pub fn from_slice`
/// - `pub fn from_uint`
/// - `fn from_uint_unchecked`
/// - `pub fn to_bytes`
/// - `pub fn to_canonical`
/// - `pub fn is_odd`
/// - `pub fn is_zero`
/// - `pub fn double`
///
/// NOTE: field implementations must provide their own inherent impls of
/// the following methods in order for the code generated by this macro to
/// compile:
///
/// - `pub fn invert`
/// - `pub fn sqrt`
///
/// # Trait impls
/// - `ConditionallySelectable`
/// - `ConstantTimeEq`
/// - `ConstantTimeGreater`
/// - `ConstantTimeLess`
/// - `Default`
/// - `DefaultIsZeroes`
/// - `Eq`
/// - `Field`
/// - `PartialEq`
///
/// ## Ops
/// - `Add`
/// - `AddAssign`
/// - `Sub`
/// - `SubAssign`
/// - `Mul`
/// - `MulAssign`
/// - `Neg`
/// - `Shr`
/// - `ShrAssign`
/// - `Invert`
#[macro_export]
macro_rules! monty_field_element {
    ($fe:tt, $params:ty, $uint:tt) => {
        impl $fe {
            /// Zero element.
            pub const ZERO: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::ZERO);

            /// Multiplicative identity.
            pub const ONE: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::ONE);

            /// Montgomery parameters constant.
            pub const PARAMS: $crate::bigint::modular::MontyParams<{ <$uint>::LIMBS }> =
                <$params>::PARAMS;

            /// Create a [`
            #[doc = stringify!($fe)]
            /// `] from a canonical big-endian representation.
            pub fn from_bytes(
                repr: &$crate::MontyFieldBytes<$params, { <$params>::LIMBS }>,
            ) -> $crate::subtle::CtOption<Self> {
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_bytes(repr).map(Self)
            }

            /// Decode [`
            #[doc = stringify!($fe)]
            /// `] from a big endian byte slice.
            pub fn from_slice(slice: &[u8]) -> Option<Self> {
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_slice(slice)
                    .map(Self)
            }

            /// Decode a [`
            #[doc = stringify!($fe)]
            /// `] from big endian hex-encoded bytes.
            ///
            /// This is primarily intended for defining constants using hex literals.
            ///
            /// # Panics
            ///
            /// - When hex is malformed
            /// - When input is the wrong length
            /// - If input overflows the modulus
            pub const fn from_hex_vartime(hex: &str) -> Self {
                Self(
                    $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_hex_vartime(hex),
                )
            }

            /// Decode [`
            #[doc = stringify!($fe)]
            /// `]
            /// from [`
            #[doc = stringify!($uint)]
            /// `] converting it into Montgomery form:
            ///
            /// ```text
            /// w * R^2 * R^-1 mod p = wR mod p
            /// ```
            pub fn from_uint(uint: &$uint) -> $crate::subtle::CtOption<Self> {
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_uint(uint).map(Self)
            }

            /// Convert a `u64` into a [`
            #[doc = stringify!($fe)]
            /// `].
            pub const fn from_u64(w: u64) -> Self {
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_u64(w))
            }

            /// Returns the big-endian encoding of this [`
            #[doc = stringify!($fe)]
            /// `].
            pub fn to_bytes(self) -> $crate::MontyFieldBytes<$params, { <$params>::LIMBS }> {
                self.0.to_bytes()
            }

            /// Determine if this [`
            #[doc = stringify!($fe)]
            /// `] is odd in the SEC1 sense: `self mod 2 == 1`.
            ///
            /// # Returns
            ///
            /// If odd, return `Choice(1)`.  Otherwise, return `Choice(0)`.
            pub fn is_odd(&self) -> $crate::subtle::Choice {
                self.0.is_odd()
            }

            /// Determine if this [`
            #[doc = stringify!($fe)]
            /// `] is even in the SEC1 sense: `self mod 2 == 0`.
            ///
            /// # Returns
            ///
            /// If even, return `Choice(1)`.  Otherwise, return `Choice(0)`.
            pub fn is_even(&self) -> $crate::subtle::Choice {
                !self.is_odd()
            }

            /// Determine if this [`
            #[doc = stringify!($fe)]
            /// `] is zero.
            ///
            /// # Returns
            ///
            /// If zero, return `Choice(1)`.  Otherwise, return `Choice(0)`.
            pub fn is_zero(&self) -> $crate::subtle::Choice {
                self.0.is_zero()
            }

            /// Returns `self^exp`, where `exp` is a little-endian integer exponent.
            ///
            /// **This operation is variable time with respect to the exponent.**
            ///
            /// If the exponent is fixed, this operation is constant time.
            pub const fn pow_vartime(&self, exp: &[u64]) -> Self {
                Self(self.0.pow_vartime(exp))
            }
        }

        impl $crate::ff::Field for $fe {
            const ZERO: Self = Self::ZERO;
            const ONE: Self = Self::ONE;

            fn try_from_rng<R: $crate::rand_core::TryRngCore + ?Sized>(
                rng: &mut R,
            ) -> ::core::result::Result<Self, R::Error> {
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::try_from_rng(rng)
                    .map(Self)
            }

            fn is_zero(&self) -> Choice {
                self.0.is_zero()
            }

            fn square(&self) -> Self {
                self.square()
            }

            fn double(&self) -> Self {
                self.double()
            }

            fn invert(&self) -> CtOption<Self> {
                self.invert()
            }

            fn sqrt(&self) -> CtOption<Self> {
                self.sqrt()
            }

            fn sqrt_ratio(num: &Self, div: &Self) -> (Choice, Self) {
                $crate::ff::helpers::sqrt_ratio_generic(num, div)
            }
        }

        impl PrimeField for $fe {
            type Repr = $crate::MontyFieldBytes<$params, { <$params>::LIMBS }>;

            const MODULUS: &'static str =
                <$params as $crate::MontyFieldParams<{ <$uint>::LIMBS }>>::MODULUS_HEX;
            const NUM_BITS: u32 =
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::NUM_BITS;
            const CAPACITY: u32 =
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::CAPACITY;
            const TWO_INV: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::TWO_INV);
            const MULTIPLICATIVE_GENERATOR: Self = Self(
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::MULTIPLICATIVE_GENERATOR,
            );
            const S: u32 = $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::S;
            const ROOT_OF_UNITY: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::ROOT_OF_UNITY);
            const ROOT_OF_UNITY_INV: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::ROOT_OF_UNITY_INV);
            const DELTA: Self =
                Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::DELTA);

            #[inline]
            fn from_repr(bytes: Self::Repr) -> $crate::subtle::CtOption<Self> {
                $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::from_repr(bytes).map(Self)
            }

            #[inline]
            fn to_repr(&self) -> Self::Repr {
                self.0.to_repr()
            }

            #[inline]
            fn is_odd(&self) -> $crate::subtle::Choice {
                self.0.is_odd()
            }
        }

        #[cfg(feature = "bits")]
        impl $crate::ff::PrimeFieldBits for $fe {
            type ReprBits = [$crate::bigint::Word; <$uint>::LIMBS];

            fn to_le_bits(&self) -> $crate::ff::FieldBits<Self::ReprBits> {
                self.to_canonical().to_words().into()
            }

            fn char_le_bits() -> $crate::ff::FieldBits<Self::ReprBits> {
                Self::PARAMS.modulus().to_words().into()
            }
        }

        // TODO(tarcieri): write `Reduce` impls
        // impl $crate::bigint::Reduce<$uint> for $fe {
        //     fn reduce(w: &$uint) -> Self {
        //         Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::reduce(w))
        //     }
        // }
        //
        // impl $crate::bigint::Reduce<$crate::MontyFieldBytes<$params, { <$params>::LIMBS }>> for $fe {
        //     #[inline]
        //     fn reduce(bytes: &$crate::MontyFieldBytes<$params, { <$params>::LIMBS }>) -> Self {
        //         Self($crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::reduce(bytes))
        //     }
        // }

        $crate::field_op!($fe, Add, add, add);
        $crate::field_op!($fe, Sub, sub, sub);
        $crate::field_op!($fe, Mul, mul, multiply);

        impl ::core::ops::AddAssign<$fe> for $fe {
            #[inline]
            fn add_assign(&mut self, other: $fe) {
                *self = *self + other;
            }
        }

        impl ::core::ops::AddAssign<&$fe> for $fe {
            #[inline]
            fn add_assign(&mut self, other: &$fe) {
                *self = *self + other;
            }
        }

        impl ::core::ops::SubAssign<$fe> for $fe {
            #[inline]
            fn sub_assign(&mut self, other: $fe) {
                *self = *self - other;
            }
        }

        impl ::core::ops::SubAssign<&$fe> for $fe {
            #[inline]
            fn sub_assign(&mut self, other: &$fe) {
                *self = *self - other;
            }
        }

        impl ::core::ops::MulAssign<&$fe> for $fe {
            #[inline]
            fn mul_assign(&mut self, other: &$fe) {
                *self = *self * other;
            }
        }

        impl ::core::ops::MulAssign for $fe {
            #[inline]
            fn mul_assign(&mut self, other: $fe) {
                *self = *self * other;
            }
        }

        impl ::core::ops::Neg for $fe {
            type Output = $fe;

            #[inline]
            fn neg(self) -> $fe {
                <$fe>::neg(&self)
            }
        }

        impl ::core::ops::Neg for &$fe {
            type Output = $fe;

            #[inline]
            fn neg(self) -> $fe {
                <$fe>::neg(self)
            }
        }

        impl ::core::fmt::Debug for $fe {
            fn fmt(&self, f: &mut ::core::fmt::Formatter<'_>) -> ::core::fmt::Result {
                write!(f, "{}(0x{:X})", stringify!($fe), &self.0)
            }
        }

        impl Default for $fe {
            fn default() -> Self {
                Self::ZERO
            }
        }

        impl Eq for $fe {}
        impl PartialEq for $fe {
            fn eq(&self, rhs: &Self) -> bool {
                self.0.ct_eq(&(rhs.0)).into()
            }
        }

        impl From<u32> for $fe {
            fn from(n: u32) -> $fe {
                Self::from_uint_unchecked(<$uint>::from(n))
            }
        }

        impl From<u64> for $fe {
            fn from(n: u64) -> $fe {
                Self::from_uint_unchecked(<$uint>::from(n))
            }
        }

        impl From<u128> for $fe {
            fn from(n: u128) -> $fe {
                Self::from_uint_unchecked(<$uint>::from(n))
            }
        }

        impl From<$fe> for $crate::MontyFieldBytes<$params, { <$params>::LIMBS }> {
            fn from(fe: $fe) -> Self {
                $crate::MontyFieldBytes::<$params, { <$params>::LIMBS }>::from(&fe)
            }
        }

        impl From<&$fe> for $crate::MontyFieldBytes<$params, { <$params>::LIMBS }> {
            fn from(fe: &$fe) -> Self {
                fe.to_repr()
            }
        }

        impl From<$crate::MontyFieldElement::<$params, { <$params>::LIMBS }>> for $fe {
            fn from(fe: $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>) -> $fe {
                $fe(fe)
            }
        }

        impl From<$fe> for $crate::MontyFieldElement<$params, { <$params>::LIMBS }> {
            fn from(fe: $fe) -> $crate::MontyFieldElement<$params, { <$params>::LIMBS }> {
                fe.0
            }
        }

        impl From<$fe> for $uint {
            fn from(fe: $fe) -> $uint {
                <$uint>::from(&fe)
            }
        }

        impl From<&$fe> for $uint {
            fn from(fe: &$fe) -> $uint {
                fe.to_canonical()
            }
        }

        impl ::core::iter::Sum for $fe {
            #[allow(unused_qualifications)]
            fn sum<I: Iterator<Item = Self>>(iter: I) -> Self {
                iter.reduce(core::ops::Add::add).unwrap_or(Self::ZERO)
            }
        }

        impl<'a> ::core::iter::Sum<&'a $fe> for $fe {
            fn sum<I: Iterator<Item = &'a $fe>>(iter: I) -> Self {
                iter.copied().sum()
            }
        }

        impl ::core::iter::Product for $fe {
            #[allow(unused_qualifications)]
            fn product<I: Iterator<Item = Self>>(iter: I) -> Self {
                iter.reduce(core::ops::Mul::mul).unwrap_or(Self::ONE)
            }
        }

        impl<'a> ::core::iter::Product<&'a $fe> for $fe {
            fn product<I: Iterator<Item = &'a Self>>(iter: I) -> Self {
                iter.copied().product()
            }
        }

        impl $crate::bigint::Invert for $fe {
            type Output = CtOption<Self>;

            fn invert(&self) -> CtOption<Self> {
                self.invert()
            }
        }

        impl $crate::subtle::ConditionallySelectable for $fe {
            fn conditional_select(a: &Self, b: &Self, choice: Choice) -> Self {
                Self(
                    $crate::MontyFieldElement::<$params, { <$params>::LIMBS }>::conditional_select(
                        &a.0, &b.0, choice,
                    ),
                )
            }
        }

        impl $crate::subtle::ConstantTimeEq for $fe {
            fn ct_eq(&self, other: &Self) -> $crate::subtle::Choice {
                self.0.ct_eq(&other.0)
            }
        }

        impl $crate::subtle::ConstantTimeGreater for $fe {
            fn ct_gt(&self, other: &Self) -> $crate::subtle::Choice {
                self.0.ct_gt(&other.0)
            }
        }

        impl $crate::subtle::ConstantTimeLess for $fe {
            fn ct_lt(&self, other: &Self) -> $crate::subtle::Choice {
                self.0.ct_lt(&other.0)
            }
        }

        impl $crate::zeroize::DefaultIsZeroes for $fe {}
    };
}

/// Emit a `core::ops` trait wrapper for an inherent method which is expected to be provided by a
/// backend arithmetic implementation (e.g. `fiat-crypto`)
#[macro_export]
macro_rules! field_op {
    ($fe:tt, $op:tt, $func:ident, $inner_func:ident) => {
        impl ::core::ops::$op for $fe {
            type Output = $fe;

            #[inline]
            fn $func(self, rhs: $fe) -> $fe {
                <$fe>::$inner_func(&self, &rhs)
            }
        }

        impl ::core::ops::$op<&$fe> for $fe {
            type Output = $fe;

            #[inline]
            fn $func(self, rhs: &$fe) -> $fe {
                <$fe>::$inner_func(&self, rhs)
            }
        }

        impl ::core::ops::$op<&$fe> for &$fe {
            type Output = $fe;

            #[inline]
            fn $func(self, rhs: &$fe) -> $fe {
                <$fe>::$inner_func(self, rhs)
            }
        }
    };
}
