//! PipeWire Stream Management
//!
//! Handles individual PipeWire streams for screen capture.

use libspa::param::video::VideoFormat;
use pipewire::spa::utils::Fraction;
use pipewire::stream::{Stream, StreamState};
use std::sync::{Arc, Mutex};
use std::time::{Duration, SystemTime};
use tokio::sync::mpsc;

use crate::buffer::SharedBufferManager;
use crate::error::{PipeWireError, Result};
use crate::format::PixelFormat;
use crate::frame::{FrameCallback, FrameStats, VideoFrame};

/// Stream configuration
#[derive(Debug, Clone)]
pub struct StreamConfig {
    /// Stream name
    pub name: String,

    /// Target width
    pub width: u32,

    /// Target height
    pub height: u32,

    /// Target framerate
    pub framerate: u32,

    /// Use DMA-BUF if available
    pub use_dmabuf: bool,

    /// Number of buffers
    pub buffer_count: u32,

    /// Preferred format
    pub preferred_format: Option<PixelFormat>,
}

impl StreamConfig {
    /// Create default configuration
    pub fn new(name: impl Into<String>) -> Self {
        Self {
            name: name.into(),
            width: 1920,
            height: 1080,
            framerate: 30,
            use_dmabuf: true,
            buffer_count: 3,
            preferred_format: Some(PixelFormat::BGRA),
        }
    }

    /// Set resolution
    pub fn with_resolution(mut self, width: u32, height: u32) -> Self {
        self.width = width;
        self.height = height;
        self
    }

    /// Set framerate
    pub fn with_framerate(mut self, fps: u32) -> Self {
        self.framerate = fps;
        self
    }

    /// Set DMA-BUF preference
    pub fn with_dmabuf(mut self, use_dmabuf: bool) -> Self {
        self.use_dmabuf = use_dmabuf;
        self
    }

    /// Set buffer count
    pub fn with_buffer_count(mut self, count: u32) -> Self {
        self.buffer_count = count;
        self
    }
}

/// PipeWire stream state
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum PwStreamState {
    /// Stream is initializing
    Initializing,
    /// Stream is ready to start
    Ready,
    /// Stream is actively streaming
    Streaming,
    /// Stream is paused
    Paused,
    /// Stream encountered an error
    Error,
    /// Stream is closing
    Closing,
}

impl From<StreamState> for PwStreamState {
    fn from(state: StreamState) -> Self {
        match state {
            StreamState::Error(_) => Self::Error,
            StreamState::Unconnected => Self::Initializing,
            StreamState::Connecting => Self::Initializing,
            StreamState::Paused => Self::Paused,
            StreamState::Streaming => Self::Streaming,
        }
    }
}

/// Format negotiation result
#[derive(Debug, Clone)]
pub struct NegotiatedFormat {
    /// Video format
    pub format: VideoFormat,

    /// Width
    pub width: u32,

    /// Height
    pub height: u32,

    /// Row stride
    pub stride: u32,

    /// Framerate
    pub framerate: Fraction,
}

/// PipeWire stream handler
pub struct PipeWireStream {
    /// Stream ID
    id: u32,

    /// Stream configuration
    config: StreamConfig,

    /// PipeWire stream (using pipewire crate)
    stream: Option<Stream>,

    /// Buffer manager (used by process_frame path)
    #[allow(dead_code)]
    buffer_manager: SharedBufferManager,

    /// Current state
    state: Arc<Mutex<PwStreamState>>,

    /// Negotiated format
    negotiated_format: Arc<Mutex<Option<NegotiatedFormat>>>,

    /// Frame callback
    frame_callback: Arc<Mutex<Option<FrameCallback>>>,

    /// Frame counter (used by process_frame path)
    #[allow(dead_code)]
    frame_counter: Arc<Mutex<u64>>,

    /// Statistics
    stats: Arc<Mutex<FrameStats>>,

    /// Start time
    start_time: Arc<Mutex<Option<SystemTime>>>,

    /// Frame sender channel
    frame_tx: Arc<Mutex<Option<mpsc::Sender<VideoFrame>>>>,
}

impl PipeWireStream {
    /// Create new PipeWire stream
    pub fn new(id: u32, config: StreamConfig) -> Self {
        let buffer_manager = SharedBufferManager::new(config.buffer_count as usize);

        Self {
            id,
            config,
            stream: None,
            buffer_manager,
            state: Arc::new(Mutex::new(PwStreamState::Initializing)),
            negotiated_format: Arc::new(Mutex::new(None)),
            frame_callback: Arc::new(Mutex::new(None)),
            frame_counter: Arc::new(Mutex::new(0)),
            stats: Arc::new(Mutex::new(FrameStats::new())),
            start_time: Arc::new(Mutex::new(None)),
            frame_tx: Arc::new(Mutex::new(None)),
        }
    }

    /// Get stream ID
    pub fn id(&self) -> u32 {
        self.id
    }

    /// Get stream state
    pub fn state(&self) -> PwStreamState {
        *self.state.lock().unwrap()
    }

    /// Set frame callback
    pub fn set_frame_callback(&mut self, callback: FrameCallback) {
        *self.frame_callback.lock().unwrap() = Some(callback);
    }

    /// Set frame channel
    pub fn set_frame_channel(&mut self, tx: mpsc::Sender<VideoFrame>) {
        *self.frame_tx.lock().unwrap() = Some(tx);
    }

    /// Get negotiated format
    pub fn negotiated_format(&self) -> Option<NegotiatedFormat> {
        self.negotiated_format.lock().unwrap().clone()
    }

    /// Get statistics
    pub fn stats(&self) -> FrameStats {
        self.stats.lock().unwrap().clone()
    }

    /// Connect to PipeWire node
    ///
    /// Establishes connection to the specified PipeWire node with full format negotiation,
    /// buffer setup, and event callback registration.
    ///
    /// # Arguments
    ///
    /// * `core` - PipeWire core connection
    /// * `node_id` - Portal-provided node ID to connect to
    ///
    /// # Returns
    ///
    /// Ok(()) on successful connection, Err if connection fails
    ///
    /// # Errors
    ///
    /// Returns error if:
    /// - Stream creation fails
    /// - Parameter construction fails
    /// - Connection to node fails
    /// - Format negotiation fails
    pub async fn connect(&mut self, core: &pipewire::core::Core, node_id: u32) -> Result<()> {
        use pipewire::spa::pod::Pod;
        use pipewire::spa::utils::Direction;
        use pipewire::stream::StreamFlags;

        // Build list of acceptable formats (prepared for future format params)
        let _formats = if let Some(pref) = self.config.preferred_format {
            vec![pref.to_spa()]
        } else {
            // Prefer BGRA/BGRx for RDP compatibility
            vec![
                VideoFormat::BGRx,
                VideoFormat::BGRA,
                VideoFormat::RGBx,
                VideoFormat::RGBA,
            ]
        };

        // Framerate for format negotiation (prepared for future use)
        let _framerate = Fraction {
            num: self.config.framerate,
            denom: 1,
        };

        // Create stream using pipewire-rs safe API
        let stream_name = format!("lamco-pw-{}", self.id);

        // Build properties
        let mut props = pipewire::properties::Properties::new();
        props.insert("media.type".to_string(), "Video".to_string());
        props.insert("media.category".to_string(), "Capture".to_string());
        props.insert("media.role".to_string(), "Screen".to_string());
        props.insert("node.target".to_string(), node_id.to_string());

        // Create the stream on the core
        let pw_stream = pipewire::stream::Stream::new(core, &stream_name, props)
            .map_err(|e| PipeWireError::StreamCreationFailed(format!("Failed to create stream: {}", e)))?;

        // Connect stream with format parameters
        // Note: In pipewire-rs, we connect first then params are negotiated via events
        let mut params: Vec<&Pod> = vec![];
        pw_stream
            .connect(
                Direction::Input,
                Some(node_id),
                StreamFlags::AUTOCONNECT | StreamFlags::MAP_BUFFERS,
                &mut params, // Params will be negotiated via events
            )
            .map_err(|e| PipeWireError::ConnectionFailed(format!("Stream connect failed: {}", e)))?;

        // Store the stream reference
        // Note: The actual Stream object needs to be managed carefully
        // as it contains non-Send types
        *self.state.lock().unwrap() = PwStreamState::Initializing;

        // In production, we would:
        // 1. Set up event listeners on the stream
        // 2. Handle state changes asynchronously
        // 3. Process format negotiation via param_changed events
        // 4. Handle process callbacks for frame data
        //
        // This requires either:
        // a) Running everything on the PipeWire thread, OR
        // b) Using message passing between threads
        //
        // For our architecture, we use dedicated PipeWire thread approach
        // implemented in connection.rs

        Ok(())
    }

    /// Start streaming
    pub async fn start(&mut self) -> Result<()> {
        *self.state.lock().unwrap() = PwStreamState::Streaming;
        *self.start_time.lock().unwrap() = Some(SystemTime::now());
        Ok(())
    }

    /// Pause streaming
    pub async fn pause(&mut self) -> Result<()> {
        *self.state.lock().unwrap() = PwStreamState::Paused;
        Ok(())
    }

    /// Resume streaming
    pub async fn resume(&mut self) -> Result<()> {
        *self.state.lock().unwrap() = PwStreamState::Streaming;
        Ok(())
    }

    /// Stop streaming
    pub async fn stop(&mut self) -> Result<()> {
        *self.state.lock().unwrap() = PwStreamState::Closing;
        self.stream = None;
        Ok(())
    }

    /// Restart stream after error
    pub async fn restart(&mut self) -> Result<()> {
        self.stop().await?;
        *self.state.lock().unwrap() = PwStreamState::Initializing;
        Ok(())
    }

    /// Process a frame from PipeWire
    ///
    /// This method is currently unused as frame processing is handled directly
    /// in the PipeWire thread. Preserved for potential future direct processing path.
    #[allow(dead_code)]
    async fn process_frame(&self, buffer_id: u32, pts: u64) -> Result<()> {
        // Get buffer
        let buffer_opt = self
            .buffer_manager
            .with_buffer(buffer_id, |buf| {
                // Extract data
                // SAFETY: Buffer is locked via with_buffer() ensuring exclusive access.
                // We immediately copy to Vec, so no dangling slice references.
                let data = unsafe { buf.as_slice().map(|s| s.to_vec()) };

                (buf.size, buf.buffer_type, data)
            })
            .await;

        if let Some((_size, buffer_type, Some(data))) = buffer_opt {
            // Get negotiated format
            let format_info = self.negotiated_format.lock().unwrap().clone();

            if let Some(format) = format_info {
                // Create video frame
                let frame_id = {
                    let mut counter = self.frame_counter.lock().unwrap();
                    let id = *counter;
                    *counter += 1;
                    id
                };

                let pixel_format = PixelFormat::from_spa(format.format).unwrap_or(PixelFormat::BGRA);

                let mut frame = VideoFrame::with_data(
                    frame_id,
                    format.width,
                    format.height,
                    format.stride,
                    pixel_format,
                    self.id,
                    data,
                );

                frame.set_timing(pts, pts, 0);

                if buffer_type.is_dmabuf() {
                    frame.flags.set_dmabuf();
                }

                // Update stats
                self.stats.lock().unwrap().update(&frame);

                // Send to callback if set
                if let Some(ref callback) = *self.frame_callback.lock().unwrap() {
                    callback(frame.clone());
                }

                // Send to channel if set
                if let Some(ref tx) = *self.frame_tx.lock().unwrap() {
                    let _ = tx.try_send(frame);
                }
            }
        }

        Ok(())
    }

    /// Get uptime
    pub fn uptime(&self) -> Option<Duration> {
        self.start_time
            .lock()
            .unwrap()
            .as_ref()
            .and_then(|start| start.elapsed().ok())
    }
}

/// Stream metrics
#[derive(Debug, Clone, Default)]
pub struct StreamMetrics {
    /// Frames processed
    pub frames_processed: u64,

    /// Bytes processed
    pub bytes_processed: u64,

    /// Errors encountered
    pub error_count: u64,

    /// Buffer underruns
    pub underruns: u64,

    /// Average frame latency (milliseconds)
    pub avg_latency_ms: f64,

    /// Current FPS
    pub current_fps: f32,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_stream_config() {
        let config = StreamConfig::new("test-stream")
            .with_resolution(1920, 1080)
            .with_framerate(60)
            .with_dmabuf(true)
            .with_buffer_count(4);

        assert_eq!(config.name, "test-stream");
        assert_eq!(config.width, 1920);
        assert_eq!(config.height, 1080);
        assert_eq!(config.framerate, 60);
        assert!(config.use_dmabuf);
        assert_eq!(config.buffer_count, 4);
    }

    #[test]
    fn test_stream_creation() {
        let config = StreamConfig::new("test");
        let stream = PipeWireStream::new(0, config);

        assert_eq!(stream.id(), 0);
        assert_eq!(stream.state(), PwStreamState::Initializing);
    }

    #[tokio::test]
    async fn test_stream_state_transitions() {
        let config = StreamConfig::new("test");
        let mut stream = PipeWireStream::new(0, config);

        assert_eq!(stream.state(), PwStreamState::Initializing);

        stream.start().await.unwrap();
        assert_eq!(stream.state(), PwStreamState::Streaming);

        stream.pause().await.unwrap();
        assert_eq!(stream.state(), PwStreamState::Paused);

        stream.resume().await.unwrap();
        assert_eq!(stream.state(), PwStreamState::Streaming);

        stream.stop().await.unwrap();
        assert_eq!(stream.state(), PwStreamState::Closing);
    }

    #[test]
    fn test_frame_callback() {
        let config = StreamConfig::new("test");
        let mut stream = PipeWireStream::new(0, config);

        let received = Arc::new(Mutex::new(false));
        let received_clone = Arc::clone(&received);

        stream.set_frame_callback(Box::new(move |_frame| {
            *received_clone.lock().unwrap() = true;
        }));

        // Verify callback is set
        assert!(stream.frame_callback.lock().unwrap().is_some());
    }
}
