use std::sync::LazyLock;

use ironrdp_core::{decode, encode_vec};

use super::cert::{RsaPublicKey, PROP_CERT_BLOBS_HEADERS_SIZE, PROP_CERT_NO_BLOBS_SIZE, RSA_KEY_SIZE_WITHOUT_MODULUS};
use super::*;
use crate::rdp::headers::{BasicSecurityHeader, BasicSecurityHeaderFlags};
use crate::rdp::server_license::{LicensePdu, PreambleFlags, PreambleVersion};

const LICENSE_HEADER_BUFFER_WITH_CERT: [u8; 8] = [0x80, 0x00, 0x00, 0x00, 0x01, 0x03, 0x9C, 0x08];
const LICENSE_HEADER_BUFFER_NO_CERT: [u8; 8] = [0x80, 0x00, 0x00, 0x00, 0x01, 0x03, 0x8A, 0x00];

const SERVER_RANDOM_BUFFER: [u8; 32] = [
    0x84, 0xef, 0xae, 0x20, 0xb1, 0xd5, 0x9e, 0x36, 0x49, 0x1a, 0xe8, 0x2e, 0x0a, 0x99, 0x89, 0xac, 0x49, 0xa6, 0x47,
    0x4f, 0x33, 0x9b, 0x5a, 0xb9, 0x95, 0x03, 0xa6, 0xc6, 0xc2, 0x3c, 0x3f, 0x61,
];

const PRODUCT_INFO_BUFFER: [u8; 64] = [
    0x00, 0x00, 0x06, 0x00, // version
    0x2c, 0x00, 0x00, 0x00, // company name len
    0x4d, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x73, 0x00, 0x6f, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20,
    0x00, 0x43, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x70, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x61, 0x00, 0x74, 0x00, 0x69, 0x00,
    0x6f, 0x00, 0x6e, 0x00, 0x00, 0x00, // company name
    0x08, 0x00, 0x00, 0x00, // product id len
    0x41, 0x00, 0x30, 0x00, 0x32, 0x00, 0x00, 0x00, // product id
];

const KEY_EXCHANGE_LIST_BUFFER: [u8; 8] = [
    0x0d, 0x00, // blob type
    0x04, 0x00, // blob len
    0x01, 0x00, 0x00, 0x00, // blob
];

const SERVER_CERTIFICATE_HEADER_BUFFER_WITH_BLOB: [u8; 8] = [
    0x03, 0x00, // blob type
    0x12, 0x08, // blob len
    0x02, 0x00, 0x00, 0x80, // certificate version
];

const CERT_HEADER_BUFFER: [u8; 8] = [
    0x02, 0x00, 0x00, 0x00, // num certificate
    0xf5, 0x02, 0x00, 0x00, // size of certificate
];

const CERT_1_BUFFER: [u8; 757] = [
    0x30, 0x82, 0x02, 0xf1, 0x30, 0x82, 0x01, 0xdd, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x01, 0x9e, 0x24, 0xa2,
    0xf2, 0xae, 0x90, 0x80, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d, 0x05, 0x00, 0x30, 0x32, 0x31, 0x30,
    0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1e, 0x0c, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x44, 0x00, 0x45, 0x00, 0x4e,
    0x00, 0x54, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x07, 0x1e, 0x12, 0x00, 0x57, 0x00, 0x4f, 0x00, 0x52, 0x00, 0x4b,
    0x00, 0x47, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x55, 0x00, 0x50, 0x30, 0x1e, 0x17, 0x0d, 0x37, 0x30, 0x30, 0x35, 0x32,
    0x37, 0x30, 0x31, 0x31, 0x31, 0x30, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x39, 0x30, 0x35, 0x32, 0x37, 0x30, 0x31, 0x31,
    0x31, 0x30, 0x33, 0x5a, 0x30, 0x32, 0x31, 0x30, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1e, 0x0c, 0x00, 0x52,
    0x00, 0x4f, 0x00, 0x44, 0x00, 0x45, 0x00, 0x4e, 0x00, 0x54, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x07, 0x1e, 0x12,
    0x00, 0x57, 0x00, 0x4f, 0x00, 0x52, 0x00, 0x4b, 0x00, 0x47, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x55, 0x00, 0x50, 0x30,
    0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03,
    0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x88, 0xad, 0x7c, 0x8f, 0x8b, 0x82,
    0x76, 0x5a, 0xbd, 0x8f, 0x6f, 0x62, 0x18, 0xe1, 0xd9, 0xaa, 0x41, 0xfd, 0xed, 0x68, 0x01, 0xc6, 0x34, 0x35, 0xb0,
    0x29, 0x04, 0xca, 0x4a, 0x4a, 0x1c, 0x7e, 0x80, 0x14, 0xf7, 0x8e, 0x77, 0xb8, 0x25, 0xff, 0x16, 0x47, 0x6f, 0xbd,
    0xe2, 0x34, 0x3d, 0x2e, 0x02, 0xb9, 0x53, 0xe4, 0x33, 0x75, 0xad, 0x73, 0x28, 0x80, 0xa0, 0x4d, 0xfc, 0x6c, 0xc0,
    0x22, 0x53, 0x1b, 0x2c, 0xf8, 0xf5, 0x01, 0x60, 0x19, 0x7e, 0x79, 0x19, 0x39, 0x8d, 0xb5, 0xce, 0x39, 0x58, 0xdd,
    0x55, 0x24, 0x3b, 0x55, 0x7b, 0x43, 0xc1, 0x7f, 0x14, 0x2f, 0xb0, 0x64, 0x3a, 0x54, 0x95, 0x2b, 0x88, 0x49, 0x0c,
    0x61, 0x2d, 0xac, 0xf8, 0x45, 0xf5, 0xda, 0x88, 0x18, 0x5f, 0xae, 0x42, 0xf8, 0x75, 0xc7, 0x26, 0x6d, 0xb5, 0xbb,
    0x39, 0x6f, 0xcc, 0x55, 0x1b, 0x32, 0x11, 0x38, 0x8d, 0xe4, 0xe9, 0x44, 0x84, 0x11, 0x36, 0xa2, 0x61, 0x76, 0xaa,
    0x4c, 0xb4, 0xe3, 0x55, 0x0f, 0xe4, 0x77, 0x8e, 0xde, 0xe3, 0xa9, 0xea, 0xb7, 0x41, 0x94, 0x00, 0x58, 0xaa, 0xc9,
    0x34, 0xa2, 0x98, 0xc6, 0x01, 0x1a, 0x76, 0x14, 0x01, 0xa8, 0xdc, 0x30, 0x7c, 0x77, 0x5a, 0x20, 0x71, 0x5a, 0xa2,
    0x3f, 0xaf, 0x13, 0x7e, 0xe8, 0xfd, 0x84, 0xa2, 0x5b, 0xcf, 0x25, 0xe9, 0xc7, 0x8f, 0xa8, 0xf2, 0x8b, 0x84, 0xc7,
    0x04, 0x5e, 0x53, 0x73, 0x4e, 0x0e, 0x89, 0xa3, 0x3c, 0xe7, 0x68, 0x5c, 0x24, 0xb7, 0x80, 0x53, 0x3c, 0x54, 0xc8,
    0xc1, 0x53, 0xaa, 0x71, 0x71, 0x3d, 0x36, 0x15, 0xd6, 0x6a, 0x9d, 0x7d, 0xde, 0xae, 0xf9, 0xe6, 0xaf, 0x57, 0xae,
    0xb9, 0x01, 0x96, 0x5d, 0xe0, 0x4d, 0xcd, 0xed, 0xc8, 0xd7, 0xf3, 0x01, 0x03, 0x38, 0x10, 0xbe, 0x7c, 0x42, 0x67,
    0x01, 0xa7, 0x23, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x13, 0x30, 0x11, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13,
    0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d,
    0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x81, 0xdd, 0xd2, 0xd3, 0x33, 0xd4, 0xa3, 0xb6, 0x8e, 0x6e, 0x7d, 0x9f,
    0xfd, 0x73, 0x9f, 0x31, 0x0b, 0xdd, 0x42, 0x82, 0x3f, 0x7e, 0x21, 0xdf, 0x28, 0xcc, 0x59, 0xca, 0x6a, 0xc0, 0xa9,
    0x3d, 0x30, 0x7d, 0xe1, 0x91, 0xdb, 0x77, 0x6b, 0x8b, 0x10, 0xe6, 0xfd, 0xbc, 0x3c, 0xa3, 0x58, 0x48, 0xc2, 0x36,
    0xdd, 0xa0, 0x0b, 0xf5, 0x8e, 0x13, 0xda, 0x7b, 0x04, 0x08, 0x44, 0xb4, 0xf2, 0xa8, 0x0d, 0x1e, 0x0b, 0x1d, 0x1a,
    0x3f, 0xf9, 0x9b, 0x4b, 0x5a, 0x54, 0xc5, 0xb3, 0xb4, 0x03, 0x93, 0x75, 0xb3, 0x72, 0x5c, 0x3d, 0xcf, 0x63, 0x0f,
    0x15, 0xe1, 0x64, 0x58, 0xde, 0x52, 0x8d, 0x97, 0x79, 0x0e, 0xa4, 0x34, 0xd5, 0x66, 0x05, 0x58, 0xb8, 0x6e, 0x79,
    0xb2, 0x09, 0x86, 0xd5, 0xf0, 0xed, 0xc4, 0x6b, 0x4c, 0xab, 0x02, 0xb8, 0x16, 0x5f, 0x3b, 0xed, 0x88, 0x5f, 0xd1,
    0xde, 0x44, 0xe3, 0x73, 0x47, 0x21, 0xf7, 0x03, 0xce, 0xe1, 0x6d, 0x10, 0x0f, 0x95, 0xcf, 0x7c, 0xa2, 0x7a, 0xa6,
    0xbf, 0x20, 0xdb, 0xe1, 0x93, 0x04, 0xc8, 0x5e, 0x6a, 0xbe, 0xc8, 0x01, 0x5d, 0x27, 0xb2, 0x03, 0x0f, 0x66, 0x75,
    0xe7, 0xcb, 0xea, 0x8d, 0x4e, 0x98, 0x9d, 0x22, 0xed, 0x28, 0x40, 0xd2, 0x7d, 0xa4, 0x4b, 0xef, 0xcc, 0xbf, 0x01,
    0x2a, 0x6d, 0x3a, 0x3e, 0xbe, 0x47, 0x38, 0xf8, 0xea, 0xa4, 0xc6, 0x30, 0x1d, 0x5e, 0x25, 0xcf, 0xfb, 0xe8, 0x3d,
    0x42, 0xdd, 0x29, 0xe8, 0x99, 0x89, 0x9e, 0xbf, 0x39, 0xee, 0x77, 0x09, 0xd9, 0x3e, 0x8b, 0x52, 0x36, 0xb6, 0xbb,
    0x8b, 0xbd, 0x0d, 0xb2, 0x52, 0xaa, 0x2c, 0xcf, 0x38, 0x4e, 0x4d, 0xcf, 0x1d, 0x6d, 0x5d, 0x25, 0x17, 0xac, 0x2c,
    0xf6, 0xf0, 0x65, 0x5a, 0xc9, 0xfe, 0x31, 0x53, 0xb4, 0xf0, 0x0c, 0x94, 0x4e, 0x0d, 0x54, 0x8e,
];

const CERT_2_BUFFER: [u8; 1277] = [
    0x30, 0x82, 0x04, 0xf9, 0x30, 0x82, 0x03, 0xe5, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x05, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d, 0x05, 0x00, 0x30, 0x32, 0x31, 0x30, 0x30, 0x13, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x1e, 0x0c, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x44, 0x00, 0x45, 0x00, 0x4e, 0x00, 0x54, 0x30,
    0x19, 0x06, 0x03, 0x55, 0x04, 0x07, 0x1e, 0x12, 0x00, 0x57, 0x00, 0x4f, 0x00, 0x52, 0x00, 0x4b, 0x00, 0x47, 0x00,
    0x52, 0x00, 0x4f, 0x00, 0x55, 0x00, 0x50, 0x30, 0x1e, 0x17, 0x0d, 0x30, 0x37, 0x30, 0x35, 0x32, 0x36, 0x31, 0x32,
    0x34, 0x35, 0x35, 0x33, 0x5a, 0x17, 0x0d, 0x33, 0x38, 0x30, 0x31, 0x31, 0x39, 0x30, 0x33, 0x31, 0x34, 0x30, 0x37,
    0x5a, 0x30, 0x81, 0x92, 0x31, 0x81, 0x8f, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1e, 0x1c, 0x00, 0x6e, 0x00,
    0x63, 0x00, 0x61, 0x00, 0x6c, 0x00, 0x72, 0x00, 0x70, 0x00, 0x63, 0x00, 0x3a, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x44,
    0x00, 0x45, 0x00, 0x4e, 0x00, 0x54, 0x30, 0x23, 0x06, 0x03, 0x55, 0x04, 0x07, 0x1e, 0x1c, 0x00, 0x6e, 0x00, 0x63,
    0x00, 0x61, 0x00, 0x6c, 0x00, 0x72, 0x00, 0x70, 0x00, 0x63, 0x00, 0x3a, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x44, 0x00,
    0x45, 0x00, 0x4e, 0x00, 0x54, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x05, 0x1e, 0x3c, 0x00, 0x31, 0x00, 0x42, 0x00,
    0x63, 0x00, 0x4b, 0x00, 0x65, 0x00, 0x62, 0x00, 0x68, 0x00, 0x70, 0x00, 0x58, 0x00, 0x5a, 0x00, 0x74, 0x00, 0x4c,
    0x00, 0x71, 0x00, 0x4f, 0x00, 0x37, 0x00, 0x53, 0x00, 0x51, 0x00, 0x6e, 0x00, 0x42, 0x00, 0x70, 0x00, 0x52, 0x00,
    0x66, 0x00, 0x75, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x59, 0x00, 0x3d, 0x00, 0x0d, 0x00, 0x0a, 0x30, 0x82,
    0x01, 0x1e, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x0f, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
    0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc8, 0x90, 0x6b, 0xf0, 0xc6, 0x58, 0x81, 0xa6, 0x89, 0x1c, 0x0e,
    0xf2, 0xf6, 0xd9, 0x82, 0x12, 0x71, 0xa5, 0x6e, 0x51, 0xdb, 0xe0, 0x32, 0x66, 0xaa, 0x91, 0x77, 0x0e, 0x88, 0xab,
    0x44, 0xb7, 0xd3, 0x97, 0xda, 0x78, 0x8f, 0x0e, 0x44, 0x26, 0x46, 0x7f, 0x16, 0xd4, 0xc6, 0x63, 0xeb, 0xca, 0x55,
    0xe5, 0x4e, 0x8b, 0x2d, 0xa6, 0x6d, 0x83, 0x95, 0xa7, 0xa8, 0x6a, 0xfa, 0xd0, 0xbe, 0x26, 0x80, 0xae, 0xab, 0x0a,
    0x64, 0x90, 0x32, 0x8c, 0xdf, 0x5c, 0xf8, 0xf9, 0xd0, 0x7e, 0xd1, 0x6b, 0x3a, 0x29, 0x7e, 0x7d, 0xbd, 0x02, 0xa3,
    0x86, 0x6c, 0xfd, 0xa5, 0x35, 0x71, 0xda, 0x21, 0xb4, 0xee, 0xa4, 0x97, 0xf3, 0xa8, 0xb2, 0x12, 0xdb, 0xa4, 0x27,
    0x57, 0x36, 0xc9, 0x08, 0x22, 0x5c, 0x54, 0xf7, 0x99, 0x7b, 0xa3, 0x2f, 0xb8, 0x5c, 0xd5, 0x16, 0xb8, 0x19, 0x27,
    0x6b, 0x71, 0x97, 0x14, 0x5b, 0xe8, 0x1f, 0x23, 0xe8, 0x5c, 0xb8, 0x1b, 0x73, 0x4b, 0x6e, 0x7a, 0x03, 0x13, 0xff,
    0x97, 0xe9, 0x62, 0xb9, 0x4a, 0xa0, 0x51, 0x23, 0xc3, 0x6c, 0x32, 0x3e, 0x02, 0xf2, 0x63, 0x97, 0x23, 0x1c, 0xc5,
    0x78, 0xd8, 0xfc, 0xb7, 0x07, 0x4b, 0xb0, 0x56, 0x0f, 0x74, 0xdf, 0xc5, 0x56, 0x28, 0xe4, 0x96, 0xfd, 0x20, 0x8e,
    0x65, 0x5a, 0xe6, 0x45, 0xed, 0xc1, 0x05, 0x3e, 0xab, 0x58, 0x55, 0x40, 0xaf, 0xe2, 0x47, 0xa0, 0x4c, 0x49, 0xa3,
    0x8d, 0x39, 0xe3, 0x66, 0x5f, 0x93, 0x33, 0x6d, 0xf8, 0x5f, 0xc5, 0x54, 0xe5, 0xfb, 0x57, 0x3a, 0xde, 0x45, 0x12,
    0xb5, 0xc7, 0x05, 0x4b, 0x88, 0x1f, 0xb4, 0x35, 0x0f, 0x7c, 0xc0, 0x75, 0x17, 0xc6, 0x67, 0xdd, 0x48, 0x80, 0xcb,
    0x0a, 0xbe, 0x9d, 0xf6, 0x93, 0x60, 0x65, 0x34, 0xeb, 0x97, 0xaf, 0x65, 0x6d, 0xdf, 0xbf, 0x6f, 0x5b, 0x02, 0x03,
    0x01, 0x00, 0x01, 0xa3, 0x82, 0x01, 0xbf, 0x30, 0x82, 0x01, 0xbb, 0x30, 0x14, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04,
    0x01, 0x82, 0x37, 0x12, 0x04, 0x01, 0x01, 0xff, 0x04, 0x04, 0x01, 0x00, 0x05, 0x00, 0x30, 0x3c, 0x06, 0x09, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x12, 0x02, 0x01, 0x01, 0xff, 0x04, 0x2c, 0x4d, 0x00, 0x69, 0x00, 0x63, 0x00,
    0x72, 0x00, 0x6f, 0x00, 0x73, 0x00, 0x6f, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x43, 0x00, 0x6f, 0x00, 0x72,
    0x00, 0x70, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x61, 0x00, 0x74, 0x00, 0x69, 0x00, 0x6f, 0x00, 0x6e, 0x00, 0x00, 0x00,
    0x30, 0x81, 0xcd, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x12, 0x05, 0x01, 0x01, 0xff, 0x04, 0x81,
    0xbc, 0x00, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x09, 0x04, 0x00, 0x00, 0x1c, 0x00,
    0x4a, 0x00, 0x66, 0x00, 0x4a, 0x00, 0xb0, 0x00, 0x01, 0x00, 0x33, 0x00, 0x64, 0x00, 0x32, 0x00, 0x36, 0x00, 0x37,
    0x00, 0x39, 0x00, 0x35, 0x00, 0x34, 0x00, 0x2d, 0x00, 0x65, 0x00, 0x65, 0x00, 0x62, 0x00, 0x37, 0x00, 0x2d, 0x00,
    0x31, 0x00, 0x31, 0x00, 0x64, 0x00, 0x31, 0x00, 0x2d, 0x00, 0x62, 0x00, 0x39, 0x00, 0x34, 0x00, 0x65, 0x00, 0x2d,
    0x00, 0x30, 0x00, 0x30, 0x00, 0x63, 0x00, 0x30, 0x00, 0x34, 0x00, 0x66, 0x00, 0x61, 0x00, 0x33, 0x00, 0x30, 0x00,
    0x38, 0x00, 0x30, 0x00, 0x64, 0x00, 0x00, 0x00, 0x33, 0x00, 0x64, 0x00, 0x32, 0x00, 0x36, 0x00, 0x37, 0x00, 0x39,
    0x00, 0x35, 0x00, 0x34, 0x00, 0x2d, 0x00, 0x65, 0x00, 0x65, 0x00, 0x62, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x31, 0x00,
    0x31, 0x00, 0x64, 0x00, 0x31, 0x00, 0x2d, 0x00, 0x62, 0x00, 0x39, 0x00, 0x34, 0x00, 0x65, 0x00, 0x2d, 0x00, 0x30,
    0x00, 0x30, 0x00, 0x63, 0x00, 0x30, 0x00, 0x34, 0x00, 0x66, 0x00, 0x61, 0x00, 0x33, 0x00, 0x30, 0x00, 0x38, 0x00,
    0x30, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x80, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0x6e, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x12, 0x06, 0x01, 0x01, 0xff, 0x04, 0x5e, 0x00, 0x30,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x3e, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x44, 0x00, 0x45, 0x00, 0x4e, 0x00, 0x54,
    0x00, 0x00, 0x00, 0x37, 0x00, 0x38, 0x00, 0x34, 0x00, 0x34, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x30, 0x00, 0x30, 0x00,
    0x36, 0x00, 0x2d, 0x00, 0x35, 0x00, 0x38, 0x00, 0x36, 0x00, 0x37, 0x00, 0x30, 0x00, 0x34, 0x00, 0x35, 0x00, 0x2d,
    0x00, 0x37, 0x00, 0x30, 0x00, 0x33, 0x00, 0x34, 0x00, 0x37, 0x00, 0x00, 0x00, 0x57, 0x00, 0x4f, 0x00, 0x52, 0x00,
    0x4b, 0x00, 0x47, 0x00, 0x52, 0x00, 0x4f, 0x00, 0x55, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x1d, 0x23, 0x01, 0x01, 0xff, 0x04, 0x1b, 0x30, 0x19, 0xa1, 0x10, 0xa4, 0x0e, 0x52, 0x00, 0x4f, 0x00,
    0x44, 0x00, 0x45, 0x00, 0x4e, 0x00, 0x54, 0x00, 0x00, 0x00, 0x82, 0x05, 0x01, 0x00, 0x00, 0x00, 0x02, 0x30, 0x09,
    0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1d, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x2e, 0xeb, 0xc7, 0x0d, 0xb8,
    0x1d, 0x47, 0x11, 0x9d, 0x09, 0x88, 0x9b, 0x51, 0xdc, 0x45, 0xdd, 0x56, 0x51, 0xe2, 0xd1, 0x23, 0x11, 0x39, 0x9b,
    0x2d, 0xda, 0xc7, 0xfe, 0x7a, 0xd7, 0x84, 0xe3, 0x3d, 0x54, 0x77, 0x97, 0x4d, 0x19, 0x92, 0x30, 0x64, 0xa0, 0x47,
    0xc6, 0x2f, 0x6d, 0x93, 0xd2, 0x64, 0x7c, 0x76, 0xc8, 0x26, 0x45, 0xad, 0x5a, 0x44, 0x54, 0xea, 0xf6, 0x4b, 0x28,
    0x77, 0x1f, 0x77, 0xea, 0xec, 0x74, 0x02, 0x38, 0x68, 0x9e, 0x79, 0x14, 0x72, 0x83, 0x34, 0x74, 0x62, 0xd2, 0xc1,
    0x0c, 0xa4, 0x0b, 0xf2, 0xa9, 0xb0, 0x38, 0xbb, 0x7c, 0xd0, 0xae, 0xbe, 0xbf, 0x74, 0x47, 0x16, 0xa0, 0xa2, 0xd3,
    0xfc, 0x1d, 0xb9, 0xba, 0x26, 0x10, 0x06, 0xef, 0xba, 0x1d, 0x43, 0x01, 0x4e, 0x4e, 0x6f, 0x56, 0xca, 0xe0, 0xee,
    0xd0, 0xf9, 0x4e, 0xa6, 0x62, 0x63, 0xff, 0xda, 0x0b, 0xc9, 0x15, 0x61, 0x6c, 0xed, 0x6b, 0x0b, 0xc4, 0x58, 0x53,
    0x86, 0x0f, 0x8c, 0x0c, 0x1a, 0x2e, 0xdf, 0xc1, 0xf2, 0x43, 0x48, 0xd4, 0xaf, 0x0a, 0x78, 0x36, 0xb2, 0x51, 0x32,
    0x28, 0x6c, 0xc2, 0x75, 0x79, 0x3f, 0x6e, 0x99, 0x66, 0x88, 0x3e, 0x34, 0xd3, 0x7f, 0x6d, 0x9d, 0x07, 0xe4, 0x6b,
    0xeb, 0x84, 0xe2, 0x0a, 0xbb, 0xca, 0x7d, 0x3a, 0x40, 0x71, 0xb0, 0xbe, 0x47, 0x9f, 0x12, 0x58, 0x31, 0x61, 0x2b,
    0x9b, 0x4a, 0x9a, 0x49, 0x8f, 0xe5, 0xb4, 0x0c, 0xf5, 0x04, 0x4d, 0x3c, 0xce, 0xbc, 0xd2, 0x79, 0x15, 0xd9, 0x28,
    0xf4, 0x23, 0x56, 0x77, 0x9f, 0x38, 0x64, 0x3e, 0x03, 0x88, 0x92, 0x04, 0x26, 0x76, 0xb9, 0xb5, 0xdf, 0x19, 0xd0,
    0x78, 0x4b, 0x7a, 0x60, 0x40, 0x23, 0x91, 0xf1, 0x15, 0x22, 0x2b, 0xb4, 0xe7, 0x02, 0x54, 0xa9, 0x16, 0x21, 0x5b,
    0x60, 0x96, 0xa9, 0x5c,
];

const CERT_2_LEN_BUFFER: [u8; 4] = [0xfd, 0x04, 0x00, 0x00];

const PADDING_BUFFER: [u8; 16] = [
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // padding
];

const SCOPE_BUFFER_WITH_COUNT: [u8; 22] = [
    0x01, 0x00, 0x00, 0x00, // scope count
    0x0e, 0x00, 0x0e, 0x00, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f, 0x6d,
    0x00, // scope list
];

const MAGIC: [u8; 4] = [0x52, 0x53, 0x41, 0x31];
const KEYLEN: [u8; 4] = [0x48, 0x00, 0x00, 0x00];
const BITLEN: [u8; 4] = [0x00, 0x02, 0x00, 0x00];
const DATALEN: [u8; 4] = [0x3f, 0x00, 0x00, 0x00];
const PUB_EXP: [u8; 4] = [0x01, 0x00, 0x01, 0x00];

const MODULUS: [u8; 72] = [
    0xcb, 0x81, 0xfe, 0xba, 0x6d, 0x61, 0xc3, 0x55, 0x05, 0xd5, 0x5f, 0x2e, 0x87, 0xf8, 0x71, 0x94, 0xd6, 0xf1, 0xa5,
    0xcb, 0xf1, 0x5f, 0x0c, 0x3d, 0xf8, 0x70, 0x02, 0x96, 0xc4, 0xfb, 0x9b, 0xc8, 0x3c, 0x2d, 0x55, 0xae, 0xe8, 0xff,
    0x32, 0x75, 0xea, 0x68, 0x79, 0xe5, 0xa2, 0x01, 0xfd, 0x31, 0xa0, 0xb1, 0x1f, 0x55, 0xa6, 0x1f, 0xc1, 0xf6, 0xd1,
    0x83, 0x88, 0x63, 0x26, 0x56, 0x12, 0xbc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
];

const CERT_HEADER_WITH_PARTIAL_RSA: [u8; 32] = [
    0x01, 0x00, 0x00, 0x00, // sig
    0x01, 0x00, 0x00, 0x00, // dwkeyal
    0x06, 0x00, // blob type
    0x5c, 0x00, // len
    0x52, 0x53, 0x41, 0x31, 0x48, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
    0x00, // rsa without modulus
];

const SIGNATURE: [u8; 72] = [
    0xe9, 0xe1, 0xd6, 0x28, 0x46, 0x8b, 0x4e, 0xf5, 0x0a, 0xdf, 0xfd, 0xee, 0x21, 0x99, 0xac, 0xb4, 0xe1, 0x8f, 0x5f,
    0x81, 0x57, 0x82, 0xef, 0x9d, 0x96, 0x52, 0x63, 0x27, 0x18, 0x29, 0xdb, 0xb3, 0x4a, 0xfd, 0x9a, 0xda, 0x42, 0xad,
    0xb5, 0x69, 0x21, 0x89, 0x0e, 0x1d, 0xc0, 0x4c, 0x1a, 0xa8, 0xaa, 0x71, 0x3e, 0x0f, 0x54, 0xb9, 0x9a, 0xe4, 0x99,
    0x68, 0x3f, 0x6c, 0xd6, 0x76, 0x84, 0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
];

const SIGNATURE_BLOB_HEADER: [u8; 4] = [
    0x08, 0x00, // sig blob type
    0x48, 0x00, // sig blob len
];

const SERVER_CERTIFICATE_HEADER_BUFFER: [u8; 4] = [
    0x02, 0x00, 0x00, 0x80, // certificate version
];

const SCOPE_BUFFER: [u8; 18] = [
    0x0e, 0x00, 0x0e, 0x00, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x63, 0x6f, 0x6d,
    0x00, // scope array
];

static PROPRIETARY_CERTIFICATE: LazyLock<ProprietaryCertificate> = LazyLock::new(|| ProprietaryCertificate {
    public_key: RsaPublicKey {
        public_exponent: 0x0001_0001,
        modulus: Vec::from(MODULUS.as_ref()),
    },
    signature: Vec::from(SIGNATURE.as_ref()),
});
static PRODUCT_INFO: LazyLock<ProductInfo> = LazyLock::new(|| ProductInfo {
    version: 0x60000,
    company_name: "Microsoft Corporation".to_owned(),
    product_id: "A02".to_owned(),
});
static PUBLIC_KEY: LazyLock<RsaPublicKey> = LazyLock::new(|| RsaPublicKey {
    public_exponent: 0x0001_0001,
    modulus: Vec::from(MODULUS.as_ref()),
});
static SERVER_LICENSE_REQUEST: LazyLock<LicensePdu> = LazyLock::new(|| {
    let mut req = ServerLicenseRequest {
        license_header: LicenseHeader {
            security_header: BasicSecurityHeader {
                flags: BasicSecurityHeaderFlags::LICENSE_PKT,
            },
            preamble_message_type: PreambleType::LicenseRequest,
            preamble_flags: PreambleFlags::empty(),
            preamble_version: PreambleVersion::V3,
            preamble_message_size: 0,
        },
        server_random: Vec::from(SERVER_RANDOM_BUFFER.as_ref()),
        product_info: ProductInfo {
            version: 0x60000,
            company_name: "Microsoft Corporation".to_owned(),
            product_id: "A02".to_owned(),
        },
        server_certificate: Some(ServerCertificate {
            issued_permanently: true,
            certificate: CertificateType::X509(X509CertificateChain {
                certificate_array: vec![Vec::from(CERT_1_BUFFER.as_ref()), Vec::from(CERT_2_BUFFER.as_ref())],
            }),
        }),
        scope_list: vec![Scope(String::from("microsoft.com"))],
    };
    req.license_header.preamble_message_size = u16::try_from(req.size()).expect("can't panic");
    req.into()
});
static X509_CERTIFICATE: LazyLock<ServerCertificate> = LazyLock::new(|| ServerCertificate {
    issued_permanently: true,
    certificate: CertificateType::X509(X509CertificateChain {
        certificate_array: vec![Vec::from(CERT_1_BUFFER.as_ref()), Vec::from(CERT_2_BUFFER.as_ref())],
    }),
});
static SCOPE: LazyLock<Scope> = LazyLock::new(|| Scope(String::from("microsoft.com")));
static CERT_CHAIN: LazyLock<X509CertificateChain> = LazyLock::new(|| X509CertificateChain {
    certificate_array: vec![Vec::from(CERT_1_BUFFER.as_ref()), Vec::from(CERT_2_BUFFER.as_ref())],
});

#[test]
fn from_buffer_correctly_parses_server_license_request() {
    let request_buffer = [
        &LICENSE_HEADER_BUFFER_WITH_CERT[..],
        &SERVER_RANDOM_BUFFER[..],
        &PRODUCT_INFO_BUFFER[..],
        &KEY_EXCHANGE_LIST_BUFFER[..],
        &SERVER_CERTIFICATE_HEADER_BUFFER_WITH_BLOB[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
        &SCOPE_BUFFER_WITH_COUNT[..],
    ]
    .concat();

    assert_eq!(*SERVER_LICENSE_REQUEST, decode(&request_buffer).unwrap());
}

#[test]
fn from_buffer_correctly_parses_server_license_request_no_certificate() {
    let server_certificate_header_buffer = [
        0x03, 0x00, // blob type
        0x00, 0x00, // blob len
    ];

    let request_buffer = [
        &LICENSE_HEADER_BUFFER_NO_CERT[..],
        &SERVER_RANDOM_BUFFER[..],
        &PRODUCT_INFO_BUFFER[..],
        &KEY_EXCHANGE_LIST_BUFFER[..],
        &server_certificate_header_buffer[..],
        &SCOPE_BUFFER_WITH_COUNT[..],
    ]
    .concat();

    let mut request = ServerLicenseRequest {
        license_header: LicenseHeader {
            security_header: BasicSecurityHeader {
                flags: BasicSecurityHeaderFlags::LICENSE_PKT,
            },
            preamble_message_type: PreambleType::LicenseRequest,
            preamble_flags: PreambleFlags::empty(),
            preamble_version: PreambleVersion::V3,
            preamble_message_size: 0,
        },
        server_random: Vec::from(SERVER_RANDOM_BUFFER.as_ref()),
        product_info: ProductInfo {
            version: 0x60000,
            company_name: "Microsoft Corporation".to_owned(),
            product_id: "A02".to_owned(),
        },
        server_certificate: None,
        scope_list: vec![Scope(String::from("microsoft.com"))],
    };
    request.license_header.preamble_message_size = u16::try_from(request.size()).expect("can't panic");
    let request: LicensePdu = request.into();

    assert_eq!(request, decode(&request_buffer).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_server_license_request() {
    let request_buffer = [
        &LICENSE_HEADER_BUFFER_WITH_CERT[..],
        &SERVER_RANDOM_BUFFER[..],
        &PRODUCT_INFO_BUFFER[..],
        &KEY_EXCHANGE_LIST_BUFFER[..],
        &SERVER_CERTIFICATE_HEADER_BUFFER_WITH_BLOB[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
        &SCOPE_BUFFER_WITH_COUNT[..],
    ]
    .concat();

    let mut request = ServerLicenseRequest {
        license_header: LicenseHeader {
            security_header: BasicSecurityHeader {
                flags: BasicSecurityHeaderFlags::LICENSE_PKT,
            },
            preamble_message_type: PreambleType::LicenseRequest,
            preamble_flags: PreambleFlags::empty(),
            preamble_version: PreambleVersion::V3,
            preamble_message_size: 0,
        },
        server_random: Vec::from(SERVER_RANDOM_BUFFER.as_ref()),
        product_info: ProductInfo {
            version: 0x60000,
            company_name: "Microsoft Corporation".to_owned(),
            product_id: "A02".to_owned(),
        },
        server_certificate: Some(ServerCertificate {
            issued_permanently: true,
            certificate: CertificateType::X509(X509CertificateChain {
                certificate_array: vec![Vec::from(CERT_1_BUFFER.as_ref()), Vec::from(CERT_2_BUFFER.as_ref())],
            }),
        }),
        scope_list: vec![Scope(String::from("microsoft.com"))],
    };
    request.license_header.preamble_message_size = u16::try_from(request.size()).unwrap();
    let request: LicensePdu = request.into();

    let serialized_request = encode_vec(&request).unwrap();

    assert_eq!(serialized_request, request_buffer);
}

#[test]
fn buffer_length_is_correct_for_server_license_request() {
    let request_buffer = [
        &LICENSE_HEADER_BUFFER_WITH_CERT[..],
        &SERVER_RANDOM_BUFFER[..],
        &PRODUCT_INFO_BUFFER[..],
        &KEY_EXCHANGE_LIST_BUFFER[..],
        &SERVER_CERTIFICATE_HEADER_BUFFER_WITH_BLOB[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
        &SCOPE_BUFFER_WITH_COUNT[..],
    ]
    .concat();

    assert_eq!(request_buffer.len(), SERVER_LICENSE_REQUEST.size());
}

#[test]
fn from_buffer_correctly_parses_rsa_public_key() {
    let buffer = [
        &MAGIC[..],
        &KEYLEN[..],
        &BITLEN[..],
        &DATALEN[..],
        &PUB_EXP[..],
        &MODULUS[..],
    ]
    .concat();

    assert_eq!(*PUBLIC_KEY, decode(&buffer).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_rsa_public_key() {
    let buffer = [
        &MAGIC[..],
        &KEYLEN[..],
        &BITLEN[..],
        &DATALEN[..],
        &PUB_EXP[..],
        &MODULUS[..],
    ]
    .concat();

    let serialized_rsa_key = encode_vec(&*PUBLIC_KEY).unwrap();

    assert_eq!(&serialized_rsa_key, &buffer);
}

#[test]
fn buffer_length_is_correct_for_rsa_public_key() {
    assert_eq!(PUBLIC_KEY.size(), RSA_KEY_SIZE_WITHOUT_MODULUS + MODULUS.len());
}

#[test]
fn from_buffer_correctly_parses_proprietary_certificate() {
    let certificate_buffer = [
        &CERT_HEADER_WITH_PARTIAL_RSA[..],
        &MODULUS[..],
        &SIGNATURE_BLOB_HEADER[..],
        &SIGNATURE[..],
    ]
    .concat();

    assert_eq!(*PROPRIETARY_CERTIFICATE, decode(&certificate_buffer).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_proprietary_certificate() {
    let certificate_buffer = [
        &CERT_HEADER_WITH_PARTIAL_RSA[..],
        &MODULUS[..],
        &SIGNATURE_BLOB_HEADER[..],
        &SIGNATURE[..],
    ]
    .concat();

    let serialized_certificate = encode_vec(&*PROPRIETARY_CERTIFICATE).unwrap();

    assert_eq!(serialized_certificate, certificate_buffer);
}

#[test]
fn buffer_length_is_correct_for_proprietary_certificate() {
    let certificate = ProprietaryCertificate {
        public_key: PUBLIC_KEY.clone(),
        signature: Vec::from(SIGNATURE.as_ref()),
    };

    assert_eq!(
        certificate.size(),
        PUBLIC_KEY.size() + SIGNATURE.len() + PROP_CERT_NO_BLOBS_SIZE + PROP_CERT_BLOBS_HEADERS_SIZE
    );
}

#[test]
fn from_buffer_correctly_parses_product_information() {
    assert_eq!(*PRODUCT_INFO, decode(&PRODUCT_INFO_BUFFER).unwrap());
}

#[test]
fn from_buffer_product_info_handles_invalid_strings_correctly() {
    let product_info_buffer: [u8; 13] = [
        0x00, 0x00, 0x06, 0x00, // version
        0x01, 0x00, 0x00, 0x00, // company name len
        0x00, // company name
        0x00, 0x00, 0x00, 0x00, // product id len
              // product id
    ];

    assert!(decode::<ProductInfo>(product_info_buffer.as_ref()).is_err());
}

#[test]
fn to_buffer_correctly_serializes_product_information() {
    let buffer = encode_vec(&*PRODUCT_INFO).unwrap();

    assert_eq!(buffer, PRODUCT_INFO_BUFFER.as_ref());
}

#[test]
fn buffer_length_is_correct_for_product_information() {
    assert_eq!(PRODUCT_INFO.size(), PRODUCT_INFO_BUFFER.len());
}

#[test]
fn get_public_key_correctly_gets_key_from_server_certificate() {
    let public_key: [u8; 270] = [
        0x30, 0x82, 0x1, 0xa, 0x2, 0x82, 0x1, 0x1, 0x0, 0xc8, 0x90, 0x6b, 0xf0, 0xc6, 0x58, 0x81, 0xa6, 0x89, 0x1c,
        0xe, 0xf2, 0xf6, 0xd9, 0x82, 0x12, 0x71, 0xa5, 0x6e, 0x51, 0xdb, 0xe0, 0x32, 0x66, 0xaa, 0x91, 0x77, 0xe, 0x88,
        0xab, 0x44, 0xb7, 0xd3, 0x97, 0xda, 0x78, 0x8f, 0xe, 0x44, 0x26, 0x46, 0x7f, 0x16, 0xd4, 0xc6, 0x63, 0xeb,
        0xca, 0x55, 0xe5, 0x4e, 0x8b, 0x2d, 0xa6, 0x6d, 0x83, 0x95, 0xa7, 0xa8, 0x6a, 0xfa, 0xd0, 0xbe, 0x26, 0x80,
        0xae, 0xab, 0xa, 0x64, 0x90, 0x32, 0x8c, 0xdf, 0x5c, 0xf8, 0xf9, 0xd0, 0x7e, 0xd1, 0x6b, 0x3a, 0x29, 0x7e,
        0x7d, 0xbd, 0x2, 0xa3, 0x86, 0x6c, 0xfd, 0xa5, 0x35, 0x71, 0xda, 0x21, 0xb4, 0xee, 0xa4, 0x97, 0xf3, 0xa8,
        0xb2, 0x12, 0xdb, 0xa4, 0x27, 0x57, 0x36, 0xc9, 0x8, 0x22, 0x5c, 0x54, 0xf7, 0x99, 0x7b, 0xa3, 0x2f, 0xb8,
        0x5c, 0xd5, 0x16, 0xb8, 0x19, 0x27, 0x6b, 0x71, 0x97, 0x14, 0x5b, 0xe8, 0x1f, 0x23, 0xe8, 0x5c, 0xb8, 0x1b,
        0x73, 0x4b, 0x6e, 0x7a, 0x3, 0x13, 0xff, 0x97, 0xe9, 0x62, 0xb9, 0x4a, 0xa0, 0x51, 0x23, 0xc3, 0x6c, 0x32,
        0x3e, 0x2, 0xf2, 0x63, 0x97, 0x23, 0x1c, 0xc5, 0x78, 0xd8, 0xfc, 0xb7, 0x7, 0x4b, 0xb0, 0x56, 0xf, 0x74, 0xdf,
        0xc5, 0x56, 0x28, 0xe4, 0x96, 0xfd, 0x20, 0x8e, 0x65, 0x5a, 0xe6, 0x45, 0xed, 0xc1, 0x5, 0x3e, 0xab, 0x58,
        0x55, 0x40, 0xaf, 0xe2, 0x47, 0xa0, 0x4c, 0x49, 0xa3, 0x8d, 0x39, 0xe3, 0x66, 0x5f, 0x93, 0x33, 0x6d, 0xf8,
        0x5f, 0xc5, 0x54, 0xe5, 0xfb, 0x57, 0x3a, 0xde, 0x45, 0x12, 0xb5, 0xc7, 0x5, 0x4b, 0x88, 0x1f, 0xb4, 0x35, 0xf,
        0x7c, 0xc0, 0x75, 0x17, 0xc6, 0x67, 0xdd, 0x48, 0x80, 0xcb, 0xa, 0xbe, 0x9d, 0xf6, 0x93, 0x60, 0x65, 0x34,
        0xeb, 0x97, 0xaf, 0x65, 0x6d, 0xdf, 0xbf, 0x6f, 0x5b, 0x2, 0x3, 0x1, 0x0, 0x1,
    ];

    assert_eq!(
        public_key.as_ref(),
        X509_CERTIFICATE.get_public_key().unwrap().as_slice()
    );
}

#[test]
fn from_buffer_correctly_parses_server_certificate() {
    let certificate_buffer = [
        &SERVER_CERTIFICATE_HEADER_BUFFER[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    assert_eq!(*X509_CERTIFICATE, decode(&certificate_buffer).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_server_certificate() {
    let certificate_buffer = [
        &SERVER_CERTIFICATE_HEADER_BUFFER[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    let serialized_certificate = encode_vec(&*X509_CERTIFICATE).unwrap();

    assert_eq!(serialized_certificate, certificate_buffer);
}

#[test]
fn buffer_length_is_correct_for_server_certificate() {
    let certificate_buffer = [
        &SERVER_CERTIFICATE_HEADER_BUFFER[..],
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    assert_eq!(X509_CERTIFICATE.size(), certificate_buffer.len());
}

#[test]
fn from_buffer_correctly_parses_scope() {
    assert_eq!(*SCOPE, decode(&SCOPE_BUFFER).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_scope() {
    let serialized_scope = encode_vec(&*SCOPE).unwrap();

    assert_eq!(serialized_scope, SCOPE_BUFFER.as_ref());
}

#[test]
fn buffer_length_is_correct_for_scope() {
    assert_eq!(SCOPE_BUFFER.len(), SCOPE.size());
}

#[test]
fn from_buffer_correctly_parses_x509_certificate_chain() {
    let chain_buffer = [
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    assert_eq!(*CERT_CHAIN, decode(&chain_buffer).unwrap());
}

#[test]
fn to_buffer_correctly_serializes_x509_certificate_chain() {
    let chain_buffer = [
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    let serialized_chain = encode_vec(&*CERT_CHAIN).unwrap();

    assert_eq!(chain_buffer, serialized_chain);
}

#[test]
fn buffer_length_is_correct_for_x509_certificate_chain() {
    let chain_buffer = [
        &CERT_HEADER_BUFFER[..],
        &CERT_1_BUFFER[..],
        &CERT_2_LEN_BUFFER[..],
        &CERT_2_BUFFER[..],
        &PADDING_BUFFER[..],
    ]
    .concat();

    assert_eq!(CERT_CHAIN.size(), chain_buffer.len());
}
