//! File Operations Module
//!
//! Handles loading, saving, and exporting configuration files.

use std::{
    fs,
    path::{Path, PathBuf},
};

use crate::config::{is_flatpak, Config};

/// Default configuration file paths for native installation
const NATIVE_CONFIG_PATHS: &[&str] = &[
    "~/.config/lamco-rdp-server/config.toml",
    "/etc/lamco-rdp-server/config.toml",
    "./config.toml",
];

/// Get configuration search paths based on deployment context.
///
/// - Flatpak: Only searches within the sandboxed config directory
/// - Native: Searches user config, system config, then current directory
pub fn get_config_paths() -> Vec<PathBuf> {
    if is_flatpak() {
        // Flatpak: dirs::config_dir() returns ~/.var/app/<app-id>/config
        if let Some(config_dir) = dirs::config_dir() {
            vec![config_dir.join("config.toml")]
        } else {
            vec![PathBuf::from("/app/config/config.toml")]
        }
    } else {
        // Native: search standard paths
        NATIVE_CONFIG_PATHS.iter().map(|p| expand_path(p)).collect()
    }
}

/// Save configuration to a TOML file
pub fn save_config(config: &Config, path: &Path) -> Result<(), String> {
    if let Some(parent) = path.parent() {
        fs::create_dir_all(parent)
            .map_err(|e| format!("Failed to create config directory: {}", e))?;
    }

    let toml_content =
        toml::to_string_pretty(config).map_err(|e| format!("Failed to serialize config: {}", e))?;

    let content_with_header = format!(
        "# Lamco RDP Server Configuration\n\
         # Generated by lamco-rdp-server-gui\n\
         # Documentation: https://lamco.ai/docs/config\n\n\
         {}\n",
        toml_content
    );

    fs::write(path, content_with_header)
        .map_err(|e| format!("Failed to write config file: {}", e))?;

    Ok(())
}

/// Load configuration from a TOML file
pub fn load_config(path: &Path) -> Result<Config, String> {
    let content =
        fs::read_to_string(path).map_err(|e| format!("Failed to read config file: {}", e))?;

    let config: Config =
        toml::from_str(&content).map_err(|e| format!("Failed to parse config file: {}", e))?;

    Ok(config)
}

/// Load configuration from the default locations
pub fn load_config_from_default_locations() -> Result<(Config, PathBuf), String> {
    let paths = get_config_paths();

    for path in &paths {
        if path.exists() {
            match load_config(path) {
                Ok(config) => return Ok((config, path.clone())),
                Err(e) => {
                    eprintln!("Warning: Failed to load config from {:?}: {}", path, e);
                }
            }
        }
    }

    let default_path = paths.first().cloned().unwrap_or_else(get_user_config_path);
    Ok((Config::default(), default_path))
}

/// Expand ~ to home directory
fn expand_path(path: &str) -> PathBuf {
    if let Some(suffix) = path.strip_prefix("~/") {
        if let Some(home) = dirs::home_dir() {
            return home.join(suffix);
        }
    }
    PathBuf::from(path)
}

/// Export configuration as a formatted string (for display/copy)
pub fn export_config_string(config: &Config) -> Result<String, String> {
    toml::to_string_pretty(config).map_err(|e| format!("Failed to serialize config: {}", e))
}

/// Check if a config file exists and is readable
pub fn validate_config_path(path: &Path) -> Result<(), String> {
    if !path.exists() {
        return Err(format!("Config file does not exist: {}", path.display()));
    }

    // Try to read the file to verify permissions
    fs::read_to_string(path).map_err(|e| format!("Cannot read config file: {}", e))?;

    Ok(())
}

pub fn get_user_config_path() -> PathBuf {
    if is_flatpak() {
        // Flatpak: dirs::config_dir() returns ~/.var/app/<app-id>/config
        dirs::config_dir()
            .map(|d| d.join("config.toml"))
            .unwrap_or_else(|| PathBuf::from("/app/config/config.toml"))
    } else if let Some(config_dir) = dirs::config_dir() {
        config_dir.join("lamco-rdp-server").join("config.toml")
    } else {
        PathBuf::from("./config.toml")
    }
}

pub fn get_system_config_path() -> PathBuf {
    PathBuf::from("/etc/lamco-rdp-server/config.toml")
}

/// Create a backup of an existing config file before overwriting
pub fn backup_config(path: &Path) -> Result<PathBuf, String> {
    if !path.exists() {
        return Err("Original file does not exist".to_string());
    }

    let timestamp = chrono::Local::now().format("%Y%m%d_%H%M%S");
    let backup_name = format!(
        "{}.backup.{}",
        path.file_name()
            .and_then(|n| n.to_str())
            .unwrap_or("config.toml"),
        timestamp
    );

    let backup_path = path.with_file_name(backup_name);

    fs::copy(path, &backup_path).map_err(|e| format!("Failed to create backup: {}", e))?;

    Ok(backup_path)
}

/// Import configuration from another file, merging with current
pub fn import_config(_current: &Config, import_path: &Path) -> Result<Config, String> {
    let imported = load_config(import_path)?;

    // For now, just replace entirely
    // Could implement selective merging in the future
    Ok(imported)
}

/// List all config files in a directory
pub fn list_config_files(dir: &Path) -> Result<Vec<PathBuf>, String> {
    if !dir.is_dir() {
        return Err(format!("{} is not a directory", dir.display()));
    }

    let mut configs = Vec::new();

    let entries = fs::read_dir(dir).map_err(|e| format!("Failed to read directory: {}", e))?;

    for entry in entries.flatten() {
        let path = entry.path();
        if path.extension().map(|e| e == "toml").unwrap_or(false) {
            configs.push(path);
        }
    }

    configs.sort();
    Ok(configs)
}

/// Export logs to a file
pub fn export_logs(logs: &[crate::gui::state::LogLine], path: &Path) -> Result<(), String> {
    if let Some(parent) = path.parent() {
        fs::create_dir_all(parent).map_err(|e| format!("Failed to create log directory: {}", e))?;
    }

    let mut content = String::new();
    content.push_str("# Lamco RDP Server Logs\n");
    content.push_str(&format!("# Exported at: {}\n\n", chrono::Local::now()));

    for log in logs {
        content.push_str(&format!(
            "{} [{}] {}\n",
            log.timestamp, log.level, log.message
        ));
    }

    fs::write(path, content).map_err(|e| format!("Failed to write log file: {}", e))?;

    Ok(())
}

/// Parse a TOML file and return validation errors without loading
pub fn validate_toml_syntax(path: &Path) -> Result<(), String> {
    let content = fs::read_to_string(path).map_err(|e| format!("Failed to read file: {}", e))?;

    let _: toml::Value =
        toml::from_str(&content).map_err(|e| format!("TOML syntax error: {}", e))?;

    Ok(())
}

/// Compare two configs and return list of differences
pub fn diff_configs(a: &Config, b: &Config) -> Vec<String> {
    let mut differences = Vec::new();

    // Server section
    if a.server.listen_addr != b.server.listen_addr {
        differences.push(format!(
            "server.listen_addr: '{}' -> '{}'",
            a.server.listen_addr, b.server.listen_addr
        ));
    }
    if a.server.max_connections != b.server.max_connections {
        differences.push(format!(
            "server.max_connections: {} -> {}",
            a.server.max_connections, b.server.max_connections
        ));
    }
    if a.server.session_timeout != b.server.session_timeout {
        differences.push(format!(
            "server.session_timeout: {} -> {}",
            a.server.session_timeout, b.server.session_timeout
        ));
    }
    if a.server.use_portals != b.server.use_portals {
        differences.push(format!(
            "server.use_portals: {} -> {}",
            a.server.use_portals, b.server.use_portals
        ));
    }

    // Security section
    if a.security.cert_path != b.security.cert_path {
        differences.push(format!(
            "security.cert_path: {:?} -> {:?}",
            a.security.cert_path, b.security.cert_path
        ));
    }
    if a.security.key_path != b.security.key_path {
        differences.push(format!(
            "security.key_path: {:?} -> {:?}",
            a.security.key_path, b.security.key_path
        ));
    }
    if a.security.enable_nla != b.security.enable_nla {
        differences.push(format!(
            "security.enable_nla: {} -> {}",
            a.security.enable_nla, b.security.enable_nla
        ));
    }
    if a.security.auth_method != b.security.auth_method {
        differences.push(format!(
            "security.auth_method: '{}' -> '{}'",
            a.security.auth_method, b.security.auth_method
        ));
    }
    if a.security.require_tls_13 != b.security.require_tls_13 {
        differences.push(format!(
            "security.require_tls_13: {} -> {}",
            a.security.require_tls_13, b.security.require_tls_13
        ));
    }

    // Video section (encoder/bitrate in dedicated sections)
    if a.video.target_fps != b.video.target_fps {
        differences.push(format!(
            "video.target_fps: {} -> {}",
            a.video.target_fps, b.video.target_fps
        ));
    }
    if a.video.cursor_mode != b.video.cursor_mode {
        differences.push(format!(
            "video.cursor_mode: '{}' -> '{}'",
            a.video.cursor_mode, b.video.cursor_mode
        ));
    }

    // EGFX section
    if a.egfx.enabled != b.egfx.enabled {
        differences.push(format!(
            "egfx.enabled: {} -> {}",
            a.egfx.enabled, b.egfx.enabled
        ));
    }
    if a.egfx.h264_bitrate != b.egfx.h264_bitrate {
        differences.push(format!(
            "egfx.h264_bitrate: {} -> {}",
            a.egfx.h264_bitrate, b.egfx.h264_bitrate
        ));
    }
    if a.egfx.codec != b.egfx.codec {
        differences.push(format!(
            "egfx.codec: '{}' -> '{}'",
            a.egfx.codec, b.egfx.codec
        ));
    }

    // Clipboard section
    if a.clipboard.enabled != b.clipboard.enabled {
        differences.push(format!(
            "clipboard.enabled: {} -> {}",
            a.clipboard.enabled, b.clipboard.enabled
        ));
    }
    if a.clipboard.max_size != b.clipboard.max_size {
        differences.push(format!(
            "clipboard.max_size: {} -> {}",
            a.clipboard.max_size, b.clipboard.max_size
        ));
    }

    // Performance section
    if a.performance.encoder_threads != b.performance.encoder_threads {
        differences.push(format!(
            "performance.encoder_threads: {} -> {}",
            a.performance.encoder_threads, b.performance.encoder_threads
        ));
    }
    if a.performance.zero_copy != b.performance.zero_copy {
        differences.push(format!(
            "performance.zero_copy: {} -> {}",
            a.performance.zero_copy, b.performance.zero_copy
        ));
    }

    // Logging section
    if a.logging.level != b.logging.level {
        differences.push(format!(
            "logging.level: '{}' -> '{}'",
            a.logging.level, b.logging.level
        ));
    }
    if a.logging.metrics != b.logging.metrics {
        differences.push(format!(
            "logging.metrics: {} -> {}",
            a.logging.metrics, b.logging.metrics
        ));
    }

    differences
}

#[cfg(test)]
mod tests {
    use tempfile::tempdir;

    use super::*;

    #[test]
    fn test_save_and_load_config() {
        let dir = tempdir().unwrap();
        let path = dir.path().join("test_config.toml");

        let config = Config::default();
        save_config(&config, &path).unwrap();

        let loaded = load_config(&path).unwrap();
        assert_eq!(config.server.listen_addr, loaded.server.listen_addr);
    }

    #[test]
    fn test_expand_path() {
        let expanded = expand_path("~/.config/test");
        assert!(!expanded.to_string_lossy().starts_with('~'));
    }

    #[test]
    fn test_get_user_config_path() {
        let path = get_user_config_path();
        assert!(path.to_string_lossy().contains("lamco-rdp-server"));
    }
}
