//! D-Bus Client for Server Management
//!
//! Connects to the lamco-rdp-server D-Bus service for remote management.
//! This provides an alternative to spawning a child process, enabling:
//! - Connection to an already-running server (systemd service)
//! - Management without process lifecycle concerns
//! - Proper service architecture (GUI as client, server as service)

use std::{collections::HashMap, time::Duration};

use tokio::sync::mpsc;
use zbus::{proxy, zvariant::OwnedValue, Connection};

use super::server_process::{LogLevel, ServerLogLine};

/// D-Bus service name (session bus)
const SERVICE_NAME: &str = "io.lamco.RdpServer";

/// D-Bus object path
const OBJECT_PATH: &str = "/io/lamco/RdpServer";

/// Proxy for the Manager interface
///
/// Generated by zbus::proxy macro for type-safe D-Bus calls.
#[proxy(
    interface = "io.lamco.RdpServer.Manager",
    default_service = "io.lamco.RdpServer",
    default_path = "/io/lamco/RdpServer"
)]
trait Manager {
    // Properties
    #[zbus(property)]
    fn version(&self) -> zbus::Result<String>;

    #[zbus(property)]
    fn status(&self) -> zbus::Result<String>;

    #[zbus(property)]
    fn listen_address(&self) -> zbus::Result<String>;

    #[zbus(property)]
    fn active_connections(&self) -> zbus::Result<u32>;

    #[zbus(property)]
    fn config_path(&self) -> zbus::Result<String>;

    #[zbus(property)]
    fn uptime(&self) -> zbus::Result<u64>;

    // Methods
    fn get_capabilities(&self) -> zbus::Result<HashMap<String, OwnedValue>>;

    fn get_service_registry(&self) -> zbus::Result<Vec<(String, String, u32)>>;

    fn get_config(&self) -> zbus::Result<String>;

    fn set_config(&self, config: &str) -> zbus::Result<(bool, String)>;

    fn reload_config(&self) -> zbus::Result<(bool, String)>;

    fn get_statistics(&self) -> zbus::Result<HashMap<String, OwnedValue>>;

    fn get_connections(&self) -> zbus::Result<Vec<(String, String, String, u64)>>;

    fn disconnect_client(&self, client_id: &str, reason: &str) -> zbus::Result<bool>;
}

/// D-Bus client for managing the server
pub struct DbusClient {
    connection: Connection,
    proxy: ManagerProxy<'static>,
    log_sender: Option<mpsc::UnboundedSender<ServerLogLine>>,
}

impl DbusClient {
    /// Try to connect to an existing D-Bus server
    ///
    /// Returns None if no server is available on the bus.
    pub async fn try_connect() -> Option<Self> {
        let connection = Connection::session().await.ok()?;

        let dbus_proxy = zbus::fdo::DBusProxy::new(&connection).await.ok()?;
        let names = dbus_proxy.list_names().await.ok()?;

        if !names.iter().any(|n| n.as_str() == SERVICE_NAME) {
            return None;
        }

        let proxy = ManagerProxy::new(&connection).await.ok()?;

        Some(Self {
            connection,
            proxy,
            log_sender: None,
        })
    }

    /// Connect to the D-Bus server, or return error
    pub async fn connect() -> Result<Self, String> {
        let connection = Connection::session()
            .await
            .map_err(|e| format!("Failed to connect to session bus: {}", e))?;

        let proxy = ManagerProxy::new(&connection)
            .await
            .map_err(|e| format!("Failed to create proxy: {}", e))?;

        let _ = proxy
            .status()
            .await
            .map_err(|e| format!("Service not responding: {}", e))?;

        Ok(Self {
            connection,
            proxy,
            log_sender: None,
        })
    }

    /// Set the log sender for forwarding log messages
    pub fn set_log_sender(&mut self, sender: mpsc::UnboundedSender<ServerLogLine>) {
        self.log_sender = Some(sender);
    }

    pub async fn version(&self) -> Result<String, String> {
        self.proxy
            .version()
            .await
            .map_err(|e| format!("Failed to get version: {}", e))
    }

    pub async fn status(&self) -> Result<String, String> {
        self.proxy
            .status()
            .await
            .map_err(|e| format!("Failed to get status: {}", e))
    }

    pub async fn is_running(&self) -> bool {
        matches!(self.status().await.as_deref(), Ok("running"))
    }

    pub async fn address(&self) -> Result<String, String> {
        self.proxy
            .listen_address()
            .await
            .map_err(|e| format!("Failed to get address: {}", e))
    }

    pub async fn active_connections(&self) -> Result<u32, String> {
        self.proxy
            .active_connections()
            .await
            .map_err(|e| format!("Failed to get connections: {}", e))
    }

    pub async fn uptime(&self) -> Result<Duration, String> {
        let secs = self
            .proxy
            .uptime()
            .await
            .map_err(|e| format!("Failed to get uptime: {}", e))?;
        Ok(Duration::from_secs(secs))
    }

    pub async fn get_config(&self) -> Result<String, String> {
        self.proxy
            .get_config()
            .await
            .map_err(|e| format!("Failed to get config: {}", e))
    }

    /// Update the server configuration
    pub async fn set_config(&self, config: &str) -> Result<(), String> {
        let (success, error) = self
            .proxy
            .set_config(config)
            .await
            .map_err(|e| format!("D-Bus call failed: {}", e))?;

        if success {
            Ok(())
        } else {
            Err(error)
        }
    }

    /// Reload configuration from disk
    pub async fn reload_config(&self) -> Result<(), String> {
        let (success, error) = self
            .proxy
            .reload_config()
            .await
            .map_err(|e| format!("D-Bus call failed: {}", e))?;

        if success {
            Ok(())
        } else {
            Err(error)
        }
    }

    /// Get runtime statistics
    pub async fn get_statistics(&self) -> Result<ServerStats, String> {
        let stats = self
            .proxy
            .get_statistics()
            .await
            .map_err(|e| format!("Failed to get statistics: {}", e))?;

        Ok(ServerStats {
            frames_encoded: extract_u64(&stats, "frames_encoded"),
            bytes_sent: extract_u64(&stats, "bytes_sent"),
            clients_total: extract_u64(&stats, "clients_total"),
            average_fps: extract_f64(&stats, "average_fps"),
            average_latency_ms: extract_f64(&stats, "average_latency_ms"),
        })
    }

    /// Get list of active connections
    pub async fn get_connections(&self) -> Result<Vec<ConnectionInfo>, String> {
        let conns = self
            .proxy
            .get_connections()
            .await
            .map_err(|e| format!("Failed to get connections: {}", e))?;

        Ok(conns
            .into_iter()
            .map(|(id, addr, user, time)| ConnectionInfo {
                client_id: id,
                peer_address: addr,
                username: user,
                connected_at: time,
            })
            .collect())
    }

    /// Disconnect a client
    pub async fn disconnect_client(&self, client_id: &str, reason: &str) -> Result<bool, String> {
        self.proxy
            .disconnect_client(client_id, reason)
            .await
            .map_err(|e| format!("Failed to disconnect client: {}", e))
    }

    /// Get service registry (all services with levels)
    pub async fn get_service_registry(&self) -> Result<Vec<ServiceInfo>, String> {
        let services = self
            .proxy
            .get_service_registry()
            .await
            .map_err(|e| format!("Failed to get service registry: {}", e))?;

        Ok(services
            .into_iter()
            .map(|(id, name, level)| ServiceInfo {
                service_id: id,
                service_name: name,
                level,
            })
            .collect())
    }

    /// Check if D-Bus service is available
    pub async fn is_service_available() -> bool {
        if let Ok(connection) = Connection::session().await {
            if let Ok(dbus_proxy) = zbus::fdo::DBusProxy::new(&connection).await {
                if let Ok(names) = dbus_proxy.list_names().await {
                    return names.iter().any(|n| n.as_str() == SERVICE_NAME);
                }
            }
        }
        false
    }

    /// Send a synthetic log message (for GUI display)
    fn send_log(&self, level: LogLevel, message: String) {
        if let Some(ref sender) = self.log_sender {
            let log_line = ServerLogLine {
                level,
                message,
                timestamp: chrono::Local::now().format("%H:%M:%S").to_string(),
            };
            let _ = sender.send(log_line);
        }
    }
}

/// Server statistics
#[derive(Debug, Clone, Default)]
pub struct ServerStats {
    pub frames_encoded: u64,
    pub bytes_sent: u64,
    pub clients_total: u64,
    pub average_fps: f64,
    pub average_latency_ms: f64,
}

/// Connection information
#[derive(Debug, Clone)]
pub struct ConnectionInfo {
    pub client_id: String,
    pub peer_address: String,
    pub username: String,
    pub connected_at: u64,
}

/// Service information from registry
#[derive(Debug, Clone)]
pub struct ServiceInfo {
    pub service_id: String,
    pub service_name: String,
    /// 0=Unavailable, 1=Degraded, 2=BestEffort, 3=Guaranteed
    pub level: u32,
}

/// Helper to extract u64 from OwnedValue HashMap
fn extract_u64(map: &HashMap<String, OwnedValue>, key: &str) -> u64 {
    map.get(key)
        .and_then(|v| v.downcast_ref::<u64>().ok())
        .unwrap_or(0)
}

/// Helper to extract f64 from OwnedValue HashMap
fn extract_f64(map: &HashMap<String, OwnedValue>, key: &str) -> f64 {
    map.get(key)
        .and_then(|v| v.downcast_ref::<f64>().ok())
        .unwrap_or(0.0)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn test_service_available_check() {
        // This test just verifies the function doesn't panic
        // Actual availability depends on whether the service is running
        let _ = DbusClient::is_service_available().await;
    }
}
