//! Capabilities Singleton
//!
//! **Execution Path:** System capability probing
//! **Status:** Active (v1.0.0+)
//! **Platform:** Universal
//! **Role:** Centralized capability detection and state management
//!
//! Provides centralized capability state management with singleton access pattern.

use std::sync::{Arc, OnceLock};

use chrono::Utc;
use thiserror::Error;
use tokio::sync::RwLock;
use tracing::{debug, info};

use crate::capabilities::{
    probes::{DisplayProbe, EncodingProbe, InputProbe, NetworkProbe, RenderingProbe, StorageProbe},
    state::{
        BlockingIssue, Degradation, MinimumViableConfig, ServiceLevel, Subsystem,
        SystemCapabilities, UserImpact,
    },
};

static INSTANCE: OnceLock<Arc<RwLock<Capabilities>>> = OnceLock::new();

/// Error during capabilities initialization
#[derive(Debug, Error)]
pub enum InitializationError {
    /// Capabilities already initialized
    #[error("Capabilities already initialized")]
    AlreadyInitialized,

    /// Probe failed during initialization
    #[error("Probe failed: {0}")]
    ProbeFailed(String),
}

/// System capabilities singleton
///
/// Manages the lifecycle of capability detection and provides access to
/// current system capabilities. Probed once at startup and accessible globally.
pub struct Capabilities {
    /// Current capability state
    pub state: SystemCapabilities,

    /// Whether initialization completed successfully
    initialized: bool,
}

impl Capabilities {
    /// Must be called once at startup before any capability queries.
    ///
    /// # Errors
    ///
    /// Returns an error if already initialized or if probing fails critically.
    pub async fn initialize() -> Result<(), InitializationError> {
        info!("Initializing capability manager...");

        let manager = Self::probe_all().await?;

        info!("{}", manager.diagnostic_summary());

        INSTANCE
            .set(Arc::new(RwLock::new(manager)))
            .map_err(|_| InitializationError::AlreadyInitialized)?;

        info!("Capability manager initialized successfully");
        Ok(())
    }

    /// # Panics
    ///
    /// Panics if called before `initialize()`.
    pub fn global() -> Arc<RwLock<Capabilities>> {
        INSTANCE
            .get()
            .expect("Capabilities::initialize() must be called first")
            .clone()
    }

    pub fn is_initialized() -> bool {
        INSTANCE.get().is_some()
    }

    async fn probe_all() -> Result<Self, InitializationError> {
        debug!("Probing display capabilities...");
        let display = DisplayProbe::probe().await;

        debug!("Probing encoding capabilities...");
        let encoding = EncodingProbe::probe().await;

        debug!("Probing input capabilities...");
        let input = InputProbe::probe().await;

        debug!("Probing storage capabilities...");
        let storage = StorageProbe::probe().await;

        debug!("Probing rendering capabilities...");
        let rendering = RenderingProbe::probe().await;

        debug!("Probing network capabilities...");
        let network = NetworkProbe::probe().await;

        let minimum_viable = Self::compute_minimum_viable(&display, &encoding, &input, &rendering);
        let degradations =
            Self::collect_degradations(&display, &encoding, &input, &storage, &rendering);
        let blocking_issues =
            Self::collect_blocking_issues(&display, &encoding, &input, &rendering);

        let state = SystemCapabilities {
            probed_at: Utc::now(),
            display,
            encoding,
            input,
            storage,
            rendering,
            network,
            minimum_viable,
            degradations,
            blocking_issues,
        };

        Ok(Self {
            state,
            initialized: true,
        })
    }

    /// Re-probe a specific subsystem
    ///
    /// Useful when conditions may have changed (e.g., after installing drivers).
    pub async fn reprobe(&mut self, subsystem: Subsystem) {
        info!("Re-probing {:?} subsystem...", subsystem);

        match subsystem {
            Subsystem::Display => self.state.display = DisplayProbe::probe().await,
            Subsystem::Encoding => self.state.encoding = EncodingProbe::probe().await,
            Subsystem::Input => self.state.input = InputProbe::probe().await,
            Subsystem::Storage => self.state.storage = StorageProbe::probe().await,
            Subsystem::Rendering => self.state.rendering = RenderingProbe::probe().await,
            Subsystem::Network => self.state.network = NetworkProbe::probe().await,
        }

        self.recompute_derived_state();
        self.state.probed_at = Utc::now();
    }

    fn recompute_derived_state(&mut self) {
        self.state.minimum_viable = Self::compute_minimum_viable(
            &self.state.display,
            &self.state.encoding,
            &self.state.input,
            &self.state.rendering,
        );
        self.state.degradations = Self::collect_degradations(
            &self.state.display,
            &self.state.encoding,
            &self.state.input,
            &self.state.storage,
            &self.state.rendering,
        );
        self.state.blocking_issues = Self::collect_blocking_issues(
            &self.state.display,
            &self.state.encoding,
            &self.state.input,
            &self.state.rendering,
        );
    }

    fn compute_minimum_viable(
        display: &crate::capabilities::probes::DisplayCapabilities,
        encoding: &crate::capabilities::probes::EncodingCapabilities,
        input: &crate::capabilities::probes::InputCapabilities,
        rendering: &crate::capabilities::probes::RenderingCapabilities,
    ) -> MinimumViableConfig {
        MinimumViableConfig {
            can_capture_screen: display.service_level.is_operational(),
            can_encode_video: encoding.service_level.is_operational(),
            can_inject_input: input.service_level.is_operational(),
            can_serve_rdp: display.service_level.is_operational()
                && encoding.service_level.is_operational(),
            can_render_gui: rendering.service_level.is_operational(),
        }
    }

    fn collect_degradations(
        display: &crate::capabilities::probes::DisplayCapabilities,
        encoding: &crate::capabilities::probes::EncodingCapabilities,
        input: &crate::capabilities::probes::InputCapabilities,
        storage: &crate::capabilities::probes::StorageCapabilities,
        rendering: &crate::capabilities::probes::RenderingCapabilities,
    ) -> Vec<Degradation> {
        let mut degradations = Vec::new();

        if display.service_level == ServiceLevel::Degraded {
            degradations.push(Degradation {
                subsystem: Subsystem::Display,
                feature: "Screen capture".into(),
                reason: display.degradation_reason.clone().unwrap_or_default(),
                fallback: "Using fallback capture method".into(),
                user_impact: UserImpact::Minor,
            });
        }

        if encoding.service_level == ServiceLevel::Fallback {
            degradations.push(Degradation {
                subsystem: Subsystem::Encoding,
                feature: "Video encoding".into(),
                reason: "No hardware encoder available".into(),
                fallback: "Using software encoding (OpenH264)".into(),
                user_impact: UserImpact::Moderate,
            });
        }

        // Clipboard degradation (Portal v1)
        if !display.portal.supports_clipboard {
            degradations.push(Degradation {
                subsystem: Subsystem::Display,
                feature: "Clipboard synchronization".into(),
                reason: "Portal version does not support clipboard".into(),
                fallback: "Clipboard disabled".into(),
                user_impact: UserImpact::Moderate,
            });
        }

        if input.service_level == ServiceLevel::Fallback {
            degradations.push(Degradation {
                subsystem: Subsystem::Input,
                feature: "Input injection".into(),
                reason: "Using Portal fallback".into(),
                fallback: "Permission dialog on each start".into(),
                user_impact: UserImpact::Minor,
            });
        }

        if rendering.service_level == ServiceLevel::Fallback {
            degradations.push(Degradation {
                subsystem: Subsystem::Rendering,
                feature: "GUI rendering".into(),
                reason: rendering.fallback_reason.clone().unwrap_or_default(),
                fallback: "Using software rendering".into(),
                user_impact: UserImpact::Minor,
            });
        }

        if storage.service_level == ServiceLevel::Fallback {
            degradations.push(Degradation {
                subsystem: Subsystem::Storage,
                feature: "Credential storage".into(),
                reason: "Secure storage not available".into(),
                fallback: "Using encrypted file storage".into(),
                user_impact: UserImpact::Minimal,
            });
        }

        degradations
    }

    fn collect_blocking_issues(
        display: &crate::capabilities::probes::DisplayCapabilities,
        encoding: &crate::capabilities::probes::EncodingCapabilities,
        input: &crate::capabilities::probes::InputCapabilities,
        _rendering: &crate::capabilities::probes::RenderingCapabilities,
    ) -> Vec<BlockingIssue> {
        let mut issues = Vec::new();

        if display.service_level == ServiceLevel::Unavailable {
            issues.push(BlockingIssue {
                subsystem: Subsystem::Display,
                issue: "Cannot capture screen".into(),
                reason: display.unavailable_reason.clone().unwrap_or("Unknown".into()),
                suggestions: vec![
                    "Ensure a Wayland compositor is running".into(),
                    "Check that XDG Desktop Portal is installed".into(),
                    "Verify portal services are running: systemctl --user status xdg-desktop-portal".into(),
                ],
            });
        }

        if encoding.service_level == ServiceLevel::Unavailable {
            issues.push(BlockingIssue {
                subsystem: Subsystem::Encoding,
                issue: "Cannot encode video".into(),
                reason: "No encoder backends available".into(),
                suggestions: vec![
                    "Install VA-API drivers for your GPU".into(),
                    "For NVIDIA: install nvidia-vaapi-driver".into(),
                    "Ensure h264 feature is compiled in".into(),
                ],
            });
        }

        if input.service_level == ServiceLevel::Unavailable {
            issues.push(BlockingIssue {
                subsystem: Subsystem::Input,
                issue: "Cannot inject input".into(),
                reason: input.unavailable_reason.clone().unwrap_or("Unknown".into()),
                suggestions: vec![
                    "Ensure XDG Desktop Portal is running".into(),
                    "Check portal RemoteDesktop interface availability".into(),
                ],
            });
        }

        issues
    }

    pub fn diagnostic_summary(&self) -> String {
        let mut summary = String::new();

        summary.push_str("╭───────────────────────────────────────────╮\n");
        summary.push_str("│         Capability Summary                │\n");
        summary.push_str("├───────────────────────────────────────────┤\n");
        summary.push_str(&format!(
            "│  🖥️  Display:   {:12} │\n",
            self.state.display.service_level.name()
        ));
        summary.push_str(&format!(
            "│  🎬 Encoding:  {:12} │\n",
            self.state.encoding.service_level.name()
        ));
        summary.push_str(&format!(
            "│  ⌨️  Input:     {:12} │\n",
            self.state.input.service_level.name()
        ));
        summary.push_str(&format!(
            "│  🔐 Storage:   {:12} │\n",
            self.state.storage.service_level.name()
        ));
        summary.push_str(&format!(
            "│  🎨 Rendering: {:12} │\n",
            self.state.rendering.service_level.name()
        ));
        summary.push_str(&format!(
            "│  🌐 Network:   {:12} │\n",
            self.state.network.service_level.name()
        ));
        summary.push_str("├───────────────────────────────────────────┤\n");

        if !self.state.degradations.is_empty() {
            summary.push_str(&format!(
                "│  ⚠️  {} degradation(s) active               │\n",
                self.state.degradations.len()
            ));
        }

        if !self.state.blocking_issues.is_empty() {
            summary.push_str(&format!(
                "│  ❌ {} blocking issue(s)                   │\n",
                self.state.blocking_issues.len()
            ));
        }

        let can_server = if self.state.minimum_viable.can_operate_server() {
            "✅"
        } else {
            "❌"
        };
        let can_gui = if self.state.minimum_viable.can_operate_gui() {
            "✅"
        } else {
            "❌"
        };
        summary.push_str(&format!(
            "│  Server: {can_server} GUI: {can_gui}                       │\n"
        ));
        summary.push_str("╰───────────────────────────────────────────╯");

        summary
    }

    pub fn to_json(&self) -> serde_json::Result<String> {
        serde_json::to_string_pretty(&self.state)
    }
}
