//! Clipboard events for async processing.
//!
//! The IronRDP [`CliprdrBackend`](ironrdp_cliprdr::backend::CliprdrBackend) trait methods
//! are called synchronously. This module provides event types that allow queueing
//! operations for asynchronous processing.

use ironrdp_cliprdr::pdu::{
    ClipboardFormat as RdpClipboardFormat, ClipboardFormatId, ClipboardGeneralCapabilityFlags, FileContentsFlags,
    FileContentsRequest, FileContentsResponse, FormatDataRequest, FormatDataResponse, LockDataId,
};
use std::collections::VecDeque;
use std::sync::{Arc, RwLock};

/// Events generated by the clipboard backend for async processing.
#[derive(Debug, Clone)]
pub enum ClipboardEvent {
    /// Backend is ready (channel initialized)
    Ready,

    /// Request to send format list
    RequestFormatList,

    /// Negotiated capabilities received
    NegotiatedCapabilities(ClipboardGeneralCapabilityFlags),

    /// Remote clipboard content changed (remote copy)
    RemoteCopy {
        /// Available formats from remote
        formats: Vec<RdpClipboardFormat>,
    },

    /// Remote requests format data
    FormatDataRequest {
        /// The requested format ID
        format_id: ClipboardFormatId,
    },

    /// Remote sent format data response
    FormatDataResponse {
        /// The format data (owned copy)
        data: Vec<u8>,
        /// Whether this is an error response
        is_error: bool,
    },

    /// Remote requests file contents
    FileContentsRequest {
        /// Stream ID for correlation
        stream_id: u32,
        /// File index
        index: u32,
        /// Byte offset
        position: u64,
        /// Requested size
        size: u32,
        /// Whether this is a size request (vs data request)
        is_size_request: bool,
    },

    /// Remote sent file contents response
    FileContentsResponse {
        /// Stream ID for correlation
        stream_id: u32,
        /// File data (owned copy)
        data: Vec<u8>,
        /// Whether this is an error response
        is_error: bool,
    },

    /// Clipboard lock request
    Lock {
        /// Data ID to lock
        data_id: u32,
    },

    /// Clipboard unlock request
    Unlock {
        /// Data ID to unlock
        data_id: u32,
    },
}

impl ClipboardEvent {
    /// Create a RemoteCopy event from RDP format list
    pub fn remote_copy(formats: &[RdpClipboardFormat]) -> Self {
        Self::RemoteCopy {
            formats: formats.to_vec(),
        }
    }

    /// Create a FormatDataRequest event
    pub fn format_data_request(request: &FormatDataRequest) -> Self {
        Self::FormatDataRequest {
            format_id: request.format,
        }
    }

    /// Create a FormatDataResponse event
    pub fn format_data_response(response: &FormatDataResponse<'_>) -> Self {
        Self::FormatDataResponse {
            data: response.data().to_vec(),
            is_error: response.is_error(),
        }
    }

    /// Create a FileContentsRequest event
    pub fn file_contents_request(request: &FileContentsRequest) -> Self {
        Self::FileContentsRequest {
            stream_id: request.stream_id,
            index: request.index,
            position: request.position,
            size: request.requested_size,
            is_size_request: request.flags.contains(FileContentsFlags::SIZE),
        }
    }

    /// Create a FileContentsResponse event
    pub fn file_contents_response(response: &FileContentsResponse<'_>) -> Self {
        // Note: FileContentsResponse doesn't expose is_error publicly,
        // so we infer error state from empty data
        Self::FileContentsResponse {
            stream_id: response.stream_id(),
            data: response.data().to_vec(),
            is_error: response.data().is_empty(),
        }
    }

    /// Create a Lock event
    pub fn lock(data_id: LockDataId) -> Self {
        Self::Lock { data_id: data_id.0 }
    }

    /// Create an Unlock event
    pub fn unlock(data_id: LockDataId) -> Self {
        Self::Unlock { data_id: data_id.0 }
    }
}

/// Sender side of the clipboard event channel.
///
/// This is a simple queue-based sender that uses `RwLock` for thread-safety.
/// Events are queued for later processing by an async task.
#[derive(Debug, Clone)]
pub struct ClipboardEventSender {
    queue: Arc<RwLock<VecDeque<ClipboardEvent>>>,
}

impl ClipboardEventSender {
    /// Create a new event sender
    pub fn new() -> Self {
        Self {
            queue: Arc::new(RwLock::new(VecDeque::new())),
        }
    }

    /// Send an event (non-blocking, queues for later processing)
    pub fn send(&self, event: ClipboardEvent) {
        if let Ok(mut queue) = self.queue.try_write() {
            queue.push_back(event);
        } else {
            tracing::warn!("Failed to acquire clipboard event queue lock");
        }
    }

    /// Create a receiver that shares the same queue
    pub fn subscribe(&self) -> ClipboardEventReceiver {
        ClipboardEventReceiver {
            queue: Arc::clone(&self.queue),
        }
    }
}

impl Default for ClipboardEventSender {
    fn default() -> Self {
        Self::new()
    }
}

/// Receiver side of the clipboard event channel.
#[derive(Debug, Clone)]
pub struct ClipboardEventReceiver {
    queue: Arc<RwLock<VecDeque<ClipboardEvent>>>,
}

impl ClipboardEventReceiver {
    /// Try to receive the next event (non-blocking)
    pub fn try_recv(&self) -> Option<ClipboardEvent> {
        self.queue.try_write().ok()?.pop_front()
    }

    /// Drain all pending events
    pub fn drain(&self) -> Vec<ClipboardEvent> {
        self.queue
            .try_write()
            .ok()
            .map(|mut q| q.drain(..).collect())
            .unwrap_or_default()
    }

    /// Check if there are pending events
    pub fn has_pending(&self) -> bool {
        self.queue.try_read().map(|q| !q.is_empty()).unwrap_or(false)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_event_channel() {
        let sender = ClipboardEventSender::new();
        let receiver = sender.subscribe();

        sender.send(ClipboardEvent::Ready);
        sender.send(ClipboardEvent::RequestFormatList);

        assert!(receiver.has_pending());

        let events = receiver.drain();
        assert_eq!(events.len(), 2);
        assert!(matches!(events[0], ClipboardEvent::Ready));
        assert!(matches!(events[1], ClipboardEvent::RequestFormatList));

        assert!(!receiver.has_pending());
    }

    #[test]
    fn test_try_recv() {
        let sender = ClipboardEventSender::new();
        let receiver = sender.subscribe();

        assert!(receiver.try_recv().is_none());

        sender.send(ClipboardEvent::Ready);

        let event = receiver.try_recv();
        assert!(matches!(event, Some(ClipboardEvent::Ready)));
        assert!(receiver.try_recv().is_none());
    }
}
