#!/usr/bin/env python3
#
#    Copyright (C) 1999-2006  Keith Dart <keith@kdart.com>
#    Copyright (C) 2008-2009  Dan O'Reilly <oreilldf@gmail.com>
#    Copyright (C) 2024       Takahiro Yoshizawa <kuro@takahiro.org>
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
"""
Managing logfile rotation. A ManagedLog object is a file-like object that
rotates itself when a maximum size is reached.
"""

import sys
import io
import os
import time
import ctypes


# -----------------------
# Helper function to flush the buffer on the C library side
def clear_c_stdio():
    """
    Flush all stream buffers on the C library side.
    Note: This does not reopen the output destination; it simply writes out
          the contents and clears them.
    """
    libc = ctypes.CDLL(None)
    libc.fflush(None)


# -----------------------
# Custom exception class
class SizeError(IOError):
    """ Raised when the log file size exceeds the maximum allowed size. """
    pass


# -----------------------
# LogFile: A file object that adds processing such as timestamps to a binary stream
class LogFile(io.FileIO):
    def __init__(self, name, mode="a", maxsize=360000):
        super(LogFile, self).__init__(name, mode)
        self.maxsize = maxsize
        self.eol = True
        try:
            self.written = os.fstat(self.fileno())[6]
        except OSError:
            self.written = 0

    def write(self, data):
        self.written += len(data)
        if len(data) <= 0:
            return
        if self.eol:
            super(LogFile, self).write(self.get_time().encode("utf-8") + b' :: ')
            self.eol = False
        if data[-1] == '\n' or (isinstance(data, bytes) and data[-1] == ord('\n')):
            self.eol = True
            data = data[:-1]
        if isinstance(data, bytes):
            timestamp = self.get_time().encode('utf-8')
            data = data.replace(b'\n', b'\n' + timestamp + b' :: ')
        else:
            timestamp = self.get_time()
            data = data.replace('\n', '\n' + timestamp + ' :: ')
            data = data.encode('utf-8')
        super(LogFile, self).write(data)
        if self.eol:
            super(LogFile, self).write(b'\n')
        self.flush()
        if self.written > self.maxsize:
            raise SizeError

    def get_time(self):
        x = time.localtime()
        return ''.join([
            str(x[0]).rjust(4, '0'), '/',
            str(x[1]).rjust(2, '0'), '/',
            str(x[2]).rjust(2, '0'), ' ',
            str(x[3]).rjust(2, '0'), ':',
            str(x[4]).rjust(2, '0'), ':',
            str(x[5]).rjust(2, '0')
        ])

    def rotate(self):
        return rotate(self)

    def note(self, text):
        self.write("\n#*===== %s =====\n" % (text,))


# -----------------------
# Log file rotation process
def shiftlogs(basename, maxsave):
    topname = "%s.%d" % (basename, maxsave)
    if os.path.isfile(topname):
        os.unlink(topname)
    for i in range(maxsave, 0, -1):
        oldname = "%s.%d" % (basename, i)
        newname = "%s.%d" % (basename, i+1)
        try:
            os.rename(oldname, newname)
        except OSError:
            pass
    try:
        os.rename(basename, "%s.1" % (basename))
    except OSError:
        pass


def rotate(fileobj, maxsave=9):
    """
    Rotate fileobj.
    Before rotation, synchronize with flush() and fsync(), then switch 
    existing file names using shiftlogs(), and create a new log file in mode "w".
    """
    name = fileobj.name
    mode = "w"
    maxsize = fileobj.maxsize
    fileobj.flush()
    try:
        os.fsync(fileobj.fileno())
    except OSError:
        pass
    fileobj.close()

    shiftlogs(name, maxsave)
    new_file = LogFile(name, mode, maxsize)
    new_file.flush()
    try:
        os.fsync(new_file.fileno())
    except OSError:
        pass
    return new_file


# -----------------------
# ManagedLog: A class that performs automatic rotation and writing
# using LogFile
class ManagedLog(object):
    def __init__(self, name, maxsize=360000, maxsave=3):
        if not os.path.exists(os.path.dirname(name)):
            os.makedirs(os.path.dirname(name))
        self._lf = LogFile(name, "a", maxsize)
        self.maxsave = maxsave

    def __repr__(self):
        return "%s(%r, %r, %r)" % (self.__class__.__name__, self._lf.name,
                                   self._lf.maxsize, self.maxsave)

    def write(self, data):
        try:
            self._lf.write(data)
        except SizeError:
            self._lf = rotate(self._lf, self.maxsave)

    def note(self, data):
        try:
            self._lf.note(data)
        except SizeError:
            self._lf = rotate(self._lf, self.maxsave)

    def written(self):
        return self._lf.written

    def rotate(self):
        self._lf = rotate(self._lf, self.maxsave)

    def __getattr__(self, name):
        return getattr(self._lf, name)


# -----------------------
# ManagedTextIOWrapper: A mechanism that inherits from TextIOWrapper to
# detect SizeError
# during writing and trigger the rotation process
class ManagedTextIOWrapper(io.TextIOWrapper):
    def __init__(self, log_obj, encoding="utf-8", managed_stdio=None):
        super().__init__(log_obj, encoding=encoding, write_through=True)
        self.managed_stdio = managed_stdio

    def write(self, s):
        try:
            result = super().write(s)
            self.flush()
            return result
        except Exception as e:
            # If a SizeError (or its wrapped exception) occurs during write(),
            # perform rotation
            if isinstance(e, SizeError):
                # Rotate `_lf` on the ManagedStdio side
                new_lf = rotate(self.managed_stdio._lf,
                                self.managed_stdio.maxsave)
                self.managed_stdio._lf = new_lf
                # Generate a new TextIOWrapper and replace the old one
                new_wrapper = ManagedTextIOWrapper(new_lf,
                                                   encoding=self.encoding,
                                                   managed_stdio=self.managed_stdio)
                self.managed_stdio._stream = new_wrapper
                sys.stdout = sys.stderr = new_wrapper
                # Write again
                return new_wrapper.write(s)
            else:
                raise


# -----------------------
# ManagedStdio: A class that manages stdout/stderr using ManagedTextIOWrapper
class ManagedStdio(ManagedLog):
    def __init__(self, name, maxsize=360000, maxsave=3):
        super().__init__(name, maxsize, maxsave)
        # Initially, wrap LogFile with TextIOWrapper and store it in _stream
        self._stream = ManagedTextIOWrapper(self._lf, encoding="utf-8",
                                            managed_stdio=self)
        sys.stdout = sys.stderr = self._stream

    def write(self, data):
        # All writes go through _stream
        try:
            result = self._stream.write(data)
            return result
        except SizeError:
            # Since the rotation process should be handled
            # within ManagedTextIOWrapper
            return self._stream.write(data)


# -----------------------
# `open` / `writelog` functions as a simple interface
def open(name, maxsize=360000, maxsave=9):
    """ Open logfile as a ManagedLog """
    return ManagedLog(name, maxsize, maxsave)


def writelog(logobj, data):
    try:
        logobj.write(data)
    except SizeError:
        return rotate(logobj)
    else:
        return logobj
