#!/usr/bin/env python3
# vim: set fileencoding=utf-8
#
#   Copyright (C) 2007 - 2009 Adam Blackburn
#   Copyright (C) 2007 - 2009 Dan O'Reilly
#   Copyright (C) 2021        Andreas Messer
#   Copyright (C) 2024        Takahiro Yoshizawa
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License Version 2 as
#   published by the Free Software Foundation.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import sys
import os
import atexit

import gi
from dbus import DBusException

# --- GTK / GObject / Pango ---
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk as gtk
from gi.repository import GLib as gobject

gi.require_version('Pango', '1.0')
from gi.repository import Pango as pango

# --- AppIndicator (optional) ---
try:
    gi.require_version('AyatanaAppIndicator3', '0.1')
    from gi.repository import AyatanaAppIndicator3 as appindicator
    USE_APP_INDICATOR = True
except Exception:
    USE_APP_INDICATOR = False

# --- Desktop notifications (optional) ---
try:
    import notify2 as pynotify
    HAS_NOTIFY = pynotify.init("Wicd")
except Exception:
    HAS_NOTIFY = False

# --- Wicd specific imports ---
import wicd.commandline
import wicd.errors
from wicd import wpath, misc
from wicd.dbus import dbus_manager
from . import gui
from .guiutil import error, can_use_notify
from wicd.translations import _

ICON_AVAIL = True
USE_EGG = False

# Use egg.trayicon only if gtk.StatusIcon is missing (legacy)
if not hasattr(gtk, "StatusIcon"):
    try:
        import egg.trayicon  # noqa: F401
        USE_EGG = True
    except Exception:
        print(
            'Unable to load tray icon: Missing both egg.trayicon and gtk.StatusIcon modules.'
        )
        ICON_AVAIL = False

misc.RenameProcess("wicd-client")

daemon = wireless = wired = lost_dbus_id = None
DBUS_AVAIL = False

theme = gtk.IconTheme.get_default()
theme.append_search_path(wpath.images)


# -------------------------
# Helpers
# -------------------------

def catchdbus(func):
    """Decorator to catch DBus exceptions and degrade gracefully."""
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except DBusException as e:
            name = None
            try:
                name = e.get_dbus_name()
            except Exception:
                pass
            if name and "AccessDenied" in name:
                error(
                    None,
                    _('Unable to contact the Wicd daemon due to an access denied error from DBus. ' \
                      'Please check that your user is in the $A group.').replace("$A", "<b>" + wpath.wicd_group + "</b>")
                )
                raise
            else:
                # Non-fatal DBus errors are ignored
                print("warning: ignoring DBus exception:", e)
                return None
    return wrapper


class NetworkMenuItem(gtk.ImageMenuItem):
    """Network menu item with optional bold label for active network."""
    def __init__(self, lbl, is_active=False):
        super().__init__()
        self.label = gtk.Label()
        self.label.set_label(lbl)
        if is_active:
            atrlist = pango.AttrList()
            attr = pango.attr_weight_new(pango.Weight.BOLD)
            attr.start_index = 0
            attr.end_index = 50
            atrlist.insert(attr)
            self.label.set_attributes(atrlist)
        self.label.set_justify(gtk.Justification.LEFT)
        self.label.set_halign(gtk.Align.START)
        self.label.set_valign(gtk.Align.START)
        self.add(self.label)
        self.label.show()


# -------------------------
# Main TrayIcon facade
# -------------------------
class TrayIcon(object):
    """Owns the UI (TrayIconGUI) and the state updater (TrayConnectionInfo)."""

    def __init__(self, animate, displaytray=True, displayapp=False):
        # bandwidth accounting (shared with TrayConnectionInfo when animate=True)
        self.cur_sndbytes = -1
        self.cur_rcvbytes = -1
        self.last_sndbytes = -1
        self.last_rcvbytes = -1
        self.max_snd_gain = 10000
        self.max_rcv_gain = 10000

        # 1) Build GUI first (it does not assume icon_info exists)
        if USE_APP_INDICATOR:
            self.tr = self.IndicatorTrayIconGUI(self)
        elif USE_EGG:
            # Historical code path (egg.trayicon). Not implemented here.
            # Fall back to StatusTrayIconGUI
            self.tr = self.StatusTrayIconGUI(self)
        else:
            self.tr = self.StatusTrayIconGUI(self)

        # 2) Then build the connection state manager and attach it to GUI
        self.icon_info = self.TrayConnectionInfo(self, self.tr, animate)
        self.tr.icon_info = self.icon_info

        if displayapp:
            self.tr.toggle_wicd_gui()

        self.tr.visible(displaytray)

    def is_embedded(self):
        if USE_EGG:
            raise NotImplementedError()
        # gtk.StatusIcon provides is_embedded; AppIndicator doesn't need this.
        return getattr(self.tr, 'is_embedded', lambda: True)()

    def get_bandwidth_bytes(self):
        """Update byte counters for bandwidth animation."""
        dev_dir = '/sys/class/net/'
        iface = daemon.GetCurrentInterface()
        path = None
        for fldr in os.listdir(dev_dir):
            if fldr == iface:
                path = os.path.join(dev_dir, fldr, 'statistics')
                break
        try:
            if not path:
                raise FileNotFoundError
            with open(os.path.join(path, 'rx_bytes')) as f:
                self.cur_rcvbytes = int(f.read().strip())
            with open(os.path.join(path, 'tx_bytes')) as f:
                self.cur_sndbytes = int(f.read().strip())
        except Exception:
            self.cur_sndbytes = -1
            self.cur_rcvbytes = -1

    # ---------------------
    # Connection Info logic
    # ---------------------
    class TrayConnectionInfo(object):
        """Updates the tray icon state, tooltip, and notifications."""

        def __init__(self, parent, tr, animate=True):
            self.parent = parent
            self.tr = tr
            self.animate = animate

            # network-related fields
            self.network_name = ''    # SSID
            self.network_type = 'none'  # 'wired' | 'wireless' | 'none' | 'killswitch' | 'no_daemon'
            self.network_str = ''     # Signal Strength (formatted)
            self.network_addr = '0.0.0.0'
            self.network_br = ''

            # DBus-driven state
            self._last_bubble = None
            self.last_state = None
            self.should_notify = True

            if DBUS_AVAIL:
                self.update_tray_icon()
            else:
                handle_no_dbus()
                self.set_not_connected_state()

            # Initial tooltip
            self.update_tooltip()

        def update_tooltip(self):
            """Set tooltip/menu text based on current connection state."""
            # AppIndicator uses a menu item in place of tooltips
            if not hasattr(self.tr, 'tooltip'):
                return True

            if self.network_type == 'none':
                self.tr.tooltip = _('Not connected')
            elif self.network_type == 'wireless':
                self.tr.tooltip = (
                    _('Connected to $A at $B (IP: $C)')
                    .replace('$A', misc.from_escape_to_str(self.network_name))
                    .replace('$B', self.network_str)
                    .replace('$C', self.network_addr)
                )
            elif self.network_type == 'wired':
                self.tr.tooltip = (
                    _('Connected to wired network (IP: $A)')
                    .replace('$A', self.network_addr)
                )
            elif self.network_type == 'killswitch':
                self.tr.tooltip = _("Not connected") + " (" + _('Wireless Kill Switch Enabled') + ")"
            elif self.network_type == 'no_daemon':
                self.tr.tooltip = _('Wicd daemon unreachable')
            return True

        def _show_notification(self, title, details, image=None):
            if not HAS_NOTIFY or not can_use_notify():
                return
            if title is None:
                title = ""
            if details is None:
                details = ""
            try:
                if not self._last_bubble:
                    self._last_bubble = pynotify.Notification(title, details, image)
                    self._last_bubble.show()
                else:
                    self._last_bubble.update(title, details, image)
                    self._last_bubble.show()
            except Exception as e:
                print("Notification error:", e)

        @catchdbus
        def wired_profile_chooser(self):
            gui.WiredProfileChooser()
            daemon.SetNeedWiredProfileChooser(False)

        def set_wired_state(self, info):
            wired_ip = info[0]
            self.network_addr = str(wired_ip)
            self.network_type = 'wired'
            self.tr.set_from_name('wired')
            self.tr.tooltip = _('Connected to wired network (IP: $A)').replace('$A', str(wired_ip))
            if self.should_notify:
                self._show_notification(_('Wired Network'), _('Connection established'), 'network-wired')
            self.update_tooltip()

        @catchdbus
        def set_wireless_state(self, info):
            if len(info) < 5:
                return
            wireless_ip = info[0]
            self.network_name = info[1]
            strength = info[2]
            cur_net_id = int(info[3])
            sig_string = daemon.FormatSignalForPrinting(str(strength))

            self.network_type = 'wireless'
            self.network_addr = str(wireless_ip)
            self.network_str = sig_string
            self.network_br = info[4]

            lock = ''
            if wireless.GetWirelessProperty(cur_net_id, 'encryption'):
                lock = '-lock'
                status_string = (
                    _('Connected to $A at $B (IP: $C)')
                    .replace('$A', misc.from_escape_to_str(self.network_name))
                    .replace('$B', sig_string)
                    .replace('$C', str(wireless_ip))
                )
            else:
                status_string = _('Not connected')

            self.tr.tooltip = status_string
            self.set_signal_image(int(strength), lock)
            if self.should_notify:
                self._show_notification(misc.from_escape_to_str(self.network_name), _('Connection established'), 'network-wireless')
            self.update_tooltip()

        def set_connecting_state(self, info):
            if info and info[0] == 'wired' and len(info) == 1:
                cur_network = _('Wired Network')
                wired = True
            else:
                cur_network = info[1] if info and len(info) > 1 else _('Network')
                wired = False
            self.tr.tooltip = _('Connecting') + " to " + cur_network + "..."
            self.tr.set_from_name('no-signal')
            self._show_notification(cur_network, _('Establishing connection...'), 'network-wired' if wired else 'network-wireless')
            self.update_tooltip()

        @catchdbus
        def set_not_connected_state(self, info=None):
            self.tr.set_from_name('no-signal')
            if not DBUS_AVAIL:
                status = _('Wicd daemon unreachable')
                self.network_type = 'no_daemon'
            else:
                status = _('Not connected')
                try:
                    if wireless.GetKillSwitchEnabled():
                        self.network_type = 'killswitch'
                        status = _('Not connected') + " (" + _('Wireless Kill Switch Enabled') + ")"
                    else:
                        self.network_type = 'none'
                except DBusException as e:
                    print("DBusException args:", e.args)
                    try:
                        print("DBusException name:", e.get_dbus_name())
                        print("DBusException message:", e.get_dbus_message())
                    except Exception as e2:
                        print("e.get_dbus_name failed:", e2)
                        self.network_type = 'no_daemon'
                        status = _('Wicd daemon unreachable')
            self.tr.tooltip = status
            self._show_notification(_('Disconnected'), None, 'stop')
            self.update_tooltip()

        @catchdbus
        def update_tray_icon(self, state=None, info=None):
            if not DBUS_AVAIL:
                return False
            if getattr(self.tr, 'submenu_open', False):
                return False
            if not state or not info:
                try:
                    state, info = daemon.GetConnectionStatus()
                except DBusException:
                    state = misc.NOT_CONNECTED
                    info = None

            self.should_notify = (can_use_notify() and self.last_state != state)
            self.last_state = state

            if state == misc.WIRED:
                self.set_wired_state(info)
            elif state == misc.WIRELESS:
                self.set_wireless_state(info)
            elif state == misc.CONNECTING:
                self.set_connecting_state(info)
            elif state in (misc.SUSPENDED, misc.NOT_CONNECTED):
                self.set_not_connected_state(info)
            else:
                print('Invalid state returned!!!')
                return False
            return True

        @catchdbus
        def set_signal_image(self, wireless_signal, lock):
            if self.animate:
                TrayIcon.get_bandwidth_bytes(self.parent)
                prefix = self.get_bandwidth_activity()
            else:
                prefix = 'idle-'

            if daemon.GetSignalDisplayType() == 0:
                # percentage scale
                if wireless_signal > 75:
                    signal_img = 'high-signal'
                elif wireless_signal > 50:
                    signal_img = 'good-signal'
                elif wireless_signal > 25:
                    signal_img = 'low-signal'
                else:
                    signal_img = 'bad-signal'
            else:
                # dBm scale
                if wireless_signal >= -60:
                    signal_img = 'high-signal'
                elif wireless_signal >= -70:
                    signal_img = 'good-signal'
                elif wireless_signal >= -80:
                    signal_img = 'low-signal'
                else:
                    signal_img = 'bad-signal'

            self.tr.set_from_name(''.join([prefix, signal_img, lock]))

        @catchdbus
        def get_bandwidth_activity(self):
            transmitting = receiving = False

            dev_dir = '/sys/class/net/'
            wiface = daemon.GetWirelessInterface()
            path = None
            for fldr in os.listdir(dev_dir):
                if fldr == wiface:
                    path = os.path.join(dev_dir, fldr, 'statistics')
                    break
            try:
                if not path:
                    raise FileNotFoundError
                rcvbytes = int(open(os.path.join(path, 'rx_bytes')).read().strip())
                sndbytes = int(open(os.path.join(path, 'tx_bytes')).read().strip())
            except Exception:
                sndbytes = None
                rcvbytes = None

            if not rcvbytes or not sndbytes:
                return 'idle-'

            # RX activity
            active, self.parent.max_rcv_gain, self.parent.last_rcvbytes = self.is_network_active(
                self.parent.cur_rcvbytes, self.parent.max_rcv_gain, self.parent.last_rcvbytes
            )
            receiving = active

            # TX activity
            active, self.parent.max_snd_gain, self.parent.last_sndbytes = self.is_network_active(
                self.parent.cur_sndbytes, self.parent.max_snd_gain, self.parent.last_sndbytes
            )
            transmitting = active

            if transmitting and receiving:
                return 'both-'
            elif transmitting:
                return 'transmitting-'
            elif receiving:
                return 'receiving-'
            else:
                return 'idle-'

        def is_network_active(self, bytes_val, max_gain, last_bytes):
            active = False
            if last_bytes == -1:
                last_bytes = bytes_val
            elif bytes_val > (last_bytes + float(max_gain / 20.0)):
                last_bytes = bytes_val
                active = True
                gain = bytes_val - last_bytes
                if gain > max_gain:
                    max_gain = gain
            return (active, max_gain, last_bytes)

    # end class TrayConnectionInfo

    # ---------------------
    # GUI base + implementations
    # ---------------------
    class TrayIconGUI(object):
        """Base class shared by StatusIcon/AppIndicator variants.
        Keeps initialization order safe (icon_info may be None at init).
        """
        def __init__(self, parent):
            super().__init__()
            self.parent = parent
            self.ind = None

            self.list = []
            self.label = None
            self.data = None

            self.gui_win = None
            self.current_icon_name = None
            self._is_scanning = False
            self.submenu_open = False

            # Build menu lazily; do not assume icon_info exists yet
            self.get_new_menu()

            if not USE_APP_INDICATOR:
                # Only for StatusIcon (hover-triggered scans)
                self.connect_item.connect("activate", self.on_net_menu_activate)

        def get_new_menu(self):
            self.menu = gtk.Menu()

            self.connect_item = gtk.MenuItem(label=_('Connect'))
            self.connect_submenu = gtk.Menu()
            self.connect_item.set_submenu(self.connect_submenu)
            self.menu.append(self.connect_item)

            info_item = gtk.MenuItem(label=_('Connection Info'))
            info_item.connect('activate', self.on_conn_info)
            self.menu.append(info_item)

            separator = gtk.SeparatorMenuItem()
            self.menu.append(separator)

            quit_item = gtk.MenuItem(label=_('Quit'))
            quit_item.connect('activate', self.on_quit)
            self.menu.append(quit_item)

            self.connect_submenu.connect('show', self.on_connect_submenu_show)
            self.connect_submenu.connect('hide', self.on_connect_submenu_hide)

            self.menu.show_all()
            self.connect_submenu.show_all()

            # Initial population (safe even if icon_info is not ready)
            self.populate_network_menu()

        def on_connect_submenu_show(self, *_):
            self.submenu_open = True

        def on_connect_submenu_hide(self, *_):
            self.submenu_open = False

        def tray_scan_started(self):
            if not DBUS_AVAIL:
                return
            self._is_scanning = True
            self.init_network_menu()

        def tray_scan_ended(self):
            if not DBUS_AVAIL:
                return
            self._is_scanning = False
            self.populate_network_menu()

        def on_activate(self, _data=None):
            if DBUS_AVAIL:
                self.toggle_wicd_gui()

        @catchdbus
        def update_tray(self, state=None, info=None):
            if not DBUS_AVAIL or self.submenu_open:
                return False
            info_list = [""] if info is None else [str(item) for item in info]
            if len(info_list) < 3:
                return
            if info_list != [''] and not self._is_scanning:
                for item in self.connect_submenu.get_children():
                    try:
                        item.set_sensitive(True)
                    except Exception:
                        pass

        def on_quit(self, _widget=None):
            sys.exit(0)

        def on_about(self, _data=None):
            dialog = gtk.AboutDialog()
            dialog.set_name('Wicd Tray Icon')
            dialog.set_version('2.0')
            dialog.set_comments('An icon that shows your network connectivity')
            dialog.set_website('http://launchpad.net/wicd')
            dialog.run()
            dialog.destroy()

        def on_conn_info(self, _data=None):
            window = gtk.Dialog(
                "Wicd Connection Info",
                None,
                0,
                (gtk.STOCK_OK, gtk.ResponseType.CLOSE)
            )

            self.label = gtk.Label()
            self.data = gtk.Label()
            self.data.set_selectable(True)
            self.label.show()
            self.data.show()
            self.list = [self.data, self.label]

            table = gtk.Table(1, 2)
            table.set_col_spacings(12)
            table.attach(self.label, 0, 1, 0, 1)
            table.attach(self.data, 1, 2, 0, 1)

            content = window.get_content_area()
            content.pack_start(table, True, True, 0)
            content.show_all()

            self.cont = 'Go'
            gobject.timeout_add(5000, self.update_conn_info_win, self.list)
            self.update_conn_info_win(self.list)

            window.run()
            window.destroy()
            self.cont = 'Stop'

        def update_conn_info_win(self, l):
            if getattr(self, 'cont', 'Stop') == 'Stop':
                return False

            [state, info] = daemon.GetConnectionStatus()
            [rx, tx] = self.get_current_bandwidth()

            if state == misc.WIRED:
                text = (_('''$A
$B KB/s
$C KB/s''')
                        .replace('$A', str(info[0]))
                        .replace('$B', str(rx))
                        .replace('$C', str(tx)))
            elif state == misc.WIRELESS:
                text = (_('''$A
$B
$C
$D
$E KB/s
$F KB/s''')
                        .replace('$A', str(misc.from_escape_to_str(info[1])))
                        .replace('$B', str(info[4]))
                        .replace('$C', str(info[0]))
                        .replace('$D', daemon.FormatSignalForPrinting(str(info[2])))
                        .replace('$E', str(rx))
                        .replace('$F', str(tx)))
            else:
                text = ''

            self.list[0].set_text('\n' + text)
            if state == misc.WIRED:
                self.list[1].set_text(_('''Wired
IP:
RX:
TX:'''))
            elif state == misc.WIRELESS:
                self.list[1].set_text(_('''Wireless
SSID:
Speed:
IP:
Strength:
RX:
TX:'''))
            elif state == misc.CONNECTING:
                self.list[1].set_text(_('Connecting'))
            elif state in (misc.SUSPENDED, misc.NOT_CONNECTED):
                self.list[1].set_text(_('Disconnected'))

            return True

        def get_current_bandwidth(self):
            self.parent.get_bandwidth_bytes()
            rxb = self.parent.cur_rcvbytes - self.parent.last_rcvbytes
            txb = self.parent.cur_sndbytes - self.parent.last_sndbytes

            self.parent.last_rcvbytes = self.parent.cur_rcvbytes
            self.parent.last_sndbytes = self.parent.cur_sndbytes

            rx_rate = float(rxb / (2 * 1024))  # time window defaults to 2s
            tx_rate = float(txb / (2 * 1024))
            return (rx_rate, tx_rate)

        @catchdbus
        def on_net_menu_activate(self, item):
            if self._is_scanning:
                return True
            self.init_network_menu()
            gobject.timeout_add(800, self._trigger_scan_if_needed, item)

        @catchdbus
        def _trigger_scan_if_needed(self, item):
            while gtk.events_pending():
                gtk.main_iteration()
            # PRELIGHT status isn't available for AppIndicator
            if getattr(item, 'state', None) != gtk.STATE_PRELIGHT:
                return True
            wireless.Scan(False)
            return False

        @catchdbus
        def populate_network_menu(self, _data=None):
            def get_prop(net_id, prop):
                return wireless.GetWirelessProperty(net_id, prop)

            self._clear_menu(self.connect_submenu)

            if not DBUS_AVAIL:
                self.connect_item.show()
                return

            is_connecting = daemon.CheckIfConnecting()
            num_networks = wireless.GetNumberOfNetworks()
            [status, info] = daemon.GetConnectionStatus()

            # Wired entry
            try:
                show_wired = daemon.GetAlwaysShowWiredInterface() or wired.CheckPluggedIn()
            except Exception:
                show_wired = True
            if show_wired:
                is_active = (status == misc.WIRED)
                self._add_item_to_menu(self.connect_submenu, "Wired Network", "__wired__", 0, is_connecting, is_active)
                sep = gtk.SeparatorMenuItem(); sep.show(); self.connect_submenu.append(sep)

            # Wi-Fi entries
            if num_networks > 0:
                skip_never_connect = not daemon.GetShowNeverConnect()
                for x in range(0, num_networks):
                    if skip_never_connect and misc.to_bool(get_prop(x, "never")):
                        continue
                    essid = get_prop(x, "essid")
                    is_active = (status == misc.WIRELESS and info and len(info) > 1 and info[1] == essid)
                    self._add_item_to_menu(self.connect_submenu, essid, "wifi", x, is_connecting, is_active)
            else:
                no_nets_item = gtk.MenuItem(_('No wireless networks found.'))
                no_nets_item.set_sensitive(False)
                no_nets_item.show()
                self.connect_submenu.append(no_nets_item)

            try:
                self.connect_submenu.reposition()
            except Exception:
                pass

            self.connect_submenu.show_all()
            self.connect_item.show()
            self.menu.show_all()

        def init_network_menu(self):
            self._clear_menu(self.connect_submenu)
            loading_item = gtk.MenuItem()
            loading_item.set_label(_('Scanning') + "...")
            loading_item.set_sensitive(False)
            loading_item.show()
            self.connect_submenu.append(loading_item)
            self.connect_item.show()

        def _clear_menu(self, menu):
            try:
                for item in menu.get_children():
                    menu.remove(item)
                    item.destroy()
            except Exception:
                pass

        def toggle_wicd_gui(self):
            if not self.gui_win:
                self.gui_win = gui.appGui(tray=self)
            elif not getattr(self.gui_win, 'is_visible', False):
                self.gui_win.show_win()
            else:
                self.gui_win.exit()
                return True

        def _add_item_to_menu(self, net_menu, lbl, type_, n_id, is_connecting, is_active):
            def network_selected(_widget, net_type, net_id):
                if net_type == "__wired__":
                    wired.ConnectWired()
                else:
                    wireless.ConnectWireless(net_id)

            item = gtk.MenuItem()
            image = gtk.Image()
            label = gtk.Label(label=lbl)
            box = gtk.Box(orientation=gtk.Orientation.HORIZONTAL, spacing=6)

            if is_active:
                atrlist = pango.AttrList()
                attr = pango.attr_weight_new(pango.Weight.BOLD)
                attr.start_index = 0
                attr.end_index = 50
                atrlist.insert(attr)
                label.set_attributes(atrlist)

            if type_ == "__wired__":
                image.set_from_icon_name("network-wired", gtk.IconSize.SMALL_TOOLBAR)
            else:
                image.set_from_icon_name(self._get_img(n_id), gtk.IconSize.SMALL_TOOLBAR)

            box.pack_start(image, False, False, 0)
            item.connect("activate", network_selected, type_, n_id)
            box.pack_start(label, False, False, 0)
            item.add(box)
            net_menu.append(item)
            item.show()
            if is_connecting:
                item.set_sensitive(False)

        @catchdbus
        def _get_img(self, net_id):
            def fix_strength(val, default):
                return val and int(float(val)) or default

            def get_prop(prop):
                return wireless.GetWirelessProperty(net_id, prop)

            strength = fix_strength(get_prop("quality"), -1)
            dbm_strength = fix_strength(get_prop('strength'), -100)

            if daemon.GetWPADriver() == 'ralink legacy' or daemon.GetSignalDisplayType() == 1:
                if dbm_strength >= -60:
                    signal_img = 'signal-100'
                elif dbm_strength >= -70:
                    signal_img = 'signal-75'
                elif dbm_strength >= -80:
                    signal_img = 'signal-50'
                else:
                    signal_img = 'signal-25'
            else:
                if strength > 75:
                    signal_img = 'signal-100'
                elif strength > 50:
                    signal_img = 'signal-75'
                elif strength > 25:
                    signal_img = 'signal-50'
                else:
                    signal_img = 'signal-25'
            return signal_img

    # end TrayIconGUI

    class StatusIcon(TrayIconGUI, gtk.StatusIcon):
        def __init__(self, parent):
            gtk.StatusIcon.__init__(self)
            TrayIcon.TrayIconGUI.__init__(self, parent)
            self.current_icon_name = ''

        @property
        def tooltip(self):
            return self.get_tooltip_text()

        @tooltip.setter
        def tooltip(self, text):
            self.set_tooltip_text(text)

    class StatusTrayIconGUI(StatusIcon):
        """GTK StatusIcon based tray."""
        def __init__(self, parent):
            super().__init__(parent)
            self.set_visible(True)
            self.connect('activate', self.on_activate)
            self.connect('popup-menu', self.on_popup_menu)
            self.set_from_name('no-signal')
            self.tooltip = "Initializing wicd..."

        def on_popup_menu(self, status, button, timestamp):
            self.init_network_menu()
            self.menu.popup(None, None, gtk.status_icon_position_menu, button, timestamp, self)

        def set_from_name(self, name=None):
            if name != getattr(self, 'current_icon_name', None):
                self.current_icon_name = name
                gtk.StatusIcon.set_from_icon_name(self, name)

        def visible(self, val):
            self.set_visible(val)

    if USE_APP_INDICATOR:
        class IndicatorTrayIconGUI(TrayIconGUI):
            """AppIndicator-based tray. Tooltips are emulated with a menu item."""
            def __init__(self, parent):
                super().__init__(parent)
                self.current_icon_name = ''
                self.ind = appindicator.Indicator.new(
                    "wicd", "network-idle", appindicator.IndicatorCategory.APPLICATION_STATUS)
                self.ind.set_status(appindicator.IndicatorStatus.ACTIVE)

                # Emulated tooltip menu item at the top; also toggles GUI on click
                self.tooltip_item = gtk.MenuItem(label=_("Initializing wicd..."))
                self.menu.prepend(self.tooltip_item)
                self.tooltip_item.connect("activate", self.on_activate)
                self.tooltip_item.show()

                self.ind.set_menu(self.menu)

            def populate_network_menu(self, data=None):
                super().populate_network_menu(data)
                if self.ind:
                    self.ind.set_menu(self.menu)

            def _clear_menu(self, menu):
                # Rebuild menu for AppIndicator contexts
                self.menu = gtk.Menu()

                # Recreate the pseudo-tooltip item
                self.tooltip_item = gtk.MenuItem()
                self.tooltip_item.set_label(getattr(self, 'tooltip', _("Initializing wicd...")))
                self.menu.append(self.tooltip_item)
                self.tooltip_item.connect("activate", self.on_activate)
                self.tooltip_item.show()

                # Connect submenu and standard items
                self.connect_item = gtk.MenuItem(label=_('Connect'))
                self.connect_submenu = gtk.Menu()
                self.connect_item.set_submenu(self.connect_submenu)
                self.menu.append(self.connect_item)

                info_item = gtk.MenuItem(label=_('Connection Info'))
                info_item.connect('activate', self.on_conn_info)
                self.menu.append(info_item)

                separator = gtk.SeparatorMenuItem(); self.menu.append(separator)

                quit_item = gtk.MenuItem(label=_('Quit'))
                quit_item.connect('activate', self.on_quit)
                self.menu.append(quit_item)

                self.connect_submenu.connect('show', self.on_connect_submenu_show)
                self.connect_submenu.connect('hide', self.on_connect_submenu_hide)

                self.menu.show_all()

                if self.ind:
                    self.ind.set_menu(self.menu)

            def on_rescan(self, *_data):
                self.init_network_menu()
                wireless.Scan(False)

            def set_from_file(self, path=None):
                if path != getattr(self, 'current_icon_path', None):
                    self.current_icon_path = path
                    self.ind.set_icon_full(path, "")

            def set_from_name(self, name=None):
                if name != getattr(self, 'current_icon_name', None):
                    self.current_icon_name = name
                    self.ind.set_icon_full(name, "")

            def visible(self, val):
                self.ind.set_status(appindicator.IndicatorStatus.ACTIVE if val else appindicator.IndicatorStatus.PASSIVE)

            @property
            def tooltip(self):
                return self.tooltip_item.get_label() if hasattr(self, 'tooltip_item') else ""

            @tooltip.setter
            def tooltip(self, text):
                if hasattr(self, 'tooltip_item'):
                    self.tooltip_item.set_label(text)


# -------------------------
# CLI / DBus bootstrap
# -------------------------

def usage():
    print((
        """
wicd %s
wireless (and wired) connection daemon front-end.

Arguments:
\t-t\t--tray\tRun the wicd tray icon only.
\t-n\t--no-tray\tRun wicd without the tray icon.
\t-h\t--help\t\tPrint this help information.
\t-a\t--no-animate\tRun the tray without network traffic tray animations.
\t-o\t--only-notifications\tDon't display anything except notifications.
""" % wpath.version
    ))


def setup_dbus(force=True):
    """Initialize DBus and connect to wicd daemon."""
    import dbus
    if getattr(dbus, "version", (0, 0, 0)) < (0, 80, 0):
        import dbus.glib  # noqa: F401
    else:
        from dbus.mainloop.glib import DBusGMainLoop
        DBusGMainLoop(set_as_default=True)

    global daemon, wireless, wired, DBUS_AVAIL, lost_dbus_id
    print("Connecting to daemon...")
    try:
        dbus_manager.connect()
    except wicd.errors.WiCDDaemonNotFound:
        if force:
            print("Can't connect to the daemon, trying to start it automatically...")
            misc.PromptToStartDaemon()
            try:
                dbus_manager.connect()
            except wicd.errors.WiCDDaemonNotFound:
                error(None, _("Could not connect to wicd's D-Bus interface. Check the wicd log for error messages."))
                return False
        else:
            return False

    if lost_dbus_id:
        gobject.source_remove(lost_dbus_id)
        lost_dbus_id = None

    dbus_ifaces = dbus_manager.get_dbus_ifaces()
    daemon = dbus_ifaces['daemon']
    wireless = dbus_ifaces['wireless']
    wired = dbus_ifaces['wired']
    DBUS_AVAIL = True
    print("Connected.")
    return True


def on_exit():
    if DBUS_AVAIL:
        try:
            daemon.SetGUIOpen(False)
        except DBusException:
            pass


def handle_no_dbus():
    global DBUS_AVAIL, lost_dbus_id
    DBUS_AVAIL = False
    gui.handle_no_dbus(from_tray=True)
    print("Wicd daemon is shutting down!")
    lost_dbus_id = misc.timeout_add(5,
        lambda: error(None,
            _('The wicd daemon has shut down. The UI will not function properly until it is restarted.'),
            block=False
        )
    )
    return False


@catchdbus
def main():
    p = wicd.commandline.get_parser()
    p.set_defaults(visibility='show_and_tray')

    p.add_argument('-t', '--tray', dest='visibility', action='store_const', const='hide_in_tray',
                   help='Minimize in tray after start')
    p.add_argument('-n', '--no-tray', dest='visibility', action='store_const', const='show',
                   help="Don't use tray area")
    p.add_argument('-o', '--only-notifications', dest='visibility', action='store_const', const='only_notifications',
                   help="Only show notifications")
    p.add_argument('-a', '--no-animate', dest='animate', action='store_false', default=True,
                   help="Don't show animations")

    args = wicd.commandline.get_args()

    print('Loading...')
    setup_dbus()
    atexit.register(on_exit)

    use_tray = 'tray' in args.visibility
    display_app = 'show' in args.visibility

    if (display_app and not use_tray) or not ICON_AVAIL:
        gui.appGui(standalone=True)
        mainloop = gobject.MainLoop()
        mainloop.run()
        sys.exit(0)

    # Set up the tray icon GUI and backend
    tray_icon = TrayIcon(args.animate, displaytray=use_tray, displayapp=display_app)

    # Wired profile chooser might have been requested before GUI launch
    if DBUS_AVAIL and daemon.GetNeedWiredProfileChooser():
        daemon.SetNeedWiredProfileChooser(False)
        tray_icon.icon_info.wired_profile_chooser()

    bus = dbus_manager.get_bus()
    bus.add_signal_receiver(tray_icon.icon_info.wired_profile_chooser,
                            'LaunchChooser', 'org.wicd.daemon')
    bus.add_signal_receiver(tray_icon.icon_info.update_tray_icon,
                            'StatusChanged', 'org.wicd.daemon')
    bus.add_signal_receiver(tray_icon.tr.update_tray,
                            'StatusChanged', 'org.wicd.daemon')
    bus.add_signal_receiver(tray_icon.tr.tray_scan_ended, 'SendEndScanSignal',
                            'org.wicd.daemon.wireless')
    bus.add_signal_receiver(tray_icon.tr.tray_scan_started,
                            'SendStartScanSignal', 'org.wicd.daemon.wireless')
    bus.add_signal_receiver(
        lambda: (handle_no_dbus() or tray_icon.icon_info.set_not_connected_state()),
        'DaemonClosing', 'org.wicd.daemon')
    bus.add_signal_receiver(lambda: setup_dbus(force=False), 'DaemonStarting', 'org.wicd.daemon')

    print('Done loading.')
    mainloop = gobject.MainLoop()
    mainloop.run()


if __name__ == '__main__':
    main()

