#!/usr/bin/env python3
# vim: set fileencoding=utf8
#
#   Copyright (C) 2007 - 2009 Adam Blackburn
#   Copyright (C) 2007 - 2009 Dan O'Reilly
#   Copyright (C) 2009        Andrew Psaltis
#   Copyright (C) 2021 - 2022 Andreas Messer
#   Copyright (C) 2024        Takahiro Yoshizawa
#
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License Version 2 as
#   published by the Free Software Foundation.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from setuptools import setup, find_namespace_packages
from setuptools.command.build_py import build_py
from setuptools.command.install import install as _install


from setuptools import setup
from setuptools.command.build_py import build_py as _build_py
import os
import subprocess
import shutil
import re
from glob import glob

config_file_template = '''
import os

etc_path        = "/etc/wicd"
resolvconf_path = "/etc/resolv.conf"
rundir_path     = "/var/run/wicd/"

wireless_conf_path = etc_path + os.sep + "wireless-settings.conf"
wired_conf_path    = etc_path + os.sep + "wired-settings.conf"
dhclient_conf_path = etc_path + os.sep + "dhclient.conf.template"

pidfile_path           = rundir_path + "wicd.pid"
resolvconf_backup_path = rundir_path + "resolv.conf.orig"
'''


class build_py(_build_py):
    def run(self):
        config_file = os.path.join(self.build_lib, 'wicd', 'config.py')
        os.makedirs(os.path.dirname(config_file), exist_ok=True)
        with open(config_file, 'wt') as f:
            print('generating ' + config_file)
            f.write(config_file_template)

        self.compile_translations()
        self.process_templates()
        super().run()

    def compile_translations(self):
        print('Compiling translations...')
        threshold = 0.8

        print("Update wicd.pot with new strings")
        subprocess.run(['pybabel', 'extract', '-F', 'babel.cfg', '-o',
                        'po/wicd.pot', '--sort-output', '.'], check=True)

        print("Update po-files with new strings from wicd.pot")
        for pofile in glob('po/*/LC_MESSAGES/*.po'):
            locale_path = os.path.dirname(pofile)
            lang = os.path.basename(os.path.dirname(locale_path))
            subprocess.run(['pybabel', 'update', '-N', '-i', 'po/wicd.pot',
                           '-d', 'po', '-D', 'wicd', '-l', lang], check=True)

        print('Compile po-files to binary mo')
        if os.path.exists('translations'):
            shutil.rmtree('translations/')
        os.makedirs('translations')

        oldlang = os.environ.get('LANG', 'C')
        os.environ['LANG'] = 'C'

        for pofile in sorted(glob('po/*/LC_MESSAGES/*.po')):
            locale_path = os.path.dirname(pofile)
            lang = os.path.basename(os.path.dirname(locale_path))
            compile_po = False
            try:
                result = subprocess.run(['msgfmt', '--statistics',
                                         pofile, '-o', '/dev/null'],
                                        stderr=subprocess.PIPE, text=True)
                output = result.stderr.strip()
                if result.returncode != 0 or not output:
                    print(f"Error processing {pofile}: {result.stderr}")
                    raise ValueError(f"Error processing {pofile}")
                else:
                    m = re.match(
                        r'^(?:'
                        r'(?:(\d+) translated messages)'
                        r'(?:, )?'
                        r'(?:(\d+) fuzzy translation(?:s)?)?'
                        r'(?:, )?'
                        r'(?:(\d+) untranslated messages?)?'
                        r')\.?$',
                        output
                    )
                    if m:
                        done = int(m.group(1) or 0)
                        fuzzy = int(m.group(2) or 0)
                        missing = int(m.group(3) or 0)
                        total = done + fuzzy + missing
                        completeness = done / total if total > 0 else 0
                        if completeness >= threshold:
                            compile_po = True
                        else:
                            print(f'Disabled {lang} ({completeness*100:.1f}% < {threshold*100:.1f}%).')
                            continue
            except Exception as e:
                print(f'An error occurred while processing {pofile}: {e}')

            if compile_po:
                mo_dir = os.path.join('translations', lang, 'LC_MESSAGES')
                os.makedirs(mo_dir, exist_ok=True)
                subprocess.run(['pybabel', 'compile', '-D', 'wicd', '-i', pofile,
                               '-l', lang, '-d', 'translations/'], check=True)
        os.environ['LANG'] = oldlang

        new_data_files = self.find_mo_files()
        existing_data_files = self.distribution.data_files or []
        self.distribution.data_files = existing_data_files + new_data_files

    def find_mo_files(self):
        data_files = []
        for root, dirs, files in os.walk('translations'):
            for file in files:
                if file.endswith('.mo'):
                    lang = os.path.relpath(root, 'translations').split(os.sep)[0]
                    mo_file = os.path.join(root, file)
                    install_dir = os.path.join('share', 'locale', lang, 'LC_MESSAGES')
                    data_files.append((install_dir, [mo_file]))
        return data_files

    def initialize_options(self):
        super().initialize_options()
        self.init = ''
        self.lib = '/usr/lib/wicd/'
        self.share = '/usr/share/wicd/'
        self.etc = '/etc/wicd/'
        self.scripts = self.share + "scripts/"
        self.encryption = self.share + 'encryption/templates/'
        self.bin = '/usr/bin/'
        self.sbin = '/usr/sbin/'
        self.daemon = self.share + 'daemon'
        self.backends = self.share + 'backends'
        self.curses = self.share + 'curses'
        self.gtk = self.share + 'gtk'
        self.cli = self.share + 'cli'
        self.gnome_shell_extensions = '/usr/share/gnome-shell/extensions/'
        self.icons = '/usr/share/icons/hicolor/'
        self.pixmaps = '/usr/share/pixmaps/'
        self.images = '/usr/share/icons/'
        self.varlib = '/var/lib/wicd/'
        self.networks = self.varlib + 'configurations/'
        self.log = '/var/log/wicd/'
        self.resume = '/etc/acpi/resume.d/'
        self.suspend = '/etc/acpi/suspend.d/'
        self.pmutils = '/usr/lib/pm-utils/sleep.d/'
        self.dbus = '/etc/dbus-1/system.d/'
        self.dbus_service = '/usr/share/dbus-1/system-services/'
        self.systemd = '/lib/systemd/system/'
        self.logrotate = '/etc/logrotate.d/'
        self.desktop = '/usr/share/applications/'
        self.translations = '/usr/share/locale/'
        self.autostart = '/etc/xdg/autostart/'
        self.docdir = '/usr/share/doc/wicd/'
        self.mandir = '/usr/share/man/'
        self.kdedir = '/usr/share/autostart/'
        self.distro = 'auto'
        
        self.no_install_init = False
        self.no_install_man = False
        self.no_install_i18n = False
        self.no_install_i18n_man = False
        self.no_install_kde = False
        self.no_install_acpi = False
        self.no_install_pmutils = False
        self.no_install_docs = False
        self.no_install_gtk = False
        self.no_install_ncurses = False
        self.no_install_cli = False
        self.no_install_gnome_shell_extensions = False
        self.no_use_notifications = False

        self.initfile = 'init/default/wicd'
        # ddistro is the detected distro
        if os.path.exists('/etc/redhat-release'):
            self.ddistro = 'redhat'
        elif os.path.exists('/etc/SuSE-release'):
            self.ddistro = 'suse'
        elif os.path.exists('/etc/fedora-release'):
            self.ddistro = 'redhat'
        elif os.path.exists('/etc/gentoo-release'):
            self.ddistro = 'gentoo'
        elif os.path.exists('/etc/debian_version'):
            self.ddistro = 'debian'
        elif os.path.exists('/etc/arch-release'):
            self.ddistro = 'arch'
        elif os.path.exists('/etc/slackware-version') or \
             os.path.exists('/etc/slamd64-version') or \
             os.path.exists('/etc/bluewhite64-version'):
            self.ddistro = 'slackware'
        elif os.path.exists('/etc/pld-release'):
            self.ddistro = 'pld'
        elif os.path.exists('/etc/lunar.release'):
            self.distro = 'lunar'
        elif os.path.exists('/etc/plamo-release'):
            self.ddistro = 'plamo'
        else:
            self.ddistro = 'FAIL'
            #self.no_install_init = True
            #self.distro_detect_failed = True
            print('WARNING: Unable to detect the distribution in use.  ' + \
                  'If you have specified --distro or --init and --initfile, configure will continue.  ' + \
                  'Please report this warning, along with the name of your ' + \
                  'distribution, to the wicd developers.')

        # Try to get the pm-utils sleep hooks directory from pkg-config and
        # the kde prefix from kde-config
        # Don't run these in a shell because it's not needed and because shell 
        # swallows the OSError we would get if {pkg,kde}-config do not exist
        # If we don't get anything from *-config, or it didn't run properly, 
        # or the path is not a proper absolute path, raise an error
        try:
            pmtemp = subprocess.Popen(["pkg-config", "--variable=pm_sleephooks", 
                                       "pm-utils"], stdout=subprocess.PIPE)
            returncode = pmtemp.wait() # let it finish, and get the exit code
            pmutils_candidate = str(pmtemp.stdout.readline().strip()) # read stdout
            if len(pmutils_candidate) == 0 or returncode != 0 or \
               not os.path.isabs(pmutils_candidate):
                raise ValueError
            else:
                self.pmutils = pmutils_candidate
        except (OSError, ValueError, FileNotFoundError):
            pass # use our default

        try:
            kdetemp = subprocess.Popen(["kde-config","--prefix"], stdout=subprocess.PIPE)
            returncode = kdetemp.wait() # let it finish, and get the exit code
            kdedir_candidate = str(kdetemp.stdout.readline().strip()) # read stdout
            if len(kdedir_candidate) == 0 or returncode != 0 or \
               not os.path.isabs(kdedir_candidate):
                raise ValueError
            else:
                self.kdedir = kdedir_candidate + '/share/autostart'
        except (OSError, ValueError, FileNotFoundError):
            # If kde-config isn't present, we'll check for kde-4.x
            try:
                kde4temp = subprocess.Popen(["kde4-config","--prefix"], stdout=subprocess.PIPE)
                returncode = kde4temp.wait() # let it finish, and get the exit code
                kde4dir_candidate = str(kde4temp.stdout.readline().strip()) # read stdout
                if len(kde4dir_candidate) == 0 or returncode != 0 or \
                   not os.path.isabs(kde4dir_candidate):
                    raise ValueError
                else:
                    self.kdedir = kde4dir_candidate + '/share/autostart'
            except (OSError, ValueError, FileNotFoundError):
                # If neither kde-config nor kde4-config are not present or 
                # return an error, then we can assume that kde isn't installed
                # on the user's system
                self.no_install_kde = True
                # If the assumption above turns out to be wrong, do this:
                #pass # use our default

        self.python = '/usr/bin/python3'
        self.pidfile = '/var/run/wicd/wicd.pid'
        self.initfilename = os.path.basename(self.initfile)
        self.wicdgroup = 'users'
        self.loggroup = ''
        self.logperms = '0600'

    def process_templates(self):
        # Retrieve version information for curses
        try:
            import curses
            print("curses version: ", curses.version)
            if isinstance(curses.version, bytes):
                curses_version = curses.version.decode('utf-8').strip().split()[0].lstrip("b'")
            else:
                curses_version = curses.version.split()[0]
        except (AttributeError, IndexError):
            curses_version = 'unknown'

        # Retrieve version information from setuptools.
        version = self.distribution.get_version()

        # Retrieve the revision from setup.cfg.
        revision_num = getattr(self.distribution.metadata, 'revision', '0')

        # Convert the attributes of self into values.
        values = {attr.upper(): getattr(self, attr) for attr in dir(self) 
                  if not callable(getattr(self, attr)) and not attr.startswith("_")}

        # Additional placeholder information.
        values.update({
            'VERSION': version,
            'REVISION': revision_num,
            'CURSES_REVNO': curses_version,
        })

        print('Processing templates...')
        in_dir = 'in'

        for item in os.listdir(in_dir):
            if item.endswith('.in'):
                original_name = os.path.join(in_dir, item)
                final_name = item[:-3].replace('=', '/')

                if final_name == 'wpath.py':
                    full_path = os.path.join(self.build_lib, 'wicd', final_name)
                else:
                    full_path = os.path.join('template', final_name)

                os.makedirs(os.path.dirname(full_path), exist_ok=True)

                try:
                    with open(original_name, 'r') as infile, open(full_path, 'w') as outfile:
                        for line in infile:
                            for key, value in values.items():
                                placeholder = f'%{key.upper()}%'
                                line = line.replace(placeholder, str(value))

                            line = line.replace('%VERSION%', str(version))
                            line = line.replace('%REVNO%', str(revision_num))
                            line = line.replace('%CURSES_REVNO%', str(curses_version))

                            outfile.write(line)

                    shutil.copymode(original_name, full_path)
                    print(f"Processed template: {original_name} -> {full_path}")

                except FileNotFoundError:
                    print(f"ERROR: Template file {original_name} not found.")
                except Exception as e:
                    print(f"ERROR: Failed to process {original_name}: {e}")


setup(
    cmdclass={
        'build_py': build_py,
    },
)
