#![allow(
    dead_code,
    improper_ctypes_definitions,
    non_camel_case_types,
    unused_variables,
    clippy::not_unsafe_ptr_arg_deref,
    clippy::too_many_arguments
)]
use crate::rc::{ConvertParam, ConvertReturnValue, Rc, RcImpl, RefGuard, WrapParamRef};
use cef_dll_sys::*;

/// Perform the conversion between CEF and Rust types in field initializers.
fn init_array_field<T, U, const N: usize>(mut value: [U; N]) -> [T; N]
where
    T: Sized,
    U: Sized + Into<T>,
{
    std::array::from_fn(move |i| {
        let mut elem = unsafe { std::mem::zeroed() };
        std::mem::swap(&mut value[i], &mut elem);
        elem.into()
    })
}

/// See [`cef_string_wide_t`] for more documentation.
pub use crate::string::CefStringUserfreeWide;

/// See [`cef_string_utf8_t`] for more documentation.
pub use crate::string::CefStringUserfreeUtf8;

/// See [`cef_string_utf16_t`] for more documentation.
pub use crate::string::CefStringUserfreeUtf16;

/// See [`char16_t`] for more documentation.
pub type Char = char16_t;

/// See [`cef_string_userfree_utf16_t`] for more documentation.
pub type CefStringUserfree = CefStringUserfreeUtf16;

/// See [`cef_string_utf16_t`] for more documentation.
pub type CefString = CefStringUtf16;

/// See [`::std::os::raw::c_ulong`] for more documentation.
pub type CursorHandle = ::std::os::raw::c_ulong;

/// See [`XEvent`] for more documentation.
pub type EventHandle = *mut XEvent;

/// See [`::std::os::raw::c_ulong`] for more documentation.
pub type WindowHandle = ::std::os::raw::c_ulong;

/// See [`_cef_accelerated_paint_native_pixmap_plane_info_t`] for more documentation.
pub type AcceleratedPaintNativePixmapPlane = AcceleratedPaintNativePixmapPlaneInfo;

/// See [`u32`] for more documentation.
pub type Color = u32;

/// See [`_cef_string_wide_t`] for more documentation.
pub use crate::string::CefStringWide;

/// See [`_cef_string_utf8_t`] for more documentation.
pub use crate::string::CefStringUtf8;

/// See [`_cef_string_utf16_t`] for more documentation.
pub use crate::string::CefStringUtf16;

/// See [`_cef_string_list_t`] for more documentation.
pub use crate::string::CefStringList;

/// See [`_cef_string_map_t`] for more documentation.
pub use crate::string::CefStringMap;

/// See [`_cef_string_multimap_t`] for more documentation.
pub use crate::string::CefStringMultimap;

/// See [`_cef_basetime_t`] for more documentation.
#[derive(Clone)]
pub struct Basetime {
    pub val: i64,
}
impl From<_cef_basetime_t> for Basetime {
    fn from(value: _cef_basetime_t) -> Self {
        Self { val: value.val }
    }
}
impl From<Basetime> for _cef_basetime_t {
    fn from(value: Basetime) -> Self {
        Self { val: value.val }
    }
}
impl Default for Basetime {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_time_t`] for more documentation.
#[derive(Clone)]
pub struct Time {
    pub year: ::std::os::raw::c_int,
    pub month: ::std::os::raw::c_int,
    pub day_of_week: ::std::os::raw::c_int,
    pub day_of_month: ::std::os::raw::c_int,
    pub hour: ::std::os::raw::c_int,
    pub minute: ::std::os::raw::c_int,
    pub second: ::std::os::raw::c_int,
    pub millisecond: ::std::os::raw::c_int,
}
impl From<_cef_time_t> for Time {
    fn from(value: _cef_time_t) -> Self {
        Self {
            year: value.year,
            month: value.month,
            day_of_week: value.day_of_week,
            day_of_month: value.day_of_month,
            hour: value.hour,
            minute: value.minute,
            second: value.second,
            millisecond: value.millisecond,
        }
    }
}
impl From<Time> for _cef_time_t {
    fn from(value: Time) -> Self {
        Self {
            year: value.year,
            month: value.month,
            day_of_week: value.day_of_week,
            day_of_month: value.day_of_month,
            hour: value.hour,
            minute: value.minute,
            second: value.second,
            millisecond: value.millisecond,
        }
    }
}
impl Default for Time {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_point_t`] for more documentation.
#[derive(Clone)]
pub struct Point {
    pub x: ::std::os::raw::c_int,
    pub y: ::std::os::raw::c_int,
}
impl From<_cef_point_t> for Point {
    fn from(value: _cef_point_t) -> Self {
        Self {
            x: value.x,
            y: value.y,
        }
    }
}
impl From<Point> for _cef_point_t {
    fn from(value: Point) -> Self {
        Self {
            x: value.x,
            y: value.y,
        }
    }
}
impl Default for Point {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_rect_t`] for more documentation.
#[derive(Clone)]
pub struct Rect {
    pub x: ::std::os::raw::c_int,
    pub y: ::std::os::raw::c_int,
    pub width: ::std::os::raw::c_int,
    pub height: ::std::os::raw::c_int,
}
impl From<_cef_rect_t> for Rect {
    fn from(value: _cef_rect_t) -> Self {
        Self {
            x: value.x,
            y: value.y,
            width: value.width,
            height: value.height,
        }
    }
}
impl From<Rect> for _cef_rect_t {
    fn from(value: Rect) -> Self {
        Self {
            x: value.x,
            y: value.y,
            width: value.width,
            height: value.height,
        }
    }
}
impl Default for Rect {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_size_t`] for more documentation.
#[derive(Clone)]
pub struct Size {
    pub width: ::std::os::raw::c_int,
    pub height: ::std::os::raw::c_int,
}
impl From<_cef_size_t> for Size {
    fn from(value: _cef_size_t) -> Self {
        Self {
            width: value.width,
            height: value.height,
        }
    }
}
impl From<Size> for _cef_size_t {
    fn from(value: Size) -> Self {
        Self {
            width: value.width,
            height: value.height,
        }
    }
}
impl Default for Size {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_insets_t`] for more documentation.
#[derive(Clone)]
pub struct Insets {
    pub top: ::std::os::raw::c_int,
    pub left: ::std::os::raw::c_int,
    pub bottom: ::std::os::raw::c_int,
    pub right: ::std::os::raw::c_int,
}
impl From<_cef_insets_t> for Insets {
    fn from(value: _cef_insets_t) -> Self {
        Self {
            top: value.top,
            left: value.left,
            bottom: value.bottom,
            right: value.right,
        }
    }
}
impl From<Insets> for _cef_insets_t {
    fn from(value: Insets) -> Self {
        Self {
            top: value.top,
            left: value.left,
            bottom: value.bottom,
            right: value.right,
        }
    }
}
impl Default for Insets {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_accelerated_paint_info_common_t`] for more documentation.
#[derive(Clone)]
pub struct AcceleratedPaintInfoCommon {
    pub size: usize,
    pub timestamp: u64,
    pub coded_size: Size,
    pub visible_rect: Rect,
    pub content_rect: Rect,
    pub source_size: Size,
    pub capture_update_rect: Rect,
    pub region_capture_rect: Rect,
    pub capture_counter: u64,
    pub has_capture_update_rect: u8,
    pub has_region_capture_rect: u8,
    pub has_source_size: u8,
    pub has_capture_counter: u8,
}
impl From<_cef_accelerated_paint_info_common_t> for AcceleratedPaintInfoCommon {
    fn from(value: _cef_accelerated_paint_info_common_t) -> Self {
        Self {
            size: value.size,
            timestamp: value.timestamp,
            coded_size: value.coded_size.into(),
            visible_rect: value.visible_rect.into(),
            content_rect: value.content_rect.into(),
            source_size: value.source_size.into(),
            capture_update_rect: value.capture_update_rect.into(),
            region_capture_rect: value.region_capture_rect.into(),
            capture_counter: value.capture_counter,
            has_capture_update_rect: value.has_capture_update_rect,
            has_region_capture_rect: value.has_region_capture_rect,
            has_source_size: value.has_source_size,
            has_capture_counter: value.has_capture_counter,
        }
    }
}
impl From<AcceleratedPaintInfoCommon> for _cef_accelerated_paint_info_common_t {
    fn from(value: AcceleratedPaintInfoCommon) -> Self {
        Self {
            size: value.size,
            timestamp: value.timestamp,
            coded_size: value.coded_size.into(),
            visible_rect: value.visible_rect.into(),
            content_rect: value.content_rect.into(),
            source_size: value.source_size.into(),
            capture_update_rect: value.capture_update_rect.into(),
            region_capture_rect: value.region_capture_rect.into(),
            capture_counter: value.capture_counter,
            has_capture_update_rect: value.has_capture_update_rect,
            has_region_capture_rect: value.has_region_capture_rect,
            has_source_size: value.has_source_size,
            has_capture_counter: value.has_capture_counter,
        }
    }
}
impl Default for AcceleratedPaintInfoCommon {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_accelerated_paint_info_common_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_main_args_t`] for more documentation.
#[derive(Clone)]
pub struct MainArgs {
    pub argc: ::std::os::raw::c_int,
    pub argv: *mut *mut ::std::os::raw::c_char,
}
impl From<_cef_main_args_t> for MainArgs {
    fn from(value: _cef_main_args_t) -> Self {
        Self {
            argc: value.argc,
            argv: value.argv,
        }
    }
}
impl From<MainArgs> for _cef_main_args_t {
    fn from(value: MainArgs) -> Self {
        Self {
            argc: value.argc,
            argv: value.argv,
        }
    }
}
impl Default for MainArgs {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_window_info_t`] for more documentation.
#[derive(Clone)]
pub struct WindowInfo {
    pub size: usize,
    pub window_name: CefString,
    pub bounds: Rect,
    pub parent_window: ::std::os::raw::c_ulong,
    pub windowless_rendering_enabled: ::std::os::raw::c_int,
    pub shared_texture_enabled: ::std::os::raw::c_int,
    pub external_begin_frame_enabled: ::std::os::raw::c_int,
    pub window: ::std::os::raw::c_ulong,
    pub runtime_style: RuntimeStyle,
}
impl From<_cef_window_info_t> for WindowInfo {
    fn from(value: _cef_window_info_t) -> Self {
        Self {
            size: value.size,
            window_name: value.window_name.into(),
            bounds: value.bounds.into(),
            parent_window: value.parent_window,
            windowless_rendering_enabled: value.windowless_rendering_enabled,
            shared_texture_enabled: value.shared_texture_enabled,
            external_begin_frame_enabled: value.external_begin_frame_enabled,
            window: value.window,
            runtime_style: value.runtime_style.into(),
        }
    }
}
impl From<WindowInfo> for _cef_window_info_t {
    fn from(value: WindowInfo) -> Self {
        Self {
            size: value.size,
            window_name: value.window_name.into(),
            bounds: value.bounds.into(),
            parent_window: value.parent_window,
            windowless_rendering_enabled: value.windowless_rendering_enabled,
            shared_texture_enabled: value.shared_texture_enabled,
            external_begin_frame_enabled: value.external_begin_frame_enabled,
            window: value.window,
            runtime_style: value.runtime_style.into(),
        }
    }
}
impl Default for WindowInfo {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_window_info_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_accelerated_paint_native_pixmap_plane_info_t`] for more documentation.
#[derive(Clone)]
pub struct AcceleratedPaintNativePixmapPlaneInfo {
    pub stride: u32,
    pub offset: u64,
    pub size: u64,
    pub fd: ::std::os::raw::c_int,
}
impl From<_cef_accelerated_paint_native_pixmap_plane_info_t>
    for AcceleratedPaintNativePixmapPlaneInfo
{
    fn from(value: _cef_accelerated_paint_native_pixmap_plane_info_t) -> Self {
        Self {
            stride: value.stride,
            offset: value.offset,
            size: value.size,
            fd: value.fd,
        }
    }
}
impl From<AcceleratedPaintNativePixmapPlaneInfo>
    for _cef_accelerated_paint_native_pixmap_plane_info_t
{
    fn from(value: AcceleratedPaintNativePixmapPlaneInfo) -> Self {
        Self {
            stride: value.stride,
            offset: value.offset,
            size: value.size,
            fd: value.fd,
        }
    }
}
impl Default for AcceleratedPaintNativePixmapPlaneInfo {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_accelerated_paint_info_t`] for more documentation.
#[derive(Clone)]
pub struct AcceleratedPaintInfo {
    pub size: usize,
    pub planes: [AcceleratedPaintNativePixmapPlaneInfo; 4usize],
    pub plane_count: ::std::os::raw::c_int,
    pub modifier: u64,
    pub format: ColorType,
    pub extra: AcceleratedPaintInfoCommon,
}
impl From<_cef_accelerated_paint_info_t> for AcceleratedPaintInfo {
    fn from(value: _cef_accelerated_paint_info_t) -> Self {
        Self {
            size: value.size,
            planes: init_array_field(value.planes),
            plane_count: value.plane_count,
            modifier: value.modifier,
            format: value.format.into(),
            extra: value.extra.into(),
        }
    }
}
impl From<AcceleratedPaintInfo> for _cef_accelerated_paint_info_t {
    fn from(value: AcceleratedPaintInfo) -> Self {
        Self {
            size: value.size,
            planes: init_array_field(value.planes),
            plane_count: value.plane_count,
            modifier: value.modifier,
            format: value.format.into(),
            extra: value.extra.into(),
        }
    }
}
impl Default for AcceleratedPaintInfo {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_accelerated_paint_info_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_settings_t`] for more documentation.
#[derive(Clone)]
pub struct Settings {
    pub size: usize,
    pub no_sandbox: ::std::os::raw::c_int,
    pub browser_subprocess_path: CefString,
    pub framework_dir_path: CefString,
    pub main_bundle_path: CefString,
    pub multi_threaded_message_loop: ::std::os::raw::c_int,
    pub external_message_pump: ::std::os::raw::c_int,
    pub windowless_rendering_enabled: ::std::os::raw::c_int,
    pub command_line_args_disabled: ::std::os::raw::c_int,
    pub cache_path: CefString,
    pub root_cache_path: CefString,
    pub persist_session_cookies: ::std::os::raw::c_int,
    pub user_agent: CefString,
    pub user_agent_product: CefString,
    pub locale: CefString,
    pub log_file: CefString,
    pub log_severity: LogSeverity,
    pub log_items: LogItems,
    pub javascript_flags: CefString,
    pub resources_dir_path: CefString,
    pub locales_dir_path: CefString,
    pub remote_debugging_port: ::std::os::raw::c_int,
    pub uncaught_exception_stack_size: ::std::os::raw::c_int,
    pub background_color: u32,
    pub accept_language_list: CefString,
    pub cookieable_schemes_list: CefString,
    pub cookieable_schemes_exclude_defaults: ::std::os::raw::c_int,
    pub chrome_policy_id: CefString,
    pub chrome_app_icon_id: ::std::os::raw::c_int,
    pub disable_signal_handlers: ::std::os::raw::c_int,
}
impl From<_cef_settings_t> for Settings {
    fn from(value: _cef_settings_t) -> Self {
        Self {
            size: value.size,
            no_sandbox: value.no_sandbox,
            browser_subprocess_path: value.browser_subprocess_path.into(),
            framework_dir_path: value.framework_dir_path.into(),
            main_bundle_path: value.main_bundle_path.into(),
            multi_threaded_message_loop: value.multi_threaded_message_loop,
            external_message_pump: value.external_message_pump,
            windowless_rendering_enabled: value.windowless_rendering_enabled,
            command_line_args_disabled: value.command_line_args_disabled,
            cache_path: value.cache_path.into(),
            root_cache_path: value.root_cache_path.into(),
            persist_session_cookies: value.persist_session_cookies,
            user_agent: value.user_agent.into(),
            user_agent_product: value.user_agent_product.into(),
            locale: value.locale.into(),
            log_file: value.log_file.into(),
            log_severity: value.log_severity.into(),
            log_items: value.log_items.into(),
            javascript_flags: value.javascript_flags.into(),
            resources_dir_path: value.resources_dir_path.into(),
            locales_dir_path: value.locales_dir_path.into(),
            remote_debugging_port: value.remote_debugging_port,
            uncaught_exception_stack_size: value.uncaught_exception_stack_size,
            background_color: value.background_color,
            accept_language_list: value.accept_language_list.into(),
            cookieable_schemes_list: value.cookieable_schemes_list.into(),
            cookieable_schemes_exclude_defaults: value.cookieable_schemes_exclude_defaults,
            chrome_policy_id: value.chrome_policy_id.into(),
            chrome_app_icon_id: value.chrome_app_icon_id,
            disable_signal_handlers: value.disable_signal_handlers,
        }
    }
}
impl From<Settings> for _cef_settings_t {
    fn from(value: Settings) -> Self {
        Self {
            size: value.size,
            no_sandbox: value.no_sandbox,
            browser_subprocess_path: value.browser_subprocess_path.into(),
            framework_dir_path: value.framework_dir_path.into(),
            main_bundle_path: value.main_bundle_path.into(),
            multi_threaded_message_loop: value.multi_threaded_message_loop,
            external_message_pump: value.external_message_pump,
            windowless_rendering_enabled: value.windowless_rendering_enabled,
            command_line_args_disabled: value.command_line_args_disabled,
            cache_path: value.cache_path.into(),
            root_cache_path: value.root_cache_path.into(),
            persist_session_cookies: value.persist_session_cookies,
            user_agent: value.user_agent.into(),
            user_agent_product: value.user_agent_product.into(),
            locale: value.locale.into(),
            log_file: value.log_file.into(),
            log_severity: value.log_severity.into(),
            log_items: value.log_items.into(),
            javascript_flags: value.javascript_flags.into(),
            resources_dir_path: value.resources_dir_path.into(),
            locales_dir_path: value.locales_dir_path.into(),
            remote_debugging_port: value.remote_debugging_port,
            uncaught_exception_stack_size: value.uncaught_exception_stack_size,
            background_color: value.background_color,
            accept_language_list: value.accept_language_list.into(),
            cookieable_schemes_list: value.cookieable_schemes_list.into(),
            cookieable_schemes_exclude_defaults: value.cookieable_schemes_exclude_defaults,
            chrome_policy_id: value.chrome_policy_id.into(),
            chrome_app_icon_id: value.chrome_app_icon_id,
            disable_signal_handlers: value.disable_signal_handlers,
        }
    }
}
impl Default for Settings {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_settings_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_request_context_settings_t`] for more documentation.
#[derive(Clone)]
pub struct RequestContextSettings {
    pub size: usize,
    pub cache_path: CefString,
    pub persist_session_cookies: ::std::os::raw::c_int,
    pub accept_language_list: CefString,
    pub cookieable_schemes_list: CefString,
    pub cookieable_schemes_exclude_defaults: ::std::os::raw::c_int,
}
impl From<_cef_request_context_settings_t> for RequestContextSettings {
    fn from(value: _cef_request_context_settings_t) -> Self {
        Self {
            size: value.size,
            cache_path: value.cache_path.into(),
            persist_session_cookies: value.persist_session_cookies,
            accept_language_list: value.accept_language_list.into(),
            cookieable_schemes_list: value.cookieable_schemes_list.into(),
            cookieable_schemes_exclude_defaults: value.cookieable_schemes_exclude_defaults,
        }
    }
}
impl From<RequestContextSettings> for _cef_request_context_settings_t {
    fn from(value: RequestContextSettings) -> Self {
        Self {
            size: value.size,
            cache_path: value.cache_path.into(),
            persist_session_cookies: value.persist_session_cookies,
            accept_language_list: value.accept_language_list.into(),
            cookieable_schemes_list: value.cookieable_schemes_list.into(),
            cookieable_schemes_exclude_defaults: value.cookieable_schemes_exclude_defaults,
        }
    }
}
impl Default for RequestContextSettings {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_request_context_settings_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_browser_settings_t`] for more documentation.
#[derive(Clone)]
pub struct BrowserSettings {
    pub size: usize,
    pub windowless_frame_rate: ::std::os::raw::c_int,
    pub standard_font_family: CefString,
    pub fixed_font_family: CefString,
    pub serif_font_family: CefString,
    pub sans_serif_font_family: CefString,
    pub cursive_font_family: CefString,
    pub fantasy_font_family: CefString,
    pub default_font_size: ::std::os::raw::c_int,
    pub default_fixed_font_size: ::std::os::raw::c_int,
    pub minimum_font_size: ::std::os::raw::c_int,
    pub minimum_logical_font_size: ::std::os::raw::c_int,
    pub default_encoding: CefString,
    pub remote_fonts: State,
    pub javascript: State,
    pub javascript_close_windows: State,
    pub javascript_access_clipboard: State,
    pub javascript_dom_paste: State,
    pub image_loading: State,
    pub image_shrink_standalone_to_fit: State,
    pub text_area_resize: State,
    pub tab_to_links: State,
    pub local_storage: State,
    pub databases_deprecated: State,
    pub webgl: State,
    pub background_color: u32,
    pub chrome_status_bubble: State,
    pub chrome_zoom_bubble: State,
}
impl From<_cef_browser_settings_t> for BrowserSettings {
    fn from(value: _cef_browser_settings_t) -> Self {
        Self {
            size: value.size,
            windowless_frame_rate: value.windowless_frame_rate,
            standard_font_family: value.standard_font_family.into(),
            fixed_font_family: value.fixed_font_family.into(),
            serif_font_family: value.serif_font_family.into(),
            sans_serif_font_family: value.sans_serif_font_family.into(),
            cursive_font_family: value.cursive_font_family.into(),
            fantasy_font_family: value.fantasy_font_family.into(),
            default_font_size: value.default_font_size,
            default_fixed_font_size: value.default_fixed_font_size,
            minimum_font_size: value.minimum_font_size,
            minimum_logical_font_size: value.minimum_logical_font_size,
            default_encoding: value.default_encoding.into(),
            remote_fonts: value.remote_fonts.into(),
            javascript: value.javascript.into(),
            javascript_close_windows: value.javascript_close_windows.into(),
            javascript_access_clipboard: value.javascript_access_clipboard.into(),
            javascript_dom_paste: value.javascript_dom_paste.into(),
            image_loading: value.image_loading.into(),
            image_shrink_standalone_to_fit: value.image_shrink_standalone_to_fit.into(),
            text_area_resize: value.text_area_resize.into(),
            tab_to_links: value.tab_to_links.into(),
            local_storage: value.local_storage.into(),
            databases_deprecated: value.databases_deprecated.into(),
            webgl: value.webgl.into(),
            background_color: value.background_color,
            chrome_status_bubble: value.chrome_status_bubble.into(),
            chrome_zoom_bubble: value.chrome_zoom_bubble.into(),
        }
    }
}
impl From<BrowserSettings> for _cef_browser_settings_t {
    fn from(value: BrowserSettings) -> Self {
        Self {
            size: value.size,
            windowless_frame_rate: value.windowless_frame_rate,
            standard_font_family: value.standard_font_family.into(),
            fixed_font_family: value.fixed_font_family.into(),
            serif_font_family: value.serif_font_family.into(),
            sans_serif_font_family: value.sans_serif_font_family.into(),
            cursive_font_family: value.cursive_font_family.into(),
            fantasy_font_family: value.fantasy_font_family.into(),
            default_font_size: value.default_font_size,
            default_fixed_font_size: value.default_fixed_font_size,
            minimum_font_size: value.minimum_font_size,
            minimum_logical_font_size: value.minimum_logical_font_size,
            default_encoding: value.default_encoding.into(),
            remote_fonts: value.remote_fonts.into(),
            javascript: value.javascript.into(),
            javascript_close_windows: value.javascript_close_windows.into(),
            javascript_access_clipboard: value.javascript_access_clipboard.into(),
            javascript_dom_paste: value.javascript_dom_paste.into(),
            image_loading: value.image_loading.into(),
            image_shrink_standalone_to_fit: value.image_shrink_standalone_to_fit.into(),
            text_area_resize: value.text_area_resize.into(),
            tab_to_links: value.tab_to_links.into(),
            local_storage: value.local_storage.into(),
            databases_deprecated: value.databases_deprecated.into(),
            webgl: value.webgl.into(),
            background_color: value.background_color,
            chrome_status_bubble: value.chrome_status_bubble.into(),
            chrome_zoom_bubble: value.chrome_zoom_bubble.into(),
        }
    }
}
impl Default for BrowserSettings {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_browser_settings_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_urlparts_t`] for more documentation.
#[derive(Clone)]
pub struct Urlparts {
    pub size: usize,
    pub spec: CefString,
    pub scheme: CefString,
    pub username: CefString,
    pub password: CefString,
    pub host: CefString,
    pub port: CefString,
    pub origin: CefString,
    pub path: CefString,
    pub query: CefString,
    pub fragment: CefString,
}
impl From<_cef_urlparts_t> for Urlparts {
    fn from(value: _cef_urlparts_t) -> Self {
        Self {
            size: value.size,
            spec: value.spec.into(),
            scheme: value.scheme.into(),
            username: value.username.into(),
            password: value.password.into(),
            host: value.host.into(),
            port: value.port.into(),
            origin: value.origin.into(),
            path: value.path.into(),
            query: value.query.into(),
            fragment: value.fragment.into(),
        }
    }
}
impl From<Urlparts> for _cef_urlparts_t {
    fn from(value: Urlparts) -> Self {
        Self {
            size: value.size,
            spec: value.spec.into(),
            scheme: value.scheme.into(),
            username: value.username.into(),
            password: value.password.into(),
            host: value.host.into(),
            port: value.port.into(),
            origin: value.origin.into(),
            path: value.path.into(),
            query: value.query.into(),
            fragment: value.fragment.into(),
        }
    }
}
impl Default for Urlparts {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_urlparts_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_cookie_t`] for more documentation.
#[derive(Clone)]
pub struct Cookie {
    pub size: usize,
    pub name: CefString,
    pub value: CefString,
    pub domain: CefString,
    pub path: CefString,
    pub secure: ::std::os::raw::c_int,
    pub httponly: ::std::os::raw::c_int,
    pub creation: Basetime,
    pub last_access: Basetime,
    pub has_expires: ::std::os::raw::c_int,
    pub expires: Basetime,
    pub same_site: CookieSameSite,
    pub priority: CookiePriority,
}
impl From<_cef_cookie_t> for Cookie {
    fn from(value: _cef_cookie_t) -> Self {
        Self {
            size: value.size,
            name: value.name.into(),
            value: value.value.into(),
            domain: value.domain.into(),
            path: value.path.into(),
            secure: value.secure,
            httponly: value.httponly,
            creation: value.creation.into(),
            last_access: value.last_access.into(),
            has_expires: value.has_expires,
            expires: value.expires.into(),
            same_site: value.same_site.into(),
            priority: value.priority.into(),
        }
    }
}
impl From<Cookie> for _cef_cookie_t {
    fn from(value: Cookie) -> Self {
        Self {
            size: value.size,
            name: value.name.into(),
            value: value.value.into(),
            domain: value.domain.into(),
            path: value.path.into(),
            secure: value.secure,
            httponly: value.httponly,
            creation: value.creation.into(),
            last_access: value.last_access.into(),
            has_expires: value.has_expires,
            expires: value.expires.into(),
            same_site: value.same_site.into(),
            priority: value.priority.into(),
        }
    }
}
impl Default for Cookie {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_cookie_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_draggable_region_t`] for more documentation.
#[derive(Clone)]
pub struct DraggableRegion {
    pub bounds: Rect,
    pub draggable: ::std::os::raw::c_int,
}
impl From<_cef_draggable_region_t> for DraggableRegion {
    fn from(value: _cef_draggable_region_t) -> Self {
        Self {
            bounds: value.bounds.into(),
            draggable: value.draggable,
        }
    }
}
impl From<DraggableRegion> for _cef_draggable_region_t {
    fn from(value: DraggableRegion) -> Self {
        Self {
            bounds: value.bounds.into(),
            draggable: value.draggable,
        }
    }
}
impl Default for DraggableRegion {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_screen_info_t`] for more documentation.
#[derive(Clone)]
pub struct ScreenInfo {
    pub size: usize,
    pub device_scale_factor: f32,
    pub depth: ::std::os::raw::c_int,
    pub depth_per_component: ::std::os::raw::c_int,
    pub is_monochrome: ::std::os::raw::c_int,
    pub rect: Rect,
    pub available_rect: Rect,
}
impl From<_cef_screen_info_t> for ScreenInfo {
    fn from(value: _cef_screen_info_t) -> Self {
        Self {
            size: value.size,
            device_scale_factor: value.device_scale_factor,
            depth: value.depth,
            depth_per_component: value.depth_per_component,
            is_monochrome: value.is_monochrome,
            rect: value.rect.into(),
            available_rect: value.available_rect.into(),
        }
    }
}
impl From<ScreenInfo> for _cef_screen_info_t {
    fn from(value: ScreenInfo) -> Self {
        Self {
            size: value.size,
            device_scale_factor: value.device_scale_factor,
            depth: value.depth,
            depth_per_component: value.depth_per_component,
            is_monochrome: value.is_monochrome,
            rect: value.rect.into(),
            available_rect: value.available_rect.into(),
        }
    }
}
impl Default for ScreenInfo {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_screen_info_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_linux_window_properties_t`] for more documentation.
#[derive(Clone)]
pub struct LinuxWindowProperties {
    pub size: usize,
    pub wayland_app_id: CefString,
    pub wm_class_class: CefString,
    pub wm_class_name: CefString,
    pub wm_role_name: CefString,
}
impl From<_cef_linux_window_properties_t> for LinuxWindowProperties {
    fn from(value: _cef_linux_window_properties_t) -> Self {
        Self {
            size: value.size,
            wayland_app_id: value.wayland_app_id.into(),
            wm_class_class: value.wm_class_class.into(),
            wm_class_name: value.wm_class_name.into(),
            wm_role_name: value.wm_role_name.into(),
        }
    }
}
impl From<LinuxWindowProperties> for _cef_linux_window_properties_t {
    fn from(value: LinuxWindowProperties) -> Self {
        Self {
            size: value.size,
            wayland_app_id: value.wayland_app_id.into(),
            wm_class_class: value.wm_class_class.into(),
            wm_class_name: value.wm_class_name.into(),
            wm_role_name: value.wm_role_name.into(),
        }
    }
}
impl Default for LinuxWindowProperties {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_linux_window_properties_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_mouse_event_t`] for more documentation.
#[derive(Clone)]
pub struct MouseEvent {
    pub x: ::std::os::raw::c_int,
    pub y: ::std::os::raw::c_int,
    pub modifiers: u32,
}
impl From<_cef_mouse_event_t> for MouseEvent {
    fn from(value: _cef_mouse_event_t) -> Self {
        Self {
            x: value.x,
            y: value.y,
            modifiers: value.modifiers,
        }
    }
}
impl From<MouseEvent> for _cef_mouse_event_t {
    fn from(value: MouseEvent) -> Self {
        Self {
            x: value.x,
            y: value.y,
            modifiers: value.modifiers,
        }
    }
}
impl Default for MouseEvent {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_touch_event_t`] for more documentation.
#[derive(Clone)]
pub struct TouchEvent {
    pub id: ::std::os::raw::c_int,
    pub x: f32,
    pub y: f32,
    pub radius_x: f32,
    pub radius_y: f32,
    pub rotation_angle: f32,
    pub pressure: f32,
    pub type_: TouchEventType,
    pub modifiers: u32,
    pub pointer_type: PointerType,
}
impl From<_cef_touch_event_t> for TouchEvent {
    fn from(value: _cef_touch_event_t) -> Self {
        Self {
            id: value.id,
            x: value.x,
            y: value.y,
            radius_x: value.radius_x,
            radius_y: value.radius_y,
            rotation_angle: value.rotation_angle,
            pressure: value.pressure,
            type_: value.type_.into(),
            modifiers: value.modifiers,
            pointer_type: value.pointer_type.into(),
        }
    }
}
impl From<TouchEvent> for _cef_touch_event_t {
    fn from(value: TouchEvent) -> Self {
        Self {
            id: value.id,
            x: value.x,
            y: value.y,
            radius_x: value.radius_x,
            radius_y: value.radius_y,
            rotation_angle: value.rotation_angle,
            pressure: value.pressure,
            type_: value.type_.into(),
            modifiers: value.modifiers,
            pointer_type: value.pointer_type.into(),
        }
    }
}
impl Default for TouchEvent {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_key_event_t`] for more documentation.
#[derive(Clone)]
pub struct KeyEvent {
    pub size: usize,
    pub type_: KeyEventType,
    pub modifiers: u32,
    pub windows_key_code: ::std::os::raw::c_int,
    pub native_key_code: ::std::os::raw::c_int,
    pub is_system_key: ::std::os::raw::c_int,
    pub character: char16_t,
    pub unmodified_character: char16_t,
    pub focus_on_editable_field: ::std::os::raw::c_int,
}
impl From<_cef_key_event_t> for KeyEvent {
    fn from(value: _cef_key_event_t) -> Self {
        Self {
            size: value.size,
            type_: value.type_.into(),
            modifiers: value.modifiers,
            windows_key_code: value.windows_key_code,
            native_key_code: value.native_key_code,
            is_system_key: value.is_system_key,
            character: value.character,
            unmodified_character: value.unmodified_character,
            focus_on_editable_field: value.focus_on_editable_field,
        }
    }
}
impl From<KeyEvent> for _cef_key_event_t {
    fn from(value: KeyEvent) -> Self {
        Self {
            size: value.size,
            type_: value.type_.into(),
            modifiers: value.modifiers,
            windows_key_code: value.windows_key_code,
            native_key_code: value.native_key_code,
            is_system_key: value.is_system_key,
            character: value.character,
            unmodified_character: value.unmodified_character,
            focus_on_editable_field: value.focus_on_editable_field,
        }
    }
}
impl Default for KeyEvent {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_key_event_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_popup_features_t`] for more documentation.
#[derive(Clone)]
pub struct PopupFeatures {
    pub size: usize,
    pub x: ::std::os::raw::c_int,
    pub x_set: ::std::os::raw::c_int,
    pub y: ::std::os::raw::c_int,
    pub y_set: ::std::os::raw::c_int,
    pub width: ::std::os::raw::c_int,
    pub width_set: ::std::os::raw::c_int,
    pub height: ::std::os::raw::c_int,
    pub height_set: ::std::os::raw::c_int,
    pub is_popup: ::std::os::raw::c_int,
}
impl From<_cef_popup_features_t> for PopupFeatures {
    fn from(value: _cef_popup_features_t) -> Self {
        Self {
            size: value.size,
            x: value.x,
            x_set: value.xSet,
            y: value.y,
            y_set: value.ySet,
            width: value.width,
            width_set: value.widthSet,
            height: value.height,
            height_set: value.heightSet,
            is_popup: value.isPopup,
        }
    }
}
impl From<PopupFeatures> for _cef_popup_features_t {
    fn from(value: PopupFeatures) -> Self {
        Self {
            size: value.size,
            x: value.x,
            xSet: value.x_set,
            y: value.y,
            ySet: value.y_set,
            width: value.width,
            widthSet: value.width_set,
            height: value.height,
            heightSet: value.height_set,
            isPopup: value.is_popup,
        }
    }
}
impl Default for PopupFeatures {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_popup_features_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_cursor_info_t`] for more documentation.
#[derive(Clone)]
pub struct CursorInfo {
    pub hotspot: Point,
    pub image_scale_factor: f32,
    pub buffer: *mut ::std::os::raw::c_void,
    pub size: Size,
}
impl From<_cef_cursor_info_t> for CursorInfo {
    fn from(value: _cef_cursor_info_t) -> Self {
        Self {
            hotspot: value.hotspot.into(),
            image_scale_factor: value.image_scale_factor,
            buffer: value.buffer,
            size: value.size.into(),
        }
    }
}
impl From<CursorInfo> for _cef_cursor_info_t {
    fn from(value: CursorInfo) -> Self {
        Self {
            hotspot: value.hotspot.into(),
            image_scale_factor: value.image_scale_factor,
            buffer: value.buffer,
            size: value.size.into(),
        }
    }
}
impl Default for CursorInfo {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_pdf_print_settings_t`] for more documentation.
#[derive(Clone)]
pub struct PdfPrintSettings {
    pub size: usize,
    pub landscape: ::std::os::raw::c_int,
    pub print_background: ::std::os::raw::c_int,
    pub scale: f64,
    pub paper_width: f64,
    pub paper_height: f64,
    pub prefer_css_page_size: ::std::os::raw::c_int,
    pub margin_type: PdfPrintMarginType,
    pub margin_top: f64,
    pub margin_right: f64,
    pub margin_bottom: f64,
    pub margin_left: f64,
    pub page_ranges: CefString,
    pub display_header_footer: ::std::os::raw::c_int,
    pub header_template: CefString,
    pub footer_template: CefString,
    pub generate_tagged_pdf: ::std::os::raw::c_int,
    pub generate_document_outline: ::std::os::raw::c_int,
}
impl From<_cef_pdf_print_settings_t> for PdfPrintSettings {
    fn from(value: _cef_pdf_print_settings_t) -> Self {
        Self {
            size: value.size,
            landscape: value.landscape,
            print_background: value.print_background,
            scale: value.scale,
            paper_width: value.paper_width,
            paper_height: value.paper_height,
            prefer_css_page_size: value.prefer_css_page_size,
            margin_type: value.margin_type.into(),
            margin_top: value.margin_top,
            margin_right: value.margin_right,
            margin_bottom: value.margin_bottom,
            margin_left: value.margin_left,
            page_ranges: value.page_ranges.into(),
            display_header_footer: value.display_header_footer,
            header_template: value.header_template.into(),
            footer_template: value.footer_template.into(),
            generate_tagged_pdf: value.generate_tagged_pdf,
            generate_document_outline: value.generate_document_outline,
        }
    }
}
impl From<PdfPrintSettings> for _cef_pdf_print_settings_t {
    fn from(value: PdfPrintSettings) -> Self {
        Self {
            size: value.size,
            landscape: value.landscape,
            print_background: value.print_background,
            scale: value.scale,
            paper_width: value.paper_width,
            paper_height: value.paper_height,
            prefer_css_page_size: value.prefer_css_page_size,
            margin_type: value.margin_type.into(),
            margin_top: value.margin_top,
            margin_right: value.margin_right,
            margin_bottom: value.margin_bottom,
            margin_left: value.margin_left,
            page_ranges: value.page_ranges.into(),
            display_header_footer: value.display_header_footer,
            header_template: value.header_template.into(),
            footer_template: value.footer_template.into(),
            generate_tagged_pdf: value.generate_tagged_pdf,
            generate_document_outline: value.generate_document_outline,
        }
    }
}
impl Default for PdfPrintSettings {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_pdf_print_settings_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_box_layout_settings_t`] for more documentation.
#[derive(Clone)]
pub struct BoxLayoutSettings {
    pub size: usize,
    pub horizontal: ::std::os::raw::c_int,
    pub inside_border_horizontal_spacing: ::std::os::raw::c_int,
    pub inside_border_vertical_spacing: ::std::os::raw::c_int,
    pub inside_border_insets: Insets,
    pub between_child_spacing: ::std::os::raw::c_int,
    pub main_axis_alignment: AxisAlignment,
    pub cross_axis_alignment: AxisAlignment,
    pub minimum_cross_axis_size: ::std::os::raw::c_int,
    pub default_flex: ::std::os::raw::c_int,
}
impl From<_cef_box_layout_settings_t> for BoxLayoutSettings {
    fn from(value: _cef_box_layout_settings_t) -> Self {
        Self {
            size: value.size,
            horizontal: value.horizontal,
            inside_border_horizontal_spacing: value.inside_border_horizontal_spacing,
            inside_border_vertical_spacing: value.inside_border_vertical_spacing,
            inside_border_insets: value.inside_border_insets.into(),
            between_child_spacing: value.between_child_spacing,
            main_axis_alignment: value.main_axis_alignment.into(),
            cross_axis_alignment: value.cross_axis_alignment.into(),
            minimum_cross_axis_size: value.minimum_cross_axis_size,
            default_flex: value.default_flex,
        }
    }
}
impl From<BoxLayoutSettings> for _cef_box_layout_settings_t {
    fn from(value: BoxLayoutSettings) -> Self {
        Self {
            size: value.size,
            horizontal: value.horizontal,
            inside_border_horizontal_spacing: value.inside_border_horizontal_spacing,
            inside_border_vertical_spacing: value.inside_border_vertical_spacing,
            inside_border_insets: value.inside_border_insets.into(),
            between_child_spacing: value.between_child_spacing,
            main_axis_alignment: value.main_axis_alignment.into(),
            cross_axis_alignment: value.cross_axis_alignment.into(),
            minimum_cross_axis_size: value.minimum_cross_axis_size,
            default_flex: value.default_flex,
        }
    }
}
impl Default for BoxLayoutSettings {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_box_layout_settings_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_range_t`] for more documentation.
#[derive(Clone)]
pub struct Range {
    pub from: u32,
    pub to: u32,
}
impl From<_cef_range_t> for Range {
    fn from(value: _cef_range_t) -> Self {
        Self {
            from: value.from,
            to: value.to,
        }
    }
}
impl From<Range> for _cef_range_t {
    fn from(value: Range) -> Self {
        Self {
            from: value.from,
            to: value.to,
        }
    }
}
impl Default for Range {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_composition_underline_t`] for more documentation.
#[derive(Clone)]
pub struct CompositionUnderline {
    pub size: usize,
    pub range: Range,
    pub color: u32,
    pub background_color: u32,
    pub thick: ::std::os::raw::c_int,
    pub style: CompositionUnderlineStyle,
}
impl From<_cef_composition_underline_t> for CompositionUnderline {
    fn from(value: _cef_composition_underline_t) -> Self {
        Self {
            size: value.size,
            range: value.range.into(),
            color: value.color,
            background_color: value.background_color,
            thick: value.thick,
            style: value.style.into(),
        }
    }
}
impl From<CompositionUnderline> for _cef_composition_underline_t {
    fn from(value: CompositionUnderline) -> Self {
        Self {
            size: value.size,
            range: value.range.into(),
            color: value.color,
            background_color: value.background_color,
            thick: value.thick,
            style: value.style.into(),
        }
    }
}
impl Default for CompositionUnderline {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_composition_underline_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_audio_parameters_t`] for more documentation.
#[derive(Clone)]
pub struct AudioParameters {
    pub size: usize,
    pub channel_layout: ChannelLayout,
    pub sample_rate: ::std::os::raw::c_int,
    pub frames_per_buffer: ::std::os::raw::c_int,
}
impl From<_cef_audio_parameters_t> for AudioParameters {
    fn from(value: _cef_audio_parameters_t) -> Self {
        Self {
            size: value.size,
            channel_layout: value.channel_layout.into(),
            sample_rate: value.sample_rate,
            frames_per_buffer: value.frames_per_buffer,
        }
    }
}
impl From<AudioParameters> for _cef_audio_parameters_t {
    fn from(value: AudioParameters) -> Self {
        Self {
            size: value.size,
            channel_layout: value.channel_layout.into(),
            sample_rate: value.sample_rate,
            frames_per_buffer: value.frames_per_buffer,
        }
    }
}
impl Default for AudioParameters {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_audio_parameters_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_media_sink_device_info_t`] for more documentation.
#[derive(Clone)]
pub struct MediaSinkDeviceInfo {
    pub size: usize,
    pub ip_address: CefString,
    pub port: ::std::os::raw::c_int,
    pub model_name: CefString,
}
impl From<_cef_media_sink_device_info_t> for MediaSinkDeviceInfo {
    fn from(value: _cef_media_sink_device_info_t) -> Self {
        Self {
            size: value.size,
            ip_address: value.ip_address.into(),
            port: value.port,
            model_name: value.model_name.into(),
        }
    }
}
impl From<MediaSinkDeviceInfo> for _cef_media_sink_device_info_t {
    fn from(value: MediaSinkDeviceInfo) -> Self {
        Self {
            size: value.size,
            ip_address: value.ip_address.into(),
            port: value.port,
            model_name: value.model_name.into(),
        }
    }
}
impl Default for MediaSinkDeviceInfo {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_media_sink_device_info_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_touch_handle_state_t`] for more documentation.
#[derive(Clone)]
pub struct TouchHandleState {
    pub size: usize,
    pub touch_handle_id: ::std::os::raw::c_int,
    pub flags: u32,
    pub enabled: ::std::os::raw::c_int,
    pub orientation: HorizontalAlignment,
    pub mirror_vertical: ::std::os::raw::c_int,
    pub mirror_horizontal: ::std::os::raw::c_int,
    pub origin: Point,
    pub alpha: f32,
}
impl From<_cef_touch_handle_state_t> for TouchHandleState {
    fn from(value: _cef_touch_handle_state_t) -> Self {
        Self {
            size: value.size,
            touch_handle_id: value.touch_handle_id,
            flags: value.flags,
            enabled: value.enabled,
            orientation: value.orientation.into(),
            mirror_vertical: value.mirror_vertical,
            mirror_horizontal: value.mirror_horizontal,
            origin: value.origin.into(),
            alpha: value.alpha,
        }
    }
}
impl From<TouchHandleState> for _cef_touch_handle_state_t {
    fn from(value: TouchHandleState) -> Self {
        Self {
            size: value.size,
            touch_handle_id: value.touch_handle_id,
            flags: value.flags,
            enabled: value.enabled,
            orientation: value.orientation.into(),
            mirror_vertical: value.mirror_vertical,
            mirror_horizontal: value.mirror_horizontal,
            origin: value.origin.into(),
            alpha: value.alpha,
        }
    }
}
impl Default for TouchHandleState {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_touch_handle_state_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_task_info_t`] for more documentation.
#[derive(Clone)]
pub struct TaskInfo {
    pub size: usize,
    pub id: i64,
    pub type_: TaskType,
    pub is_killable: ::std::os::raw::c_int,
    pub title: CefString,
    pub cpu_usage: f64,
    pub number_of_processors: ::std::os::raw::c_int,
    pub memory: i64,
    pub gpu_memory: i64,
    pub is_gpu_memory_inflated: ::std::os::raw::c_int,
}
impl From<_cef_task_info_t> for TaskInfo {
    fn from(value: _cef_task_info_t) -> Self {
        Self {
            size: value.size,
            id: value.id,
            type_: value.type_.into(),
            is_killable: value.is_killable,
            title: value.title.into(),
            cpu_usage: value.cpu_usage,
            number_of_processors: value.number_of_processors,
            memory: value.memory,
            gpu_memory: value.gpu_memory,
            is_gpu_memory_inflated: value.is_gpu_memory_inflated,
        }
    }
}
impl From<TaskInfo> for _cef_task_info_t {
    fn from(value: TaskInfo) -> Self {
        Self {
            size: value.size,
            id: value.id,
            type_: value.type_.into(),
            is_killable: value.is_killable,
            title: value.title.into(),
            cpu_usage: value.cpu_usage,
            number_of_processors: value.number_of_processors,
            memory: value.memory,
            gpu_memory: value.gpu_memory,
            is_gpu_memory_inflated: value.is_gpu_memory_inflated,
        }
    }
}
impl Default for TaskInfo {
    fn default() -> Self {
        Self {
            size: std::mem::size_of::<_cef_task_info_t>(),
            ..unsafe { std::mem::zeroed() }
        }
    }
}

/// See [`_cef_base_ref_counted_t`] for more documentation.
#[derive(Clone)]
pub struct BaseRefCounted(RefGuard<_cef_base_ref_counted_t>);
impl BaseRefCounted {
    fn get_raw(&self) -> *mut _cef_base_ref_counted_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for BaseRefCounted {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_base_ref_counted_t> for &BaseRefCounted {
    fn into_raw(self) -> *mut _cef_base_ref_counted_t {
        self.get_raw()
    }
}
impl ConvertParam<*mut _cef_base_ref_counted_t> for &mut BaseRefCounted {
    fn into_raw(self) -> *mut _cef_base_ref_counted_t {
        self.get_raw()
    }
}
impl ConvertReturnValue<BaseRefCounted> for *mut _cef_base_ref_counted_t {
    fn wrap_result(self) -> BaseRefCounted {
        BaseRefCounted(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<BaseRefCounted> for *mut _cef_base_ref_counted_t {
    fn from(value: BaseRefCounted) -> Self {
        let object = value.get_raw();
        std::mem::forget(value);
        object
    }
}
impl Default for BaseRefCounted {
    fn default() -> Self {
        Self(unsafe { RefGuard::from_raw(std::ptr::null_mut()) })
    }
}

/// See [`_cef_base_scoped_t`] for more documentation.
#[derive(Clone, Copy)]
pub struct BaseScoped(*mut _cef_base_scoped_t);
impl BaseScoped {
    fn get_raw(&self) -> *mut _cef_base_scoped_t {
        self.0
    }
}
impl ConvertParam<*mut _cef_base_scoped_t> for &BaseScoped {
    fn into_raw(self) -> *mut _cef_base_scoped_t {
        self.get_raw()
    }
}
impl ConvertParam<*mut _cef_base_scoped_t> for &mut BaseScoped {
    fn into_raw(self) -> *mut _cef_base_scoped_t {
        self.get_raw()
    }
}
impl ConvertReturnValue<BaseScoped> for *mut _cef_base_scoped_t {
    fn wrap_result(self) -> BaseScoped {
        BaseScoped(self)
    }
}
impl From<BaseScoped> for *mut _cef_base_scoped_t {
    fn from(value: BaseScoped) -> Self {
        value.get_raw()
    }
}
impl Default for BaseScoped {
    fn default() -> Self {
        Self(std::ptr::null_mut())
    }
}

/// See [`_cef_dev_tools_message_observer_t`] for more documentation.
#[derive(Clone)]
pub struct DevToolsMessageObserver(RefGuard<_cef_dev_tools_message_observer_t>);
impl DevToolsMessageObserver {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDevToolsMessageObserver,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDevToolsMessageObserver>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDevToolsMessageObserver>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_dev_tools_message_observer_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDevToolsMessageObserver: ImplDevToolsMessageObserver {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_dev_tools_message_observer_t, Self>);
}
pub trait ImplDevToolsMessageObserver: Clone + Sized + Rc {
    #[doc = "See [`_cef_dev_tools_message_observer_t::on_dev_tools_message`] for more documentation."]
    fn on_dev_tools_message(
        &self,
        browser: Option<&mut Browser>,
        message: Option<&[u8]>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_dev_tools_message_observer_t::on_dev_tools_method_result`] for more documentation."]
    fn on_dev_tools_method_result(
        &self,
        browser: Option<&mut Browser>,
        message_id: ::std::os::raw::c_int,
        success: ::std::os::raw::c_int,
        result: Option<&[u8]>,
    ) {
    }
    #[doc = "See [`_cef_dev_tools_message_observer_t::on_dev_tools_event`] for more documentation."]
    fn on_dev_tools_event(
        &self,
        browser: Option<&mut Browser>,
        method: Option<&CefString>,
        params: Option<&[u8]>,
    ) {
    }
    #[doc = "See [`_cef_dev_tools_message_observer_t::on_dev_tools_agent_attached`] for more documentation."]
    fn on_dev_tools_agent_attached(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_dev_tools_message_observer_t::on_dev_tools_agent_detached`] for more documentation."]
    fn on_dev_tools_agent_detached(&self, browser: Option<&mut Browser>) {}
    fn init_methods(object: &mut _cef_dev_tools_message_observer_t) {
        impl_cef_dev_tools_message_observer_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_dev_tools_message_observer_t;
}
mod impl_cef_dev_tools_message_observer_t {
    use super::*;
    pub fn init_methods<I: ImplDevToolsMessageObserver>(
        object: &mut _cef_dev_tools_message_observer_t,
    ) {
        object.on_dev_tools_message = Some(on_dev_tools_message::<I>);
        object.on_dev_tools_method_result = Some(on_dev_tools_method_result::<I>);
        object.on_dev_tools_event = Some(on_dev_tools_event::<I>);
        object.on_dev_tools_agent_attached = Some(on_dev_tools_agent_attached::<I>);
        object.on_dev_tools_agent_detached = Some(on_dev_tools_agent_detached::<I>);
    }
    extern "C" fn on_dev_tools_message<I: ImplDevToolsMessageObserver>(
        self_: *mut _cef_dev_tools_message_observer_t,
        browser: *mut _cef_browser_t,
        message: *const ::std::os::raw::c_void,
        message_size: usize,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_message, arg_message_size) =
            (self_, browser, message, message_size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_message = (!arg_message.is_null() && arg_message_size > 0)
            .then(|| unsafe { std::slice::from_raw_parts(arg_message.cast(), arg_message_size) });
        ImplDevToolsMessageObserver::on_dev_tools_message(
            &arg_self_.interface,
            arg_browser,
            arg_message,
        )
    }
    extern "C" fn on_dev_tools_method_result<I: ImplDevToolsMessageObserver>(
        self_: *mut _cef_dev_tools_message_observer_t,
        browser: *mut _cef_browser_t,
        message_id: ::std::os::raw::c_int,
        success: ::std::os::raw::c_int,
        result: *const ::std::os::raw::c_void,
        result_size: usize,
    ) {
        let (arg_self_, arg_browser, arg_message_id, arg_success, arg_result, arg_result_size) =
            (self_, browser, message_id, success, result, result_size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_message_id = arg_message_id.into_raw();
        let arg_success = arg_success.into_raw();
        let arg_result = (!arg_result.is_null() && arg_result_size > 0)
            .then(|| unsafe { std::slice::from_raw_parts(arg_result.cast(), arg_result_size) });
        ImplDevToolsMessageObserver::on_dev_tools_method_result(
            &arg_self_.interface,
            arg_browser,
            arg_message_id,
            arg_success,
            arg_result,
        )
    }
    extern "C" fn on_dev_tools_event<I: ImplDevToolsMessageObserver>(
        self_: *mut _cef_dev_tools_message_observer_t,
        browser: *mut _cef_browser_t,
        method: *const cef_string_t,
        params: *const ::std::os::raw::c_void,
        params_size: usize,
    ) {
        let (arg_self_, arg_browser, arg_method, arg_params, arg_params_size) =
            (self_, browser, method, params, params_size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_method = if arg_method.is_null() {
            None
        } else {
            Some(arg_method.into())
        };
        let arg_method = arg_method.as_ref();
        let arg_params = (!arg_params.is_null() && arg_params_size > 0)
            .then(|| unsafe { std::slice::from_raw_parts(arg_params.cast(), arg_params_size) });
        ImplDevToolsMessageObserver::on_dev_tools_event(
            &arg_self_.interface,
            arg_browser,
            arg_method,
            arg_params,
        )
    }
    extern "C" fn on_dev_tools_agent_attached<I: ImplDevToolsMessageObserver>(
        self_: *mut _cef_dev_tools_message_observer_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplDevToolsMessageObserver::on_dev_tools_agent_attached(&arg_self_.interface, arg_browser)
    }
    extern "C" fn on_dev_tools_agent_detached<I: ImplDevToolsMessageObserver>(
        self_: *mut _cef_dev_tools_message_observer_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplDevToolsMessageObserver::on_dev_tools_agent_detached(&arg_self_.interface, arg_browser)
    }
}
impl ImplDevToolsMessageObserver for DevToolsMessageObserver {
    fn on_dev_tools_message(
        &self,
        browser: Option<&mut Browser>,
        message: Option<&[u8]>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_dev_tools_message
                .map(|f| {
                    let (arg_browser, arg_message) = (browser, message);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_message_size = arg_message
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let arg_message = arg_message
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_browser, arg_message, arg_message_size);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_dev_tools_method_result(
        &self,
        browser: Option<&mut Browser>,
        message_id: ::std::os::raw::c_int,
        success: ::std::os::raw::c_int,
        result: Option<&[u8]>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_dev_tools_method_result {
                let (arg_browser, arg_message_id, arg_success, arg_result) =
                    (browser, message_id, success, result);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_result_size = arg_result.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let arg_result = arg_result
                    .and_then(|arg| {
                        if arg.is_empty() {
                            None
                        } else {
                            Some(arg.as_ptr().cast())
                        }
                    })
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_message_id,
                    arg_success,
                    arg_result,
                    arg_result_size,
                );
            }
        }
    }
    fn on_dev_tools_event(
        &self,
        browser: Option<&mut Browser>,
        method: Option<&CefString>,
        params: Option<&[u8]>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_dev_tools_event {
                let (arg_browser, arg_method, arg_params) = (browser, method, params);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_method = arg_method
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_params_size = arg_params.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let arg_params = arg_params
                    .and_then(|arg| {
                        if arg.is_empty() {
                            None
                        } else {
                            Some(arg.as_ptr().cast())
                        }
                    })
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_method,
                    arg_params,
                    arg_params_size,
                );
            }
        }
    }
    fn on_dev_tools_agent_attached(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_dev_tools_agent_attached {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn on_dev_tools_agent_detached(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_dev_tools_agent_detached {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_dev_tools_message_observer_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_dev_tools_message_observer_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DevToolsMessageObserver {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_dev_tools_message_observer_t> for &DevToolsMessageObserver {
    fn into_raw(self) -> *mut _cef_dev_tools_message_observer_t {
        ImplDevToolsMessageObserver::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_dev_tools_message_observer_t> for &mut DevToolsMessageObserver {
    fn into_raw(self) -> *mut _cef_dev_tools_message_observer_t {
        ImplDevToolsMessageObserver::get_raw(self)
    }
}
impl ConvertReturnValue<DevToolsMessageObserver> for *mut _cef_dev_tools_message_observer_t {
    fn wrap_result(self) -> DevToolsMessageObserver {
        DevToolsMessageObserver(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DevToolsMessageObserver> for *mut _cef_dev_tools_message_observer_t {
    fn from(value: DevToolsMessageObserver) -> Self {
        let object = ImplDevToolsMessageObserver::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DevToolsMessageObserver {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_value_t`] for more documentation.
#[derive(Clone)]
pub struct Value(RefGuard<_cef_value_t>);
pub trait ImplValue: Clone + Sized + Rc {
    #[doc = "See [`_cef_value_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::is_owned`] for more documentation."]
    fn is_owned(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut Value>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::is_equal`] for more documentation."]
    fn is_equal(&self, that: Option<&mut Value>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::copy`] for more documentation."]
    fn copy(&self) -> Option<Value>;
    #[doc = "See [`_cef_value_t::get_type`] for more documentation."]
    fn get_type(&self) -> ValueType;
    #[doc = "See [`_cef_value_t::get_bool`] for more documentation."]
    fn bool(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::get_int`] for more documentation."]
    fn int(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::get_double`] for more documentation."]
    fn double(&self) -> f64;
    #[doc = "See [`_cef_value_t::get_string`] for more documentation."]
    fn string(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_value_t::get_binary`] for more documentation."]
    fn binary(&self) -> Option<BinaryValue>;
    #[doc = "See [`_cef_value_t::get_dictionary`] for more documentation."]
    fn dictionary(&self) -> Option<DictionaryValue>;
    #[doc = "See [`_cef_value_t::get_list`] for more documentation."]
    fn list(&self) -> Option<ListValue>;
    #[doc = "See [`_cef_value_t::set_null`] for more documentation."]
    fn set_null(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_bool`] for more documentation."]
    fn set_bool(&self, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_int`] for more documentation."]
    fn set_int(&self, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_double`] for more documentation."]
    fn set_double(&self, value: f64) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_string`] for more documentation."]
    fn set_string(&self, value: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_binary`] for more documentation."]
    fn set_binary(&self, value: Option<&mut BinaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_dictionary`] for more documentation."]
    fn set_dictionary(&self, value: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_value_t::set_list`] for more documentation."]
    fn set_list(&self, value: Option<&mut ListValue>) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_value_t;
}
impl ImplValue for Value {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_owned(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_owned
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut Value>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_equal(&self, that: Option<&mut Value>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_equal
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn copy(&self) -> Option<Value> {
        unsafe {
            self.0
                .copy
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_type(&self) -> ValueType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn bool(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_bool
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn int(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_int
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn double(&self) -> f64 {
        unsafe {
            self.0
                .get_double
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn string(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_string
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn binary(&self) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_binary
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn dictionary(&self) -> Option<DictionaryValue> {
        unsafe {
            self.0
                .get_dictionary
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn list(&self) -> Option<ListValue> {
        unsafe {
            self.0
                .get_list
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_null(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_null
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_bool(&self, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_bool
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_int(&self, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_int
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_double(&self, value: f64) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_double
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_string(&self, value: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_string
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_binary(&self, value: Option<&mut BinaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_binary
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplBinaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_dictionary(&self, value: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_dictionary
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_list(&self, value: Option<&mut ListValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_list
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplListValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_value_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_value_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Value {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_value_t> for &Value {
    fn into_raw(self) -> *mut _cef_value_t {
        ImplValue::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_value_t> for &mut Value {
    fn into_raw(self) -> *mut _cef_value_t {
        ImplValue::get_raw(self)
    }
}
impl ConvertReturnValue<Value> for *mut _cef_value_t {
    fn wrap_result(self) -> Value {
        Value(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Value> for *mut _cef_value_t {
    fn from(value: Value) -> Self {
        let object = ImplValue::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Value {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_binary_value_t`] for more documentation.
#[derive(Clone)]
pub struct BinaryValue(RefGuard<_cef_binary_value_t>);
pub trait ImplBinaryValue: Clone + Sized + Rc {
    #[doc = "See [`_cef_binary_value_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_binary_value_t::is_owned`] for more documentation."]
    fn is_owned(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_binary_value_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut BinaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_binary_value_t::is_equal`] for more documentation."]
    fn is_equal(&self, that: Option<&mut BinaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_binary_value_t::copy`] for more documentation."]
    fn copy(&self) -> Option<BinaryValue>;
    #[doc = "See [`_cef_binary_value_t::get_raw_data`] for more documentation."]
    fn raw_data(&self) -> *const ::std::os::raw::c_void;
    #[doc = "See [`_cef_binary_value_t::get_size`] for more documentation."]
    fn size(&self) -> usize;
    #[doc = "See [`_cef_binary_value_t::get_data`] for more documentation."]
    fn data(&self, buffer: Option<&mut Vec<u8>>, data_offset: usize) -> usize;
    fn get_raw(&self) -> *mut _cef_binary_value_t;
}
impl ImplBinaryValue for BinaryValue {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_owned(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_owned
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut BinaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplBinaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_equal(&self, that: Option<&mut BinaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_equal
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplBinaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn copy(&self) -> Option<BinaryValue> {
        unsafe {
            self.0
                .copy
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn raw_data(&self) -> *const ::std::os::raw::c_void {
        unsafe {
            self.0
                .get_raw_data
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_else(|| std::mem::zeroed())
        }
    }
    fn size(&self) -> usize {
        unsafe {
            self.0
                .get_size
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn data(&self, buffer: Option<&mut Vec<u8>>, data_offset: usize) -> usize {
        unsafe {
            self.0
                .get_data
                .map(|f| {
                    let (arg_buffer, arg_data_offset) = (buffer, data_offset);
                    let arg_self_ = self.into_raw();
                    let arg_buffer_size =
                        arg_buffer.as_ref().map(|arg| arg.len()).unwrap_or_default();
                    let mut out_buffer = arg_buffer;
                    let arg_buffer = out_buffer
                        .as_mut()
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_mut_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_buffer, arg_buffer_size, arg_data_offset);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_binary_value_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_binary_value_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for BinaryValue {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_binary_value_t> for &BinaryValue {
    fn into_raw(self) -> *mut _cef_binary_value_t {
        ImplBinaryValue::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_binary_value_t> for &mut BinaryValue {
    fn into_raw(self) -> *mut _cef_binary_value_t {
        ImplBinaryValue::get_raw(self)
    }
}
impl ConvertReturnValue<BinaryValue> for *mut _cef_binary_value_t {
    fn wrap_result(self) -> BinaryValue {
        BinaryValue(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<BinaryValue> for *mut _cef_binary_value_t {
    fn from(value: BinaryValue) -> Self {
        let object = ImplBinaryValue::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for BinaryValue {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_dictionary_value_t`] for more documentation.
#[derive(Clone)]
pub struct DictionaryValue(RefGuard<_cef_dictionary_value_t>);
pub trait ImplDictionaryValue: Clone + Sized + Rc {
    #[doc = "See [`_cef_dictionary_value_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::is_owned`] for more documentation."]
    fn is_owned(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::is_equal`] for more documentation."]
    fn is_equal(&self, that: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::copy`] for more documentation."]
    fn copy(&self, exclude_empty_children: ::std::os::raw::c_int) -> Option<DictionaryValue>;
    #[doc = "See [`_cef_dictionary_value_t::get_size`] for more documentation."]
    fn size(&self) -> usize;
    #[doc = "See [`_cef_dictionary_value_t::clear`] for more documentation."]
    fn clear(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::has_key`] for more documentation."]
    fn has_key(&self, key: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::get_keys`] for more documentation."]
    fn keys(&self, keys: Option<&mut CefStringList>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::remove`] for more documentation."]
    fn remove(&self, key: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::get_type`] for more documentation."]
    fn get_type(&self, key: Option<&CefString>) -> ValueType;
    #[doc = "See [`_cef_dictionary_value_t::get_value`] for more documentation."]
    fn value(&self, key: Option<&CefString>) -> Option<Value>;
    #[doc = "See [`_cef_dictionary_value_t::get_bool`] for more documentation."]
    fn bool(&self, key: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::get_int`] for more documentation."]
    fn int(&self, key: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::get_double`] for more documentation."]
    fn double(&self, key: Option<&CefString>) -> f64;
    #[doc = "See [`_cef_dictionary_value_t::get_string`] for more documentation."]
    fn string(&self, key: Option<&CefString>) -> CefStringUserfree;
    #[doc = "See [`_cef_dictionary_value_t::get_binary`] for more documentation."]
    fn binary(&self, key: Option<&CefString>) -> Option<BinaryValue>;
    #[doc = "See [`_cef_dictionary_value_t::get_dictionary`] for more documentation."]
    fn dictionary(&self, key: Option<&CefString>) -> Option<DictionaryValue>;
    #[doc = "See [`_cef_dictionary_value_t::get_list`] for more documentation."]
    fn list(&self, key: Option<&CefString>) -> Option<ListValue>;
    #[doc = "See [`_cef_dictionary_value_t::set_value`] for more documentation."]
    fn set_value(
        &self,
        key: Option<&CefString>,
        value: Option<&mut Value>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_null`] for more documentation."]
    fn set_null(&self, key: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_bool`] for more documentation."]
    fn set_bool(
        &self,
        key: Option<&CefString>,
        value: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_int`] for more documentation."]
    fn set_int(
        &self,
        key: Option<&CefString>,
        value: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_double`] for more documentation."]
    fn set_double(&self, key: Option<&CefString>, value: f64) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_string`] for more documentation."]
    fn set_string(
        &self,
        key: Option<&CefString>,
        value: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_binary`] for more documentation."]
    fn set_binary(
        &self,
        key: Option<&CefString>,
        value: Option<&mut BinaryValue>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_dictionary`] for more documentation."]
    fn set_dictionary(
        &self,
        key: Option<&CefString>,
        value: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_dictionary_value_t::set_list`] for more documentation."]
    fn set_list(
        &self,
        key: Option<&CefString>,
        value: Option<&mut ListValue>,
    ) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_dictionary_value_t;
}
impl ImplDictionaryValue for DictionaryValue {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_owned(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_owned
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_equal(&self, that: Option<&mut DictionaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_equal
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn copy(&self, exclude_empty_children: ::std::os::raw::c_int) -> Option<DictionaryValue> {
        unsafe {
            self.0
                .copy
                .map(|f| {
                    let arg_exclude_empty_children = exclude_empty_children;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_exclude_empty_children);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn size(&self) -> usize {
        unsafe {
            self.0
                .get_size
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn clear(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .clear
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_key(&self, key: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_key
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn keys(&self, keys: Option<&mut CefStringList>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_keys
                .map(|f| {
                    let arg_keys = keys;
                    let arg_self_ = self.into_raw();
                    let arg_keys = arg_keys
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_keys);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove(&self, key: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_type(&self, key: Option<&CefString>) -> ValueType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn value(&self, key: Option<&CefString>) -> Option<Value> {
        unsafe {
            self.0
                .get_value
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn bool(&self, key: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_bool
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn int(&self, key: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_int
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn double(&self, key: Option<&CefString>) -> f64 {
        unsafe {
            self.0
                .get_double
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn string(&self, key: Option<&CefString>) -> CefStringUserfree {
        unsafe {
            self.0
                .get_string
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn binary(&self, key: Option<&CefString>) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_binary
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn dictionary(&self, key: Option<&CefString>) -> Option<DictionaryValue> {
        unsafe {
            self.0
                .get_dictionary
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn list(&self, key: Option<&CefString>) -> Option<ListValue> {
        unsafe {
            self.0
                .get_list
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_value(
        &self,
        key: Option<&CefString>,
        value: Option<&mut Value>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_value
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_null(&self, key: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_null
                .map(|f| {
                    let arg_key = key;
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_bool(
        &self,
        key: Option<&CefString>,
        value: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_bool
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_int(
        &self,
        key: Option<&CefString>,
        value: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_int
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_double(&self, key: Option<&CefString>, value: f64) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_double
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_string(
        &self,
        key: Option<&CefString>,
        value: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_string
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_binary(
        &self,
        key: Option<&CefString>,
        value: Option<&mut BinaryValue>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_binary
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplBinaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_dictionary(
        &self,
        key: Option<&CefString>,
        value: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_dictionary
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_list(
        &self,
        key: Option<&CefString>,
        value: Option<&mut ListValue>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_list
                .map(|f| {
                    let (arg_key, arg_value) = (key, value);
                    let arg_self_ = self.into_raw();
                    let arg_key = arg_key
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplListValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_key, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_dictionary_value_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_dictionary_value_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DictionaryValue {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_dictionary_value_t> for &DictionaryValue {
    fn into_raw(self) -> *mut _cef_dictionary_value_t {
        ImplDictionaryValue::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_dictionary_value_t> for &mut DictionaryValue {
    fn into_raw(self) -> *mut _cef_dictionary_value_t {
        ImplDictionaryValue::get_raw(self)
    }
}
impl ConvertReturnValue<DictionaryValue> for *mut _cef_dictionary_value_t {
    fn wrap_result(self) -> DictionaryValue {
        DictionaryValue(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DictionaryValue> for *mut _cef_dictionary_value_t {
    fn from(value: DictionaryValue) -> Self {
        let object = ImplDictionaryValue::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DictionaryValue {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_list_value_t`] for more documentation.
#[derive(Clone)]
pub struct ListValue(RefGuard<_cef_list_value_t>);
pub trait ImplListValue: Clone + Sized + Rc {
    #[doc = "See [`_cef_list_value_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::is_owned`] for more documentation."]
    fn is_owned(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut ListValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::is_equal`] for more documentation."]
    fn is_equal(&self, that: Option<&mut ListValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::copy`] for more documentation."]
    fn copy(&self) -> Option<ListValue>;
    #[doc = "See [`_cef_list_value_t::set_size`] for more documentation."]
    fn set_size(&self, size: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::get_size`] for more documentation."]
    fn size(&self) -> usize;
    #[doc = "See [`_cef_list_value_t::clear`] for more documentation."]
    fn clear(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::remove`] for more documentation."]
    fn remove(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::get_type`] for more documentation."]
    fn get_type(&self, index: usize) -> ValueType;
    #[doc = "See [`_cef_list_value_t::get_value`] for more documentation."]
    fn value(&self, index: usize) -> Option<Value>;
    #[doc = "See [`_cef_list_value_t::get_bool`] for more documentation."]
    fn bool(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::get_int`] for more documentation."]
    fn int(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::get_double`] for more documentation."]
    fn double(&self, index: usize) -> f64;
    #[doc = "See [`_cef_list_value_t::get_string`] for more documentation."]
    fn string(&self, index: usize) -> CefStringUserfree;
    #[doc = "See [`_cef_list_value_t::get_binary`] for more documentation."]
    fn binary(&self, index: usize) -> Option<BinaryValue>;
    #[doc = "See [`_cef_list_value_t::get_dictionary`] for more documentation."]
    fn dictionary(&self, index: usize) -> Option<DictionaryValue>;
    #[doc = "See [`_cef_list_value_t::get_list`] for more documentation."]
    fn list(&self, index: usize) -> Option<ListValue>;
    #[doc = "See [`_cef_list_value_t::set_value`] for more documentation."]
    fn set_value(&self, index: usize, value: Option<&mut Value>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_null`] for more documentation."]
    fn set_null(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_bool`] for more documentation."]
    fn set_bool(&self, index: usize, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_int`] for more documentation."]
    fn set_int(&self, index: usize, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_double`] for more documentation."]
    fn set_double(&self, index: usize, value: f64) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_string`] for more documentation."]
    fn set_string(&self, index: usize, value: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_binary`] for more documentation."]
    fn set_binary(&self, index: usize, value: Option<&mut BinaryValue>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_dictionary`] for more documentation."]
    fn set_dictionary(
        &self,
        index: usize,
        value: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_list_value_t::set_list`] for more documentation."]
    fn set_list(&self, index: usize, value: Option<&mut ListValue>) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_list_value_t;
}
impl ImplListValue for ListValue {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_owned(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_owned
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut ListValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplListValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_equal(&self, that: Option<&mut ListValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_equal
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplListValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn copy(&self) -> Option<ListValue> {
        unsafe {
            self.0
                .copy
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_size(&self, size: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_size
                .map(|f| {
                    let arg_size = size;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_size);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn size(&self) -> usize {
        unsafe {
            self.0
                .get_size
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn clear(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .clear
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_type(&self, index: usize) -> ValueType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn value(&self, index: usize) -> Option<Value> {
        unsafe {
            self.0
                .get_value
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn bool(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_bool
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn int(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_int
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn double(&self, index: usize) -> f64 {
        unsafe {
            self.0
                .get_double
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn string(&self, index: usize) -> CefStringUserfree {
        unsafe {
            self.0
                .get_string
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn binary(&self, index: usize) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_binary
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn dictionary(&self, index: usize) -> Option<DictionaryValue> {
        unsafe {
            self.0
                .get_dictionary
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn list(&self, index: usize) -> Option<ListValue> {
        unsafe {
            self.0
                .get_list
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_value(&self, index: usize, value: Option<&mut Value>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_value
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_null(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_null
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_bool(&self, index: usize, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_bool
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_int(&self, index: usize, value: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_int
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_double(&self, index: usize, value: f64) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_double
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_string(&self, index: usize, value: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_string
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_binary(&self, index: usize, value: Option<&mut BinaryValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_binary
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplBinaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_dictionary(
        &self,
        index: usize,
        value: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_dictionary
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_list(&self, index: usize, value: Option<&mut ListValue>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_list
                .map(|f| {
                    let (arg_index, arg_value) = (index, value);
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplListValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_index, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_list_value_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_list_value_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ListValue {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_list_value_t> for &ListValue {
    fn into_raw(self) -> *mut _cef_list_value_t {
        ImplListValue::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_list_value_t> for &mut ListValue {
    fn into_raw(self) -> *mut _cef_list_value_t {
        ImplListValue::get_raw(self)
    }
}
impl ConvertReturnValue<ListValue> for *mut _cef_list_value_t {
    fn wrap_result(self) -> ListValue {
        ListValue(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ListValue> for *mut _cef_list_value_t {
    fn from(value: ListValue) -> Self {
        let object = ImplListValue::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ListValue {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_image_t`] for more documentation.
#[derive(Clone)]
pub struct Image(RefGuard<_cef_image_t>);
pub trait ImplImage: Clone + Sized + Rc {
    #[doc = "See [`_cef_image_t::is_empty`] for more documentation."]
    fn is_empty(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut Image>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::add_bitmap`] for more documentation."]
    fn add_bitmap(
        &self,
        scale_factor: f32,
        pixel_width: ::std::os::raw::c_int,
        pixel_height: ::std::os::raw::c_int,
        color_type: ColorType,
        alpha_type: AlphaType,
        pixel_data: Option<&[u8]>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::add_png`] for more documentation."]
    fn add_png(&self, scale_factor: f32, png_data: Option<&[u8]>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::add_jpeg`] for more documentation."]
    fn add_jpeg(&self, scale_factor: f32, jpeg_data: Option<&[u8]>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::get_width`] for more documentation."]
    fn width(&self) -> usize;
    #[doc = "See [`_cef_image_t::get_height`] for more documentation."]
    fn height(&self) -> usize;
    #[doc = "See [`_cef_image_t::has_representation`] for more documentation."]
    fn has_representation(&self, scale_factor: f32) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::remove_representation`] for more documentation."]
    fn remove_representation(&self, scale_factor: f32) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::get_representation_info`] for more documentation."]
    fn representation_info(
        &self,
        scale_factor: f32,
        actual_scale_factor: Option<&mut f32>,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_image_t::get_as_bitmap`] for more documentation."]
    fn as_bitmap(
        &self,
        scale_factor: f32,
        color_type: ColorType,
        alpha_type: AlphaType,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue>;
    #[doc = "See [`_cef_image_t::get_as_png`] for more documentation."]
    fn as_png(
        &self,
        scale_factor: f32,
        with_transparency: ::std::os::raw::c_int,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue>;
    #[doc = "See [`_cef_image_t::get_as_jpeg`] for more documentation."]
    fn as_jpeg(
        &self,
        scale_factor: f32,
        quality: ::std::os::raw::c_int,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue>;
    fn get_raw(&self) -> *mut _cef_image_t;
}
impl ImplImage for Image {
    fn is_empty(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_empty
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut Image>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplImage::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_bitmap(
        &self,
        scale_factor: f32,
        pixel_width: ::std::os::raw::c_int,
        pixel_height: ::std::os::raw::c_int,
        color_type: ColorType,
        alpha_type: AlphaType,
        pixel_data: Option<&[u8]>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_bitmap
                .map(|f| {
                    let (
                        arg_scale_factor,
                        arg_pixel_width,
                        arg_pixel_height,
                        arg_color_type,
                        arg_alpha_type,
                        arg_pixel_data,
                    ) = (
                        scale_factor,
                        pixel_width,
                        pixel_height,
                        color_type,
                        alpha_type,
                        pixel_data,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let arg_alpha_type = arg_alpha_type.into_raw();
                    let arg_pixel_data_size = arg_pixel_data
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let arg_pixel_data = arg_pixel_data
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_pixel_width,
                        arg_pixel_height,
                        arg_color_type,
                        arg_alpha_type,
                        arg_pixel_data,
                        arg_pixel_data_size,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_png(&self, scale_factor: f32, png_data: Option<&[u8]>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_png
                .map(|f| {
                    let (arg_scale_factor, arg_png_data) = (scale_factor, png_data);
                    let arg_self_ = self.into_raw();
                    let arg_png_data_size = arg_png_data
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let arg_png_data = arg_png_data
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_scale_factor, arg_png_data, arg_png_data_size);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_jpeg(&self, scale_factor: f32, jpeg_data: Option<&[u8]>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_jpeg
                .map(|f| {
                    let (arg_scale_factor, arg_jpeg_data) = (scale_factor, jpeg_data);
                    let arg_self_ = self.into_raw();
                    let arg_jpeg_data_size = arg_jpeg_data
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let arg_jpeg_data = arg_jpeg_data
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_jpeg_data,
                        arg_jpeg_data_size,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn width(&self) -> usize {
        unsafe {
            self.0
                .get_width
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn height(&self) -> usize {
        unsafe {
            self.0
                .get_height
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_representation(&self, scale_factor: f32) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_representation
                .map(|f| {
                    let arg_scale_factor = scale_factor;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_scale_factor);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove_representation(&self, scale_factor: f32) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove_representation
                .map(|f| {
                    let arg_scale_factor = scale_factor;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_scale_factor);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn representation_info(
        &self,
        scale_factor: f32,
        actual_scale_factor: Option<&mut f32>,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_representation_info
                .map(|f| {
                    let (
                        arg_scale_factor,
                        arg_actual_scale_factor,
                        arg_pixel_width,
                        arg_pixel_height,
                    ) = (scale_factor, actual_scale_factor, pixel_width, pixel_height);
                    let arg_self_ = self.into_raw();
                    let arg_actual_scale_factor = arg_actual_scale_factor
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_pixel_width = arg_pixel_width
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_pixel_height = arg_pixel_height
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_actual_scale_factor,
                        arg_pixel_width,
                        arg_pixel_height,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn as_bitmap(
        &self,
        scale_factor: f32,
        color_type: ColorType,
        alpha_type: AlphaType,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_as_bitmap
                .map(|f| {
                    let (
                        arg_scale_factor,
                        arg_color_type,
                        arg_alpha_type,
                        arg_pixel_width,
                        arg_pixel_height,
                    ) = (
                        scale_factor,
                        color_type,
                        alpha_type,
                        pixel_width,
                        pixel_height,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let arg_alpha_type = arg_alpha_type.into_raw();
                    let arg_pixel_width = arg_pixel_width
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_pixel_height = arg_pixel_height
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_color_type,
                        arg_alpha_type,
                        arg_pixel_width,
                        arg_pixel_height,
                    );
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn as_png(
        &self,
        scale_factor: f32,
        with_transparency: ::std::os::raw::c_int,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_as_png
                .map(|f| {
                    let (
                        arg_scale_factor,
                        arg_with_transparency,
                        arg_pixel_width,
                        arg_pixel_height,
                    ) = (scale_factor, with_transparency, pixel_width, pixel_height);
                    let arg_self_ = self.into_raw();
                    let arg_pixel_width = arg_pixel_width
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_pixel_height = arg_pixel_height
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_with_transparency,
                        arg_pixel_width,
                        arg_pixel_height,
                    );
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn as_jpeg(
        &self,
        scale_factor: f32,
        quality: ::std::os::raw::c_int,
        pixel_width: Option<&mut ::std::os::raw::c_int>,
        pixel_height: Option<&mut ::std::os::raw::c_int>,
    ) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_as_jpeg
                .map(|f| {
                    let (arg_scale_factor, arg_quality, arg_pixel_width, arg_pixel_height) =
                        (scale_factor, quality, pixel_width, pixel_height);
                    let arg_self_ = self.into_raw();
                    let arg_pixel_width = arg_pixel_width
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_pixel_height = arg_pixel_height
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_scale_factor,
                        arg_quality,
                        arg_pixel_width,
                        arg_pixel_height,
                    );
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_image_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_image_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Image {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_image_t> for &Image {
    fn into_raw(self) -> *mut _cef_image_t {
        ImplImage::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_image_t> for &mut Image {
    fn into_raw(self) -> *mut _cef_image_t {
        ImplImage::get_raw(self)
    }
}
impl ConvertReturnValue<Image> for *mut _cef_image_t {
    fn wrap_result(self) -> Image {
        Image(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Image> for *mut _cef_image_t {
    fn from(value: Image) -> Self {
        let object = ImplImage::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Image {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_read_handler_t`] for more documentation.
#[derive(Clone)]
pub struct ReadHandler(RefGuard<_cef_read_handler_t>);
impl ReadHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapReadHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplReadHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapReadHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_read_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapReadHandler: ImplReadHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_read_handler_t, Self>);
}
pub trait ImplReadHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_read_handler_t::read`] for more documentation."]
    fn read(&self, ptr: *mut u8, size: usize, n: usize) -> usize {
        Default::default()
    }
    #[doc = "See [`_cef_read_handler_t::seek`] for more documentation."]
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_read_handler_t::tell`] for more documentation."]
    fn tell(&self) -> i64 {
        Default::default()
    }
    #[doc = "See [`_cef_read_handler_t::eof`] for more documentation."]
    fn eof(&self) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_read_handler_t::may_block`] for more documentation."]
    fn may_block(&self) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_read_handler_t) {
        impl_cef_read_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_read_handler_t;
}
mod impl_cef_read_handler_t {
    use super::*;
    pub fn init_methods<I: ImplReadHandler>(object: &mut _cef_read_handler_t) {
        object.read = Some(read::<I>);
        object.seek = Some(seek::<I>);
        object.tell = Some(tell::<I>);
        object.eof = Some(eof::<I>);
        object.may_block = Some(may_block::<I>);
    }
    extern "C" fn read<I: ImplReadHandler>(
        self_: *mut _cef_read_handler_t,
        ptr: *mut ::std::os::raw::c_void,
        size: usize,
        n: usize,
    ) -> usize {
        let (arg_self_, arg_ptr, arg_size, arg_n) = (self_, ptr, size, n);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_ptr = arg_ptr.cast();
        let arg_size = arg_size.into_raw();
        let arg_n = arg_n.into_raw();
        ImplReadHandler::read(&arg_self_.interface, arg_ptr, arg_size, arg_n)
    }
    extern "C" fn seek<I: ImplReadHandler>(
        self_: *mut _cef_read_handler_t,
        offset: i64,
        whence: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_offset, arg_whence) = (self_, offset, whence);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_offset = arg_offset.into_raw();
        let arg_whence = arg_whence.into_raw();
        ImplReadHandler::seek(&arg_self_.interface, arg_offset, arg_whence)
    }
    extern "C" fn tell<I: ImplReadHandler>(self_: *mut _cef_read_handler_t) -> i64 {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplReadHandler::tell(&arg_self_.interface)
    }
    extern "C" fn eof<I: ImplReadHandler>(
        self_: *mut _cef_read_handler_t,
    ) -> ::std::os::raw::c_int {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplReadHandler::eof(&arg_self_.interface)
    }
    extern "C" fn may_block<I: ImplReadHandler>(
        self_: *mut _cef_read_handler_t,
    ) -> ::std::os::raw::c_int {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplReadHandler::may_block(&arg_self_.interface)
    }
}
impl ImplReadHandler for ReadHandler {
    fn read(&self, ptr: *mut u8, size: usize, n: usize) -> usize {
        unsafe {
            self.0
                .read
                .map(|f| {
                    let (arg_ptr, arg_size, arg_n) = (ptr, size, n);
                    let arg_self_ = self.into_raw();
                    let arg_ptr = arg_ptr.cast();
                    let result = f(arg_self_, arg_ptr, arg_size, arg_n);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .seek
                .map(|f| {
                    let (arg_offset, arg_whence) = (offset, whence);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_offset, arg_whence);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn tell(&self) -> i64 {
        unsafe {
            self.0
                .tell
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn eof(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .eof
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn may_block(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .may_block
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_read_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_read_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ReadHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_read_handler_t> for &ReadHandler {
    fn into_raw(self) -> *mut _cef_read_handler_t {
        ImplReadHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_read_handler_t> for &mut ReadHandler {
    fn into_raw(self) -> *mut _cef_read_handler_t {
        ImplReadHandler::get_raw(self)
    }
}
impl ConvertReturnValue<ReadHandler> for *mut _cef_read_handler_t {
    fn wrap_result(self) -> ReadHandler {
        ReadHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ReadHandler> for *mut _cef_read_handler_t {
    fn from(value: ReadHandler) -> Self {
        let object = ImplReadHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ReadHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_stream_reader_t`] for more documentation.
#[derive(Clone)]
pub struct StreamReader(RefGuard<_cef_stream_reader_t>);
pub trait ImplStreamReader: Clone + Sized + Rc {
    #[doc = "See [`_cef_stream_reader_t::read`] for more documentation."]
    fn read(&self, ptr: *mut u8, size: usize, n: usize) -> usize;
    #[doc = "See [`_cef_stream_reader_t::seek`] for more documentation."]
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_stream_reader_t::tell`] for more documentation."]
    fn tell(&self) -> i64;
    #[doc = "See [`_cef_stream_reader_t::eof`] for more documentation."]
    fn eof(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_stream_reader_t::may_block`] for more documentation."]
    fn may_block(&self) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_stream_reader_t;
}
impl ImplStreamReader for StreamReader {
    fn read(&self, ptr: *mut u8, size: usize, n: usize) -> usize {
        unsafe {
            self.0
                .read
                .map(|f| {
                    let (arg_ptr, arg_size, arg_n) = (ptr, size, n);
                    let arg_self_ = self.into_raw();
                    let arg_ptr = arg_ptr.cast();
                    let result = f(arg_self_, arg_ptr, arg_size, arg_n);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .seek
                .map(|f| {
                    let (arg_offset, arg_whence) = (offset, whence);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_offset, arg_whence);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn tell(&self) -> i64 {
        unsafe {
            self.0
                .tell
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn eof(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .eof
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn may_block(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .may_block
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_stream_reader_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_stream_reader_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for StreamReader {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_stream_reader_t> for &StreamReader {
    fn into_raw(self) -> *mut _cef_stream_reader_t {
        ImplStreamReader::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_stream_reader_t> for &mut StreamReader {
    fn into_raw(self) -> *mut _cef_stream_reader_t {
        ImplStreamReader::get_raw(self)
    }
}
impl ConvertReturnValue<StreamReader> for *mut _cef_stream_reader_t {
    fn wrap_result(self) -> StreamReader {
        StreamReader(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<StreamReader> for *mut _cef_stream_reader_t {
    fn from(value: StreamReader) -> Self {
        let object = ImplStreamReader::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for StreamReader {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_write_handler_t`] for more documentation.
#[derive(Clone)]
pub struct WriteHandler(RefGuard<_cef_write_handler_t>);
impl WriteHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapWriteHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplWriteHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapWriteHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_write_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapWriteHandler: ImplWriteHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_write_handler_t, Self>);
}
pub trait ImplWriteHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_write_handler_t::write`] for more documentation."]
    fn write(&self, ptr: *const u8, size: usize, n: usize) -> usize {
        Default::default()
    }
    #[doc = "See [`_cef_write_handler_t::seek`] for more documentation."]
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_write_handler_t::tell`] for more documentation."]
    fn tell(&self) -> i64 {
        Default::default()
    }
    #[doc = "See [`_cef_write_handler_t::flush`] for more documentation."]
    fn flush(&self) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_write_handler_t::may_block`] for more documentation."]
    fn may_block(&self) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_write_handler_t) {
        impl_cef_write_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_write_handler_t;
}
mod impl_cef_write_handler_t {
    use super::*;
    pub fn init_methods<I: ImplWriteHandler>(object: &mut _cef_write_handler_t) {
        object.write = Some(write::<I>);
        object.seek = Some(seek::<I>);
        object.tell = Some(tell::<I>);
        object.flush = Some(flush::<I>);
        object.may_block = Some(may_block::<I>);
    }
    extern "C" fn write<I: ImplWriteHandler>(
        self_: *mut _cef_write_handler_t,
        ptr: *const ::std::os::raw::c_void,
        size: usize,
        n: usize,
    ) -> usize {
        let (arg_self_, arg_ptr, arg_size, arg_n) = (self_, ptr, size, n);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_ptr = arg_ptr.cast();
        let arg_size = arg_size.into_raw();
        let arg_n = arg_n.into_raw();
        ImplWriteHandler::write(&arg_self_.interface, arg_ptr, arg_size, arg_n)
    }
    extern "C" fn seek<I: ImplWriteHandler>(
        self_: *mut _cef_write_handler_t,
        offset: i64,
        whence: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_offset, arg_whence) = (self_, offset, whence);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_offset = arg_offset.into_raw();
        let arg_whence = arg_whence.into_raw();
        ImplWriteHandler::seek(&arg_self_.interface, arg_offset, arg_whence)
    }
    extern "C" fn tell<I: ImplWriteHandler>(self_: *mut _cef_write_handler_t) -> i64 {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplWriteHandler::tell(&arg_self_.interface)
    }
    extern "C" fn flush<I: ImplWriteHandler>(
        self_: *mut _cef_write_handler_t,
    ) -> ::std::os::raw::c_int {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplWriteHandler::flush(&arg_self_.interface)
    }
    extern "C" fn may_block<I: ImplWriteHandler>(
        self_: *mut _cef_write_handler_t,
    ) -> ::std::os::raw::c_int {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplWriteHandler::may_block(&arg_self_.interface)
    }
}
impl ImplWriteHandler for WriteHandler {
    fn write(&self, ptr: *const u8, size: usize, n: usize) -> usize {
        unsafe {
            self.0
                .write
                .map(|f| {
                    let (arg_ptr, arg_size, arg_n) = (ptr, size, n);
                    let arg_self_ = self.into_raw();
                    let arg_ptr = arg_ptr.cast();
                    let result = f(arg_self_, arg_ptr, arg_size, arg_n);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .seek
                .map(|f| {
                    let (arg_offset, arg_whence) = (offset, whence);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_offset, arg_whence);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn tell(&self) -> i64 {
        unsafe {
            self.0
                .tell
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn flush(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .flush
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn may_block(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .may_block
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_write_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_write_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for WriteHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_write_handler_t> for &WriteHandler {
    fn into_raw(self) -> *mut _cef_write_handler_t {
        ImplWriteHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_write_handler_t> for &mut WriteHandler {
    fn into_raw(self) -> *mut _cef_write_handler_t {
        ImplWriteHandler::get_raw(self)
    }
}
impl ConvertReturnValue<WriteHandler> for *mut _cef_write_handler_t {
    fn wrap_result(self) -> WriteHandler {
        WriteHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<WriteHandler> for *mut _cef_write_handler_t {
    fn from(value: WriteHandler) -> Self {
        let object = ImplWriteHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for WriteHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_stream_writer_t`] for more documentation.
#[derive(Clone)]
pub struct StreamWriter(RefGuard<_cef_stream_writer_t>);
pub trait ImplStreamWriter: Clone + Sized + Rc {
    #[doc = "See [`_cef_stream_writer_t::write`] for more documentation."]
    fn write(&self, ptr: *const u8, size: usize, n: usize) -> usize;
    #[doc = "See [`_cef_stream_writer_t::seek`] for more documentation."]
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_stream_writer_t::tell`] for more documentation."]
    fn tell(&self) -> i64;
    #[doc = "See [`_cef_stream_writer_t::flush`] for more documentation."]
    fn flush(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_stream_writer_t::may_block`] for more documentation."]
    fn may_block(&self) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_stream_writer_t;
}
impl ImplStreamWriter for StreamWriter {
    fn write(&self, ptr: *const u8, size: usize, n: usize) -> usize {
        unsafe {
            self.0
                .write
                .map(|f| {
                    let (arg_ptr, arg_size, arg_n) = (ptr, size, n);
                    let arg_self_ = self.into_raw();
                    let arg_ptr = arg_ptr.cast();
                    let result = f(arg_self_, arg_ptr, arg_size, arg_n);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn seek(&self, offset: i64, whence: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .seek
                .map(|f| {
                    let (arg_offset, arg_whence) = (offset, whence);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_offset, arg_whence);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn tell(&self) -> i64 {
        unsafe {
            self.0
                .tell
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn flush(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .flush
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn may_block(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .may_block
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_stream_writer_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_stream_writer_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for StreamWriter {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_stream_writer_t> for &StreamWriter {
    fn into_raw(self) -> *mut _cef_stream_writer_t {
        ImplStreamWriter::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_stream_writer_t> for &mut StreamWriter {
    fn into_raw(self) -> *mut _cef_stream_writer_t {
        ImplStreamWriter::get_raw(self)
    }
}
impl ConvertReturnValue<StreamWriter> for *mut _cef_stream_writer_t {
    fn wrap_result(self) -> StreamWriter {
        StreamWriter(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<StreamWriter> for *mut _cef_stream_writer_t {
    fn from(value: StreamWriter) -> Self {
        let object = ImplStreamWriter::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for StreamWriter {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_drag_data_t`] for more documentation.
#[derive(Clone)]
pub struct DragData(RefGuard<_cef_drag_data_t>);
pub trait ImplDragData: Clone + Sized + Rc {
    #[doc = "See [`_cef_drag_data_t::clone`] for more documentation."]
    fn clone(&self) -> Option<DragData>;
    #[doc = "See [`_cef_drag_data_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::is_link`] for more documentation."]
    fn is_link(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::is_fragment`] for more documentation."]
    fn is_fragment(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::is_file`] for more documentation."]
    fn is_file(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::get_link_url`] for more documentation."]
    fn link_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_link_title`] for more documentation."]
    fn link_title(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_link_metadata`] for more documentation."]
    fn link_metadata(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_fragment_text`] for more documentation."]
    fn fragment_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_fragment_html`] for more documentation."]
    fn fragment_html(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_fragment_base_url`] for more documentation."]
    fn fragment_base_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_file_name`] for more documentation."]
    fn file_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_drag_data_t::get_file_contents`] for more documentation."]
    fn file_contents(&self, writer: Option<&mut StreamWriter>) -> usize;
    #[doc = "See [`_cef_drag_data_t::get_file_names`] for more documentation."]
    fn file_names(&self, names: Option<&mut CefStringList>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::get_file_paths`] for more documentation."]
    fn file_paths(&self, paths: Option<&mut CefStringList>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_drag_data_t::set_link_url`] for more documentation."]
    fn set_link_url(&self, url: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::set_link_title`] for more documentation."]
    fn set_link_title(&self, title: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::set_link_metadata`] for more documentation."]
    fn set_link_metadata(&self, data: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::set_fragment_text`] for more documentation."]
    fn set_fragment_text(&self, text: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::set_fragment_html`] for more documentation."]
    fn set_fragment_html(&self, html: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::set_fragment_base_url`] for more documentation."]
    fn set_fragment_base_url(&self, base_url: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::reset_file_contents`] for more documentation."]
    fn reset_file_contents(&self);
    #[doc = "See [`_cef_drag_data_t::add_file`] for more documentation."]
    fn add_file(&self, path: Option<&CefString>, display_name: Option<&CefString>);
    #[doc = "See [`_cef_drag_data_t::clear_filenames`] for more documentation."]
    fn clear_filenames(&self);
    #[doc = "See [`_cef_drag_data_t::get_image`] for more documentation."]
    fn image(&self) -> Option<Image>;
    #[doc = "See [`_cef_drag_data_t::get_image_hotspot`] for more documentation."]
    fn image_hotspot(&self) -> Point;
    #[doc = "See [`_cef_drag_data_t::has_image`] for more documentation."]
    fn has_image(&self) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_drag_data_t;
}
impl ImplDragData for DragData {
    fn clone(&self) -> Option<DragData> {
        unsafe {
            self.0
                .clone
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_link(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_link
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_fragment(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_fragment
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_file(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_file
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn link_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_link_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn link_title(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_link_title
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn link_metadata(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_link_metadata
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn fragment_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_fragment_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn fragment_html(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_fragment_html
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn fragment_base_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_fragment_base_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn file_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_file_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn file_contents(&self, writer: Option<&mut StreamWriter>) -> usize {
        unsafe {
            self.0
                .get_file_contents
                .map(|f| {
                    let arg_writer = writer;
                    let arg_self_ = self.into_raw();
                    let arg_writer = arg_writer
                        .map(|arg| {
                            arg.add_ref();
                            ImplStreamWriter::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_writer);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn file_names(&self, names: Option<&mut CefStringList>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_file_names
                .map(|f| {
                    let arg_names = names;
                    let arg_self_ = self.into_raw();
                    let arg_names = arg_names
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_names);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn file_paths(&self, paths: Option<&mut CefStringList>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_file_paths
                .map(|f| {
                    let arg_paths = paths;
                    let arg_self_ = self.into_raw();
                    let arg_paths = arg_paths
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_paths);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_link_url(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_link_url {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn set_link_title(&self, title: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_link_title {
                let arg_title = title;
                let arg_self_ = self.into_raw();
                let arg_title = arg_title
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_title);
            }
        }
    }
    fn set_link_metadata(&self, data: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_link_metadata {
                let arg_data = data;
                let arg_self_ = self.into_raw();
                let arg_data = arg_data
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_data);
            }
        }
    }
    fn set_fragment_text(&self, text: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_fragment_text {
                let arg_text = text;
                let arg_self_ = self.into_raw();
                let arg_text = arg_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_text);
            }
        }
    }
    fn set_fragment_html(&self, html: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_fragment_html {
                let arg_html = html;
                let arg_self_ = self.into_raw();
                let arg_html = arg_html
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_html);
            }
        }
    }
    fn set_fragment_base_url(&self, base_url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_fragment_base_url {
                let arg_base_url = base_url;
                let arg_self_ = self.into_raw();
                let arg_base_url = arg_base_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_base_url);
            }
        }
    }
    fn reset_file_contents(&self) {
        unsafe {
            if let Some(f) = self.0.reset_file_contents {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn add_file(&self, path: Option<&CefString>, display_name: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.add_file {
                let (arg_path, arg_display_name) = (path, display_name);
                let arg_self_ = self.into_raw();
                let arg_path = arg_path
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_display_name = arg_display_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_path, arg_display_name);
            }
        }
    }
    fn clear_filenames(&self) {
        unsafe {
            if let Some(f) = self.0.clear_filenames {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn image(&self) -> Option<Image> {
        unsafe {
            self.0
                .get_image
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn image_hotspot(&self) -> Point {
        unsafe {
            self.0
                .get_image_hotspot
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_image(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_image
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_drag_data_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_drag_data_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DragData {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_drag_data_t> for &DragData {
    fn into_raw(self) -> *mut _cef_drag_data_t {
        ImplDragData::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_drag_data_t> for &mut DragData {
    fn into_raw(self) -> *mut _cef_drag_data_t {
        ImplDragData::get_raw(self)
    }
}
impl ConvertReturnValue<DragData> for *mut _cef_drag_data_t {
    fn wrap_result(self) -> DragData {
        DragData(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DragData> for *mut _cef_drag_data_t {
    fn from(value: DragData) -> Self {
        let object = ImplDragData::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DragData {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_domvisitor_t`] for more documentation.
#[derive(Clone)]
pub struct Domvisitor(RefGuard<_cef_domvisitor_t>);
impl Domvisitor {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDomvisitor,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDomvisitor>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDomvisitor>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_domvisitor_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDomvisitor: ImplDomvisitor {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_domvisitor_t, Self>);
}
pub trait ImplDomvisitor: Clone + Sized + Rc {
    #[doc = "See [`_cef_domvisitor_t::visit`] for more documentation."]
    fn visit(&self, document: Option<&mut Domdocument>) {}
    fn init_methods(object: &mut _cef_domvisitor_t) {
        impl_cef_domvisitor_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_domvisitor_t;
}
mod impl_cef_domvisitor_t {
    use super::*;
    pub fn init_methods<I: ImplDomvisitor>(object: &mut _cef_domvisitor_t) {
        object.visit = Some(visit::<I>);
    }
    extern "C" fn visit<I: ImplDomvisitor>(
        self_: *mut _cef_domvisitor_t,
        document: *mut _cef_domdocument_t,
    ) {
        let (arg_self_, arg_document) = (self_, document);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_document = unsafe { arg_document.as_mut() }
            .map(|arg| Domdocument(unsafe { RefGuard::from_raw(arg) }));
        let arg_document = arg_document.as_mut();
        ImplDomvisitor::visit(&arg_self_.interface, arg_document)
    }
}
impl ImplDomvisitor for Domvisitor {
    fn visit(&self, document: Option<&mut Domdocument>) {
        unsafe {
            if let Some(f) = self.0.visit {
                let arg_document = document;
                let arg_self_ = self.into_raw();
                let arg_document = arg_document
                    .map(|arg| {
                        arg.add_ref();
                        ImplDomdocument::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_document);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_domvisitor_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_domvisitor_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Domvisitor {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_domvisitor_t> for &Domvisitor {
    fn into_raw(self) -> *mut _cef_domvisitor_t {
        ImplDomvisitor::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_domvisitor_t> for &mut Domvisitor {
    fn into_raw(self) -> *mut _cef_domvisitor_t {
        ImplDomvisitor::get_raw(self)
    }
}
impl ConvertReturnValue<Domvisitor> for *mut _cef_domvisitor_t {
    fn wrap_result(self) -> Domvisitor {
        Domvisitor(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Domvisitor> for *mut _cef_domvisitor_t {
    fn from(value: Domvisitor) -> Self {
        let object = ImplDomvisitor::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Domvisitor {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_domdocument_t`] for more documentation.
#[derive(Clone)]
pub struct Domdocument(RefGuard<_cef_domdocument_t>);
pub trait ImplDomdocument: Clone + Sized + Rc {
    #[doc = "See [`_cef_domdocument_t::get_type`] for more documentation."]
    fn get_type(&self) -> DomDocumentType;
    #[doc = "See [`_cef_domdocument_t::get_document`] for more documentation."]
    fn document(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domdocument_t::get_body`] for more documentation."]
    fn body(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domdocument_t::get_head`] for more documentation."]
    fn head(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domdocument_t::get_title`] for more documentation."]
    fn title(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domdocument_t::get_element_by_id`] for more documentation."]
    fn element_by_id(&self, id: Option<&CefString>) -> Option<Domnode>;
    #[doc = "See [`_cef_domdocument_t::get_focused_node`] for more documentation."]
    fn focused_node(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domdocument_t::has_selection`] for more documentation."]
    fn has_selection(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domdocument_t::get_selection_start_offset`] for more documentation."]
    fn selection_start_offset(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domdocument_t::get_selection_end_offset`] for more documentation."]
    fn selection_end_offset(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domdocument_t::get_selection_as_markup`] for more documentation."]
    fn selection_as_markup(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domdocument_t::get_selection_as_text`] for more documentation."]
    fn selection_as_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domdocument_t::get_base_url`] for more documentation."]
    fn base_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domdocument_t::get_complete_url`] for more documentation."]
    fn complete_url(&self, partial_url: Option<&CefString>) -> CefStringUserfree;
    fn get_raw(&self) -> *mut _cef_domdocument_t;
}
impl ImplDomdocument for Domdocument {
    fn get_type(&self) -> DomDocumentType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn document(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_document
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn body(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_body
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn head(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_head
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn title(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_title
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_by_id(&self, id: Option<&CefString>) -> Option<Domnode> {
        unsafe {
            self.0
                .get_element_by_id
                .map(|f| {
                    let arg_id = id;
                    let arg_self_ = self.into_raw();
                    let arg_id = arg_id.map(|arg| arg.into_raw()).unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_id);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn focused_node(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_focused_node
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn has_selection(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_selection
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn selection_start_offset(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_selection_start_offset
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn selection_end_offset(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_selection_end_offset
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn selection_as_markup(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_selection_as_markup
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn selection_as_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_selection_as_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn base_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_base_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn complete_url(&self, partial_url: Option<&CefString>) -> CefStringUserfree {
        unsafe {
            self.0
                .get_complete_url
                .map(|f| {
                    let arg_partial_url = partial_url;
                    let arg_self_ = self.into_raw();
                    let arg_partial_url = arg_partial_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_partial_url);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_domdocument_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_domdocument_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Domdocument {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_domdocument_t> for &Domdocument {
    fn into_raw(self) -> *mut _cef_domdocument_t {
        ImplDomdocument::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_domdocument_t> for &mut Domdocument {
    fn into_raw(self) -> *mut _cef_domdocument_t {
        ImplDomdocument::get_raw(self)
    }
}
impl ConvertReturnValue<Domdocument> for *mut _cef_domdocument_t {
    fn wrap_result(self) -> Domdocument {
        Domdocument(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Domdocument> for *mut _cef_domdocument_t {
    fn from(value: Domdocument) -> Self {
        let object = ImplDomdocument::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Domdocument {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_domnode_t`] for more documentation.
#[derive(Clone)]
pub struct Domnode(RefGuard<_cef_domnode_t>);
pub trait ImplDomnode: Clone + Sized + Rc {
    #[doc = "See [`_cef_domnode_t::get_type`] for more documentation."]
    fn get_type(&self) -> DomNodeType;
    #[doc = "See [`_cef_domnode_t::is_text`] for more documentation."]
    fn is_text(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::is_element`] for more documentation."]
    fn is_element(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::is_editable`] for more documentation."]
    fn is_editable(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::is_form_control_element`] for more documentation."]
    fn is_form_control_element(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_form_control_element_type`] for more documentation."]
    fn form_control_element_type(&self) -> DomFormControlType;
    #[doc = "See [`_cef_domnode_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut Domnode>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_name`] for more documentation."]
    fn name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::get_value`] for more documentation."]
    fn value(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::set_value`] for more documentation."]
    fn set_value(&self, value: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_as_markup`] for more documentation."]
    fn as_markup(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::get_document`] for more documentation."]
    fn document(&self) -> Option<Domdocument>;
    #[doc = "See [`_cef_domnode_t::get_parent`] for more documentation."]
    fn parent(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domnode_t::get_previous_sibling`] for more documentation."]
    fn previous_sibling(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domnode_t::get_next_sibling`] for more documentation."]
    fn next_sibling(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domnode_t::has_children`] for more documentation."]
    fn has_children(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_first_child`] for more documentation."]
    fn first_child(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domnode_t::get_last_child`] for more documentation."]
    fn last_child(&self) -> Option<Domnode>;
    #[doc = "See [`_cef_domnode_t::get_element_tag_name`] for more documentation."]
    fn element_tag_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::has_element_attributes`] for more documentation."]
    fn has_element_attributes(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::has_element_attribute`] for more documentation."]
    fn has_element_attribute(&self, attr_name: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_element_attribute`] for more documentation."]
    fn element_attribute(&self, attr_name: Option<&CefString>) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::get_element_attributes`] for more documentation."]
    fn element_attributes(&self, attr_map: Option<&mut CefStringMap>);
    #[doc = "See [`_cef_domnode_t::set_element_attribute`] for more documentation."]
    fn set_element_attribute(
        &self,
        attr_name: Option<&CefString>,
        value: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_domnode_t::get_element_inner_text`] for more documentation."]
    fn element_inner_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_domnode_t::get_element_bounds`] for more documentation."]
    fn element_bounds(&self) -> Rect;
    fn get_raw(&self) -> *mut _cef_domnode_t;
}
impl ImplDomnode for Domnode {
    fn get_type(&self) -> DomNodeType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_text(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_element(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_element
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_editable(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_editable
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_form_control_element(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_form_control_element
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn form_control_element_type(&self) -> DomFormControlType {
        unsafe {
            self.0
                .get_form_control_element_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut Domnode>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplDomnode::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn value(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_value
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_value(&self, value: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_value
                .map(|f| {
                    let arg_value = value;
                    let arg_self_ = self.into_raw();
                    let arg_value = arg_value
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn as_markup(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_as_markup
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn document(&self) -> Option<Domdocument> {
        unsafe {
            self.0
                .get_document
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn parent(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_parent
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn previous_sibling(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_previous_sibling
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn next_sibling(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_next_sibling
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn has_children(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_children
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn first_child(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_first_child
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn last_child(&self) -> Option<Domnode> {
        unsafe {
            self.0
                .get_last_child
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn element_tag_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_element_tag_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_element_attributes(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_element_attributes
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_element_attribute(&self, attr_name: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_element_attribute
                .map(|f| {
                    let arg_attr_name = attr_name;
                    let arg_self_ = self.into_raw();
                    let arg_attr_name = arg_attr_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_attr_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_attribute(&self, attr_name: Option<&CefString>) -> CefStringUserfree {
        unsafe {
            self.0
                .get_element_attribute
                .map(|f| {
                    let arg_attr_name = attr_name;
                    let arg_self_ = self.into_raw();
                    let arg_attr_name = arg_attr_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_attr_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_attributes(&self, attr_map: Option<&mut CefStringMap>) {
        unsafe {
            if let Some(f) = self.0.get_element_attributes {
                let arg_attr_map = attr_map;
                let arg_self_ = self.into_raw();
                let arg_attr_map = arg_attr_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_attr_map);
            }
        }
    }
    fn set_element_attribute(
        &self,
        attr_name: Option<&CefString>,
        value: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_element_attribute
                .map(|f| {
                    let (arg_attr_name, arg_value) = (attr_name, value);
                    let arg_self_ = self.into_raw();
                    let arg_attr_name = arg_attr_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_attr_name, arg_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_inner_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_element_inner_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_bounds(&self) -> Rect {
        unsafe {
            self.0
                .get_element_bounds
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_domnode_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_domnode_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Domnode {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_domnode_t> for &Domnode {
    fn into_raw(self) -> *mut _cef_domnode_t {
        ImplDomnode::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_domnode_t> for &mut Domnode {
    fn into_raw(self) -> *mut _cef_domnode_t {
        ImplDomnode::get_raw(self)
    }
}
impl ConvertReturnValue<Domnode> for *mut _cef_domnode_t {
    fn wrap_result(self) -> Domnode {
        Domnode(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Domnode> for *mut _cef_domnode_t {
    fn from(value: Domnode) -> Self {
        let object = ImplDomnode::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Domnode {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_shared_memory_region_t`] for more documentation.
#[derive(Clone)]
pub struct SharedMemoryRegion(RefGuard<_cef_shared_memory_region_t>);
pub trait ImplSharedMemoryRegion: Clone + Sized + Rc {
    #[doc = "See [`_cef_shared_memory_region_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_shared_memory_region_t::size`] for more documentation."]
    fn size(&self) -> usize;
    #[doc = "See [`_cef_shared_memory_region_t::memory`] for more documentation."]
    fn memory(&self) -> *mut ::std::os::raw::c_void;
    fn get_raw(&self) -> *mut _cef_shared_memory_region_t;
}
impl ImplSharedMemoryRegion for SharedMemoryRegion {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn size(&self) -> usize {
        unsafe {
            self.0
                .size
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn memory(&self) -> *mut ::std::os::raw::c_void {
        unsafe {
            self.0
                .memory
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_else(|| std::mem::zeroed())
        }
    }
    fn get_raw(&self) -> *mut _cef_shared_memory_region_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_shared_memory_region_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for SharedMemoryRegion {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_shared_memory_region_t> for &SharedMemoryRegion {
    fn into_raw(self) -> *mut _cef_shared_memory_region_t {
        ImplSharedMemoryRegion::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_shared_memory_region_t> for &mut SharedMemoryRegion {
    fn into_raw(self) -> *mut _cef_shared_memory_region_t {
        ImplSharedMemoryRegion::get_raw(self)
    }
}
impl ConvertReturnValue<SharedMemoryRegion> for *mut _cef_shared_memory_region_t {
    fn wrap_result(self) -> SharedMemoryRegion {
        SharedMemoryRegion(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<SharedMemoryRegion> for *mut _cef_shared_memory_region_t {
    fn from(value: SharedMemoryRegion) -> Self {
        let object = ImplSharedMemoryRegion::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for SharedMemoryRegion {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_process_message_t`] for more documentation.
#[derive(Clone)]
pub struct ProcessMessage(RefGuard<_cef_process_message_t>);
pub trait ImplProcessMessage: Clone + Sized + Rc {
    #[doc = "See [`_cef_process_message_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_process_message_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_process_message_t::copy`] for more documentation."]
    fn copy(&self) -> Option<ProcessMessage>;
    #[doc = "See [`_cef_process_message_t::get_name`] for more documentation."]
    fn name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_process_message_t::get_argument_list`] for more documentation."]
    fn argument_list(&self) -> Option<ListValue>;
    #[doc = "See [`_cef_process_message_t::get_shared_memory_region`] for more documentation."]
    fn shared_memory_region(&self) -> Option<SharedMemoryRegion>;
    fn get_raw(&self) -> *mut _cef_process_message_t;
}
impl ImplProcessMessage for ProcessMessage {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn copy(&self) -> Option<ProcessMessage> {
        unsafe {
            self.0
                .copy
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn argument_list(&self) -> Option<ListValue> {
        unsafe {
            self.0
                .get_argument_list
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn shared_memory_region(&self) -> Option<SharedMemoryRegion> {
        unsafe {
            self.0
                .get_shared_memory_region
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_process_message_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_process_message_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ProcessMessage {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_process_message_t> for &ProcessMessage {
    fn into_raw(self) -> *mut _cef_process_message_t {
        ImplProcessMessage::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_process_message_t> for &mut ProcessMessage {
    fn into_raw(self) -> *mut _cef_process_message_t {
        ImplProcessMessage::get_raw(self)
    }
}
impl ConvertReturnValue<ProcessMessage> for *mut _cef_process_message_t {
    fn wrap_result(self) -> ProcessMessage {
        ProcessMessage(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ProcessMessage> for *mut _cef_process_message_t {
    fn from(value: ProcessMessage) -> Self {
        let object = ImplProcessMessage::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ProcessMessage {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_request_t`] for more documentation.
#[derive(Clone)]
pub struct Request(RefGuard<_cef_request_t>);
pub trait ImplRequest: Clone + Sized + Rc {
    #[doc = "See [`_cef_request_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_t::get_url`] for more documentation."]
    fn url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_request_t::set_url`] for more documentation."]
    fn set_url(&self, url: Option<&CefString>);
    #[doc = "See [`_cef_request_t::get_method`] for more documentation."]
    fn method(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_request_t::set_method`] for more documentation."]
    fn set_method(&self, method: Option<&CefString>);
    #[doc = "See [`_cef_request_t::set_referrer`] for more documentation."]
    fn set_referrer(&self, referrer_url: Option<&CefString>, policy: ReferrerPolicy);
    #[doc = "See [`_cef_request_t::get_referrer_url`] for more documentation."]
    fn referrer_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_request_t::get_referrer_policy`] for more documentation."]
    fn referrer_policy(&self) -> ReferrerPolicy;
    #[doc = "See [`_cef_request_t::get_post_data`] for more documentation."]
    fn post_data(&self) -> Option<PostData>;
    #[doc = "See [`_cef_request_t::set_post_data`] for more documentation."]
    fn set_post_data(&self, post_data: Option<&mut PostData>);
    #[doc = "See [`_cef_request_t::get_header_map`] for more documentation."]
    fn header_map(&self, header_map: Option<&mut CefStringMultimap>);
    #[doc = "See [`_cef_request_t::set_header_map`] for more documentation."]
    fn set_header_map(&self, header_map: Option<&mut CefStringMultimap>);
    #[doc = "See [`_cef_request_t::get_header_by_name`] for more documentation."]
    fn header_by_name(&self, name: Option<&CefString>) -> CefStringUserfree;
    #[doc = "See [`_cef_request_t::set_header_by_name`] for more documentation."]
    fn set_header_by_name(
        &self,
        name: Option<&CefString>,
        value: Option<&CefString>,
        overwrite: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_request_t::set`] for more documentation."]
    fn set(
        &self,
        url: Option<&CefString>,
        method: Option<&CefString>,
        post_data: Option<&mut PostData>,
        header_map: Option<&mut CefStringMultimap>,
    );
    #[doc = "See [`_cef_request_t::get_flags`] for more documentation."]
    fn flags(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_t::set_flags`] for more documentation."]
    fn set_flags(&self, flags: ::std::os::raw::c_int);
    #[doc = "See [`_cef_request_t::get_first_party_for_cookies`] for more documentation."]
    fn first_party_for_cookies(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_request_t::set_first_party_for_cookies`] for more documentation."]
    fn set_first_party_for_cookies(&self, url: Option<&CefString>);
    #[doc = "See [`_cef_request_t::get_resource_type`] for more documentation."]
    fn resource_type(&self) -> ResourceType;
    #[doc = "See [`_cef_request_t::get_transition_type`] for more documentation."]
    fn transition_type(&self) -> TransitionType;
    #[doc = "See [`_cef_request_t::get_identifier`] for more documentation."]
    fn identifier(&self) -> u64;
    fn get_raw(&self) -> *mut _cef_request_t;
}
impl ImplRequest for Request {
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_url(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_url {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn method(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_method
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_method(&self, method: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_method {
                let arg_method = method;
                let arg_self_ = self.into_raw();
                let arg_method = arg_method
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_method);
            }
        }
    }
    fn set_referrer(&self, referrer_url: Option<&CefString>, policy: ReferrerPolicy) {
        unsafe {
            if let Some(f) = self.0.set_referrer {
                let (arg_referrer_url, arg_policy) = (referrer_url, policy);
                let arg_self_ = self.into_raw();
                let arg_referrer_url = arg_referrer_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_policy = arg_policy.into_raw();
                f(arg_self_, arg_referrer_url, arg_policy);
            }
        }
    }
    fn referrer_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_referrer_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn referrer_policy(&self) -> ReferrerPolicy {
        unsafe {
            self.0
                .get_referrer_policy
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn post_data(&self) -> Option<PostData> {
        unsafe {
            self.0
                .get_post_data
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_post_data(&self, post_data: Option<&mut PostData>) {
        unsafe {
            if let Some(f) = self.0.set_post_data {
                let arg_post_data = post_data;
                let arg_self_ = self.into_raw();
                let arg_post_data = arg_post_data
                    .map(|arg| {
                        arg.add_ref();
                        ImplPostData::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_post_data);
            }
        }
    }
    fn header_map(&self, header_map: Option<&mut CefStringMultimap>) {
        unsafe {
            if let Some(f) = self.0.get_header_map {
                let arg_header_map = header_map;
                let arg_self_ = self.into_raw();
                let arg_header_map = arg_header_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_header_map);
            }
        }
    }
    fn set_header_map(&self, header_map: Option<&mut CefStringMultimap>) {
        unsafe {
            if let Some(f) = self.0.set_header_map {
                let arg_header_map = header_map;
                let arg_self_ = self.into_raw();
                let arg_header_map = arg_header_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_header_map);
            }
        }
    }
    fn header_by_name(&self, name: Option<&CefString>) -> CefStringUserfree {
        unsafe {
            self.0
                .get_header_by_name
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_header_by_name(
        &self,
        name: Option<&CefString>,
        value: Option<&CefString>,
        overwrite: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.set_header_by_name {
                let (arg_name, arg_value, arg_overwrite) = (name, value, overwrite);
                let arg_self_ = self.into_raw();
                let arg_name = arg_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_value = arg_value
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_name, arg_value, arg_overwrite);
            }
        }
    }
    fn set(
        &self,
        url: Option<&CefString>,
        method: Option<&CefString>,
        post_data: Option<&mut PostData>,
        header_map: Option<&mut CefStringMultimap>,
    ) {
        unsafe {
            if let Some(f) = self.0.set {
                let (arg_url, arg_method, arg_post_data, arg_header_map) =
                    (url, method, post_data, header_map);
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_method = arg_method
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_post_data = arg_post_data
                    .map(|arg| {
                        arg.add_ref();
                        ImplPostData::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_header_map = arg_header_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_url,
                    arg_method,
                    arg_post_data,
                    arg_header_map,
                );
            }
        }
    }
    fn flags(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_flags
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_flags(&self, flags: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_flags {
                let arg_flags = flags;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_flags);
            }
        }
    }
    fn first_party_for_cookies(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_first_party_for_cookies
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_first_party_for_cookies(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_first_party_for_cookies {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn resource_type(&self) -> ResourceType {
        unsafe {
            self.0
                .get_resource_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn transition_type(&self) -> TransitionType {
        unsafe {
            self.0
                .get_transition_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn identifier(&self) -> u64 {
        unsafe {
            self.0
                .get_identifier
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_request_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_request_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Request {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_request_t> for &Request {
    fn into_raw(self) -> *mut _cef_request_t {
        ImplRequest::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_request_t> for &mut Request {
    fn into_raw(self) -> *mut _cef_request_t {
        ImplRequest::get_raw(self)
    }
}
impl ConvertReturnValue<Request> for *mut _cef_request_t {
    fn wrap_result(self) -> Request {
        Request(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Request> for *mut _cef_request_t {
    fn from(value: Request) -> Self {
        let object = ImplRequest::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Request {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_post_data_t`] for more documentation.
#[derive(Clone)]
pub struct PostData(RefGuard<_cef_post_data_t>);
pub trait ImplPostData: Clone + Sized + Rc {
    #[doc = "See [`_cef_post_data_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_post_data_t::has_excluded_elements`] for more documentation."]
    fn has_excluded_elements(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_post_data_t::get_element_count`] for more documentation."]
    fn element_count(&self) -> usize;
    #[doc = "See [`_cef_post_data_t::get_elements`] for more documentation."]
    fn elements(&self, elements: Option<&mut Vec<Option<PostDataElement>>>);
    #[doc = "See [`_cef_post_data_t::remove_element`] for more documentation."]
    fn remove_element(&self, element: Option<&mut PostDataElement>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_post_data_t::add_element`] for more documentation."]
    fn add_element(&self, element: Option<&mut PostDataElement>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_post_data_t::remove_elements`] for more documentation."]
    fn remove_elements(&self);
    fn get_raw(&self) -> *mut _cef_post_data_t;
}
impl ImplPostData for PostData {
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_excluded_elements(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_excluded_elements
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn element_count(&self) -> usize {
        unsafe {
            self.0
                .get_element_count
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn elements(&self, elements: Option<&mut Vec<Option<PostDataElement>>>) {
        unsafe {
            if let Some(f) = self.0.get_elements {
                let arg_elements = elements;
                let arg_self_ = self.into_raw();
                let mut out_elements_count = arg_elements
                    .as_ref()
                    .map(|arg| arg.len())
                    .unwrap_or_default();
                let arg_elements_count = &mut out_elements_count;
                let out_elements = arg_elements;
                let mut vec_elements = out_elements
                    .as_ref()
                    .map(|arg| {
                        arg.iter()
                            .map(|elem| {
                                elem.as_ref()
                                    .map(|elem| {
                                        elem.add_ref();
                                        elem.get_raw()
                                    })
                                    .unwrap_or(std::ptr::null_mut())
                            })
                            .collect::<Vec<_>>()
                    })
                    .unwrap_or_default();
                let arg_elements = if vec_elements.is_empty() {
                    std::ptr::null_mut()
                } else {
                    vec_elements.as_mut_ptr()
                };
                f(arg_self_, arg_elements_count, arg_elements);
                if let Some(out_elements) = out_elements {
                    *out_elements = vec_elements
                        .into_iter()
                        .take(out_elements_count)
                        .map(|elem| {
                            if elem.is_null() {
                                None
                            } else {
                                Some(elem.wrap_result())
                            }
                        })
                        .collect();
                }
            }
        }
    }
    fn remove_element(&self, element: Option<&mut PostDataElement>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove_element
                .map(|f| {
                    let arg_element = element;
                    let arg_self_ = self.into_raw();
                    let arg_element = arg_element
                        .map(|arg| {
                            arg.add_ref();
                            ImplPostDataElement::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_element);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_element(&self, element: Option<&mut PostDataElement>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_element
                .map(|f| {
                    let arg_element = element;
                    let arg_self_ = self.into_raw();
                    let arg_element = arg_element
                        .map(|arg| {
                            arg.add_ref();
                            ImplPostDataElement::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_element);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove_elements(&self) {
        unsafe {
            if let Some(f) = self.0.remove_elements {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_post_data_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_post_data_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PostData {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_post_data_t> for &PostData {
    fn into_raw(self) -> *mut _cef_post_data_t {
        ImplPostData::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_post_data_t> for &mut PostData {
    fn into_raw(self) -> *mut _cef_post_data_t {
        ImplPostData::get_raw(self)
    }
}
impl ConvertReturnValue<PostData> for *mut _cef_post_data_t {
    fn wrap_result(self) -> PostData {
        PostData(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PostData> for *mut _cef_post_data_t {
    fn from(value: PostData) -> Self {
        let object = ImplPostData::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PostData {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_post_data_element_t`] for more documentation.
#[derive(Clone)]
pub struct PostDataElement(RefGuard<_cef_post_data_element_t>);
pub trait ImplPostDataElement: Clone + Sized + Rc {
    #[doc = "See [`_cef_post_data_element_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_post_data_element_t::set_to_empty`] for more documentation."]
    fn set_to_empty(&self);
    #[doc = "See [`_cef_post_data_element_t::set_to_file`] for more documentation."]
    fn set_to_file(&self, file_name: Option<&CefString>);
    #[doc = "See [`_cef_post_data_element_t::set_to_bytes`] for more documentation."]
    fn set_to_bytes(&self, size: usize, bytes: *const u8);
    #[doc = "See [`_cef_post_data_element_t::get_type`] for more documentation."]
    fn get_type(&self) -> PostdataelementType;
    #[doc = "See [`_cef_post_data_element_t::get_file`] for more documentation."]
    fn file(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_post_data_element_t::get_bytes_count`] for more documentation."]
    fn bytes_count(&self) -> usize;
    #[doc = "See [`_cef_post_data_element_t::get_bytes`] for more documentation."]
    fn bytes(&self, size: usize, bytes: *mut u8) -> usize;
    fn get_raw(&self) -> *mut _cef_post_data_element_t;
}
impl ImplPostDataElement for PostDataElement {
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_to_empty(&self) {
        unsafe {
            if let Some(f) = self.0.set_to_empty {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn set_to_file(&self, file_name: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_to_file {
                let arg_file_name = file_name;
                let arg_self_ = self.into_raw();
                let arg_file_name = arg_file_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_file_name);
            }
        }
    }
    fn set_to_bytes(&self, size: usize, bytes: *const u8) {
        unsafe {
            if let Some(f) = self.0.set_to_bytes {
                let (arg_size, arg_bytes) = (size, bytes);
                let arg_self_ = self.into_raw();
                let arg_bytes = arg_bytes.cast();
                f(arg_self_, arg_size, arg_bytes);
            }
        }
    }
    fn get_type(&self) -> PostdataelementType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn file(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_file
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn bytes_count(&self) -> usize {
        unsafe {
            self.0
                .get_bytes_count
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn bytes(&self, size: usize, bytes: *mut u8) -> usize {
        unsafe {
            self.0
                .get_bytes
                .map(|f| {
                    let (arg_size, arg_bytes) = (size, bytes);
                    let arg_self_ = self.into_raw();
                    let arg_bytes = arg_bytes.cast();
                    let result = f(arg_self_, arg_size, arg_bytes);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_post_data_element_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_post_data_element_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PostDataElement {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_post_data_element_t> for &PostDataElement {
    fn into_raw(self) -> *mut _cef_post_data_element_t {
        ImplPostDataElement::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_post_data_element_t> for &mut PostDataElement {
    fn into_raw(self) -> *mut _cef_post_data_element_t {
        ImplPostDataElement::get_raw(self)
    }
}
impl ConvertReturnValue<PostDataElement> for *mut _cef_post_data_element_t {
    fn wrap_result(self) -> PostDataElement {
        PostDataElement(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PostDataElement> for *mut _cef_post_data_element_t {
    fn from(value: PostDataElement) -> Self {
        let object = ImplPostDataElement::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PostDataElement {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_string_visitor_t`] for more documentation.
#[derive(Clone)]
pub struct CefStringVisitor(RefGuard<_cef_string_visitor_t>);
impl CefStringVisitor {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapCefStringVisitor,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplCefStringVisitor>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapCefStringVisitor>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_string_visitor_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapCefStringVisitor: ImplCefStringVisitor {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_string_visitor_t, Self>);
}
pub trait ImplCefStringVisitor: Clone + Sized + Rc {
    #[doc = "See [`_cef_string_visitor_t::visit`] for more documentation."]
    fn visit(&self, string: Option<&CefString>) {}
    fn init_methods(object: &mut _cef_string_visitor_t) {
        impl_cef_string_visitor_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_string_visitor_t;
}
mod impl_cef_string_visitor_t {
    use super::*;
    pub fn init_methods<I: ImplCefStringVisitor>(object: &mut _cef_string_visitor_t) {
        object.visit = Some(visit::<I>);
    }
    extern "C" fn visit<I: ImplCefStringVisitor>(
        self_: *mut _cef_string_visitor_t,
        string: *const cef_string_t,
    ) {
        let (arg_self_, arg_string) = (self_, string);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_string = if arg_string.is_null() {
            None
        } else {
            Some(arg_string.into())
        };
        let arg_string = arg_string.as_ref();
        ImplCefStringVisitor::visit(&arg_self_.interface, arg_string)
    }
}
impl ImplCefStringVisitor for CefStringVisitor {
    fn visit(&self, string: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.visit {
                let arg_string = string;
                let arg_self_ = self.into_raw();
                let arg_string = arg_string
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_string);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_string_visitor_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_string_visitor_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CefStringVisitor {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_string_visitor_t> for &CefStringVisitor {
    fn into_raw(self) -> *mut _cef_string_visitor_t {
        ImplCefStringVisitor::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_string_visitor_t> for &mut CefStringVisitor {
    fn into_raw(self) -> *mut _cef_string_visitor_t {
        ImplCefStringVisitor::get_raw(self)
    }
}
impl ConvertReturnValue<CefStringVisitor> for *mut _cef_string_visitor_t {
    fn wrap_result(self) -> CefStringVisitor {
        CefStringVisitor(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CefStringVisitor> for *mut _cef_string_visitor_t {
    fn from(value: CefStringVisitor) -> Self {
        let object = ImplCefStringVisitor::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CefStringVisitor {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_frame_t`] for more documentation.
#[derive(Clone)]
pub struct Frame(RefGuard<_cef_frame_t>);
pub trait ImplFrame: Clone + Sized + Rc {
    #[doc = "See [`_cef_frame_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_frame_t::undo`] for more documentation."]
    fn undo(&self);
    #[doc = "See [`_cef_frame_t::redo`] for more documentation."]
    fn redo(&self);
    #[doc = "See [`_cef_frame_t::cut`] for more documentation."]
    fn cut(&self);
    #[doc = "See [`_cef_frame_t::copy`] for more documentation."]
    fn copy(&self);
    #[doc = "See [`_cef_frame_t::paste`] for more documentation."]
    fn paste(&self);
    #[doc = "See [`_cef_frame_t::paste_and_match_style`] for more documentation."]
    fn paste_and_match_style(&self);
    #[doc = "See [`_cef_frame_t::del`] for more documentation."]
    fn del(&self);
    #[doc = "See [`_cef_frame_t::select_all`] for more documentation."]
    fn select_all(&self);
    #[doc = "See [`_cef_frame_t::view_source`] for more documentation."]
    fn view_source(&self);
    #[doc = "See [`_cef_frame_t::get_source`] for more documentation."]
    fn source(&self, visitor: Option<&mut impl ImplCefStringVisitor>);
    #[doc = "See [`_cef_frame_t::get_text`] for more documentation."]
    fn text(&self, visitor: Option<&mut impl ImplCefStringVisitor>);
    #[doc = "See [`_cef_frame_t::load_request`] for more documentation."]
    fn load_request(&self, request: Option<&mut Request>);
    #[doc = "See [`_cef_frame_t::load_url`] for more documentation."]
    fn load_url(&self, url: Option<&CefString>);
    #[doc = "See [`_cef_frame_t::execute_java_script`] for more documentation."]
    fn execute_java_script(
        &self,
        code: Option<&CefString>,
        script_url: Option<&CefString>,
        start_line: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_frame_t::is_main`] for more documentation."]
    fn is_main(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_frame_t::is_focused`] for more documentation."]
    fn is_focused(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_frame_t::get_name`] for more documentation."]
    fn name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_frame_t::get_identifier`] for more documentation."]
    fn identifier(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_frame_t::get_parent`] for more documentation."]
    fn parent(&self) -> Option<Frame>;
    #[doc = "See [`_cef_frame_t::get_url`] for more documentation."]
    fn url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_frame_t::get_browser`] for more documentation."]
    fn browser(&self) -> Option<Browser>;
    #[doc = "See [`_cef_frame_t::get_v8_context`] for more documentation."]
    fn v8_context(&self) -> Option<V8Context>;
    #[doc = "See [`_cef_frame_t::visit_dom`] for more documentation."]
    fn visit_dom(&self, visitor: Option<&mut impl ImplDomvisitor>);
    #[doc = "See [`_cef_frame_t::create_urlrequest`] for more documentation."]
    fn create_urlrequest(
        &self,
        request: Option<&mut Request>,
        client: Option<&mut impl ImplUrlrequestClient>,
    ) -> Option<Urlrequest>;
    #[doc = "See [`_cef_frame_t::send_process_message`] for more documentation."]
    fn send_process_message(&self, target_process: ProcessId, message: Option<&mut ProcessMessage>);
    fn get_raw(&self) -> *mut _cef_frame_t;
}
impl ImplFrame for Frame {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn undo(&self) {
        unsafe {
            if let Some(f) = self.0.undo {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn redo(&self) {
        unsafe {
            if let Some(f) = self.0.redo {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn cut(&self) {
        unsafe {
            if let Some(f) = self.0.cut {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn copy(&self) {
        unsafe {
            if let Some(f) = self.0.copy {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn paste(&self) {
        unsafe {
            if let Some(f) = self.0.paste {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn paste_and_match_style(&self) {
        unsafe {
            if let Some(f) = self.0.paste_and_match_style {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn del(&self) {
        unsafe {
            if let Some(f) = self.0.del {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn select_all(&self) {
        unsafe {
            if let Some(f) = self.0.select_all {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn view_source(&self) {
        unsafe {
            if let Some(f) = self.0.view_source {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn source(&self, visitor: Option<&mut impl ImplCefStringVisitor>) {
        unsafe {
            if let Some(f) = self.0.get_source {
                let arg_visitor = visitor;
                let arg_self_ = self.into_raw();
                let arg_visitor = arg_visitor
                    .map(|arg| {
                        arg.add_ref();
                        ImplCefStringVisitor::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_visitor);
            }
        }
    }
    fn text(&self, visitor: Option<&mut impl ImplCefStringVisitor>) {
        unsafe {
            if let Some(f) = self.0.get_text {
                let arg_visitor = visitor;
                let arg_self_ = self.into_raw();
                let arg_visitor = arg_visitor
                    .map(|arg| {
                        arg.add_ref();
                        ImplCefStringVisitor::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_visitor);
            }
        }
    }
    fn load_request(&self, request: Option<&mut Request>) {
        unsafe {
            if let Some(f) = self.0.load_request {
                let arg_request = request;
                let arg_self_ = self.into_raw();
                let arg_request = arg_request
                    .map(|arg| {
                        arg.add_ref();
                        ImplRequest::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_request);
            }
        }
    }
    fn load_url(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.load_url {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn execute_java_script(
        &self,
        code: Option<&CefString>,
        script_url: Option<&CefString>,
        start_line: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.execute_java_script {
                let (arg_code, arg_script_url, arg_start_line) = (code, script_url, start_line);
                let arg_self_ = self.into_raw();
                let arg_code = arg_code
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_script_url = arg_script_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_code, arg_script_url, arg_start_line);
            }
        }
    }
    fn is_main(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_main
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_focused(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_focused
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn identifier(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_identifier
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn parent(&self) -> Option<Frame> {
        unsafe {
            self.0
                .get_parent
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn browser(&self) -> Option<Browser> {
        unsafe {
            self.0
                .get_browser
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn v8_context(&self) -> Option<V8Context> {
        unsafe {
            self.0
                .get_v8_context
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn visit_dom(&self, visitor: Option<&mut impl ImplDomvisitor>) {
        unsafe {
            if let Some(f) = self.0.visit_dom {
                let arg_visitor = visitor;
                let arg_self_ = self.into_raw();
                let arg_visitor = arg_visitor
                    .map(|arg| {
                        arg.add_ref();
                        ImplDomvisitor::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_visitor);
            }
        }
    }
    fn create_urlrequest(
        &self,
        request: Option<&mut Request>,
        client: Option<&mut impl ImplUrlrequestClient>,
    ) -> Option<Urlrequest> {
        unsafe {
            self.0
                .create_urlrequest
                .map(|f| {
                    let (arg_request, arg_client) = (request, client);
                    let arg_self_ = self.into_raw();
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_client = arg_client
                        .map(|arg| {
                            arg.add_ref();
                            ImplUrlrequestClient::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_request, arg_client);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn send_process_message(
        &self,
        target_process: ProcessId,
        message: Option<&mut ProcessMessage>,
    ) {
        unsafe {
            if let Some(f) = self.0.send_process_message {
                let (arg_target_process, arg_message) = (target_process, message);
                let arg_self_ = self.into_raw();
                let arg_target_process = arg_target_process.into_raw();
                let arg_message = arg_message
                    .map(|arg| {
                        arg.add_ref();
                        ImplProcessMessage::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_target_process, arg_message);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_frame_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_frame_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Frame {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_frame_t> for &Frame {
    fn into_raw(self) -> *mut _cef_frame_t {
        ImplFrame::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_frame_t> for &mut Frame {
    fn into_raw(self) -> *mut _cef_frame_t {
        ImplFrame::get_raw(self)
    }
}
impl ConvertReturnValue<Frame> for *mut _cef_frame_t {
    fn wrap_result(self) -> Frame {
        Frame(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Frame> for *mut _cef_frame_t {
    fn from(value: Frame) -> Self {
        let object = ImplFrame::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Frame {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_x509_cert_principal_t`] for more documentation.
#[derive(Clone)]
pub struct X509CertPrincipal(RefGuard<_cef_x509_cert_principal_t>);
pub trait ImplX509CertPrincipal: Clone + Sized + Rc {
    #[doc = "See [`_cef_x509_cert_principal_t::get_display_name`] for more documentation."]
    fn display_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_x509_cert_principal_t::get_common_name`] for more documentation."]
    fn common_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_x509_cert_principal_t::get_locality_name`] for more documentation."]
    fn locality_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_x509_cert_principal_t::get_state_or_province_name`] for more documentation."]
    fn state_or_province_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_x509_cert_principal_t::get_country_name`] for more documentation."]
    fn country_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_x509_cert_principal_t::get_organization_names`] for more documentation."]
    fn organization_names(&self, names: Option<&mut CefStringList>);
    #[doc = "See [`_cef_x509_cert_principal_t::get_organization_unit_names`] for more documentation."]
    fn organization_unit_names(&self, names: Option<&mut CefStringList>);
    fn get_raw(&self) -> *mut _cef_x509_cert_principal_t;
}
impl ImplX509CertPrincipal for X509CertPrincipal {
    fn display_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_display_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn common_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_common_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn locality_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_locality_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn state_or_province_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_state_or_province_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn country_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_country_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn organization_names(&self, names: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.get_organization_names {
                let arg_names = names;
                let arg_self_ = self.into_raw();
                let arg_names = arg_names
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_names);
            }
        }
    }
    fn organization_unit_names(&self, names: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.get_organization_unit_names {
                let arg_names = names;
                let arg_self_ = self.into_raw();
                let arg_names = arg_names
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_names);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_x509_cert_principal_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_x509_cert_principal_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for X509CertPrincipal {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_x509_cert_principal_t> for &X509CertPrincipal {
    fn into_raw(self) -> *mut _cef_x509_cert_principal_t {
        ImplX509CertPrincipal::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_x509_cert_principal_t> for &mut X509CertPrincipal {
    fn into_raw(self) -> *mut _cef_x509_cert_principal_t {
        ImplX509CertPrincipal::get_raw(self)
    }
}
impl ConvertReturnValue<X509CertPrincipal> for *mut _cef_x509_cert_principal_t {
    fn wrap_result(self) -> X509CertPrincipal {
        X509CertPrincipal(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<X509CertPrincipal> for *mut _cef_x509_cert_principal_t {
    fn from(value: X509CertPrincipal) -> Self {
        let object = ImplX509CertPrincipal::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for X509CertPrincipal {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_x509_certificate_t`] for more documentation.
#[derive(Clone)]
pub struct X509Certificate(RefGuard<_cef_x509_certificate_t>);
pub trait ImplX509Certificate: Clone + Sized + Rc {
    #[doc = "See [`_cef_x509_certificate_t::get_subject`] for more documentation."]
    fn subject(&self) -> Option<X509CertPrincipal>;
    #[doc = "See [`_cef_x509_certificate_t::get_issuer`] for more documentation."]
    fn issuer(&self) -> Option<X509CertPrincipal>;
    #[doc = "See [`_cef_x509_certificate_t::get_serial_number`] for more documentation."]
    fn serial_number(&self) -> Option<BinaryValue>;
    #[doc = "See [`_cef_x509_certificate_t::get_valid_start`] for more documentation."]
    fn valid_start(&self) -> Basetime;
    #[doc = "See [`_cef_x509_certificate_t::get_valid_expiry`] for more documentation."]
    fn valid_expiry(&self) -> Basetime;
    #[doc = "See [`_cef_x509_certificate_t::get_derencoded`] for more documentation."]
    fn derencoded(&self) -> Option<BinaryValue>;
    #[doc = "See [`_cef_x509_certificate_t::get_pemencoded`] for more documentation."]
    fn pemencoded(&self) -> Option<BinaryValue>;
    #[doc = "See [`_cef_x509_certificate_t::get_issuer_chain_size`] for more documentation."]
    fn issuer_chain_size(&self) -> usize;
    #[doc = "See [`_cef_x509_certificate_t::get_derencoded_issuer_chain`] for more documentation."]
    fn derencoded_issuer_chain(&self, chain: Option<&mut Vec<Option<BinaryValue>>>);
    #[doc = "See [`_cef_x509_certificate_t::get_pemencoded_issuer_chain`] for more documentation."]
    fn pemencoded_issuer_chain(&self, chain: Option<&mut Vec<Option<BinaryValue>>>);
    fn get_raw(&self) -> *mut _cef_x509_certificate_t;
}
impl ImplX509Certificate for X509Certificate {
    fn subject(&self) -> Option<X509CertPrincipal> {
        unsafe {
            self.0
                .get_subject
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn issuer(&self) -> Option<X509CertPrincipal> {
        unsafe {
            self.0
                .get_issuer
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn serial_number(&self) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_serial_number
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn valid_start(&self) -> Basetime {
        unsafe {
            self.0
                .get_valid_start
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn valid_expiry(&self) -> Basetime {
        unsafe {
            self.0
                .get_valid_expiry
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn derencoded(&self) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_derencoded
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn pemencoded(&self) -> Option<BinaryValue> {
        unsafe {
            self.0
                .get_pemencoded
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn issuer_chain_size(&self) -> usize {
        unsafe {
            self.0
                .get_issuer_chain_size
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn derencoded_issuer_chain(&self, chain: Option<&mut Vec<Option<BinaryValue>>>) {
        unsafe {
            if let Some(f) = self.0.get_derencoded_issuer_chain {
                let arg_chain = chain;
                let arg_self_ = self.into_raw();
                let mut out_chain_count =
                    arg_chain.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let arg_chain_count = &mut out_chain_count;
                let out_chain = arg_chain;
                let mut vec_chain = out_chain
                    .as_ref()
                    .map(|arg| {
                        arg.iter()
                            .map(|elem| {
                                elem.as_ref()
                                    .map(|elem| {
                                        elem.add_ref();
                                        elem.get_raw()
                                    })
                                    .unwrap_or(std::ptr::null_mut())
                            })
                            .collect::<Vec<_>>()
                    })
                    .unwrap_or_default();
                let arg_chain = if vec_chain.is_empty() {
                    std::ptr::null_mut()
                } else {
                    vec_chain.as_mut_ptr()
                };
                f(arg_self_, arg_chain_count, arg_chain);
                if let Some(out_chain) = out_chain {
                    *out_chain = vec_chain
                        .into_iter()
                        .take(out_chain_count)
                        .map(|elem| {
                            if elem.is_null() {
                                None
                            } else {
                                Some(elem.wrap_result())
                            }
                        })
                        .collect();
                }
            }
        }
    }
    fn pemencoded_issuer_chain(&self, chain: Option<&mut Vec<Option<BinaryValue>>>) {
        unsafe {
            if let Some(f) = self.0.get_pemencoded_issuer_chain {
                let arg_chain = chain;
                let arg_self_ = self.into_raw();
                let mut out_chain_count =
                    arg_chain.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let arg_chain_count = &mut out_chain_count;
                let out_chain = arg_chain;
                let mut vec_chain = out_chain
                    .as_ref()
                    .map(|arg| {
                        arg.iter()
                            .map(|elem| {
                                elem.as_ref()
                                    .map(|elem| {
                                        elem.add_ref();
                                        elem.get_raw()
                                    })
                                    .unwrap_or(std::ptr::null_mut())
                            })
                            .collect::<Vec<_>>()
                    })
                    .unwrap_or_default();
                let arg_chain = if vec_chain.is_empty() {
                    std::ptr::null_mut()
                } else {
                    vec_chain.as_mut_ptr()
                };
                f(arg_self_, arg_chain_count, arg_chain);
                if let Some(out_chain) = out_chain {
                    *out_chain = vec_chain
                        .into_iter()
                        .take(out_chain_count)
                        .map(|elem| {
                            if elem.is_null() {
                                None
                            } else {
                                Some(elem.wrap_result())
                            }
                        })
                        .collect();
                }
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_x509_certificate_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_x509_certificate_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for X509Certificate {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_x509_certificate_t> for &X509Certificate {
    fn into_raw(self) -> *mut _cef_x509_certificate_t {
        ImplX509Certificate::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_x509_certificate_t> for &mut X509Certificate {
    fn into_raw(self) -> *mut _cef_x509_certificate_t {
        ImplX509Certificate::get_raw(self)
    }
}
impl ConvertReturnValue<X509Certificate> for *mut _cef_x509_certificate_t {
    fn wrap_result(self) -> X509Certificate {
        X509Certificate(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<X509Certificate> for *mut _cef_x509_certificate_t {
    fn from(value: X509Certificate) -> Self {
        let object = ImplX509Certificate::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for X509Certificate {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_sslstatus_t`] for more documentation.
#[derive(Clone)]
pub struct Sslstatus(RefGuard<_cef_sslstatus_t>);
pub trait ImplSslstatus: Clone + Sized + Rc {
    #[doc = "See [`_cef_sslstatus_t::is_secure_connection`] for more documentation."]
    fn is_secure_connection(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_sslstatus_t::get_cert_status`] for more documentation."]
    fn cert_status(&self) -> CertStatus;
    #[doc = "See [`_cef_sslstatus_t::get_sslversion`] for more documentation."]
    fn sslversion(&self) -> SslVersion;
    #[doc = "See [`_cef_sslstatus_t::get_content_status`] for more documentation."]
    fn content_status(&self) -> SslContentStatus;
    #[doc = "See [`_cef_sslstatus_t::get_x509_certificate`] for more documentation."]
    fn x509_certificate(&self) -> Option<X509Certificate>;
    fn get_raw(&self) -> *mut _cef_sslstatus_t;
}
impl ImplSslstatus for Sslstatus {
    fn is_secure_connection(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_secure_connection
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn cert_status(&self) -> CertStatus {
        unsafe {
            self.0
                .get_cert_status
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn sslversion(&self) -> SslVersion {
        unsafe {
            self.0
                .get_sslversion
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn content_status(&self) -> SslContentStatus {
        unsafe {
            self.0
                .get_content_status
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn x509_certificate(&self) -> Option<X509Certificate> {
        unsafe {
            self.0
                .get_x509_certificate
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_sslstatus_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_sslstatus_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Sslstatus {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_sslstatus_t> for &Sslstatus {
    fn into_raw(self) -> *mut _cef_sslstatus_t {
        ImplSslstatus::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_sslstatus_t> for &mut Sslstatus {
    fn into_raw(self) -> *mut _cef_sslstatus_t {
        ImplSslstatus::get_raw(self)
    }
}
impl ConvertReturnValue<Sslstatus> for *mut _cef_sslstatus_t {
    fn wrap_result(self) -> Sslstatus {
        Sslstatus(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Sslstatus> for *mut _cef_sslstatus_t {
    fn from(value: Sslstatus) -> Self {
        let object = ImplSslstatus::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Sslstatus {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_navigation_entry_t`] for more documentation.
#[derive(Clone)]
pub struct NavigationEntry(RefGuard<_cef_navigation_entry_t>);
pub trait ImplNavigationEntry: Clone + Sized + Rc {
    #[doc = "See [`_cef_navigation_entry_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_navigation_entry_t::get_url`] for more documentation."]
    fn url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_navigation_entry_t::get_display_url`] for more documentation."]
    fn display_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_navigation_entry_t::get_original_url`] for more documentation."]
    fn original_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_navigation_entry_t::get_title`] for more documentation."]
    fn title(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_navigation_entry_t::get_transition_type`] for more documentation."]
    fn transition_type(&self) -> TransitionType;
    #[doc = "See [`_cef_navigation_entry_t::has_post_data`] for more documentation."]
    fn has_post_data(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_navigation_entry_t::get_completion_time`] for more documentation."]
    fn completion_time(&self) -> Basetime;
    #[doc = "See [`_cef_navigation_entry_t::get_http_status_code`] for more documentation."]
    fn http_status_code(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_navigation_entry_t::get_sslstatus`] for more documentation."]
    fn sslstatus(&self) -> Option<Sslstatus>;
    fn get_raw(&self) -> *mut _cef_navigation_entry_t;
}
impl ImplNavigationEntry for NavigationEntry {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn display_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_display_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn original_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_original_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn title(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_title
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn transition_type(&self) -> TransitionType {
        unsafe {
            self.0
                .get_transition_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_post_data(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_post_data
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn completion_time(&self) -> Basetime {
        unsafe {
            self.0
                .get_completion_time
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn http_status_code(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_http_status_code
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn sslstatus(&self) -> Option<Sslstatus> {
        unsafe {
            self.0
                .get_sslstatus
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_navigation_entry_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_navigation_entry_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for NavigationEntry {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_navigation_entry_t> for &NavigationEntry {
    fn into_raw(self) -> *mut _cef_navigation_entry_t {
        ImplNavigationEntry::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_navigation_entry_t> for &mut NavigationEntry {
    fn into_raw(self) -> *mut _cef_navigation_entry_t {
        ImplNavigationEntry::get_raw(self)
    }
}
impl ConvertReturnValue<NavigationEntry> for *mut _cef_navigation_entry_t {
    fn wrap_result(self) -> NavigationEntry {
        NavigationEntry(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<NavigationEntry> for *mut _cef_navigation_entry_t {
    fn from(value: NavigationEntry) -> Self {
        let object = ImplNavigationEntry::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for NavigationEntry {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_registration_t`] for more documentation.
#[derive(Clone)]
pub struct Registration(RefGuard<_cef_registration_t>);
pub trait ImplRegistration: Clone + Sized + Rc {
    fn get_raw(&self) -> *mut _cef_registration_t;
}
impl ImplRegistration for Registration {
    fn get_raw(&self) -> *mut _cef_registration_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_registration_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Registration {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_registration_t> for &Registration {
    fn into_raw(self) -> *mut _cef_registration_t {
        ImplRegistration::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_registration_t> for &mut Registration {
    fn into_raw(self) -> *mut _cef_registration_t {
        ImplRegistration::get_raw(self)
    }
}
impl ConvertReturnValue<Registration> for *mut _cef_registration_t {
    fn wrap_result(self) -> Registration {
        Registration(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Registration> for *mut _cef_registration_t {
    fn from(value: Registration) -> Self {
        let object = ImplRegistration::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Registration {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_callback_t`] for more documentation.
#[derive(Clone)]
pub struct Callback(RefGuard<_cef_callback_t>);
pub trait ImplCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_callback_t::cont`] for more documentation."]
    fn cont(&self);
    #[doc = "See [`_cef_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_callback_t;
}
impl ImplCallback for Callback {
    fn cont(&self) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Callback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_callback_t> for &Callback {
    fn into_raw(self) -> *mut _cef_callback_t {
        ImplCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_callback_t> for &mut Callback {
    fn into_raw(self) -> *mut _cef_callback_t {
        ImplCallback::get_raw(self)
    }
}
impl ConvertReturnValue<Callback> for *mut _cef_callback_t {
    fn wrap_result(self) -> Callback {
        Callback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Callback> for *mut _cef_callback_t {
    fn from(value: Callback) -> Self {
        let object = ImplCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Callback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_completion_callback_t`] for more documentation.
#[derive(Clone)]
pub struct CompletionCallback(RefGuard<_cef_completion_callback_t>);
impl CompletionCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapCompletionCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplCompletionCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapCompletionCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_completion_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapCompletionCallback: ImplCompletionCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_completion_callback_t, Self>);
}
pub trait ImplCompletionCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_completion_callback_t::on_complete`] for more documentation."]
    fn on_complete(&self) {}
    fn init_methods(object: &mut _cef_completion_callback_t) {
        impl_cef_completion_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_completion_callback_t;
}
mod impl_cef_completion_callback_t {
    use super::*;
    pub fn init_methods<I: ImplCompletionCallback>(object: &mut _cef_completion_callback_t) {
        object.on_complete = Some(on_complete::<I>);
    }
    extern "C" fn on_complete<I: ImplCompletionCallback>(self_: *mut _cef_completion_callback_t) {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplCompletionCallback::on_complete(&arg_self_.interface)
    }
}
impl ImplCompletionCallback for CompletionCallback {
    fn on_complete(&self) {
        unsafe {
            if let Some(f) = self.0.on_complete {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_completion_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_completion_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CompletionCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_completion_callback_t> for &CompletionCallback {
    fn into_raw(self) -> *mut _cef_completion_callback_t {
        ImplCompletionCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_completion_callback_t> for &mut CompletionCallback {
    fn into_raw(self) -> *mut _cef_completion_callback_t {
        ImplCompletionCallback::get_raw(self)
    }
}
impl ConvertReturnValue<CompletionCallback> for *mut _cef_completion_callback_t {
    fn wrap_result(self) -> CompletionCallback {
        CompletionCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CompletionCallback> for *mut _cef_completion_callback_t {
    fn from(value: CompletionCallback) -> Self {
        let object = ImplCompletionCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CompletionCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_cookie_manager_t`] for more documentation.
#[derive(Clone)]
pub struct CookieManager(RefGuard<_cef_cookie_manager_t>);
pub trait ImplCookieManager: Clone + Sized + Rc {
    #[doc = "See [`_cef_cookie_manager_t::visit_all_cookies`] for more documentation."]
    fn visit_all_cookies(
        &self,
        visitor: Option<&mut impl ImplCookieVisitor>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_cookie_manager_t::visit_url_cookies`] for more documentation."]
    fn visit_url_cookies(
        &self,
        url: Option<&CefString>,
        include_http_only: ::std::os::raw::c_int,
        visitor: Option<&mut impl ImplCookieVisitor>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_cookie_manager_t::set_cookie`] for more documentation."]
    fn set_cookie(
        &self,
        url: Option<&CefString>,
        cookie: Option<&Cookie>,
        callback: Option<&mut impl ImplSetCookieCallback>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_cookie_manager_t::delete_cookies`] for more documentation."]
    fn delete_cookies(
        &self,
        url: Option<&CefString>,
        cookie_name: Option<&CefString>,
        callback: Option<&mut impl ImplDeleteCookiesCallback>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_cookie_manager_t::flush_store`] for more documentation."]
    fn flush_store(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_cookie_manager_t;
}
impl ImplCookieManager for CookieManager {
    fn visit_all_cookies(
        &self,
        visitor: Option<&mut impl ImplCookieVisitor>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .visit_all_cookies
                .map(|f| {
                    let arg_visitor = visitor;
                    let arg_self_ = self.into_raw();
                    let arg_visitor = arg_visitor
                        .map(|arg| {
                            arg.add_ref();
                            ImplCookieVisitor::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_visitor);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn visit_url_cookies(
        &self,
        url: Option<&CefString>,
        include_http_only: ::std::os::raw::c_int,
        visitor: Option<&mut impl ImplCookieVisitor>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .visit_url_cookies
                .map(|f| {
                    let (arg_url, arg_include_http_only, arg_visitor) =
                        (url, include_http_only, visitor);
                    let arg_self_ = self.into_raw();
                    let arg_url = arg_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_visitor = arg_visitor
                        .map(|arg| {
                            arg.add_ref();
                            ImplCookieVisitor::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_url, arg_include_http_only, arg_visitor);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_cookie(
        &self,
        url: Option<&CefString>,
        cookie: Option<&Cookie>,
        callback: Option<&mut impl ImplSetCookieCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_cookie
                .map(|f| {
                    let (arg_url, arg_cookie, arg_callback) = (url, cookie, callback);
                    let arg_self_ = self.into_raw();
                    let arg_url = arg_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_cookie = arg_cookie.cloned().map(|arg| arg.into());
                    let arg_cookie = arg_cookie
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplSetCookieCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_url, arg_cookie, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn delete_cookies(
        &self,
        url: Option<&CefString>,
        cookie_name: Option<&CefString>,
        callback: Option<&mut impl ImplDeleteCookiesCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .delete_cookies
                .map(|f| {
                    let (arg_url, arg_cookie_name, arg_callback) = (url, cookie_name, callback);
                    let arg_self_ = self.into_raw();
                    let arg_url = arg_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_cookie_name = arg_cookie_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplDeleteCookiesCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_url, arg_cookie_name, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn flush_store(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .flush_store
                .map(|f| {
                    let arg_callback = callback;
                    let arg_self_ = self.into_raw();
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCompletionCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_cookie_manager_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_cookie_manager_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CookieManager {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_cookie_manager_t> for &CookieManager {
    fn into_raw(self) -> *mut _cef_cookie_manager_t {
        ImplCookieManager::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_cookie_manager_t> for &mut CookieManager {
    fn into_raw(self) -> *mut _cef_cookie_manager_t {
        ImplCookieManager::get_raw(self)
    }
}
impl ConvertReturnValue<CookieManager> for *mut _cef_cookie_manager_t {
    fn wrap_result(self) -> CookieManager {
        CookieManager(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CookieManager> for *mut _cef_cookie_manager_t {
    fn from(value: CookieManager) -> Self {
        let object = ImplCookieManager::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CookieManager {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_cookie_visitor_t`] for more documentation.
#[derive(Clone)]
pub struct CookieVisitor(RefGuard<_cef_cookie_visitor_t>);
impl CookieVisitor {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapCookieVisitor,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplCookieVisitor>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapCookieVisitor>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_cookie_visitor_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapCookieVisitor: ImplCookieVisitor {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_cookie_visitor_t, Self>);
}
pub trait ImplCookieVisitor: Clone + Sized + Rc {
    #[doc = "See [`_cef_cookie_visitor_t::visit`] for more documentation."]
    fn visit(
        &self,
        cookie: Option<&Cookie>,
        count: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
        delete_cookie: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_cookie_visitor_t) {
        impl_cef_cookie_visitor_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_cookie_visitor_t;
}
mod impl_cef_cookie_visitor_t {
    use super::*;
    pub fn init_methods<I: ImplCookieVisitor>(object: &mut _cef_cookie_visitor_t) {
        object.visit = Some(visit::<I>);
    }
    extern "C" fn visit<I: ImplCookieVisitor>(
        self_: *mut _cef_cookie_visitor_t,
        cookie: *const _cef_cookie_t,
        count: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
        delete_cookie: *mut ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_cookie, arg_count, arg_total, arg_delete_cookie) =
            (self_, cookie, count, total, delete_cookie);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_cookie = if arg_cookie.is_null() {
            None
        } else {
            Some(WrapParamRef::<Cookie, _>::from(arg_cookie))
        };
        let arg_cookie = arg_cookie.as_ref().map(|arg| arg.as_ref());
        let arg_count = arg_count.into_raw();
        let arg_total = arg_total.into_raw();
        let mut arg_delete_cookie = if arg_delete_cookie.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_delete_cookie,
            ))
        };
        let arg_delete_cookie = arg_delete_cookie.as_mut().map(|arg| arg.as_mut());
        ImplCookieVisitor::visit(
            &arg_self_.interface,
            arg_cookie,
            arg_count,
            arg_total,
            arg_delete_cookie,
        )
    }
}
impl ImplCookieVisitor for CookieVisitor {
    fn visit(
        &self,
        cookie: Option<&Cookie>,
        count: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
        delete_cookie: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .visit
                .map(|f| {
                    let (arg_cookie, arg_count, arg_total, arg_delete_cookie) =
                        (cookie, count, total, delete_cookie);
                    let arg_self_ = self.into_raw();
                    let arg_cookie = arg_cookie.cloned().map(|arg| arg.into());
                    let arg_cookie = arg_cookie
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_delete_cookie = arg_delete_cookie
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_cookie,
                        arg_count,
                        arg_total,
                        arg_delete_cookie,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_cookie_visitor_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_cookie_visitor_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CookieVisitor {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_cookie_visitor_t> for &CookieVisitor {
    fn into_raw(self) -> *mut _cef_cookie_visitor_t {
        ImplCookieVisitor::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_cookie_visitor_t> for &mut CookieVisitor {
    fn into_raw(self) -> *mut _cef_cookie_visitor_t {
        ImplCookieVisitor::get_raw(self)
    }
}
impl ConvertReturnValue<CookieVisitor> for *mut _cef_cookie_visitor_t {
    fn wrap_result(self) -> CookieVisitor {
        CookieVisitor(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CookieVisitor> for *mut _cef_cookie_visitor_t {
    fn from(value: CookieVisitor) -> Self {
        let object = ImplCookieVisitor::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CookieVisitor {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_set_cookie_callback_t`] for more documentation.
#[derive(Clone)]
pub struct SetCookieCallback(RefGuard<_cef_set_cookie_callback_t>);
impl SetCookieCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapSetCookieCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplSetCookieCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapSetCookieCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_set_cookie_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapSetCookieCallback: ImplSetCookieCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_set_cookie_callback_t, Self>);
}
pub trait ImplSetCookieCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_set_cookie_callback_t::on_complete`] for more documentation."]
    fn on_complete(&self, success: ::std::os::raw::c_int) {}
    fn init_methods(object: &mut _cef_set_cookie_callback_t) {
        impl_cef_set_cookie_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_set_cookie_callback_t;
}
mod impl_cef_set_cookie_callback_t {
    use super::*;
    pub fn init_methods<I: ImplSetCookieCallback>(object: &mut _cef_set_cookie_callback_t) {
        object.on_complete = Some(on_complete::<I>);
    }
    extern "C" fn on_complete<I: ImplSetCookieCallback>(
        self_: *mut _cef_set_cookie_callback_t,
        success: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_success) = (self_, success);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_success = arg_success.into_raw();
        ImplSetCookieCallback::on_complete(&arg_self_.interface, arg_success)
    }
}
impl ImplSetCookieCallback for SetCookieCallback {
    fn on_complete(&self, success: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.on_complete {
                let arg_success = success;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_success);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_set_cookie_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_set_cookie_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for SetCookieCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_set_cookie_callback_t> for &SetCookieCallback {
    fn into_raw(self) -> *mut _cef_set_cookie_callback_t {
        ImplSetCookieCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_set_cookie_callback_t> for &mut SetCookieCallback {
    fn into_raw(self) -> *mut _cef_set_cookie_callback_t {
        ImplSetCookieCallback::get_raw(self)
    }
}
impl ConvertReturnValue<SetCookieCallback> for *mut _cef_set_cookie_callback_t {
    fn wrap_result(self) -> SetCookieCallback {
        SetCookieCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<SetCookieCallback> for *mut _cef_set_cookie_callback_t {
    fn from(value: SetCookieCallback) -> Self {
        let object = ImplSetCookieCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for SetCookieCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_delete_cookies_callback_t`] for more documentation.
#[derive(Clone)]
pub struct DeleteCookiesCallback(RefGuard<_cef_delete_cookies_callback_t>);
impl DeleteCookiesCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDeleteCookiesCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDeleteCookiesCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDeleteCookiesCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_delete_cookies_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDeleteCookiesCallback: ImplDeleteCookiesCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_delete_cookies_callback_t, Self>);
}
pub trait ImplDeleteCookiesCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_delete_cookies_callback_t::on_complete`] for more documentation."]
    fn on_complete(&self, num_deleted: ::std::os::raw::c_int) {}
    fn init_methods(object: &mut _cef_delete_cookies_callback_t) {
        impl_cef_delete_cookies_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_delete_cookies_callback_t;
}
mod impl_cef_delete_cookies_callback_t {
    use super::*;
    pub fn init_methods<I: ImplDeleteCookiesCallback>(object: &mut _cef_delete_cookies_callback_t) {
        object.on_complete = Some(on_complete::<I>);
    }
    extern "C" fn on_complete<I: ImplDeleteCookiesCallback>(
        self_: *mut _cef_delete_cookies_callback_t,
        num_deleted: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_num_deleted) = (self_, num_deleted);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_num_deleted = arg_num_deleted.into_raw();
        ImplDeleteCookiesCallback::on_complete(&arg_self_.interface, arg_num_deleted)
    }
}
impl ImplDeleteCookiesCallback for DeleteCookiesCallback {
    fn on_complete(&self, num_deleted: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.on_complete {
                let arg_num_deleted = num_deleted;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_num_deleted);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_delete_cookies_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_delete_cookies_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DeleteCookiesCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_delete_cookies_callback_t> for &DeleteCookiesCallback {
    fn into_raw(self) -> *mut _cef_delete_cookies_callback_t {
        ImplDeleteCookiesCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_delete_cookies_callback_t> for &mut DeleteCookiesCallback {
    fn into_raw(self) -> *mut _cef_delete_cookies_callback_t {
        ImplDeleteCookiesCallback::get_raw(self)
    }
}
impl ConvertReturnValue<DeleteCookiesCallback> for *mut _cef_delete_cookies_callback_t {
    fn wrap_result(self) -> DeleteCookiesCallback {
        DeleteCookiesCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DeleteCookiesCallback> for *mut _cef_delete_cookies_callback_t {
    fn from(value: DeleteCookiesCallback) -> Self {
        let object = ImplDeleteCookiesCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DeleteCookiesCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_router_t`] for more documentation.
#[derive(Clone)]
pub struct MediaRouter(RefGuard<_cef_media_router_t>);
pub trait ImplMediaRouter: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_router_t::add_observer`] for more documentation."]
    fn add_observer(&self, observer: Option<&mut impl ImplMediaObserver>) -> Option<Registration>;
    #[doc = "See [`_cef_media_router_t::get_source`] for more documentation."]
    fn source(&self, urn: Option<&CefString>) -> Option<MediaSource>;
    #[doc = "See [`_cef_media_router_t::notify_current_sinks`] for more documentation."]
    fn notify_current_sinks(&self);
    #[doc = "See [`_cef_media_router_t::create_route`] for more documentation."]
    fn create_route(
        &self,
        source: Option<&mut MediaSource>,
        sink: Option<&mut MediaSink>,
        callback: Option<&mut impl ImplMediaRouteCreateCallback>,
    );
    #[doc = "See [`_cef_media_router_t::notify_current_routes`] for more documentation."]
    fn notify_current_routes(&self);
    fn get_raw(&self) -> *mut _cef_media_router_t;
}
impl ImplMediaRouter for MediaRouter {
    fn add_observer(&self, observer: Option<&mut impl ImplMediaObserver>) -> Option<Registration> {
        unsafe {
            self.0
                .add_observer
                .map(|f| {
                    let arg_observer = observer;
                    let arg_self_ = self.into_raw();
                    let arg_observer = arg_observer
                        .map(|arg| {
                            arg.add_ref();
                            ImplMediaObserver::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_observer);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn source(&self, urn: Option<&CefString>) -> Option<MediaSource> {
        unsafe {
            self.0
                .get_source
                .map(|f| {
                    let arg_urn = urn;
                    let arg_self_ = self.into_raw();
                    let arg_urn = arg_urn
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_urn);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn notify_current_sinks(&self) {
        unsafe {
            if let Some(f) = self.0.notify_current_sinks {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn create_route(
        &self,
        source: Option<&mut MediaSource>,
        sink: Option<&mut MediaSink>,
        callback: Option<&mut impl ImplMediaRouteCreateCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.create_route {
                let (arg_source, arg_sink, arg_callback) = (source, sink, callback);
                let arg_self_ = self.into_raw();
                let arg_source = arg_source
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaSource::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_sink = arg_sink
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaSink::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaRouteCreateCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_source, arg_sink, arg_callback);
            }
        }
    }
    fn notify_current_routes(&self) {
        unsafe {
            if let Some(f) = self.0.notify_current_routes {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_router_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_router_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaRouter {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_router_t> for &MediaRouter {
    fn into_raw(self) -> *mut _cef_media_router_t {
        ImplMediaRouter::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_router_t> for &mut MediaRouter {
    fn into_raw(self) -> *mut _cef_media_router_t {
        ImplMediaRouter::get_raw(self)
    }
}
impl ConvertReturnValue<MediaRouter> for *mut _cef_media_router_t {
    fn wrap_result(self) -> MediaRouter {
        MediaRouter(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaRouter> for *mut _cef_media_router_t {
    fn from(value: MediaRouter) -> Self {
        let object = ImplMediaRouter::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaRouter {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_observer_t`] for more documentation.
#[derive(Clone)]
pub struct MediaObserver(RefGuard<_cef_media_observer_t>);
impl MediaObserver {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapMediaObserver,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplMediaObserver>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapMediaObserver>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_media_observer_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapMediaObserver: ImplMediaObserver {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_media_observer_t, Self>);
}
pub trait ImplMediaObserver: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_observer_t::on_sinks`] for more documentation."]
    fn on_sinks(&self, sinks: Option<&[Option<MediaSink>]>) {}
    #[doc = "See [`_cef_media_observer_t::on_routes`] for more documentation."]
    fn on_routes(&self, routes: Option<&[Option<MediaRoute>]>) {}
    #[doc = "See [`_cef_media_observer_t::on_route_state_changed`] for more documentation."]
    fn on_route_state_changed(
        &self,
        route: Option<&mut MediaRoute>,
        state: MediaRouteConnectionState,
    ) {
    }
    #[doc = "See [`_cef_media_observer_t::on_route_message_received`] for more documentation."]
    fn on_route_message_received(&self, route: Option<&mut MediaRoute>, message: Option<&[u8]>) {}
    fn init_methods(object: &mut _cef_media_observer_t) {
        impl_cef_media_observer_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_media_observer_t;
}
mod impl_cef_media_observer_t {
    use super::*;
    pub fn init_methods<I: ImplMediaObserver>(object: &mut _cef_media_observer_t) {
        object.on_sinks = Some(on_sinks::<I>);
        object.on_routes = Some(on_routes::<I>);
        object.on_route_state_changed = Some(on_route_state_changed::<I>);
        object.on_route_message_received = Some(on_route_message_received::<I>);
    }
    extern "C" fn on_sinks<I: ImplMediaObserver>(
        self_: *mut _cef_media_observer_t,
        sinks_count: usize,
        sinks: *const *mut _cef_media_sink_t,
    ) {
        let (arg_self_, arg_sinks_count, arg_sinks) = (self_, sinks_count, sinks);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let vec_sinks = unsafe { arg_sinks.as_ref() }.map(|arg| {
            let arg =
                unsafe { std::slice::from_raw_parts(std::ptr::from_ref(arg), arg_sinks_count) };
            arg.iter()
                .map(|arg| {
                    if arg.is_null() {
                        None
                    } else {
                        Some(MediaSink(unsafe { RefGuard::from_raw(*arg) }))
                    }
                })
                .collect::<Vec<_>>()
        });
        let arg_sinks = vec_sinks.as_deref();
        ImplMediaObserver::on_sinks(&arg_self_.interface, arg_sinks)
    }
    extern "C" fn on_routes<I: ImplMediaObserver>(
        self_: *mut _cef_media_observer_t,
        routes_count: usize,
        routes: *const *mut _cef_media_route_t,
    ) {
        let (arg_self_, arg_routes_count, arg_routes) = (self_, routes_count, routes);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let vec_routes = unsafe { arg_routes.as_ref() }.map(|arg| {
            let arg =
                unsafe { std::slice::from_raw_parts(std::ptr::from_ref(arg), arg_routes_count) };
            arg.iter()
                .map(|arg| {
                    if arg.is_null() {
                        None
                    } else {
                        Some(MediaRoute(unsafe { RefGuard::from_raw(*arg) }))
                    }
                })
                .collect::<Vec<_>>()
        });
        let arg_routes = vec_routes.as_deref();
        ImplMediaObserver::on_routes(&arg_self_.interface, arg_routes)
    }
    extern "C" fn on_route_state_changed<I: ImplMediaObserver>(
        self_: *mut _cef_media_observer_t,
        route: *mut _cef_media_route_t,
        state: cef_media_route_connection_state_t,
    ) {
        let (arg_self_, arg_route, arg_state) = (self_, route, state);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_route =
            unsafe { arg_route.as_mut() }.map(|arg| MediaRoute(unsafe { RefGuard::from_raw(arg) }));
        let arg_route = arg_route.as_mut();
        let arg_state = arg_state.into_raw();
        ImplMediaObserver::on_route_state_changed(&arg_self_.interface, arg_route, arg_state)
    }
    extern "C" fn on_route_message_received<I: ImplMediaObserver>(
        self_: *mut _cef_media_observer_t,
        route: *mut _cef_media_route_t,
        message: *const ::std::os::raw::c_void,
        message_size: usize,
    ) {
        let (arg_self_, arg_route, arg_message, arg_message_size) =
            (self_, route, message, message_size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_route =
            unsafe { arg_route.as_mut() }.map(|arg| MediaRoute(unsafe { RefGuard::from_raw(arg) }));
        let arg_route = arg_route.as_mut();
        let arg_message = (!arg_message.is_null() && arg_message_size > 0)
            .then(|| unsafe { std::slice::from_raw_parts(arg_message.cast(), arg_message_size) });
        ImplMediaObserver::on_route_message_received(&arg_self_.interface, arg_route, arg_message)
    }
}
impl ImplMediaObserver for MediaObserver {
    fn on_sinks(&self, sinks: Option<&[Option<MediaSink>]>) {
        unsafe {
            if let Some(f) = self.0.on_sinks {
                let arg_sinks = sinks;
                let arg_self_ = self.into_raw();
                let arg_sinks_count = arg_sinks.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let vec_sinks = arg_sinks
                    .as_ref()
                    .map(|arg| {
                        arg.iter()
                            .map(|elem| {
                                elem.as_ref()
                                    .map(|elem| {
                                        elem.add_ref();
                                        elem.get_raw()
                                    })
                                    .unwrap_or(std::ptr::null_mut())
                            })
                            .collect::<Vec<_>>()
                    })
                    .unwrap_or_default();
                let arg_sinks = if vec_sinks.is_empty() {
                    std::ptr::null()
                } else {
                    vec_sinks.as_ptr()
                };
                f(arg_self_, arg_sinks_count, arg_sinks);
            }
        }
    }
    fn on_routes(&self, routes: Option<&[Option<MediaRoute>]>) {
        unsafe {
            if let Some(f) = self.0.on_routes {
                let arg_routes = routes;
                let arg_self_ = self.into_raw();
                let arg_routes_count = arg_routes.as_ref().map(|arg| arg.len()).unwrap_or_default();
                let vec_routes = arg_routes
                    .as_ref()
                    .map(|arg| {
                        arg.iter()
                            .map(|elem| {
                                elem.as_ref()
                                    .map(|elem| {
                                        elem.add_ref();
                                        elem.get_raw()
                                    })
                                    .unwrap_or(std::ptr::null_mut())
                            })
                            .collect::<Vec<_>>()
                    })
                    .unwrap_or_default();
                let arg_routes = if vec_routes.is_empty() {
                    std::ptr::null()
                } else {
                    vec_routes.as_ptr()
                };
                f(arg_self_, arg_routes_count, arg_routes);
            }
        }
    }
    fn on_route_state_changed(
        &self,
        route: Option<&mut MediaRoute>,
        state: MediaRouteConnectionState,
    ) {
        unsafe {
            if let Some(f) = self.0.on_route_state_changed {
                let (arg_route, arg_state) = (route, state);
                let arg_self_ = self.into_raw();
                let arg_route = arg_route
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaRoute::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_state = arg_state.into_raw();
                f(arg_self_, arg_route, arg_state);
            }
        }
    }
    fn on_route_message_received(&self, route: Option<&mut MediaRoute>, message: Option<&[u8]>) {
        unsafe {
            if let Some(f) = self.0.on_route_message_received {
                let (arg_route, arg_message) = (route, message);
                let arg_self_ = self.into_raw();
                let arg_route = arg_route
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaRoute::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_message_size = arg_message
                    .as_ref()
                    .map(|arg| arg.len())
                    .unwrap_or_default();
                let arg_message = arg_message
                    .and_then(|arg| {
                        if arg.is_empty() {
                            None
                        } else {
                            Some(arg.as_ptr().cast())
                        }
                    })
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_route, arg_message, arg_message_size);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_observer_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_observer_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaObserver {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_observer_t> for &MediaObserver {
    fn into_raw(self) -> *mut _cef_media_observer_t {
        ImplMediaObserver::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_observer_t> for &mut MediaObserver {
    fn into_raw(self) -> *mut _cef_media_observer_t {
        ImplMediaObserver::get_raw(self)
    }
}
impl ConvertReturnValue<MediaObserver> for *mut _cef_media_observer_t {
    fn wrap_result(self) -> MediaObserver {
        MediaObserver(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaObserver> for *mut _cef_media_observer_t {
    fn from(value: MediaObserver) -> Self {
        let object = ImplMediaObserver::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaObserver {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_route_t`] for more documentation.
#[derive(Clone)]
pub struct MediaRoute(RefGuard<_cef_media_route_t>);
pub trait ImplMediaRoute: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_route_t::get_id`] for more documentation."]
    fn id(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_media_route_t::get_source`] for more documentation."]
    fn source(&self) -> Option<MediaSource>;
    #[doc = "See [`_cef_media_route_t::get_sink`] for more documentation."]
    fn sink(&self) -> Option<MediaSink>;
    #[doc = "See [`_cef_media_route_t::send_route_message`] for more documentation."]
    fn send_route_message(&self, message: Option<&[u8]>);
    #[doc = "See [`_cef_media_route_t::terminate`] for more documentation."]
    fn terminate(&self);
    fn get_raw(&self) -> *mut _cef_media_route_t;
}
impl ImplMediaRoute for MediaRoute {
    fn id(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_id
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn source(&self) -> Option<MediaSource> {
        unsafe {
            self.0
                .get_source
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn sink(&self) -> Option<MediaSink> {
        unsafe {
            self.0
                .get_sink
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn send_route_message(&self, message: Option<&[u8]>) {
        unsafe {
            if let Some(f) = self.0.send_route_message {
                let arg_message = message;
                let arg_self_ = self.into_raw();
                let arg_message_size = arg_message
                    .as_ref()
                    .map(|arg| arg.len())
                    .unwrap_or_default();
                let arg_message = arg_message
                    .and_then(|arg| {
                        if arg.is_empty() {
                            None
                        } else {
                            Some(arg.as_ptr().cast())
                        }
                    })
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_message, arg_message_size);
            }
        }
    }
    fn terminate(&self) {
        unsafe {
            if let Some(f) = self.0.terminate {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_route_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_route_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaRoute {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_route_t> for &MediaRoute {
    fn into_raw(self) -> *mut _cef_media_route_t {
        ImplMediaRoute::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_route_t> for &mut MediaRoute {
    fn into_raw(self) -> *mut _cef_media_route_t {
        ImplMediaRoute::get_raw(self)
    }
}
impl ConvertReturnValue<MediaRoute> for *mut _cef_media_route_t {
    fn wrap_result(self) -> MediaRoute {
        MediaRoute(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaRoute> for *mut _cef_media_route_t {
    fn from(value: MediaRoute) -> Self {
        let object = ImplMediaRoute::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaRoute {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_route_create_callback_t`] for more documentation.
#[derive(Clone)]
pub struct MediaRouteCreateCallback(RefGuard<_cef_media_route_create_callback_t>);
impl MediaRouteCreateCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapMediaRouteCreateCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplMediaRouteCreateCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapMediaRouteCreateCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_media_route_create_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapMediaRouteCreateCallback: ImplMediaRouteCreateCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_media_route_create_callback_t, Self>);
}
pub trait ImplMediaRouteCreateCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_route_create_callback_t::on_media_route_create_finished`] for more documentation."]
    fn on_media_route_create_finished(
        &self,
        result: MediaRouteCreateResult,
        error: Option<&CefString>,
        route: Option<&mut MediaRoute>,
    ) {
    }
    fn init_methods(object: &mut _cef_media_route_create_callback_t) {
        impl_cef_media_route_create_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_media_route_create_callback_t;
}
mod impl_cef_media_route_create_callback_t {
    use super::*;
    pub fn init_methods<I: ImplMediaRouteCreateCallback>(
        object: &mut _cef_media_route_create_callback_t,
    ) {
        object.on_media_route_create_finished = Some(on_media_route_create_finished::<I>);
    }
    extern "C" fn on_media_route_create_finished<I: ImplMediaRouteCreateCallback>(
        self_: *mut _cef_media_route_create_callback_t,
        result: cef_media_route_create_result_t,
        error: *const cef_string_t,
        route: *mut _cef_media_route_t,
    ) {
        let (arg_self_, arg_result, arg_error, arg_route) = (self_, result, error, route);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_result = arg_result.into_raw();
        let arg_error = if arg_error.is_null() {
            None
        } else {
            Some(arg_error.into())
        };
        let arg_error = arg_error.as_ref();
        let mut arg_route =
            unsafe { arg_route.as_mut() }.map(|arg| MediaRoute(unsafe { RefGuard::from_raw(arg) }));
        let arg_route = arg_route.as_mut();
        ImplMediaRouteCreateCallback::on_media_route_create_finished(
            &arg_self_.interface,
            arg_result,
            arg_error,
            arg_route,
        )
    }
}
impl ImplMediaRouteCreateCallback for MediaRouteCreateCallback {
    fn on_media_route_create_finished(
        &self,
        result: MediaRouteCreateResult,
        error: Option<&CefString>,
        route: Option<&mut MediaRoute>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_media_route_create_finished {
                let (arg_result, arg_error, arg_route) = (result, error, route);
                let arg_self_ = self.into_raw();
                let arg_result = arg_result.into_raw();
                let arg_error = arg_error
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_route = arg_route
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaRoute::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_result, arg_error, arg_route);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_route_create_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_route_create_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaRouteCreateCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_route_create_callback_t> for &MediaRouteCreateCallback {
    fn into_raw(self) -> *mut _cef_media_route_create_callback_t {
        ImplMediaRouteCreateCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_route_create_callback_t> for &mut MediaRouteCreateCallback {
    fn into_raw(self) -> *mut _cef_media_route_create_callback_t {
        ImplMediaRouteCreateCallback::get_raw(self)
    }
}
impl ConvertReturnValue<MediaRouteCreateCallback> for *mut _cef_media_route_create_callback_t {
    fn wrap_result(self) -> MediaRouteCreateCallback {
        MediaRouteCreateCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaRouteCreateCallback> for *mut _cef_media_route_create_callback_t {
    fn from(value: MediaRouteCreateCallback) -> Self {
        let object = ImplMediaRouteCreateCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaRouteCreateCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_sink_t`] for more documentation.
#[derive(Clone)]
pub struct MediaSink(RefGuard<_cef_media_sink_t>);
pub trait ImplMediaSink: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_sink_t::get_id`] for more documentation."]
    fn id(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_media_sink_t::get_name`] for more documentation."]
    fn name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_media_sink_t::get_icon_type`] for more documentation."]
    fn icon_type(&self) -> MediaSinkIconType;
    #[doc = "See [`_cef_media_sink_t::get_device_info`] for more documentation."]
    fn device_info(&self, callback: Option<&mut impl ImplMediaSinkDeviceInfoCallback>);
    #[doc = "See [`_cef_media_sink_t::is_cast_sink`] for more documentation."]
    fn is_cast_sink(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_media_sink_t::is_dial_sink`] for more documentation."]
    fn is_dial_sink(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_media_sink_t::is_compatible_with`] for more documentation."]
    fn is_compatible_with(&self, source: Option<&mut MediaSource>) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_media_sink_t;
}
impl ImplMediaSink for MediaSink {
    fn id(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_id
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn icon_type(&self) -> MediaSinkIconType {
        unsafe {
            self.0
                .get_icon_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn device_info(&self, callback: Option<&mut impl ImplMediaSinkDeviceInfoCallback>) {
        unsafe {
            if let Some(f) = self.0.get_device_info {
                let arg_callback = callback;
                let arg_self_ = self.into_raw();
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplMediaSinkDeviceInfoCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_callback);
            }
        }
    }
    fn is_cast_sink(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_cast_sink
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_dial_sink(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_dial_sink
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_compatible_with(&self, source: Option<&mut MediaSource>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_compatible_with
                .map(|f| {
                    let arg_source = source;
                    let arg_self_ = self.into_raw();
                    let arg_source = arg_source
                        .map(|arg| {
                            arg.add_ref();
                            ImplMediaSource::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_source);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_media_sink_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_sink_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaSink {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_sink_t> for &MediaSink {
    fn into_raw(self) -> *mut _cef_media_sink_t {
        ImplMediaSink::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_sink_t> for &mut MediaSink {
    fn into_raw(self) -> *mut _cef_media_sink_t {
        ImplMediaSink::get_raw(self)
    }
}
impl ConvertReturnValue<MediaSink> for *mut _cef_media_sink_t {
    fn wrap_result(self) -> MediaSink {
        MediaSink(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaSink> for *mut _cef_media_sink_t {
    fn from(value: MediaSink) -> Self {
        let object = ImplMediaSink::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaSink {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_sink_device_info_callback_t`] for more documentation.
#[derive(Clone)]
pub struct MediaSinkDeviceInfoCallback(RefGuard<_cef_media_sink_device_info_callback_t>);
impl MediaSinkDeviceInfoCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapMediaSinkDeviceInfoCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplMediaSinkDeviceInfoCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapMediaSinkDeviceInfoCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_media_sink_device_info_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapMediaSinkDeviceInfoCallback: ImplMediaSinkDeviceInfoCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_media_sink_device_info_callback_t, Self>);
}
pub trait ImplMediaSinkDeviceInfoCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_sink_device_info_callback_t::on_media_sink_device_info`] for more documentation."]
    fn on_media_sink_device_info(&self, device_info: Option<&MediaSinkDeviceInfo>) {}
    fn init_methods(object: &mut _cef_media_sink_device_info_callback_t) {
        impl_cef_media_sink_device_info_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_media_sink_device_info_callback_t;
}
mod impl_cef_media_sink_device_info_callback_t {
    use super::*;
    pub fn init_methods<I: ImplMediaSinkDeviceInfoCallback>(
        object: &mut _cef_media_sink_device_info_callback_t,
    ) {
        object.on_media_sink_device_info = Some(on_media_sink_device_info::<I>);
    }
    extern "C" fn on_media_sink_device_info<I: ImplMediaSinkDeviceInfoCallback>(
        self_: *mut _cef_media_sink_device_info_callback_t,
        device_info: *const _cef_media_sink_device_info_t,
    ) {
        let (arg_self_, arg_device_info) = (self_, device_info);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_device_info = if arg_device_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<MediaSinkDeviceInfo, _>::from(
                arg_device_info,
            ))
        };
        let arg_device_info = arg_device_info.as_ref().map(|arg| arg.as_ref());
        ImplMediaSinkDeviceInfoCallback::on_media_sink_device_info(
            &arg_self_.interface,
            arg_device_info,
        )
    }
}
impl ImplMediaSinkDeviceInfoCallback for MediaSinkDeviceInfoCallback {
    fn on_media_sink_device_info(&self, device_info: Option<&MediaSinkDeviceInfo>) {
        unsafe {
            if let Some(f) = self.0.on_media_sink_device_info {
                let arg_device_info = device_info;
                let arg_self_ = self.into_raw();
                let arg_device_info = arg_device_info.cloned().map(|arg| arg.into());
                let arg_device_info = arg_device_info
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_device_info);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_sink_device_info_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_sink_device_info_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaSinkDeviceInfoCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_sink_device_info_callback_t> for &MediaSinkDeviceInfoCallback {
    fn into_raw(self) -> *mut _cef_media_sink_device_info_callback_t {
        ImplMediaSinkDeviceInfoCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_sink_device_info_callback_t>
    for &mut MediaSinkDeviceInfoCallback
{
    fn into_raw(self) -> *mut _cef_media_sink_device_info_callback_t {
        ImplMediaSinkDeviceInfoCallback::get_raw(self)
    }
}
impl ConvertReturnValue<MediaSinkDeviceInfoCallback>
    for *mut _cef_media_sink_device_info_callback_t
{
    fn wrap_result(self) -> MediaSinkDeviceInfoCallback {
        MediaSinkDeviceInfoCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaSinkDeviceInfoCallback> for *mut _cef_media_sink_device_info_callback_t {
    fn from(value: MediaSinkDeviceInfoCallback) -> Self {
        let object = ImplMediaSinkDeviceInfoCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaSinkDeviceInfoCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_source_t`] for more documentation.
#[derive(Clone)]
pub struct MediaSource(RefGuard<_cef_media_source_t>);
pub trait ImplMediaSource: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_source_t::get_id`] for more documentation."]
    fn id(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_media_source_t::is_cast_source`] for more documentation."]
    fn is_cast_source(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_media_source_t::is_dial_source`] for more documentation."]
    fn is_dial_source(&self) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_media_source_t;
}
impl ImplMediaSource for MediaSource {
    fn id(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_id
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_cast_source(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_cast_source
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_dial_source(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_dial_source
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_media_source_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_source_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaSource {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_source_t> for &MediaSource {
    fn into_raw(self) -> *mut _cef_media_source_t {
        ImplMediaSource::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_source_t> for &mut MediaSource {
    fn into_raw(self) -> *mut _cef_media_source_t {
        ImplMediaSource::get_raw(self)
    }
}
impl ConvertReturnValue<MediaSource> for *mut _cef_media_source_t {
    fn wrap_result(self) -> MediaSource {
        MediaSource(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaSource> for *mut _cef_media_source_t {
    fn from(value: MediaSource) -> Self {
        let object = ImplMediaSource::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaSource {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_preference_registrar_t`] for more documentation.
#[derive(Clone, Copy)]
pub struct PreferenceRegistrar(*mut _cef_preference_registrar_t);
pub trait ImplPreferenceRegistrar: Sized {
    #[doc = "See [`_cef_preference_registrar_t::add_preference`] for more documentation."]
    fn add_preference(
        &self,
        name: Option<&CefString>,
        default_value: Option<&mut Value>,
    ) -> ::std::os::raw::c_int;
    fn init_methods(object: &mut _cef_preference_registrar_t) {
        impl_cef_preference_registrar_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_preference_registrar_t;
}
mod impl_cef_preference_registrar_t {
    use super::*;
    pub fn init_methods<I: ImplPreferenceRegistrar>(object: &mut _cef_preference_registrar_t) {
        object.add_preference = Some(add_preference::<I>);
    }
    extern "C" fn add_preference<I: ImplPreferenceRegistrar>(
        self_: *mut _cef_preference_registrar_t,
        name: *const cef_string_t,
        default_value: *mut _cef_value_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_name, arg_default_value) = (self_, name, default_value);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_name = if arg_name.is_null() {
            None
        } else {
            Some(arg_name.into())
        };
        let arg_name = arg_name.as_ref();
        let mut arg_default_value = unsafe { arg_default_value.as_mut() }
            .map(|arg| Value(unsafe { RefGuard::from_raw(arg) }));
        let arg_default_value = arg_default_value.as_mut();
        ImplPreferenceRegistrar::add_preference(&arg_self_.interface, arg_name, arg_default_value)
    }
}
impl ImplPreferenceRegistrar for PreferenceRegistrar {
    fn add_preference(
        &self,
        name: Option<&CefString>,
        default_value: Option<&mut Value>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .as_ref()
                .and_then(|this| this.add_preference)
                .map(|f| {
                    let (arg_name, arg_default_value) = (name, default_value);
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_default_value = arg_default_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_name, arg_default_value);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_preference_registrar_t {
        self.0
    }
}
impl ConvertParam<*mut _cef_preference_registrar_t> for &PreferenceRegistrar {
    fn into_raw(self) -> *mut _cef_preference_registrar_t {
        ImplPreferenceRegistrar::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_preference_registrar_t> for &mut PreferenceRegistrar {
    fn into_raw(self) -> *mut _cef_preference_registrar_t {
        ImplPreferenceRegistrar::get_raw(self)
    }
}
impl ConvertReturnValue<PreferenceRegistrar> for *mut _cef_preference_registrar_t {
    fn wrap_result(self) -> PreferenceRegistrar {
        PreferenceRegistrar(self)
    }
}
impl From<PreferenceRegistrar> for *mut _cef_preference_registrar_t {
    fn from(value: PreferenceRegistrar) -> Self {
        ImplPreferenceRegistrar::get_raw(&value)
    }
}
impl Default for PreferenceRegistrar {
    fn default() -> Self {
        Self(std::ptr::null_mut())
    }
}

/// See [`_cef_preference_observer_t`] for more documentation.
#[derive(Clone)]
pub struct PreferenceObserver(RefGuard<_cef_preference_observer_t>);
impl PreferenceObserver {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapPreferenceObserver,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplPreferenceObserver>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapPreferenceObserver>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_preference_observer_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapPreferenceObserver: ImplPreferenceObserver {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_preference_observer_t, Self>);
}
pub trait ImplPreferenceObserver: Clone + Sized + Rc {
    #[doc = "See [`_cef_preference_observer_t::on_preference_changed`] for more documentation."]
    fn on_preference_changed(&self, name: Option<&CefString>) {}
    fn init_methods(object: &mut _cef_preference_observer_t) {
        impl_cef_preference_observer_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_preference_observer_t;
}
mod impl_cef_preference_observer_t {
    use super::*;
    pub fn init_methods<I: ImplPreferenceObserver>(object: &mut _cef_preference_observer_t) {
        object.on_preference_changed = Some(on_preference_changed::<I>);
    }
    extern "C" fn on_preference_changed<I: ImplPreferenceObserver>(
        self_: *mut _cef_preference_observer_t,
        name: *const cef_string_t,
    ) {
        let (arg_self_, arg_name) = (self_, name);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_name = if arg_name.is_null() {
            None
        } else {
            Some(arg_name.into())
        };
        let arg_name = arg_name.as_ref();
        ImplPreferenceObserver::on_preference_changed(&arg_self_.interface, arg_name)
    }
}
impl ImplPreferenceObserver for PreferenceObserver {
    fn on_preference_changed(&self, name: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.on_preference_changed {
                let arg_name = name;
                let arg_self_ = self.into_raw();
                let arg_name = arg_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_name);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_preference_observer_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_preference_observer_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PreferenceObserver {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_preference_observer_t> for &PreferenceObserver {
    fn into_raw(self) -> *mut _cef_preference_observer_t {
        ImplPreferenceObserver::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_preference_observer_t> for &mut PreferenceObserver {
    fn into_raw(self) -> *mut _cef_preference_observer_t {
        ImplPreferenceObserver::get_raw(self)
    }
}
impl ConvertReturnValue<PreferenceObserver> for *mut _cef_preference_observer_t {
    fn wrap_result(self) -> PreferenceObserver {
        PreferenceObserver(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PreferenceObserver> for *mut _cef_preference_observer_t {
    fn from(value: PreferenceObserver) -> Self {
        let object = ImplPreferenceObserver::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PreferenceObserver {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_preference_manager_t`] for more documentation.
#[derive(Clone)]
pub struct PreferenceManager(RefGuard<_cef_preference_manager_t>);
pub trait ImplPreferenceManager: Clone + Sized + Rc {
    #[doc = "See [`_cef_preference_manager_t::has_preference`] for more documentation."]
    fn has_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_preference_manager_t::get_preference`] for more documentation."]
    fn preference(&self, name: Option<&CefString>) -> Option<Value>;
    #[doc = "See [`_cef_preference_manager_t::get_all_preferences`] for more documentation."]
    fn all_preferences(&self, include_defaults: ::std::os::raw::c_int) -> Option<DictionaryValue>;
    #[doc = "See [`_cef_preference_manager_t::can_set_preference`] for more documentation."]
    fn can_set_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_preference_manager_t::set_preference`] for more documentation."]
    fn set_preference(
        &self,
        name: Option<&CefString>,
        value: Option<&mut Value>,
        error: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_preference_manager_t::add_preference_observer`] for more documentation."]
    fn add_preference_observer(
        &self,
        name: Option<&CefString>,
        observer: Option<&mut impl ImplPreferenceObserver>,
    ) -> Option<Registration>;
    fn get_raw(&self) -> *mut _cef_preference_manager_t;
}
impl ImplPreferenceManager for PreferenceManager {
    fn has_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_preference
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn preference(&self, name: Option<&CefString>) -> Option<Value> {
        unsafe {
            self.0
                .get_preference
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn all_preferences(&self, include_defaults: ::std::os::raw::c_int) -> Option<DictionaryValue> {
        unsafe {
            self.0
                .get_all_preferences
                .map(|f| {
                    let arg_include_defaults = include_defaults;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_include_defaults);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn can_set_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_set_preference
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_preference(
        &self,
        name: Option<&CefString>,
        value: Option<&mut Value>,
        error: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_preference
                .map(|f| {
                    let (arg_name, arg_value, arg_error) = (name, value, error);
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_value = arg_value
                        .map(|arg| {
                            arg.add_ref();
                            ImplValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_error = arg_error
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_name, arg_value, arg_error);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_preference_observer(
        &self,
        name: Option<&CefString>,
        observer: Option<&mut impl ImplPreferenceObserver>,
    ) -> Option<Registration> {
        unsafe {
            self.0
                .add_preference_observer
                .map(|f| {
                    let (arg_name, arg_observer) = (name, observer);
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_observer = arg_observer
                        .map(|arg| {
                            arg.add_ref();
                            ImplPreferenceObserver::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_name, arg_observer);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_preference_manager_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_preference_manager_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PreferenceManager {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_preference_manager_t> for &PreferenceManager {
    fn into_raw(self) -> *mut _cef_preference_manager_t {
        ImplPreferenceManager::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_preference_manager_t> for &mut PreferenceManager {
    fn into_raw(self) -> *mut _cef_preference_manager_t {
        ImplPreferenceManager::get_raw(self)
    }
}
impl ConvertReturnValue<PreferenceManager> for *mut _cef_preference_manager_t {
    fn wrap_result(self) -> PreferenceManager {
        PreferenceManager(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PreferenceManager> for *mut _cef_preference_manager_t {
    fn from(value: PreferenceManager) -> Self {
        let object = ImplPreferenceManager::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PreferenceManager {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_resolve_callback_t`] for more documentation.
#[derive(Clone)]
pub struct ResolveCallback(RefGuard<_cef_resolve_callback_t>);
impl ResolveCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapResolveCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplResolveCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapResolveCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_resolve_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapResolveCallback: ImplResolveCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_resolve_callback_t, Self>);
}
pub trait ImplResolveCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_resolve_callback_t::on_resolve_completed`] for more documentation."]
    fn on_resolve_completed(&self, result: Errorcode, resolved_ips: Option<&mut CefStringList>) {}
    fn init_methods(object: &mut _cef_resolve_callback_t) {
        impl_cef_resolve_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_resolve_callback_t;
}
mod impl_cef_resolve_callback_t {
    use super::*;
    pub fn init_methods<I: ImplResolveCallback>(object: &mut _cef_resolve_callback_t) {
        object.on_resolve_completed = Some(on_resolve_completed::<I>);
    }
    extern "C" fn on_resolve_completed<I: ImplResolveCallback>(
        self_: *mut _cef_resolve_callback_t,
        result: cef_errorcode_t,
        resolved_ips: *mut _cef_string_list_t,
    ) {
        let (arg_self_, arg_result, arg_resolved_ips) = (self_, result, resolved_ips);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_result = arg_result.into_raw();
        let mut arg_resolved_ips = if arg_resolved_ips.is_null() {
            None
        } else {
            Some(arg_resolved_ips.into())
        };
        let arg_resolved_ips = arg_resolved_ips.as_mut();
        ImplResolveCallback::on_resolve_completed(
            &arg_self_.interface,
            arg_result,
            arg_resolved_ips,
        )
    }
}
impl ImplResolveCallback for ResolveCallback {
    fn on_resolve_completed(&self, result: Errorcode, resolved_ips: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.on_resolve_completed {
                let (arg_result, arg_resolved_ips) = (result, resolved_ips);
                let arg_self_ = self.into_raw();
                let arg_result = arg_result.into_raw();
                let arg_resolved_ips = arg_resolved_ips
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_result, arg_resolved_ips);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_resolve_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_resolve_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResolveCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_resolve_callback_t> for &ResolveCallback {
    fn into_raw(self) -> *mut _cef_resolve_callback_t {
        ImplResolveCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_resolve_callback_t> for &mut ResolveCallback {
    fn into_raw(self) -> *mut _cef_resolve_callback_t {
        ImplResolveCallback::get_raw(self)
    }
}
impl ConvertReturnValue<ResolveCallback> for *mut _cef_resolve_callback_t {
    fn wrap_result(self) -> ResolveCallback {
        ResolveCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResolveCallback> for *mut _cef_resolve_callback_t {
    fn from(value: ResolveCallback) -> Self {
        let object = ImplResolveCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResolveCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_setting_observer_t`] for more documentation.
#[derive(Clone)]
pub struct SettingObserver(RefGuard<_cef_setting_observer_t>);
impl SettingObserver {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapSettingObserver,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplSettingObserver>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapSettingObserver>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_setting_observer_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapSettingObserver: ImplSettingObserver {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_setting_observer_t, Self>);
}
pub trait ImplSettingObserver: Clone + Sized + Rc {
    #[doc = "See [`_cef_setting_observer_t::on_setting_changed`] for more documentation."]
    fn on_setting_changed(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) {
    }
    fn init_methods(object: &mut _cef_setting_observer_t) {
        impl_cef_setting_observer_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_setting_observer_t;
}
mod impl_cef_setting_observer_t {
    use super::*;
    pub fn init_methods<I: ImplSettingObserver>(object: &mut _cef_setting_observer_t) {
        object.on_setting_changed = Some(on_setting_changed::<I>);
    }
    extern "C" fn on_setting_changed<I: ImplSettingObserver>(
        self_: *mut _cef_setting_observer_t,
        requesting_url: *const cef_string_t,
        top_level_url: *const cef_string_t,
        content_type: cef_content_setting_types_t,
    ) {
        let (arg_self_, arg_requesting_url, arg_top_level_url, arg_content_type) =
            (self_, requesting_url, top_level_url, content_type);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_requesting_url = if arg_requesting_url.is_null() {
            None
        } else {
            Some(arg_requesting_url.into())
        };
        let arg_requesting_url = arg_requesting_url.as_ref();
        let arg_top_level_url = if arg_top_level_url.is_null() {
            None
        } else {
            Some(arg_top_level_url.into())
        };
        let arg_top_level_url = arg_top_level_url.as_ref();
        let arg_content_type = arg_content_type.into_raw();
        ImplSettingObserver::on_setting_changed(
            &arg_self_.interface,
            arg_requesting_url,
            arg_top_level_url,
            arg_content_type,
        )
    }
}
impl ImplSettingObserver for SettingObserver {
    fn on_setting_changed(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) {
        unsafe {
            if let Some(f) = self.0.on_setting_changed {
                let (arg_requesting_url, arg_top_level_url, arg_content_type) =
                    (requesting_url, top_level_url, content_type);
                let arg_self_ = self.into_raw();
                let arg_requesting_url = arg_requesting_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_top_level_url = arg_top_level_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_content_type = arg_content_type.into_raw();
                f(
                    arg_self_,
                    arg_requesting_url,
                    arg_top_level_url,
                    arg_content_type,
                );
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_setting_observer_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_setting_observer_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for SettingObserver {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_setting_observer_t> for &SettingObserver {
    fn into_raw(self) -> *mut _cef_setting_observer_t {
        ImplSettingObserver::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_setting_observer_t> for &mut SettingObserver {
    fn into_raw(self) -> *mut _cef_setting_observer_t {
        ImplSettingObserver::get_raw(self)
    }
}
impl ConvertReturnValue<SettingObserver> for *mut _cef_setting_observer_t {
    fn wrap_result(self) -> SettingObserver {
        SettingObserver(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<SettingObserver> for *mut _cef_setting_observer_t {
    fn from(value: SettingObserver) -> Self {
        let object = ImplSettingObserver::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for SettingObserver {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_request_context_t`] for more documentation.
#[derive(Clone)]
pub struct RequestContext(RefGuard<_cef_request_context_t>);
pub trait ImplRequestContext: ImplPreferenceManager {
    #[doc = "See [`_cef_request_context_t::is_same`] for more documentation."]
    fn is_same(&self, other: Option<&mut RequestContext>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_context_t::is_sharing_with`] for more documentation."]
    fn is_sharing_with(&self, other: Option<&mut RequestContext>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_context_t::is_global`] for more documentation."]
    fn is_global(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_context_t::get_handler`] for more documentation."]
    fn handler(&self) -> Option<RequestContextHandler>;
    #[doc = "See [`_cef_request_context_t::get_cache_path`] for more documentation."]
    fn cache_path(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_request_context_t::get_cookie_manager`] for more documentation."]
    fn cookie_manager(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> Option<CookieManager>;
    #[doc = "See [`_cef_request_context_t::register_scheme_handler_factory`] for more documentation."]
    fn register_scheme_handler_factory(
        &self,
        scheme_name: Option<&CefString>,
        domain_name: Option<&CefString>,
        factory: Option<&mut impl ImplSchemeHandlerFactory>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_context_t::clear_scheme_handler_factories`] for more documentation."]
    fn clear_scheme_handler_factories(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_request_context_t::clear_certificate_exceptions`] for more documentation."]
    fn clear_certificate_exceptions(&self, callback: Option<&mut impl ImplCompletionCallback>);
    #[doc = "See [`_cef_request_context_t::clear_http_auth_credentials`] for more documentation."]
    fn clear_http_auth_credentials(&self, callback: Option<&mut impl ImplCompletionCallback>);
    #[doc = "See [`_cef_request_context_t::close_all_connections`] for more documentation."]
    fn close_all_connections(&self, callback: Option<&mut impl ImplCompletionCallback>);
    #[doc = "See [`_cef_request_context_t::resolve_host`] for more documentation."]
    fn resolve_host(
        &self,
        origin: Option<&CefString>,
        callback: Option<&mut impl ImplResolveCallback>,
    );
    #[doc = "See [`_cef_request_context_t::get_media_router`] for more documentation."]
    fn media_router(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> Option<MediaRouter>;
    #[doc = "See [`_cef_request_context_t::get_website_setting`] for more documentation."]
    fn website_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) -> Option<Value>;
    #[doc = "See [`_cef_request_context_t::set_website_setting`] for more documentation."]
    fn set_website_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
        value: Option<&mut Value>,
    );
    #[doc = "See [`_cef_request_context_t::get_content_setting`] for more documentation."]
    fn content_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) -> ContentSettingValues;
    #[doc = "See [`_cef_request_context_t::set_content_setting`] for more documentation."]
    fn set_content_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
        value: ContentSettingValues,
    );
    #[doc = "See [`_cef_request_context_t::set_chrome_color_scheme`] for more documentation."]
    fn set_chrome_color_scheme(&self, variant: ColorVariant, user_color: u32);
    #[doc = "See [`_cef_request_context_t::get_chrome_color_scheme_mode`] for more documentation."]
    fn chrome_color_scheme_mode(&self) -> ColorVariant;
    #[doc = "See [`_cef_request_context_t::get_chrome_color_scheme_color`] for more documentation."]
    fn chrome_color_scheme_color(&self) -> cef_color_t;
    #[doc = "See [`_cef_request_context_t::get_chrome_color_scheme_variant`] for more documentation."]
    fn chrome_color_scheme_variant(&self) -> ColorVariant;
    #[doc = "See [`_cef_request_context_t::add_setting_observer`] for more documentation."]
    fn add_setting_observer(
        &self,
        observer: Option<&mut impl ImplSettingObserver>,
    ) -> Option<Registration>;
    fn get_raw(&self) -> *mut _cef_request_context_t {
        <Self as ImplPreferenceManager>::get_raw(self).cast()
    }
}
impl ImplPreferenceManager for RequestContext {
    fn has_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int {
        PreferenceManager::from(self).has_preference(name)
    }
    fn preference(&self, name: Option<&CefString>) -> Option<Value> {
        PreferenceManager::from(self).preference(name)
    }
    fn all_preferences(&self, include_defaults: ::std::os::raw::c_int) -> Option<DictionaryValue> {
        PreferenceManager::from(self).all_preferences(include_defaults)
    }
    fn can_set_preference(&self, name: Option<&CefString>) -> ::std::os::raw::c_int {
        PreferenceManager::from(self).can_set_preference(name)
    }
    fn set_preference(
        &self,
        name: Option<&CefString>,
        value: Option<&mut Value>,
        error: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        PreferenceManager::from(self).set_preference(name, value, error)
    }
    fn add_preference_observer(
        &self,
        name: Option<&CefString>,
        observer: Option<&mut impl ImplPreferenceObserver>,
    ) -> Option<Registration> {
        PreferenceManager::from(self).add_preference_observer(name, observer)
    }
    fn get_raw(&self) -> *mut _cef_preference_manager_t {
        unsafe { RefGuard::into_raw(&self.0).cast() }
    }
}
impl std::convert::From<&RequestContext> for PreferenceManager {
    fn from(from: &RequestContext) -> Self {
        PreferenceManager(unsafe { RefGuard::from_raw_add_ref(RefGuard::into_raw(&from.0).cast()) })
    }
}
impl ImplRequestContext for RequestContext {
    fn is_same(&self, other: Option<&mut RequestContext>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_other = other;
                    let arg_self_ = self.into_raw();
                    let arg_other = arg_other
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequestContext::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_other);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_sharing_with(&self, other: Option<&mut RequestContext>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_sharing_with
                .map(|f| {
                    let arg_other = other;
                    let arg_self_ = self.into_raw();
                    let arg_other = arg_other
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequestContext::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_other);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_global(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_global
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn handler(&self) -> Option<RequestContextHandler> {
        unsafe {
            self.0
                .get_handler
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn cache_path(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_cache_path
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn cookie_manager(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> Option<CookieManager> {
        unsafe {
            self.0
                .get_cookie_manager
                .map(|f| {
                    let arg_callback = callback;
                    let arg_self_ = self.into_raw();
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCompletionCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_callback);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn register_scheme_handler_factory(
        &self,
        scheme_name: Option<&CefString>,
        domain_name: Option<&CefString>,
        factory: Option<&mut impl ImplSchemeHandlerFactory>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .register_scheme_handler_factory
                .map(|f| {
                    let (arg_scheme_name, arg_domain_name, arg_factory) =
                        (scheme_name, domain_name, factory);
                    let arg_self_ = self.into_raw();
                    let arg_scheme_name = arg_scheme_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_domain_name = arg_domain_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_factory = arg_factory
                        .map(|arg| {
                            arg.add_ref();
                            ImplSchemeHandlerFactory::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_scheme_name, arg_domain_name, arg_factory);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn clear_scheme_handler_factories(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .clear_scheme_handler_factories
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn clear_certificate_exceptions(&self, callback: Option<&mut impl ImplCompletionCallback>) {
        unsafe {
            if let Some(f) = self.0.clear_certificate_exceptions {
                let arg_callback = callback;
                let arg_self_ = self.into_raw();
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplCompletionCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_callback);
            }
        }
    }
    fn clear_http_auth_credentials(&self, callback: Option<&mut impl ImplCompletionCallback>) {
        unsafe {
            if let Some(f) = self.0.clear_http_auth_credentials {
                let arg_callback = callback;
                let arg_self_ = self.into_raw();
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplCompletionCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_callback);
            }
        }
    }
    fn close_all_connections(&self, callback: Option<&mut impl ImplCompletionCallback>) {
        unsafe {
            if let Some(f) = self.0.close_all_connections {
                let arg_callback = callback;
                let arg_self_ = self.into_raw();
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplCompletionCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_callback);
            }
        }
    }
    fn resolve_host(
        &self,
        origin: Option<&CefString>,
        callback: Option<&mut impl ImplResolveCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.resolve_host {
                let (arg_origin, arg_callback) = (origin, callback);
                let arg_self_ = self.into_raw();
                let arg_origin = arg_origin
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplResolveCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_origin, arg_callback);
            }
        }
    }
    fn media_router(
        &self,
        callback: Option<&mut impl ImplCompletionCallback>,
    ) -> Option<MediaRouter> {
        unsafe {
            self.0
                .get_media_router
                .map(|f| {
                    let arg_callback = callback;
                    let arg_self_ = self.into_raw();
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCompletionCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_callback);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn website_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) -> Option<Value> {
        unsafe {
            self.0
                .get_website_setting
                .map(|f| {
                    let (arg_requesting_url, arg_top_level_url, arg_content_type) =
                        (requesting_url, top_level_url, content_type);
                    let arg_self_ = self.into_raw();
                    let arg_requesting_url = arg_requesting_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_top_level_url = arg_top_level_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_content_type = arg_content_type.into_raw();
                    let result = f(
                        arg_self_,
                        arg_requesting_url,
                        arg_top_level_url,
                        arg_content_type,
                    );
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_website_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
        value: Option<&mut Value>,
    ) {
        unsafe {
            if let Some(f) = self.0.set_website_setting {
                let (arg_requesting_url, arg_top_level_url, arg_content_type, arg_value) =
                    (requesting_url, top_level_url, content_type, value);
                let arg_self_ = self.into_raw();
                let arg_requesting_url = arg_requesting_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_top_level_url = arg_top_level_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_content_type = arg_content_type.into_raw();
                let arg_value = arg_value
                    .map(|arg| {
                        arg.add_ref();
                        ImplValue::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_requesting_url,
                    arg_top_level_url,
                    arg_content_type,
                    arg_value,
                );
            }
        }
    }
    fn content_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
    ) -> ContentSettingValues {
        unsafe {
            self.0
                .get_content_setting
                .map(|f| {
                    let (arg_requesting_url, arg_top_level_url, arg_content_type) =
                        (requesting_url, top_level_url, content_type);
                    let arg_self_ = self.into_raw();
                    let arg_requesting_url = arg_requesting_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_top_level_url = arg_top_level_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_content_type = arg_content_type.into_raw();
                    let result = f(
                        arg_self_,
                        arg_requesting_url,
                        arg_top_level_url,
                        arg_content_type,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_content_setting(
        &self,
        requesting_url: Option<&CefString>,
        top_level_url: Option<&CefString>,
        content_type: ContentSettingTypes,
        value: ContentSettingValues,
    ) {
        unsafe {
            if let Some(f) = self.0.set_content_setting {
                let (arg_requesting_url, arg_top_level_url, arg_content_type, arg_value) =
                    (requesting_url, top_level_url, content_type, value);
                let arg_self_ = self.into_raw();
                let arg_requesting_url = arg_requesting_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_top_level_url = arg_top_level_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_content_type = arg_content_type.into_raw();
                let arg_value = arg_value.into_raw();
                f(
                    arg_self_,
                    arg_requesting_url,
                    arg_top_level_url,
                    arg_content_type,
                    arg_value,
                );
            }
        }
    }
    fn set_chrome_color_scheme(&self, variant: ColorVariant, user_color: u32) {
        unsafe {
            if let Some(f) = self.0.set_chrome_color_scheme {
                let (arg_variant, arg_user_color) = (variant, user_color);
                let arg_self_ = self.into_raw();
                let arg_variant = arg_variant.into_raw();
                f(arg_self_, arg_variant, arg_user_color);
            }
        }
    }
    fn chrome_color_scheme_mode(&self) -> ColorVariant {
        unsafe {
            self.0
                .get_chrome_color_scheme_mode
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn chrome_color_scheme_color(&self) -> cef_color_t {
        unsafe {
            self.0
                .get_chrome_color_scheme_color
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn chrome_color_scheme_variant(&self) -> ColorVariant {
        unsafe {
            self.0
                .get_chrome_color_scheme_variant
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_setting_observer(
        &self,
        observer: Option<&mut impl ImplSettingObserver>,
    ) -> Option<Registration> {
        unsafe {
            self.0
                .add_setting_observer
                .map(|f| {
                    let arg_observer = observer;
                    let arg_self_ = self.into_raw();
                    let arg_observer = arg_observer
                        .map(|arg| {
                            arg.add_ref();
                            ImplSettingObserver::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_observer);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_request_context_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_request_context_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for RequestContext {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_request_context_t> for &RequestContext {
    fn into_raw(self) -> *mut _cef_request_context_t {
        ImplRequestContext::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_request_context_t> for &mut RequestContext {
    fn into_raw(self) -> *mut _cef_request_context_t {
        ImplRequestContext::get_raw(self)
    }
}
impl ConvertReturnValue<RequestContext> for *mut _cef_request_context_t {
    fn wrap_result(self) -> RequestContext {
        RequestContext(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<RequestContext> for *mut _cef_request_context_t {
    fn from(value: RequestContext) -> Self {
        let object = ImplRequestContext::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for RequestContext {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_browser_t`] for more documentation.
#[derive(Clone)]
pub struct Browser(RefGuard<_cef_browser_t>);
pub trait ImplBrowser: Clone + Sized + Rc {
    #[doc = "See [`_cef_browser_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::get_host`] for more documentation."]
    fn host(&self) -> Option<BrowserHost>;
    #[doc = "See [`_cef_browser_t::can_go_back`] for more documentation."]
    fn can_go_back(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::go_back`] for more documentation."]
    fn go_back(&self);
    #[doc = "See [`_cef_browser_t::can_go_forward`] for more documentation."]
    fn can_go_forward(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::go_forward`] for more documentation."]
    fn go_forward(&self);
    #[doc = "See [`_cef_browser_t::is_loading`] for more documentation."]
    fn is_loading(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::reload`] for more documentation."]
    fn reload(&self);
    #[doc = "See [`_cef_browser_t::reload_ignore_cache`] for more documentation."]
    fn reload_ignore_cache(&self);
    #[doc = "See [`_cef_browser_t::stop_load`] for more documentation."]
    fn stop_load(&self);
    #[doc = "See [`_cef_browser_t::get_identifier`] for more documentation."]
    fn identifier(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::is_same`] for more documentation."]
    fn is_same(&self, that: Option<&mut Browser>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::is_popup`] for more documentation."]
    fn is_popup(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::has_document`] for more documentation."]
    fn has_document(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_t::get_main_frame`] for more documentation."]
    fn main_frame(&self) -> Option<Frame>;
    #[doc = "See [`_cef_browser_t::get_focused_frame`] for more documentation."]
    fn focused_frame(&self) -> Option<Frame>;
    #[doc = "See [`_cef_browser_t::get_frame_by_identifier`] for more documentation."]
    fn frame_by_identifier(&self, identifier: Option<&CefString>) -> Option<Frame>;
    #[doc = "See [`_cef_browser_t::get_frame_by_name`] for more documentation."]
    fn frame_by_name(&self, name: Option<&CefString>) -> Option<Frame>;
    #[doc = "See [`_cef_browser_t::get_frame_count`] for more documentation."]
    fn frame_count(&self) -> usize;
    #[doc = "See [`_cef_browser_t::get_frame_identifiers`] for more documentation."]
    fn frame_identifiers(&self, identifiers: Option<&mut CefStringList>);
    #[doc = "See [`_cef_browser_t::get_frame_names`] for more documentation."]
    fn frame_names(&self, names: Option<&mut CefStringList>);
    fn get_raw(&self) -> *mut _cef_browser_t;
}
impl ImplBrowser for Browser {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn host(&self) -> Option<BrowserHost> {
        unsafe {
            self.0
                .get_host
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn can_go_back(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_go_back
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn go_back(&self) {
        unsafe {
            if let Some(f) = self.0.go_back {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn can_go_forward(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_go_forward
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn go_forward(&self) {
        unsafe {
            if let Some(f) = self.0.go_forward {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn is_loading(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_loading
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn reload(&self) {
        unsafe {
            if let Some(f) = self.0.reload {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn reload_ignore_cache(&self) {
        unsafe {
            if let Some(f) = self.0.reload_ignore_cache {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn stop_load(&self) {
        unsafe {
            if let Some(f) = self.0.stop_load {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn identifier(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_identifier
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_same(&self, that: Option<&mut Browser>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_same
                .map(|f| {
                    let arg_that = that;
                    let arg_self_ = self.into_raw();
                    let arg_that = arg_that
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_that);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_popup(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_popup
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_document(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_document
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn main_frame(&self) -> Option<Frame> {
        unsafe {
            self.0
                .get_main_frame
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn focused_frame(&self) -> Option<Frame> {
        unsafe {
            self.0
                .get_focused_frame
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn frame_by_identifier(&self, identifier: Option<&CefString>) -> Option<Frame> {
        unsafe {
            self.0
                .get_frame_by_identifier
                .map(|f| {
                    let arg_identifier = identifier;
                    let arg_self_ = self.into_raw();
                    let arg_identifier = arg_identifier
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_identifier);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn frame_by_name(&self, name: Option<&CefString>) -> Option<Frame> {
        unsafe {
            self.0
                .get_frame_by_name
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn frame_count(&self) -> usize {
        unsafe {
            self.0
                .get_frame_count
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn frame_identifiers(&self, identifiers: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.get_frame_identifiers {
                let arg_identifiers = identifiers;
                let arg_self_ = self.into_raw();
                let arg_identifiers = arg_identifiers
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_identifiers);
            }
        }
    }
    fn frame_names(&self, names: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.get_frame_names {
                let arg_names = names;
                let arg_self_ = self.into_raw();
                let arg_names = arg_names
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_names);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_browser_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_browser_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Browser {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_browser_t> for &Browser {
    fn into_raw(self) -> *mut _cef_browser_t {
        ImplBrowser::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_browser_t> for &mut Browser {
    fn into_raw(self) -> *mut _cef_browser_t {
        ImplBrowser::get_raw(self)
    }
}
impl ConvertReturnValue<Browser> for *mut _cef_browser_t {
    fn wrap_result(self) -> Browser {
        Browser(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Browser> for *mut _cef_browser_t {
    fn from(value: Browser) -> Self {
        let object = ImplBrowser::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Browser {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_run_file_dialog_callback_t`] for more documentation.
#[derive(Clone)]
pub struct RunFileDialogCallback(RefGuard<_cef_run_file_dialog_callback_t>);
impl RunFileDialogCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapRunFileDialogCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplRunFileDialogCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapRunFileDialogCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_run_file_dialog_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapRunFileDialogCallback: ImplRunFileDialogCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_run_file_dialog_callback_t, Self>);
}
pub trait ImplRunFileDialogCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_run_file_dialog_callback_t::on_file_dialog_dismissed`] for more documentation."]
    fn on_file_dialog_dismissed(&self, file_paths: Option<&mut CefStringList>) {}
    fn init_methods(object: &mut _cef_run_file_dialog_callback_t) {
        impl_cef_run_file_dialog_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_run_file_dialog_callback_t;
}
mod impl_cef_run_file_dialog_callback_t {
    use super::*;
    pub fn init_methods<I: ImplRunFileDialogCallback>(
        object: &mut _cef_run_file_dialog_callback_t,
    ) {
        object.on_file_dialog_dismissed = Some(on_file_dialog_dismissed::<I>);
    }
    extern "C" fn on_file_dialog_dismissed<I: ImplRunFileDialogCallback>(
        self_: *mut _cef_run_file_dialog_callback_t,
        file_paths: *mut _cef_string_list_t,
    ) {
        let (arg_self_, arg_file_paths) = (self_, file_paths);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_file_paths = if arg_file_paths.is_null() {
            None
        } else {
            Some(arg_file_paths.into())
        };
        let arg_file_paths = arg_file_paths.as_mut();
        ImplRunFileDialogCallback::on_file_dialog_dismissed(&arg_self_.interface, arg_file_paths)
    }
}
impl ImplRunFileDialogCallback for RunFileDialogCallback {
    fn on_file_dialog_dismissed(&self, file_paths: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.on_file_dialog_dismissed {
                let arg_file_paths = file_paths;
                let arg_self_ = self.into_raw();
                let arg_file_paths = arg_file_paths
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_file_paths);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_run_file_dialog_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_run_file_dialog_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for RunFileDialogCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_run_file_dialog_callback_t> for &RunFileDialogCallback {
    fn into_raw(self) -> *mut _cef_run_file_dialog_callback_t {
        ImplRunFileDialogCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_run_file_dialog_callback_t> for &mut RunFileDialogCallback {
    fn into_raw(self) -> *mut _cef_run_file_dialog_callback_t {
        ImplRunFileDialogCallback::get_raw(self)
    }
}
impl ConvertReturnValue<RunFileDialogCallback> for *mut _cef_run_file_dialog_callback_t {
    fn wrap_result(self) -> RunFileDialogCallback {
        RunFileDialogCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<RunFileDialogCallback> for *mut _cef_run_file_dialog_callback_t {
    fn from(value: RunFileDialogCallback) -> Self {
        let object = ImplRunFileDialogCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for RunFileDialogCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_navigation_entry_visitor_t`] for more documentation.
#[derive(Clone)]
pub struct NavigationEntryVisitor(RefGuard<_cef_navigation_entry_visitor_t>);
impl NavigationEntryVisitor {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapNavigationEntryVisitor,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplNavigationEntryVisitor>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapNavigationEntryVisitor>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_navigation_entry_visitor_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapNavigationEntryVisitor: ImplNavigationEntryVisitor {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_navigation_entry_visitor_t, Self>);
}
pub trait ImplNavigationEntryVisitor: Clone + Sized + Rc {
    #[doc = "See [`_cef_navigation_entry_visitor_t::visit`] for more documentation."]
    fn visit(
        &self,
        entry: Option<&mut NavigationEntry>,
        current: ::std::os::raw::c_int,
        index: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_navigation_entry_visitor_t) {
        impl_cef_navigation_entry_visitor_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_navigation_entry_visitor_t;
}
mod impl_cef_navigation_entry_visitor_t {
    use super::*;
    pub fn init_methods<I: ImplNavigationEntryVisitor>(
        object: &mut _cef_navigation_entry_visitor_t,
    ) {
        object.visit = Some(visit::<I>);
    }
    extern "C" fn visit<I: ImplNavigationEntryVisitor>(
        self_: *mut _cef_navigation_entry_visitor_t,
        entry: *mut _cef_navigation_entry_t,
        current: ::std::os::raw::c_int,
        index: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_entry, arg_current, arg_index, arg_total) =
            (self_, entry, current, index, total);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_entry = unsafe { arg_entry.as_mut() }
            .map(|arg| NavigationEntry(unsafe { RefGuard::from_raw(arg) }));
        let arg_entry = arg_entry.as_mut();
        let arg_current = arg_current.into_raw();
        let arg_index = arg_index.into_raw();
        let arg_total = arg_total.into_raw();
        ImplNavigationEntryVisitor::visit(
            &arg_self_.interface,
            arg_entry,
            arg_current,
            arg_index,
            arg_total,
        )
    }
}
impl ImplNavigationEntryVisitor for NavigationEntryVisitor {
    fn visit(
        &self,
        entry: Option<&mut NavigationEntry>,
        current: ::std::os::raw::c_int,
        index: ::std::os::raw::c_int,
        total: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .visit
                .map(|f| {
                    let (arg_entry, arg_current, arg_index, arg_total) =
                        (entry, current, index, total);
                    let arg_self_ = self.into_raw();
                    let arg_entry = arg_entry
                        .map(|arg| {
                            arg.add_ref();
                            ImplNavigationEntry::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_entry, arg_current, arg_index, arg_total);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_navigation_entry_visitor_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_navigation_entry_visitor_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for NavigationEntryVisitor {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_navigation_entry_visitor_t> for &NavigationEntryVisitor {
    fn into_raw(self) -> *mut _cef_navigation_entry_visitor_t {
        ImplNavigationEntryVisitor::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_navigation_entry_visitor_t> for &mut NavigationEntryVisitor {
    fn into_raw(self) -> *mut _cef_navigation_entry_visitor_t {
        ImplNavigationEntryVisitor::get_raw(self)
    }
}
impl ConvertReturnValue<NavigationEntryVisitor> for *mut _cef_navigation_entry_visitor_t {
    fn wrap_result(self) -> NavigationEntryVisitor {
        NavigationEntryVisitor(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<NavigationEntryVisitor> for *mut _cef_navigation_entry_visitor_t {
    fn from(value: NavigationEntryVisitor) -> Self {
        let object = ImplNavigationEntryVisitor::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for NavigationEntryVisitor {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_pdf_print_callback_t`] for more documentation.
#[derive(Clone)]
pub struct PdfPrintCallback(RefGuard<_cef_pdf_print_callback_t>);
impl PdfPrintCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapPdfPrintCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplPdfPrintCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapPdfPrintCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_pdf_print_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapPdfPrintCallback: ImplPdfPrintCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_pdf_print_callback_t, Self>);
}
pub trait ImplPdfPrintCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_pdf_print_callback_t::on_pdf_print_finished`] for more documentation."]
    fn on_pdf_print_finished(&self, path: Option<&CefString>, ok: ::std::os::raw::c_int) {}
    fn init_methods(object: &mut _cef_pdf_print_callback_t) {
        impl_cef_pdf_print_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_pdf_print_callback_t;
}
mod impl_cef_pdf_print_callback_t {
    use super::*;
    pub fn init_methods<I: ImplPdfPrintCallback>(object: &mut _cef_pdf_print_callback_t) {
        object.on_pdf_print_finished = Some(on_pdf_print_finished::<I>);
    }
    extern "C" fn on_pdf_print_finished<I: ImplPdfPrintCallback>(
        self_: *mut _cef_pdf_print_callback_t,
        path: *const cef_string_t,
        ok: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_path, arg_ok) = (self_, path, ok);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_path = if arg_path.is_null() {
            None
        } else {
            Some(arg_path.into())
        };
        let arg_path = arg_path.as_ref();
        let arg_ok = arg_ok.into_raw();
        ImplPdfPrintCallback::on_pdf_print_finished(&arg_self_.interface, arg_path, arg_ok)
    }
}
impl ImplPdfPrintCallback for PdfPrintCallback {
    fn on_pdf_print_finished(&self, path: Option<&CefString>, ok: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.on_pdf_print_finished {
                let (arg_path, arg_ok) = (path, ok);
                let arg_self_ = self.into_raw();
                let arg_path = arg_path
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_path, arg_ok);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_pdf_print_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_pdf_print_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PdfPrintCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_pdf_print_callback_t> for &PdfPrintCallback {
    fn into_raw(self) -> *mut _cef_pdf_print_callback_t {
        ImplPdfPrintCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_pdf_print_callback_t> for &mut PdfPrintCallback {
    fn into_raw(self) -> *mut _cef_pdf_print_callback_t {
        ImplPdfPrintCallback::get_raw(self)
    }
}
impl ConvertReturnValue<PdfPrintCallback> for *mut _cef_pdf_print_callback_t {
    fn wrap_result(self) -> PdfPrintCallback {
        PdfPrintCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PdfPrintCallback> for *mut _cef_pdf_print_callback_t {
    fn from(value: PdfPrintCallback) -> Self {
        let object = ImplPdfPrintCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PdfPrintCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_download_image_callback_t`] for more documentation.
#[derive(Clone)]
pub struct DownloadImageCallback(RefGuard<_cef_download_image_callback_t>);
impl DownloadImageCallback {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDownloadImageCallback,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDownloadImageCallback>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDownloadImageCallback>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_download_image_callback_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDownloadImageCallback: ImplDownloadImageCallback {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_download_image_callback_t, Self>);
}
pub trait ImplDownloadImageCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_download_image_callback_t::on_download_image_finished`] for more documentation."]
    fn on_download_image_finished(
        &self,
        image_url: Option<&CefString>,
        http_status_code: ::std::os::raw::c_int,
        image: Option<&mut Image>,
    ) {
    }
    fn init_methods(object: &mut _cef_download_image_callback_t) {
        impl_cef_download_image_callback_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_download_image_callback_t;
}
mod impl_cef_download_image_callback_t {
    use super::*;
    pub fn init_methods<I: ImplDownloadImageCallback>(object: &mut _cef_download_image_callback_t) {
        object.on_download_image_finished = Some(on_download_image_finished::<I>);
    }
    extern "C" fn on_download_image_finished<I: ImplDownloadImageCallback>(
        self_: *mut _cef_download_image_callback_t,
        image_url: *const cef_string_t,
        http_status_code: ::std::os::raw::c_int,
        image: *mut _cef_image_t,
    ) {
        let (arg_self_, arg_image_url, arg_http_status_code, arg_image) =
            (self_, image_url, http_status_code, image);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_image_url = if arg_image_url.is_null() {
            None
        } else {
            Some(arg_image_url.into())
        };
        let arg_image_url = arg_image_url.as_ref();
        let arg_http_status_code = arg_http_status_code.into_raw();
        let mut arg_image =
            unsafe { arg_image.as_mut() }.map(|arg| Image(unsafe { RefGuard::from_raw(arg) }));
        let arg_image = arg_image.as_mut();
        ImplDownloadImageCallback::on_download_image_finished(
            &arg_self_.interface,
            arg_image_url,
            arg_http_status_code,
            arg_image,
        )
    }
}
impl ImplDownloadImageCallback for DownloadImageCallback {
    fn on_download_image_finished(
        &self,
        image_url: Option<&CefString>,
        http_status_code: ::std::os::raw::c_int,
        image: Option<&mut Image>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_download_image_finished {
                let (arg_image_url, arg_http_status_code, arg_image) =
                    (image_url, http_status_code, image);
                let arg_self_ = self.into_raw();
                let arg_image_url = arg_image_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_image = arg_image
                    .map(|arg| {
                        arg.add_ref();
                        ImplImage::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_image_url, arg_http_status_code, arg_image);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_download_image_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_download_image_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DownloadImageCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_download_image_callback_t> for &DownloadImageCallback {
    fn into_raw(self) -> *mut _cef_download_image_callback_t {
        ImplDownloadImageCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_download_image_callback_t> for &mut DownloadImageCallback {
    fn into_raw(self) -> *mut _cef_download_image_callback_t {
        ImplDownloadImageCallback::get_raw(self)
    }
}
impl ConvertReturnValue<DownloadImageCallback> for *mut _cef_download_image_callback_t {
    fn wrap_result(self) -> DownloadImageCallback {
        DownloadImageCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DownloadImageCallback> for *mut _cef_download_image_callback_t {
    fn from(value: DownloadImageCallback) -> Self {
        let object = ImplDownloadImageCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DownloadImageCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_browser_host_t`] for more documentation.
#[derive(Clone)]
pub struct BrowserHost(RefGuard<_cef_browser_host_t>);
pub trait ImplBrowserHost: Clone + Sized + Rc {
    #[doc = "See [`_cef_browser_host_t::get_browser`] for more documentation."]
    fn browser(&self) -> Option<Browser>;
    #[doc = "See [`_cef_browser_host_t::close_browser`] for more documentation."]
    fn close_browser(&self, force_close: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::try_close_browser`] for more documentation."]
    fn try_close_browser(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::is_ready_to_be_closed`] for more documentation."]
    fn is_ready_to_be_closed(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::set_focus`] for more documentation."]
    fn set_focus(&self, focus: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::get_window_handle`] for more documentation."]
    fn window_handle(&self) -> cef_window_handle_t;
    #[doc = "See [`_cef_browser_host_t::get_opener_window_handle`] for more documentation."]
    fn opener_window_handle(&self) -> cef_window_handle_t;
    #[doc = "See [`_cef_browser_host_t::get_opener_identifier`] for more documentation."]
    fn opener_identifier(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::has_view`] for more documentation."]
    fn has_view(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::get_client`] for more documentation."]
    fn client(&self) -> Option<Client>;
    #[doc = "See [`_cef_browser_host_t::get_request_context`] for more documentation."]
    fn request_context(&self) -> Option<RequestContext>;
    #[doc = "See [`_cef_browser_host_t::can_zoom`] for more documentation."]
    fn can_zoom(&self, command: ZoomCommand) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::zoom`] for more documentation."]
    fn zoom(&self, command: ZoomCommand);
    #[doc = "See [`_cef_browser_host_t::get_default_zoom_level`] for more documentation."]
    fn default_zoom_level(&self) -> f64;
    #[doc = "See [`_cef_browser_host_t::get_zoom_level`] for more documentation."]
    fn zoom_level(&self) -> f64;
    #[doc = "See [`_cef_browser_host_t::set_zoom_level`] for more documentation."]
    fn set_zoom_level(&self, zoom_level: f64);
    #[doc = "See [`_cef_browser_host_t::run_file_dialog`] for more documentation."]
    fn run_file_dialog(
        &self,
        mode: FileDialogMode,
        title: Option<&CefString>,
        default_file_path: Option<&CefString>,
        accept_filters: Option<&mut CefStringList>,
        callback: Option<&mut impl ImplRunFileDialogCallback>,
    );
    #[doc = "See [`_cef_browser_host_t::start_download`] for more documentation."]
    fn start_download(&self, url: Option<&CefString>);
    #[doc = "See [`_cef_browser_host_t::download_image`] for more documentation."]
    fn download_image(
        &self,
        image_url: Option<&CefString>,
        is_favicon: ::std::os::raw::c_int,
        max_image_size: u32,
        bypass_cache: ::std::os::raw::c_int,
        callback: Option<&mut impl ImplDownloadImageCallback>,
    );
    #[doc = "See [`_cef_browser_host_t::print`] for more documentation."]
    fn print(&self);
    #[doc = "See [`_cef_browser_host_t::print_to_pdf`] for more documentation."]
    fn print_to_pdf(
        &self,
        path: Option<&CefString>,
        settings: Option<&PdfPrintSettings>,
        callback: Option<&mut impl ImplPdfPrintCallback>,
    );
    #[doc = "See [`_cef_browser_host_t::find`] for more documentation."]
    fn find(
        &self,
        search_text: Option<&CefString>,
        forward: ::std::os::raw::c_int,
        match_case: ::std::os::raw::c_int,
        find_next: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_browser_host_t::stop_finding`] for more documentation."]
    fn stop_finding(&self, clear_selection: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::show_dev_tools`] for more documentation."]
    fn show_dev_tools(
        &self,
        window_info: Option<&WindowInfo>,
        client: Option<&mut impl ImplClient>,
        settings: Option<&BrowserSettings>,
        inspect_element_at: Option<&Point>,
    );
    #[doc = "See [`_cef_browser_host_t::close_dev_tools`] for more documentation."]
    fn close_dev_tools(&self);
    #[doc = "See [`_cef_browser_host_t::has_dev_tools`] for more documentation."]
    fn has_dev_tools(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::send_dev_tools_message`] for more documentation."]
    fn send_dev_tools_message(&self, message: Option<&[u8]>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::execute_dev_tools_method`] for more documentation."]
    fn execute_dev_tools_method(
        &self,
        message_id: ::std::os::raw::c_int,
        method: Option<&CefString>,
        params: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::add_dev_tools_message_observer`] for more documentation."]
    fn add_dev_tools_message_observer(
        &self,
        observer: Option<&mut impl ImplDevToolsMessageObserver>,
    ) -> Option<Registration>;
    #[doc = "See [`_cef_browser_host_t::get_navigation_entries`] for more documentation."]
    fn navigation_entries(
        &self,
        visitor: Option<&mut impl ImplNavigationEntryVisitor>,
        current_only: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_browser_host_t::replace_misspelling`] for more documentation."]
    fn replace_misspelling(&self, word: Option<&CefString>);
    #[doc = "See [`_cef_browser_host_t::add_word_to_dictionary`] for more documentation."]
    fn add_word_to_dictionary(&self, word: Option<&CefString>);
    #[doc = "See [`_cef_browser_host_t::is_window_rendering_disabled`] for more documentation."]
    fn is_window_rendering_disabled(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::was_resized`] for more documentation."]
    fn was_resized(&self);
    #[doc = "See [`_cef_browser_host_t::was_hidden`] for more documentation."]
    fn was_hidden(&self, hidden: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::notify_screen_info_changed`] for more documentation."]
    fn notify_screen_info_changed(&self);
    #[doc = "See [`_cef_browser_host_t::invalidate`] for more documentation."]
    fn invalidate(&self, type_: PaintElementType);
    #[doc = "See [`_cef_browser_host_t::send_external_begin_frame`] for more documentation."]
    fn send_external_begin_frame(&self);
    #[doc = "See [`_cef_browser_host_t::send_key_event`] for more documentation."]
    fn send_key_event(&self, event: Option<&KeyEvent>);
    #[doc = "See [`_cef_browser_host_t::send_mouse_click_event`] for more documentation."]
    fn send_mouse_click_event(
        &self,
        event: Option<&MouseEvent>,
        type_: MouseButtonType,
        mouse_up: ::std::os::raw::c_int,
        click_count: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_browser_host_t::send_mouse_move_event`] for more documentation."]
    fn send_mouse_move_event(&self, event: Option<&MouseEvent>, mouse_leave: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::send_mouse_wheel_event`] for more documentation."]
    fn send_mouse_wheel_event(
        &self,
        event: Option<&MouseEvent>,
        delta_x: ::std::os::raw::c_int,
        delta_y: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_browser_host_t::send_touch_event`] for more documentation."]
    fn send_touch_event(&self, event: Option<&TouchEvent>);
    #[doc = "See [`_cef_browser_host_t::send_capture_lost_event`] for more documentation."]
    fn send_capture_lost_event(&self);
    #[doc = "See [`_cef_browser_host_t::notify_move_or_resize_started`] for more documentation."]
    fn notify_move_or_resize_started(&self);
    #[doc = "See [`_cef_browser_host_t::get_windowless_frame_rate`] for more documentation."]
    fn windowless_frame_rate(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::set_windowless_frame_rate`] for more documentation."]
    fn set_windowless_frame_rate(&self, frame_rate: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::ime_set_composition`] for more documentation."]
    fn ime_set_composition(
        &self,
        text: Option<&CefString>,
        underlines_count: usize,
        underlines: Option<&CompositionUnderline>,
        replacement_range: Option<&Range>,
        selection_range: Option<&Range>,
    );
    #[doc = "See [`_cef_browser_host_t::ime_commit_text`] for more documentation."]
    fn ime_commit_text(
        &self,
        text: Option<&CefString>,
        replacement_range: Option<&Range>,
        relative_cursor_pos: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_browser_host_t::ime_finish_composing_text`] for more documentation."]
    fn ime_finish_composing_text(&self, keep_selection: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::ime_cancel_composition`] for more documentation."]
    fn ime_cancel_composition(&self);
    #[doc = "See [`_cef_browser_host_t::drag_target_drag_enter`] for more documentation."]
    fn drag_target_drag_enter(
        &self,
        drag_data: Option<&mut DragData>,
        event: Option<&MouseEvent>,
        allowed_ops: DragOperationsMask,
    );
    #[doc = "See [`_cef_browser_host_t::drag_target_drag_over`] for more documentation."]
    fn drag_target_drag_over(&self, event: Option<&MouseEvent>, allowed_ops: DragOperationsMask);
    #[doc = "See [`_cef_browser_host_t::drag_target_drag_leave`] for more documentation."]
    fn drag_target_drag_leave(&self);
    #[doc = "See [`_cef_browser_host_t::drag_target_drop`] for more documentation."]
    fn drag_target_drop(&self, event: Option<&MouseEvent>);
    #[doc = "See [`_cef_browser_host_t::drag_source_ended_at`] for more documentation."]
    fn drag_source_ended_at(
        &self,
        x: ::std::os::raw::c_int,
        y: ::std::os::raw::c_int,
        op: DragOperationsMask,
    );
    #[doc = "See [`_cef_browser_host_t::drag_source_system_drag_ended`] for more documentation."]
    fn drag_source_system_drag_ended(&self);
    #[doc = "See [`_cef_browser_host_t::get_visible_navigation_entry`] for more documentation."]
    fn visible_navigation_entry(&self) -> Option<NavigationEntry>;
    #[doc = "See [`_cef_browser_host_t::set_accessibility_state`] for more documentation."]
    fn set_accessibility_state(&self, accessibility_state: State);
    #[doc = "See [`_cef_browser_host_t::set_auto_resize_enabled`] for more documentation."]
    fn set_auto_resize_enabled(
        &self,
        enabled: ::std::os::raw::c_int,
        min_size: Option<&Size>,
        max_size: Option<&Size>,
    );
    #[doc = "See [`_cef_browser_host_t::set_audio_muted`] for more documentation."]
    fn set_audio_muted(&self, mute: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::is_audio_muted`] for more documentation."]
    fn is_audio_muted(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::is_fullscreen`] for more documentation."]
    fn is_fullscreen(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::exit_fullscreen`] for more documentation."]
    fn exit_fullscreen(&self, will_cause_resize: ::std::os::raw::c_int);
    #[doc = "See [`_cef_browser_host_t::can_execute_chrome_command`] for more documentation."]
    fn can_execute_chrome_command(
        &self,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::execute_chrome_command`] for more documentation."]
    fn execute_chrome_command(
        &self,
        command_id: ::std::os::raw::c_int,
        disposition: WindowOpenDisposition,
    );
    #[doc = "See [`_cef_browser_host_t::is_render_process_unresponsive`] for more documentation."]
    fn is_render_process_unresponsive(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_browser_host_t::get_runtime_style`] for more documentation."]
    fn runtime_style(&self) -> RuntimeStyle;
    fn get_raw(&self) -> *mut _cef_browser_host_t;
}
impl ImplBrowserHost for BrowserHost {
    fn browser(&self) -> Option<Browser> {
        unsafe {
            self.0
                .get_browser
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn close_browser(&self, force_close: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.close_browser {
                let arg_force_close = force_close;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_force_close);
            }
        }
    }
    fn try_close_browser(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .try_close_browser
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_ready_to_be_closed(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_ready_to_be_closed
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_focus(&self, focus: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_focus {
                let arg_focus = focus;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_focus);
            }
        }
    }
    fn window_handle(&self) -> cef_window_handle_t {
        unsafe {
            self.0
                .get_window_handle
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn opener_window_handle(&self) -> cef_window_handle_t {
        unsafe {
            self.0
                .get_opener_window_handle
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn opener_identifier(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_opener_identifier
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_view(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_view
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn client(&self) -> Option<Client> {
        unsafe {
            self.0
                .get_client
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn request_context(&self) -> Option<RequestContext> {
        unsafe {
            self.0
                .get_request_context
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn can_zoom(&self, command: ZoomCommand) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_zoom
                .map(|f| {
                    let arg_command = command;
                    let arg_self_ = self.into_raw();
                    let arg_command = arg_command.into_raw();
                    let result = f(arg_self_, arg_command);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn zoom(&self, command: ZoomCommand) {
        unsafe {
            if let Some(f) = self.0.zoom {
                let arg_command = command;
                let arg_self_ = self.into_raw();
                let arg_command = arg_command.into_raw();
                f(arg_self_, arg_command);
            }
        }
    }
    fn default_zoom_level(&self) -> f64 {
        unsafe {
            self.0
                .get_default_zoom_level
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn zoom_level(&self) -> f64 {
        unsafe {
            self.0
                .get_zoom_level
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_zoom_level(&self, zoom_level: f64) {
        unsafe {
            if let Some(f) = self.0.set_zoom_level {
                let arg_zoom_level = zoom_level;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_zoom_level);
            }
        }
    }
    fn run_file_dialog(
        &self,
        mode: FileDialogMode,
        title: Option<&CefString>,
        default_file_path: Option<&CefString>,
        accept_filters: Option<&mut CefStringList>,
        callback: Option<&mut impl ImplRunFileDialogCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.run_file_dialog {
                let (arg_mode, arg_title, arg_default_file_path, arg_accept_filters, arg_callback) =
                    (mode, title, default_file_path, accept_filters, callback);
                let arg_self_ = self.into_raw();
                let arg_mode = arg_mode.into_raw();
                let arg_title = arg_title
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_default_file_path = arg_default_file_path
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_accept_filters = arg_accept_filters
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplRunFileDialogCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_mode,
                    arg_title,
                    arg_default_file_path,
                    arg_accept_filters,
                    arg_callback,
                );
            }
        }
    }
    fn start_download(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.start_download {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn download_image(
        &self,
        image_url: Option<&CefString>,
        is_favicon: ::std::os::raw::c_int,
        max_image_size: u32,
        bypass_cache: ::std::os::raw::c_int,
        callback: Option<&mut impl ImplDownloadImageCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.download_image {
                let (
                    arg_image_url,
                    arg_is_favicon,
                    arg_max_image_size,
                    arg_bypass_cache,
                    arg_callback,
                ) = (
                    image_url,
                    is_favicon,
                    max_image_size,
                    bypass_cache,
                    callback,
                );
                let arg_self_ = self.into_raw();
                let arg_image_url = arg_image_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplDownloadImageCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_image_url,
                    arg_is_favicon,
                    arg_max_image_size,
                    arg_bypass_cache,
                    arg_callback,
                );
            }
        }
    }
    fn print(&self) {
        unsafe {
            if let Some(f) = self.0.print {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn print_to_pdf(
        &self,
        path: Option<&CefString>,
        settings: Option<&PdfPrintSettings>,
        callback: Option<&mut impl ImplPdfPrintCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.print_to_pdf {
                let (arg_path, arg_settings, arg_callback) = (path, settings, callback);
                let arg_self_ = self.into_raw();
                let arg_path = arg_path
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_settings = arg_settings.cloned().map(|arg| arg.into());
                let arg_settings = arg_settings
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplPdfPrintCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_path, arg_settings, arg_callback);
            }
        }
    }
    fn find(
        &self,
        search_text: Option<&CefString>,
        forward: ::std::os::raw::c_int,
        match_case: ::std::os::raw::c_int,
        find_next: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.find {
                let (arg_search_text, arg_forward, arg_match_case, arg_find_next) =
                    (search_text, forward, match_case, find_next);
                let arg_self_ = self.into_raw();
                let arg_search_text = arg_search_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_search_text,
                    arg_forward,
                    arg_match_case,
                    arg_find_next,
                );
            }
        }
    }
    fn stop_finding(&self, clear_selection: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.stop_finding {
                let arg_clear_selection = clear_selection;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_clear_selection);
            }
        }
    }
    fn show_dev_tools(
        &self,
        window_info: Option<&WindowInfo>,
        client: Option<&mut impl ImplClient>,
        settings: Option<&BrowserSettings>,
        inspect_element_at: Option<&Point>,
    ) {
        unsafe {
            if let Some(f) = self.0.show_dev_tools {
                let (arg_window_info, arg_client, arg_settings, arg_inspect_element_at) =
                    (window_info, client, settings, inspect_element_at);
                let arg_self_ = self.into_raw();
                let arg_window_info = arg_window_info.cloned().map(|arg| arg.into());
                let arg_window_info = arg_window_info
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_client = arg_client
                    .map(|arg| {
                        arg.add_ref();
                        ImplClient::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_settings = arg_settings.cloned().map(|arg| arg.into());
                let arg_settings = arg_settings
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_inspect_element_at = arg_inspect_element_at.cloned().map(|arg| arg.into());
                let arg_inspect_element_at = arg_inspect_element_at
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_window_info,
                    arg_client,
                    arg_settings,
                    arg_inspect_element_at,
                );
            }
        }
    }
    fn close_dev_tools(&self) {
        unsafe {
            if let Some(f) = self.0.close_dev_tools {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn has_dev_tools(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_dev_tools
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn send_dev_tools_message(&self, message: Option<&[u8]>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .send_dev_tools_message
                .map(|f| {
                    let arg_message = message;
                    let arg_self_ = self.into_raw();
                    let arg_message_size = arg_message
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let arg_message = arg_message
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_message, arg_message_size);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn execute_dev_tools_method(
        &self,
        message_id: ::std::os::raw::c_int,
        method: Option<&CefString>,
        params: Option<&mut DictionaryValue>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .execute_dev_tools_method
                .map(|f| {
                    let (arg_message_id, arg_method, arg_params) = (message_id, method, params);
                    let arg_self_ = self.into_raw();
                    let arg_method = arg_method
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_params = arg_params
                        .map(|arg| {
                            arg.add_ref();
                            ImplDictionaryValue::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_message_id, arg_method, arg_params);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_dev_tools_message_observer(
        &self,
        observer: Option<&mut impl ImplDevToolsMessageObserver>,
    ) -> Option<Registration> {
        unsafe {
            self.0
                .add_dev_tools_message_observer
                .map(|f| {
                    let arg_observer = observer;
                    let arg_self_ = self.into_raw();
                    let arg_observer = arg_observer
                        .map(|arg| {
                            arg.add_ref();
                            ImplDevToolsMessageObserver::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_observer);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn navigation_entries(
        &self,
        visitor: Option<&mut impl ImplNavigationEntryVisitor>,
        current_only: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.get_navigation_entries {
                let (arg_visitor, arg_current_only) = (visitor, current_only);
                let arg_self_ = self.into_raw();
                let arg_visitor = arg_visitor
                    .map(|arg| {
                        arg.add_ref();
                        ImplNavigationEntryVisitor::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_visitor, arg_current_only);
            }
        }
    }
    fn replace_misspelling(&self, word: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.replace_misspelling {
                let arg_word = word;
                let arg_self_ = self.into_raw();
                let arg_word = arg_word
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_word);
            }
        }
    }
    fn add_word_to_dictionary(&self, word: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.add_word_to_dictionary {
                let arg_word = word;
                let arg_self_ = self.into_raw();
                let arg_word = arg_word
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_word);
            }
        }
    }
    fn is_window_rendering_disabled(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_window_rendering_disabled
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn was_resized(&self) {
        unsafe {
            if let Some(f) = self.0.was_resized {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn was_hidden(&self, hidden: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.was_hidden {
                let arg_hidden = hidden;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_hidden);
            }
        }
    }
    fn notify_screen_info_changed(&self) {
        unsafe {
            if let Some(f) = self.0.notify_screen_info_changed {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn invalidate(&self, type_: PaintElementType) {
        unsafe {
            if let Some(f) = self.0.invalidate {
                let arg_type_ = type_;
                let arg_self_ = self.into_raw();
                let arg_type_ = arg_type_.into_raw();
                f(arg_self_, arg_type_);
            }
        }
    }
    fn send_external_begin_frame(&self) {
        unsafe {
            if let Some(f) = self.0.send_external_begin_frame {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn send_key_event(&self, event: Option<&KeyEvent>) {
        unsafe {
            if let Some(f) = self.0.send_key_event {
                let arg_event = event;
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_event);
            }
        }
    }
    fn send_mouse_click_event(
        &self,
        event: Option<&MouseEvent>,
        type_: MouseButtonType,
        mouse_up: ::std::os::raw::c_int,
        click_count: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.send_mouse_click_event {
                let (arg_event, arg_type_, arg_mouse_up, arg_click_count) =
                    (event, type_, mouse_up, click_count);
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_type_ = arg_type_.into_raw();
                f(
                    arg_self_,
                    arg_event,
                    arg_type_,
                    arg_mouse_up,
                    arg_click_count,
                );
            }
        }
    }
    fn send_mouse_move_event(
        &self,
        event: Option<&MouseEvent>,
        mouse_leave: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.send_mouse_move_event {
                let (arg_event, arg_mouse_leave) = (event, mouse_leave);
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_event, arg_mouse_leave);
            }
        }
    }
    fn send_mouse_wheel_event(
        &self,
        event: Option<&MouseEvent>,
        delta_x: ::std::os::raw::c_int,
        delta_y: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.send_mouse_wheel_event {
                let (arg_event, arg_delta_x, arg_delta_y) = (event, delta_x, delta_y);
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_event, arg_delta_x, arg_delta_y);
            }
        }
    }
    fn send_touch_event(&self, event: Option<&TouchEvent>) {
        unsafe {
            if let Some(f) = self.0.send_touch_event {
                let arg_event = event;
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_event);
            }
        }
    }
    fn send_capture_lost_event(&self) {
        unsafe {
            if let Some(f) = self.0.send_capture_lost_event {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn notify_move_or_resize_started(&self) {
        unsafe {
            if let Some(f) = self.0.notify_move_or_resize_started {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn windowless_frame_rate(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_windowless_frame_rate
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_windowless_frame_rate(&self, frame_rate: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_windowless_frame_rate {
                let arg_frame_rate = frame_rate;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_frame_rate);
            }
        }
    }
    fn ime_set_composition(
        &self,
        text: Option<&CefString>,
        underlines_count: usize,
        underlines: Option<&CompositionUnderline>,
        replacement_range: Option<&Range>,
        selection_range: Option<&Range>,
    ) {
        unsafe {
            if let Some(f) = self.0.ime_set_composition {
                let (
                    arg_text,
                    arg_underlines_count,
                    arg_underlines,
                    arg_replacement_range,
                    arg_selection_range,
                ) = (
                    text,
                    underlines_count,
                    underlines,
                    replacement_range,
                    selection_range,
                );
                let arg_self_ = self.into_raw();
                let arg_text = arg_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_underlines = arg_underlines.cloned().map(|arg| arg.into());
                let arg_underlines = arg_underlines
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_replacement_range = arg_replacement_range.cloned().map(|arg| arg.into());
                let arg_replacement_range = arg_replacement_range
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_selection_range = arg_selection_range.cloned().map(|arg| arg.into());
                let arg_selection_range = arg_selection_range
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_text,
                    arg_underlines_count,
                    arg_underlines,
                    arg_replacement_range,
                    arg_selection_range,
                );
            }
        }
    }
    fn ime_commit_text(
        &self,
        text: Option<&CefString>,
        replacement_range: Option<&Range>,
        relative_cursor_pos: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.ime_commit_text {
                let (arg_text, arg_replacement_range, arg_relative_cursor_pos) =
                    (text, replacement_range, relative_cursor_pos);
                let arg_self_ = self.into_raw();
                let arg_text = arg_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_replacement_range = arg_replacement_range.cloned().map(|arg| arg.into());
                let arg_replacement_range = arg_replacement_range
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_text,
                    arg_replacement_range,
                    arg_relative_cursor_pos,
                );
            }
        }
    }
    fn ime_finish_composing_text(&self, keep_selection: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.ime_finish_composing_text {
                let arg_keep_selection = keep_selection;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_keep_selection);
            }
        }
    }
    fn ime_cancel_composition(&self) {
        unsafe {
            if let Some(f) = self.0.ime_cancel_composition {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn drag_target_drag_enter(
        &self,
        drag_data: Option<&mut DragData>,
        event: Option<&MouseEvent>,
        allowed_ops: DragOperationsMask,
    ) {
        unsafe {
            if let Some(f) = self.0.drag_target_drag_enter {
                let (arg_drag_data, arg_event, arg_allowed_ops) = (drag_data, event, allowed_ops);
                let arg_self_ = self.into_raw();
                let arg_drag_data = arg_drag_data
                    .map(|arg| {
                        arg.add_ref();
                        ImplDragData::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_allowed_ops = arg_allowed_ops.into_raw();
                f(arg_self_, arg_drag_data, arg_event, arg_allowed_ops);
            }
        }
    }
    fn drag_target_drag_over(&self, event: Option<&MouseEvent>, allowed_ops: DragOperationsMask) {
        unsafe {
            if let Some(f) = self.0.drag_target_drag_over {
                let (arg_event, arg_allowed_ops) = (event, allowed_ops);
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_allowed_ops = arg_allowed_ops.into_raw();
                f(arg_self_, arg_event, arg_allowed_ops);
            }
        }
    }
    fn drag_target_drag_leave(&self) {
        unsafe {
            if let Some(f) = self.0.drag_target_drag_leave {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn drag_target_drop(&self, event: Option<&MouseEvent>) {
        unsafe {
            if let Some(f) = self.0.drag_target_drop {
                let arg_event = event;
                let arg_self_ = self.into_raw();
                let arg_event = arg_event.cloned().map(|arg| arg.into());
                let arg_event = arg_event
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_event);
            }
        }
    }
    fn drag_source_ended_at(
        &self,
        x: ::std::os::raw::c_int,
        y: ::std::os::raw::c_int,
        op: DragOperationsMask,
    ) {
        unsafe {
            if let Some(f) = self.0.drag_source_ended_at {
                let (arg_x, arg_y, arg_op) = (x, y, op);
                let arg_self_ = self.into_raw();
                let arg_op = arg_op.into_raw();
                f(arg_self_, arg_x, arg_y, arg_op);
            }
        }
    }
    fn drag_source_system_drag_ended(&self) {
        unsafe {
            if let Some(f) = self.0.drag_source_system_drag_ended {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn visible_navigation_entry(&self) -> Option<NavigationEntry> {
        unsafe {
            self.0
                .get_visible_navigation_entry
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn set_accessibility_state(&self, accessibility_state: State) {
        unsafe {
            if let Some(f) = self.0.set_accessibility_state {
                let arg_accessibility_state = accessibility_state;
                let arg_self_ = self.into_raw();
                let arg_accessibility_state = arg_accessibility_state.into_raw();
                f(arg_self_, arg_accessibility_state);
            }
        }
    }
    fn set_auto_resize_enabled(
        &self,
        enabled: ::std::os::raw::c_int,
        min_size: Option<&Size>,
        max_size: Option<&Size>,
    ) {
        unsafe {
            if let Some(f) = self.0.set_auto_resize_enabled {
                let (arg_enabled, arg_min_size, arg_max_size) = (enabled, min_size, max_size);
                let arg_self_ = self.into_raw();
                let arg_min_size = arg_min_size.cloned().map(|arg| arg.into());
                let arg_min_size = arg_min_size
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_max_size = arg_max_size.cloned().map(|arg| arg.into());
                let arg_max_size = arg_max_size
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_enabled, arg_min_size, arg_max_size);
            }
        }
    }
    fn set_audio_muted(&self, mute: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_audio_muted {
                let arg_mute = mute;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_mute);
            }
        }
    }
    fn is_audio_muted(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_audio_muted
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_fullscreen(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_fullscreen
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn exit_fullscreen(&self, will_cause_resize: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.exit_fullscreen {
                let arg_will_cause_resize = will_cause_resize;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_will_cause_resize);
            }
        }
    }
    fn can_execute_chrome_command(
        &self,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_execute_chrome_command
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn execute_chrome_command(
        &self,
        command_id: ::std::os::raw::c_int,
        disposition: WindowOpenDisposition,
    ) {
        unsafe {
            if let Some(f) = self.0.execute_chrome_command {
                let (arg_command_id, arg_disposition) = (command_id, disposition);
                let arg_self_ = self.into_raw();
                let arg_disposition = arg_disposition.into_raw();
                f(arg_self_, arg_command_id, arg_disposition);
            }
        }
    }
    fn is_render_process_unresponsive(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_render_process_unresponsive
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn runtime_style(&self) -> RuntimeStyle {
        unsafe {
            self.0
                .get_runtime_style
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_browser_host_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_browser_host_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for BrowserHost {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_browser_host_t> for &BrowserHost {
    fn into_raw(self) -> *mut _cef_browser_host_t {
        ImplBrowserHost::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_browser_host_t> for &mut BrowserHost {
    fn into_raw(self) -> *mut _cef_browser_host_t {
        ImplBrowserHost::get_raw(self)
    }
}
impl ConvertReturnValue<BrowserHost> for *mut _cef_browser_host_t {
    fn wrap_result(self) -> BrowserHost {
        BrowserHost(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<BrowserHost> for *mut _cef_browser_host_t {
    fn from(value: BrowserHost) -> Self {
        let object = ImplBrowserHost::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for BrowserHost {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_audio_handler_t`] for more documentation.
#[derive(Clone)]
pub struct AudioHandler(RefGuard<_cef_audio_handler_t>);
impl AudioHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapAudioHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplAudioHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapAudioHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_audio_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapAudioHandler: ImplAudioHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_audio_handler_t, Self>);
}
pub trait ImplAudioHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_audio_handler_t::get_audio_parameters`] for more documentation."]
    fn audio_parameters(
        &self,
        browser: Option<&mut Browser>,
        params: Option<&mut AudioParameters>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_audio_handler_t::on_audio_stream_started`] for more documentation."]
    fn on_audio_stream_started(
        &self,
        browser: Option<&mut Browser>,
        params: Option<&AudioParameters>,
        channels: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_audio_handler_t::on_audio_stream_packet`] for more documentation."]
    fn on_audio_stream_packet(
        &self,
        browser: Option<&mut Browser>,
        data: *mut *const f32,
        frames: ::std::os::raw::c_int,
        pts: i64,
    ) {
    }
    #[doc = "See [`_cef_audio_handler_t::on_audio_stream_stopped`] for more documentation."]
    fn on_audio_stream_stopped(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_audio_handler_t::on_audio_stream_error`] for more documentation."]
    fn on_audio_stream_error(&self, browser: Option<&mut Browser>, message: Option<&CefString>) {}
    fn init_methods(object: &mut _cef_audio_handler_t) {
        impl_cef_audio_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_audio_handler_t;
}
mod impl_cef_audio_handler_t {
    use super::*;
    pub fn init_methods<I: ImplAudioHandler>(object: &mut _cef_audio_handler_t) {
        object.get_audio_parameters = Some(get_audio_parameters::<I>);
        object.on_audio_stream_started = Some(on_audio_stream_started::<I>);
        object.on_audio_stream_packet = Some(on_audio_stream_packet::<I>);
        object.on_audio_stream_stopped = Some(on_audio_stream_stopped::<I>);
        object.on_audio_stream_error = Some(on_audio_stream_error::<I>);
    }
    extern "C" fn get_audio_parameters<I: ImplAudioHandler>(
        self_: *mut _cef_audio_handler_t,
        browser: *mut _cef_browser_t,
        params: *mut _cef_audio_parameters_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_params) = (self_, browser, params);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_params = if arg_params.is_null() {
            None
        } else {
            Some(WrapParamRef::<AudioParameters, _>::from(arg_params))
        };
        let arg_params = arg_params.as_mut().map(|arg| arg.as_mut());
        ImplAudioHandler::audio_parameters(&arg_self_.interface, arg_browser, arg_params)
    }
    extern "C" fn on_audio_stream_started<I: ImplAudioHandler>(
        self_: *mut _cef_audio_handler_t,
        browser: *mut _cef_browser_t,
        params: *const _cef_audio_parameters_t,
        channels: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_params, arg_channels) = (self_, browser, params, channels);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_params = if arg_params.is_null() {
            None
        } else {
            Some(WrapParamRef::<AudioParameters, _>::from(arg_params))
        };
        let arg_params = arg_params.as_ref().map(|arg| arg.as_ref());
        let arg_channels = arg_channels.into_raw();
        ImplAudioHandler::on_audio_stream_started(
            &arg_self_.interface,
            arg_browser,
            arg_params,
            arg_channels,
        )
    }
    extern "C" fn on_audio_stream_packet<I: ImplAudioHandler>(
        self_: *mut _cef_audio_handler_t,
        browser: *mut _cef_browser_t,
        data: *mut *const f32,
        frames: ::std::os::raw::c_int,
        pts: i64,
    ) {
        let (arg_self_, arg_browser, arg_data, arg_frames, arg_pts) =
            (self_, browser, data, frames, pts);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_data = arg_data.into_raw();
        let arg_frames = arg_frames.into_raw();
        let arg_pts = arg_pts.into_raw();
        ImplAudioHandler::on_audio_stream_packet(
            &arg_self_.interface,
            arg_browser,
            arg_data,
            arg_frames,
            arg_pts,
        )
    }
    extern "C" fn on_audio_stream_stopped<I: ImplAudioHandler>(
        self_: *mut _cef_audio_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplAudioHandler::on_audio_stream_stopped(&arg_self_.interface, arg_browser)
    }
    extern "C" fn on_audio_stream_error<I: ImplAudioHandler>(
        self_: *mut _cef_audio_handler_t,
        browser: *mut _cef_browser_t,
        message: *const cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_message) = (self_, browser, message);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_message = if arg_message.is_null() {
            None
        } else {
            Some(arg_message.into())
        };
        let arg_message = arg_message.as_ref();
        ImplAudioHandler::on_audio_stream_error(&arg_self_.interface, arg_browser, arg_message)
    }
}
impl ImplAudioHandler for AudioHandler {
    fn audio_parameters(
        &self,
        browser: Option<&mut Browser>,
        params: Option<&mut AudioParameters>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_audio_parameters
                .map(|f| {
                    let (arg_browser, arg_params) = (browser, params);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let mut arg_params = arg_params.cloned().map(|arg| arg.into());
                    let arg_params = arg_params
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_params);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_audio_stream_started(
        &self,
        browser: Option<&mut Browser>,
        params: Option<&AudioParameters>,
        channels: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_audio_stream_started {
                let (arg_browser, arg_params, arg_channels) = (browser, params, channels);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_params = arg_params.cloned().map(|arg| arg.into());
                let arg_params = arg_params
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_params, arg_channels);
            }
        }
    }
    fn on_audio_stream_packet(
        &self,
        browser: Option<&mut Browser>,
        data: *mut *const f32,
        frames: ::std::os::raw::c_int,
        pts: i64,
    ) {
        unsafe {
            if let Some(f) = self.0.on_audio_stream_packet {
                let (arg_browser, arg_data, arg_frames, arg_pts) = (browser, data, frames, pts);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_data = arg_data.cast();
                f(arg_self_, arg_browser, arg_data, arg_frames, arg_pts);
            }
        }
    }
    fn on_audio_stream_stopped(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_audio_stream_stopped {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn on_audio_stream_error(&self, browser: Option<&mut Browser>, message: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.on_audio_stream_error {
                let (arg_browser, arg_message) = (browser, message);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_message = arg_message
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_message);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_audio_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_audio_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for AudioHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_audio_handler_t> for &AudioHandler {
    fn into_raw(self) -> *mut _cef_audio_handler_t {
        ImplAudioHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_audio_handler_t> for &mut AudioHandler {
    fn into_raw(self) -> *mut _cef_audio_handler_t {
        ImplAudioHandler::get_raw(self)
    }
}
impl ConvertReturnValue<AudioHandler> for *mut _cef_audio_handler_t {
    fn wrap_result(self) -> AudioHandler {
        AudioHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<AudioHandler> for *mut _cef_audio_handler_t {
    fn from(value: AudioHandler) -> Self {
        let object = ImplAudioHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for AudioHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_command_handler_t`] for more documentation.
#[derive(Clone)]
pub struct CommandHandler(RefGuard<_cef_command_handler_t>);
impl CommandHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapCommandHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplCommandHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapCommandHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_command_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapCommandHandler: ImplCommandHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_command_handler_t, Self>);
}
pub trait ImplCommandHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_command_handler_t::on_chrome_command`] for more documentation."]
    fn on_chrome_command(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
        disposition: WindowOpenDisposition,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_command_handler_t::is_chrome_app_menu_item_visible`] for more documentation."]
    fn is_chrome_app_menu_item_visible(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_command_handler_t::is_chrome_app_menu_item_enabled`] for more documentation."]
    fn is_chrome_app_menu_item_enabled(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_command_handler_t::is_chrome_page_action_icon_visible`] for more documentation."]
    fn is_chrome_page_action_icon_visible(
        &self,
        icon_type: ChromePageActionIconType,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_command_handler_t::is_chrome_toolbar_button_visible`] for more documentation."]
    fn is_chrome_toolbar_button_visible(
        &self,
        button_type: ChromeToolbarButtonType,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_command_handler_t) {
        impl_cef_command_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_command_handler_t;
}
mod impl_cef_command_handler_t {
    use super::*;
    pub fn init_methods<I: ImplCommandHandler>(object: &mut _cef_command_handler_t) {
        object.on_chrome_command = Some(on_chrome_command::<I>);
        object.is_chrome_app_menu_item_visible = Some(is_chrome_app_menu_item_visible::<I>);
        object.is_chrome_app_menu_item_enabled = Some(is_chrome_app_menu_item_enabled::<I>);
        object.is_chrome_page_action_icon_visible = Some(is_chrome_page_action_icon_visible::<I>);
        object.is_chrome_toolbar_button_visible = Some(is_chrome_toolbar_button_visible::<I>);
    }
    extern "C" fn on_chrome_command<I: ImplCommandHandler>(
        self_: *mut _cef_command_handler_t,
        browser: *mut _cef_browser_t,
        command_id: ::std::os::raw::c_int,
        disposition: cef_window_open_disposition_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_command_id, arg_disposition) =
            (self_, browser, command_id, disposition);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        let arg_disposition = arg_disposition.into_raw();
        ImplCommandHandler::on_chrome_command(
            &arg_self_.interface,
            arg_browser,
            arg_command_id,
            arg_disposition,
        )
    }
    extern "C" fn is_chrome_app_menu_item_visible<I: ImplCommandHandler>(
        self_: *mut _cef_command_handler_t,
        browser: *mut _cef_browser_t,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_command_id) = (self_, browser, command_id);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        ImplCommandHandler::is_chrome_app_menu_item_visible(
            &arg_self_.interface,
            arg_browser,
            arg_command_id,
        )
    }
    extern "C" fn is_chrome_app_menu_item_enabled<I: ImplCommandHandler>(
        self_: *mut _cef_command_handler_t,
        browser: *mut _cef_browser_t,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_command_id) = (self_, browser, command_id);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        ImplCommandHandler::is_chrome_app_menu_item_enabled(
            &arg_self_.interface,
            arg_browser,
            arg_command_id,
        )
    }
    extern "C" fn is_chrome_page_action_icon_visible<I: ImplCommandHandler>(
        self_: *mut _cef_command_handler_t,
        icon_type: cef_chrome_page_action_icon_type_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_icon_type) = (self_, icon_type);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_icon_type = arg_icon_type.into_raw();
        ImplCommandHandler::is_chrome_page_action_icon_visible(&arg_self_.interface, arg_icon_type)
    }
    extern "C" fn is_chrome_toolbar_button_visible<I: ImplCommandHandler>(
        self_: *mut _cef_command_handler_t,
        button_type: cef_chrome_toolbar_button_type_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_button_type) = (self_, button_type);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_button_type = arg_button_type.into_raw();
        ImplCommandHandler::is_chrome_toolbar_button_visible(&arg_self_.interface, arg_button_type)
    }
}
impl ImplCommandHandler for CommandHandler {
    fn on_chrome_command(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
        disposition: WindowOpenDisposition,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_chrome_command
                .map(|f| {
                    let (arg_browser, arg_command_id, arg_disposition) =
                        (browser, command_id, disposition);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_disposition = arg_disposition.into_raw();
                    let result = f(arg_self_, arg_browser, arg_command_id, arg_disposition);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_chrome_app_menu_item_visible(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_chrome_app_menu_item_visible
                .map(|f| {
                    let (arg_browser, arg_command_id) = (browser, command_id);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_chrome_app_menu_item_enabled(
        &self,
        browser: Option<&mut Browser>,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_chrome_app_menu_item_enabled
                .map(|f| {
                    let (arg_browser, arg_command_id) = (browser, command_id);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_chrome_page_action_icon_visible(
        &self,
        icon_type: ChromePageActionIconType,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_chrome_page_action_icon_visible
                .map(|f| {
                    let arg_icon_type = icon_type;
                    let arg_self_ = self.into_raw();
                    let arg_icon_type = arg_icon_type.into_raw();
                    let result = f(arg_self_, arg_icon_type);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_chrome_toolbar_button_visible(
        &self,
        button_type: ChromeToolbarButtonType,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_chrome_toolbar_button_visible
                .map(|f| {
                    let arg_button_type = button_type;
                    let arg_self_ = self.into_raw();
                    let arg_button_type = arg_button_type.into_raw();
                    let result = f(arg_self_, arg_button_type);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_command_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_command_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CommandHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_command_handler_t> for &CommandHandler {
    fn into_raw(self) -> *mut _cef_command_handler_t {
        ImplCommandHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_command_handler_t> for &mut CommandHandler {
    fn into_raw(self) -> *mut _cef_command_handler_t {
        ImplCommandHandler::get_raw(self)
    }
}
impl ConvertReturnValue<CommandHandler> for *mut _cef_command_handler_t {
    fn wrap_result(self) -> CommandHandler {
        CommandHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CommandHandler> for *mut _cef_command_handler_t {
    fn from(value: CommandHandler) -> Self {
        let object = ImplCommandHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CommandHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_menu_model_delegate_t`] for more documentation.
#[derive(Clone)]
pub struct MenuModelDelegate(RefGuard<_cef_menu_model_delegate_t>);
impl MenuModelDelegate {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapMenuModelDelegate,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplMenuModelDelegate>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapMenuModelDelegate>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_menu_model_delegate_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapMenuModelDelegate: ImplMenuModelDelegate {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_menu_model_delegate_t, Self>);
}
pub trait ImplMenuModelDelegate: Clone + Sized + Rc {
    #[doc = "See [`_cef_menu_model_delegate_t::execute_command`] for more documentation."]
    fn execute_command(
        &self,
        menu_model: Option<&mut MenuModel>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) {
    }
    #[doc = "See [`_cef_menu_model_delegate_t::mouse_outside_menu`] for more documentation."]
    fn mouse_outside_menu(&self, menu_model: Option<&mut MenuModel>, screen_point: Option<&Point>) {
    }
    #[doc = "See [`_cef_menu_model_delegate_t::unhandled_open_submenu`] for more documentation."]
    fn unhandled_open_submenu(
        &self,
        menu_model: Option<&mut MenuModel>,
        is_rtl: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_menu_model_delegate_t::unhandled_close_submenu`] for more documentation."]
    fn unhandled_close_submenu(
        &self,
        menu_model: Option<&mut MenuModel>,
        is_rtl: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_menu_model_delegate_t::menu_will_show`] for more documentation."]
    fn menu_will_show(&self, menu_model: Option<&mut MenuModel>) {}
    #[doc = "See [`_cef_menu_model_delegate_t::menu_closed`] for more documentation."]
    fn menu_closed(&self, menu_model: Option<&mut MenuModel>) {}
    #[doc = "See [`_cef_menu_model_delegate_t::format_label`] for more documentation."]
    fn format_label(
        &self,
        menu_model: Option<&mut MenuModel>,
        label: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_menu_model_delegate_t) {
        impl_cef_menu_model_delegate_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_menu_model_delegate_t;
}
mod impl_cef_menu_model_delegate_t {
    use super::*;
    pub fn init_methods<I: ImplMenuModelDelegate>(object: &mut _cef_menu_model_delegate_t) {
        object.execute_command = Some(execute_command::<I>);
        object.mouse_outside_menu = Some(mouse_outside_menu::<I>);
        object.unhandled_open_submenu = Some(unhandled_open_submenu::<I>);
        object.unhandled_close_submenu = Some(unhandled_close_submenu::<I>);
        object.menu_will_show = Some(menu_will_show::<I>);
        object.menu_closed = Some(menu_closed::<I>);
        object.format_label = Some(format_label::<I>);
    }
    extern "C" fn execute_command<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
        command_id: ::std::os::raw::c_int,
        event_flags: cef_event_flags_t,
    ) {
        let (arg_self_, arg_menu_model, arg_command_id, arg_event_flags) =
            (self_, menu_model, command_id, event_flags);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        let arg_event_flags = arg_event_flags.into_raw();
        ImplMenuModelDelegate::execute_command(
            &arg_self_.interface,
            arg_menu_model,
            arg_command_id,
            arg_event_flags,
        )
    }
    extern "C" fn mouse_outside_menu<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
        screen_point: *const _cef_point_t,
    ) {
        let (arg_self_, arg_menu_model, arg_screen_point) = (self_, menu_model, screen_point);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        let arg_screen_point = if arg_screen_point.is_null() {
            None
        } else {
            Some(WrapParamRef::<Point, _>::from(arg_screen_point))
        };
        let arg_screen_point = arg_screen_point.as_ref().map(|arg| arg.as_ref());
        ImplMenuModelDelegate::mouse_outside_menu(
            &arg_self_.interface,
            arg_menu_model,
            arg_screen_point,
        )
    }
    extern "C" fn unhandled_open_submenu<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
        is_rtl: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_menu_model, arg_is_rtl) = (self_, menu_model, is_rtl);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        let arg_is_rtl = arg_is_rtl.into_raw();
        ImplMenuModelDelegate::unhandled_open_submenu(
            &arg_self_.interface,
            arg_menu_model,
            arg_is_rtl,
        )
    }
    extern "C" fn unhandled_close_submenu<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
        is_rtl: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_menu_model, arg_is_rtl) = (self_, menu_model, is_rtl);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        let arg_is_rtl = arg_is_rtl.into_raw();
        ImplMenuModelDelegate::unhandled_close_submenu(
            &arg_self_.interface,
            arg_menu_model,
            arg_is_rtl,
        )
    }
    extern "C" fn menu_will_show<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
    ) {
        let (arg_self_, arg_menu_model) = (self_, menu_model);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        ImplMenuModelDelegate::menu_will_show(&arg_self_.interface, arg_menu_model)
    }
    extern "C" fn menu_closed<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
    ) {
        let (arg_self_, arg_menu_model) = (self_, menu_model);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        ImplMenuModelDelegate::menu_closed(&arg_self_.interface, arg_menu_model)
    }
    extern "C" fn format_label<I: ImplMenuModelDelegate>(
        self_: *mut _cef_menu_model_delegate_t,
        menu_model: *mut _cef_menu_model_t,
        label: *mut cef_string_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_menu_model, arg_label) = (self_, menu_model, label);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_menu_model = unsafe { arg_menu_model.as_mut() }
            .map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_menu_model = arg_menu_model.as_mut();
        let mut arg_label = if arg_label.is_null() {
            None
        } else {
            Some(arg_label.into())
        };
        let arg_label = arg_label.as_mut();
        ImplMenuModelDelegate::format_label(&arg_self_.interface, arg_menu_model, arg_label)
    }
}
impl ImplMenuModelDelegate for MenuModelDelegate {
    fn execute_command(
        &self,
        menu_model: Option<&mut MenuModel>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) {
        unsafe {
            if let Some(f) = self.0.execute_command {
                let (arg_menu_model, arg_command_id, arg_event_flags) =
                    (menu_model, command_id, event_flags);
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_event_flags = arg_event_flags.into_raw();
                f(arg_self_, arg_menu_model, arg_command_id, arg_event_flags);
            }
        }
    }
    fn mouse_outside_menu(&self, menu_model: Option<&mut MenuModel>, screen_point: Option<&Point>) {
        unsafe {
            if let Some(f) = self.0.mouse_outside_menu {
                let (arg_menu_model, arg_screen_point) = (menu_model, screen_point);
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_screen_point = arg_screen_point.cloned().map(|arg| arg.into());
                let arg_screen_point = arg_screen_point
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_menu_model, arg_screen_point);
            }
        }
    }
    fn unhandled_open_submenu(
        &self,
        menu_model: Option<&mut MenuModel>,
        is_rtl: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.unhandled_open_submenu {
                let (arg_menu_model, arg_is_rtl) = (menu_model, is_rtl);
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_menu_model, arg_is_rtl);
            }
        }
    }
    fn unhandled_close_submenu(
        &self,
        menu_model: Option<&mut MenuModel>,
        is_rtl: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.unhandled_close_submenu {
                let (arg_menu_model, arg_is_rtl) = (menu_model, is_rtl);
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_menu_model, arg_is_rtl);
            }
        }
    }
    fn menu_will_show(&self, menu_model: Option<&mut MenuModel>) {
        unsafe {
            if let Some(f) = self.0.menu_will_show {
                let arg_menu_model = menu_model;
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_menu_model);
            }
        }
    }
    fn menu_closed(&self, menu_model: Option<&mut MenuModel>) {
        unsafe {
            if let Some(f) = self.0.menu_closed {
                let arg_menu_model = menu_model;
                let arg_self_ = self.into_raw();
                let arg_menu_model = arg_menu_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_menu_model);
            }
        }
    }
    fn format_label(
        &self,
        menu_model: Option<&mut MenuModel>,
        label: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .format_label
                .map(|f| {
                    let (arg_menu_model, arg_label) = (menu_model, label);
                    let arg_self_ = self.into_raw();
                    let arg_menu_model = arg_menu_model
                        .map(|arg| {
                            arg.add_ref();
                            ImplMenuModel::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_menu_model, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_menu_model_delegate_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_menu_model_delegate_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MenuModelDelegate {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_menu_model_delegate_t> for &MenuModelDelegate {
    fn into_raw(self) -> *mut _cef_menu_model_delegate_t {
        ImplMenuModelDelegate::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_menu_model_delegate_t> for &mut MenuModelDelegate {
    fn into_raw(self) -> *mut _cef_menu_model_delegate_t {
        ImplMenuModelDelegate::get_raw(self)
    }
}
impl ConvertReturnValue<MenuModelDelegate> for *mut _cef_menu_model_delegate_t {
    fn wrap_result(self) -> MenuModelDelegate {
        MenuModelDelegate(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MenuModelDelegate> for *mut _cef_menu_model_delegate_t {
    fn from(value: MenuModelDelegate) -> Self {
        let object = ImplMenuModelDelegate::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MenuModelDelegate {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_menu_model_t`] for more documentation.
#[derive(Clone)]
pub struct MenuModel(RefGuard<_cef_menu_model_t>);
pub trait ImplMenuModel: Clone + Sized + Rc {
    #[doc = "See [`_cef_menu_model_t::is_sub_menu`] for more documentation."]
    fn is_sub_menu(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::clear`] for more documentation."]
    fn clear(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_count`] for more documentation."]
    fn count(&self) -> usize;
    #[doc = "See [`_cef_menu_model_t::add_separator`] for more documentation."]
    fn add_separator(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::add_item`] for more documentation."]
    fn add_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::add_check_item`] for more documentation."]
    fn add_check_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::add_radio_item`] for more documentation."]
    fn add_radio_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::add_sub_menu`] for more documentation."]
    fn add_sub_menu(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> Option<MenuModel>;
    #[doc = "See [`_cef_menu_model_t::insert_separator_at`] for more documentation."]
    fn insert_separator_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::insert_item_at`] for more documentation."]
    fn insert_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::insert_check_item_at`] for more documentation."]
    fn insert_check_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::insert_radio_item_at`] for more documentation."]
    fn insert_radio_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::insert_sub_menu_at`] for more documentation."]
    fn insert_sub_menu_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> Option<MenuModel>;
    #[doc = "See [`_cef_menu_model_t::remove`] for more documentation."]
    fn remove(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::remove_at`] for more documentation."]
    fn remove_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_index_of`] for more documentation."]
    fn index_of(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_command_id_at`] for more documentation."]
    fn command_id_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_command_id_at`] for more documentation."]
    fn set_command_id_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_label`] for more documentation."]
    fn label(&self, command_id: ::std::os::raw::c_int) -> CefStringUserfree;
    #[doc = "See [`_cef_menu_model_t::get_label_at`] for more documentation."]
    fn label_at(&self, index: usize) -> CefStringUserfree;
    #[doc = "See [`_cef_menu_model_t::set_label`] for more documentation."]
    fn set_label(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_label_at`] for more documentation."]
    fn set_label_at(&self, index: usize, label: Option<&CefString>) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_type`] for more documentation."]
    fn get_type(&self, command_id: ::std::os::raw::c_int) -> MenuItemType;
    #[doc = "See [`_cef_menu_model_t::get_type_at`] for more documentation."]
    fn type_at(&self, index: usize) -> MenuItemType;
    #[doc = "See [`_cef_menu_model_t::get_group_id`] for more documentation."]
    fn group_id(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_group_id_at`] for more documentation."]
    fn group_id_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_group_id`] for more documentation."]
    fn set_group_id(
        &self,
        command_id: ::std::os::raw::c_int,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_group_id_at`] for more documentation."]
    fn set_group_id_at(
        &self,
        index: usize,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_sub_menu`] for more documentation."]
    fn sub_menu(&self, command_id: ::std::os::raw::c_int) -> Option<MenuModel>;
    #[doc = "See [`_cef_menu_model_t::get_sub_menu_at`] for more documentation."]
    fn sub_menu_at(&self, index: usize) -> Option<MenuModel>;
    #[doc = "See [`_cef_menu_model_t::is_visible`] for more documentation."]
    fn is_visible(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::is_visible_at`] for more documentation."]
    fn is_visible_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_visible`] for more documentation."]
    fn set_visible(
        &self,
        command_id: ::std::os::raw::c_int,
        visible: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_visible_at`] for more documentation."]
    fn set_visible_at(&self, index: usize, visible: ::std::os::raw::c_int)
        -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::is_enabled`] for more documentation."]
    fn is_enabled(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::is_enabled_at`] for more documentation."]
    fn is_enabled_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_enabled`] for more documentation."]
    fn set_enabled(
        &self,
        command_id: ::std::os::raw::c_int,
        enabled: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_enabled_at`] for more documentation."]
    fn set_enabled_at(&self, index: usize, enabled: ::std::os::raw::c_int)
        -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::is_checked`] for more documentation."]
    fn is_checked(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::is_checked_at`] for more documentation."]
    fn is_checked_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_checked`] for more documentation."]
    fn set_checked(
        &self,
        command_id: ::std::os::raw::c_int,
        checked: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_checked_at`] for more documentation."]
    fn set_checked_at(&self, index: usize, checked: ::std::os::raw::c_int)
        -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::has_accelerator`] for more documentation."]
    fn has_accelerator(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::has_accelerator_at`] for more documentation."]
    fn has_accelerator_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_accelerator`] for more documentation."]
    fn set_accelerator(
        &self,
        command_id: ::std::os::raw::c_int,
        key_code: ::std::os::raw::c_int,
        shift_pressed: ::std::os::raw::c_int,
        ctrl_pressed: ::std::os::raw::c_int,
        alt_pressed: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_accelerator_at`] for more documentation."]
    fn set_accelerator_at(
        &self,
        index: usize,
        key_code: ::std::os::raw::c_int,
        shift_pressed: ::std::os::raw::c_int,
        ctrl_pressed: ::std::os::raw::c_int,
        alt_pressed: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::remove_accelerator`] for more documentation."]
    fn remove_accelerator(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::remove_accelerator_at`] for more documentation."]
    fn remove_accelerator_at(&self, index: usize) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_accelerator`] for more documentation."]
    fn accelerator(
        &self,
        command_id: ::std::os::raw::c_int,
        key_code: Option<&mut ::std::os::raw::c_int>,
        shift_pressed: Option<&mut ::std::os::raw::c_int>,
        ctrl_pressed: Option<&mut ::std::os::raw::c_int>,
        alt_pressed: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_accelerator_at`] for more documentation."]
    fn accelerator_at(
        &self,
        index: usize,
        key_code: Option<&mut ::std::os::raw::c_int>,
        shift_pressed: Option<&mut ::std::os::raw::c_int>,
        ctrl_pressed: Option<&mut ::std::os::raw::c_int>,
        alt_pressed: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_color`] for more documentation."]
    fn set_color(
        &self,
        command_id: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: u32,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_color_at`] for more documentation."]
    fn set_color_at(
        &self,
        index: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: u32,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_color`] for more documentation."]
    fn color(
        &self,
        command_id: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: Option<&mut u32>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::get_color_at`] for more documentation."]
    fn color_at(
        &self,
        index: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: Option<&mut u32>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_font_list`] for more documentation."]
    fn set_font_list(
        &self,
        command_id: ::std::os::raw::c_int,
        font_list: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_menu_model_t::set_font_list_at`] for more documentation."]
    fn set_font_list_at(
        &self,
        index: ::std::os::raw::c_int,
        font_list: Option<&CefString>,
    ) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_menu_model_t;
}
impl ImplMenuModel for MenuModel {
    fn is_sub_menu(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_sub_menu
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn clear(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .clear
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn count(&self) -> usize {
        unsafe {
            self.0
                .get_count
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_separator(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_separator
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_item
                .map(|f| {
                    let (arg_command_id, arg_label) = (command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_check_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_check_item
                .map(|f| {
                    let (arg_command_id, arg_label) = (command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_radio_item(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .add_radio_item
                .map(|f| {
                    let (arg_command_id, arg_label, arg_group_id) = (command_id, label, group_id);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_label, arg_group_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn add_sub_menu(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> Option<MenuModel> {
        unsafe {
            self.0
                .add_sub_menu
                .map(|f| {
                    let (arg_command_id, arg_label) = (command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_label);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn insert_separator_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .insert_separator_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn insert_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .insert_item_at
                .map(|f| {
                    let (arg_index, arg_command_id, arg_label) = (index, command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_command_id, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn insert_check_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .insert_check_item_at
                .map(|f| {
                    let (arg_index, arg_command_id, arg_label) = (index, command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_command_id, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn insert_radio_item_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .insert_radio_item_at
                .map(|f| {
                    let (arg_index, arg_command_id, arg_label, arg_group_id) =
                        (index, command_id, label, group_id);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_index,
                        arg_command_id,
                        arg_label,
                        arg_group_id,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn insert_sub_menu_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> Option<MenuModel> {
        unsafe {
            self.0
                .insert_sub_menu_at
                .map(|f| {
                    let (arg_index, arg_command_id, arg_label) = (index, command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_command_id, arg_label);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn remove(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn index_of(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_index_of
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn command_id_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_command_id_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_command_id_at(
        &self,
        index: usize,
        command_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_command_id_at
                .map(|f| {
                    let (arg_index, arg_command_id) = (index, command_id);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn label(&self, command_id: ::std::os::raw::c_int) -> CefStringUserfree {
        unsafe {
            self.0
                .get_label
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn label_at(&self, index: usize) -> CefStringUserfree {
        unsafe {
            self.0
                .get_label_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_label(
        &self,
        command_id: ::std::os::raw::c_int,
        label: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_label
                .map(|f| {
                    let (arg_command_id, arg_label) = (command_id, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_label_at(&self, index: usize, label: Option<&CefString>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_label_at
                .map(|f| {
                    let (arg_index, arg_label) = (index, label);
                    let arg_self_ = self.into_raw();
                    let arg_label = arg_label
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_label);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_type(&self, command_id: ::std::os::raw::c_int) -> MenuItemType {
        unsafe {
            self.0
                .get_type
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn type_at(&self, index: usize) -> MenuItemType {
        unsafe {
            self.0
                .get_type_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn group_id(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_group_id
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn group_id_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_group_id_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_group_id(
        &self,
        command_id: ::std::os::raw::c_int,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_group_id
                .map(|f| {
                    let (arg_command_id, arg_group_id) = (command_id, group_id);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id, arg_group_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_group_id_at(
        &self,
        index: usize,
        group_id: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_group_id_at
                .map(|f| {
                    let (arg_index, arg_group_id) = (index, group_id);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_group_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn sub_menu(&self, command_id: ::std::os::raw::c_int) -> Option<MenuModel> {
        unsafe {
            self.0
                .get_sub_menu
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn sub_menu_at(&self, index: usize) -> Option<MenuModel> {
        unsafe {
            self.0
                .get_sub_menu_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn is_visible(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_visible
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_visible_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_visible_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_visible(
        &self,
        command_id: ::std::os::raw::c_int,
        visible: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_visible
                .map(|f| {
                    let (arg_command_id, arg_visible) = (command_id, visible);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id, arg_visible);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_visible_at(
        &self,
        index: usize,
        visible: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_visible_at
                .map(|f| {
                    let (arg_index, arg_visible) = (index, visible);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_visible);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_enabled(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_enabled
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_enabled_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_enabled_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_enabled(
        &self,
        command_id: ::std::os::raw::c_int,
        enabled: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_enabled
                .map(|f| {
                    let (arg_command_id, arg_enabled) = (command_id, enabled);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id, arg_enabled);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_enabled_at(
        &self,
        index: usize,
        enabled: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_enabled_at
                .map(|f| {
                    let (arg_index, arg_enabled) = (index, enabled);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_enabled);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_checked(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_checked
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_checked_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_checked_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_checked(
        &self,
        command_id: ::std::os::raw::c_int,
        checked: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_checked
                .map(|f| {
                    let (arg_command_id, arg_checked) = (command_id, checked);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id, arg_checked);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_checked_at(
        &self,
        index: usize,
        checked: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_checked_at
                .map(|f| {
                    let (arg_index, arg_checked) = (index, checked);
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index, arg_checked);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_accelerator(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_accelerator
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_accelerator_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_accelerator_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_accelerator(
        &self,
        command_id: ::std::os::raw::c_int,
        key_code: ::std::os::raw::c_int,
        shift_pressed: ::std::os::raw::c_int,
        ctrl_pressed: ::std::os::raw::c_int,
        alt_pressed: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_accelerator
                .map(|f| {
                    let (
                        arg_command_id,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    ) = (
                        command_id,
                        key_code,
                        shift_pressed,
                        ctrl_pressed,
                        alt_pressed,
                    );
                    let arg_self_ = self.into_raw();
                    let result = f(
                        arg_self_,
                        arg_command_id,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_accelerator_at(
        &self,
        index: usize,
        key_code: ::std::os::raw::c_int,
        shift_pressed: ::std::os::raw::c_int,
        ctrl_pressed: ::std::os::raw::c_int,
        alt_pressed: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_accelerator_at
                .map(|f| {
                    let (
                        arg_index,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    ) = (index, key_code, shift_pressed, ctrl_pressed, alt_pressed);
                    let arg_self_ = self.into_raw();
                    let result = f(
                        arg_self_,
                        arg_index,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove_accelerator(&self, command_id: ::std::os::raw::c_int) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove_accelerator
                .map(|f| {
                    let arg_command_id = command_id;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_command_id);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn remove_accelerator_at(&self, index: usize) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .remove_accelerator_at
                .map(|f| {
                    let arg_index = index;
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_, arg_index);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn accelerator(
        &self,
        command_id: ::std::os::raw::c_int,
        key_code: Option<&mut ::std::os::raw::c_int>,
        shift_pressed: Option<&mut ::std::os::raw::c_int>,
        ctrl_pressed: Option<&mut ::std::os::raw::c_int>,
        alt_pressed: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_accelerator
                .map(|f| {
                    let (
                        arg_command_id,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    ) = (
                        command_id,
                        key_code,
                        shift_pressed,
                        ctrl_pressed,
                        alt_pressed,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_key_code = arg_key_code
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_shift_pressed = arg_shift_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_ctrl_pressed = arg_ctrl_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_alt_pressed = arg_alt_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_command_id,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn accelerator_at(
        &self,
        index: usize,
        key_code: Option<&mut ::std::os::raw::c_int>,
        shift_pressed: Option<&mut ::std::os::raw::c_int>,
        ctrl_pressed: Option<&mut ::std::os::raw::c_int>,
        alt_pressed: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_accelerator_at
                .map(|f| {
                    let (
                        arg_index,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    ) = (index, key_code, shift_pressed, ctrl_pressed, alt_pressed);
                    let arg_self_ = self.into_raw();
                    let arg_key_code = arg_key_code
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_shift_pressed = arg_shift_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_ctrl_pressed = arg_ctrl_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_alt_pressed = arg_alt_pressed
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_index,
                        arg_key_code,
                        arg_shift_pressed,
                        arg_ctrl_pressed,
                        arg_alt_pressed,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_color(
        &self,
        command_id: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: u32,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_color
                .map(|f| {
                    let (arg_command_id, arg_color_type, arg_color) =
                        (command_id, color_type, color);
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let result = f(arg_self_, arg_command_id, arg_color_type, arg_color);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_color_at(
        &self,
        index: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: u32,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_color_at
                .map(|f| {
                    let (arg_index, arg_color_type, arg_color) = (index, color_type, color);
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let result = f(arg_self_, arg_index, arg_color_type, arg_color);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn color(
        &self,
        command_id: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: Option<&mut u32>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_color
                .map(|f| {
                    let (arg_command_id, arg_color_type, arg_color) =
                        (command_id, color_type, color);
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let arg_color = arg_color
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_command_id, arg_color_type, arg_color);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn color_at(
        &self,
        index: ::std::os::raw::c_int,
        color_type: MenuColorType,
        color: Option<&mut u32>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_color_at
                .map(|f| {
                    let (arg_index, arg_color_type, arg_color) = (index, color_type, color);
                    let arg_self_ = self.into_raw();
                    let arg_color_type = arg_color_type.into_raw();
                    let arg_color = arg_color
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_index, arg_color_type, arg_color);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_font_list(
        &self,
        command_id: ::std::os::raw::c_int,
        font_list: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_font_list
                .map(|f| {
                    let (arg_command_id, arg_font_list) = (command_id, font_list);
                    let arg_self_ = self.into_raw();
                    let arg_font_list = arg_font_list
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_command_id, arg_font_list);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_font_list_at(
        &self,
        index: ::std::os::raw::c_int,
        font_list: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .set_font_list_at
                .map(|f| {
                    let (arg_index, arg_font_list) = (index, font_list);
                    let arg_self_ = self.into_raw();
                    let arg_font_list = arg_font_list
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_index, arg_font_list);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_menu_model_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_menu_model_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MenuModel {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_menu_model_t> for &MenuModel {
    fn into_raw(self) -> *mut _cef_menu_model_t {
        ImplMenuModel::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_menu_model_t> for &mut MenuModel {
    fn into_raw(self) -> *mut _cef_menu_model_t {
        ImplMenuModel::get_raw(self)
    }
}
impl ConvertReturnValue<MenuModel> for *mut _cef_menu_model_t {
    fn wrap_result(self) -> MenuModel {
        MenuModel(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MenuModel> for *mut _cef_menu_model_t {
    fn from(value: MenuModel) -> Self {
        let object = ImplMenuModel::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MenuModel {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_run_context_menu_callback_t`] for more documentation.
#[derive(Clone)]
pub struct RunContextMenuCallback(RefGuard<_cef_run_context_menu_callback_t>);
pub trait ImplRunContextMenuCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_run_context_menu_callback_t::cont`] for more documentation."]
    fn cont(&self, command_id: ::std::os::raw::c_int, event_flags: EventFlags);
    #[doc = "See [`_cef_run_context_menu_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_run_context_menu_callback_t;
}
impl ImplRunContextMenuCallback for RunContextMenuCallback {
    fn cont(&self, command_id: ::std::os::raw::c_int, event_flags: EventFlags) {
        unsafe {
            if let Some(f) = self.0.cont {
                let (arg_command_id, arg_event_flags) = (command_id, event_flags);
                let arg_self_ = self.into_raw();
                let arg_event_flags = arg_event_flags.into_raw();
                f(arg_self_, arg_command_id, arg_event_flags);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_run_context_menu_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_run_context_menu_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for RunContextMenuCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_run_context_menu_callback_t> for &RunContextMenuCallback {
    fn into_raw(self) -> *mut _cef_run_context_menu_callback_t {
        ImplRunContextMenuCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_run_context_menu_callback_t> for &mut RunContextMenuCallback {
    fn into_raw(self) -> *mut _cef_run_context_menu_callback_t {
        ImplRunContextMenuCallback::get_raw(self)
    }
}
impl ConvertReturnValue<RunContextMenuCallback> for *mut _cef_run_context_menu_callback_t {
    fn wrap_result(self) -> RunContextMenuCallback {
        RunContextMenuCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<RunContextMenuCallback> for *mut _cef_run_context_menu_callback_t {
    fn from(value: RunContextMenuCallback) -> Self {
        let object = ImplRunContextMenuCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for RunContextMenuCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_run_quick_menu_callback_t`] for more documentation.
#[derive(Clone)]
pub struct RunQuickMenuCallback(RefGuard<_cef_run_quick_menu_callback_t>);
pub trait ImplRunQuickMenuCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_run_quick_menu_callback_t::cont`] for more documentation."]
    fn cont(&self, command_id: ::std::os::raw::c_int, event_flags: EventFlags);
    #[doc = "See [`_cef_run_quick_menu_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_run_quick_menu_callback_t;
}
impl ImplRunQuickMenuCallback for RunQuickMenuCallback {
    fn cont(&self, command_id: ::std::os::raw::c_int, event_flags: EventFlags) {
        unsafe {
            if let Some(f) = self.0.cont {
                let (arg_command_id, arg_event_flags) = (command_id, event_flags);
                let arg_self_ = self.into_raw();
                let arg_event_flags = arg_event_flags.into_raw();
                f(arg_self_, arg_command_id, arg_event_flags);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_run_quick_menu_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_run_quick_menu_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for RunQuickMenuCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_run_quick_menu_callback_t> for &RunQuickMenuCallback {
    fn into_raw(self) -> *mut _cef_run_quick_menu_callback_t {
        ImplRunQuickMenuCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_run_quick_menu_callback_t> for &mut RunQuickMenuCallback {
    fn into_raw(self) -> *mut _cef_run_quick_menu_callback_t {
        ImplRunQuickMenuCallback::get_raw(self)
    }
}
impl ConvertReturnValue<RunQuickMenuCallback> for *mut _cef_run_quick_menu_callback_t {
    fn wrap_result(self) -> RunQuickMenuCallback {
        RunQuickMenuCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<RunQuickMenuCallback> for *mut _cef_run_quick_menu_callback_t {
    fn from(value: RunQuickMenuCallback) -> Self {
        let object = ImplRunQuickMenuCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for RunQuickMenuCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_context_menu_handler_t`] for more documentation.
#[derive(Clone)]
pub struct ContextMenuHandler(RefGuard<_cef_context_menu_handler_t>);
impl ContextMenuHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapContextMenuHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplContextMenuHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapContextMenuHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_context_menu_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapContextMenuHandler: ImplContextMenuHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_context_menu_handler_t, Self>);
}
pub trait ImplContextMenuHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_context_menu_handler_t::on_before_context_menu`] for more documentation."]
    fn on_before_context_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        model: Option<&mut MenuModel>,
    ) {
    }
    #[doc = "See [`_cef_context_menu_handler_t::run_context_menu`] for more documentation."]
    fn run_context_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        model: Option<&mut MenuModel>,
        callback: Option<&mut RunContextMenuCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_context_menu_handler_t::on_context_menu_command`] for more documentation."]
    fn on_context_menu_command(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_context_menu_handler_t::on_context_menu_dismissed`] for more documentation."]
    fn on_context_menu_dismissed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {}
    #[doc = "See [`_cef_context_menu_handler_t::run_quick_menu`] for more documentation."]
    fn run_quick_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        location: Option<&Point>,
        size: Option<&Size>,
        edit_state_flags: QuickMenuEditStateFlags,
        callback: Option<&mut RunQuickMenuCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_context_menu_handler_t::on_quick_menu_command`] for more documentation."]
    fn on_quick_menu_command(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_context_menu_handler_t::on_quick_menu_dismissed`] for more documentation."]
    fn on_quick_menu_dismissed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {}
    fn init_methods(object: &mut _cef_context_menu_handler_t) {
        impl_cef_context_menu_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_context_menu_handler_t;
}
mod impl_cef_context_menu_handler_t {
    use super::*;
    pub fn init_methods<I: ImplContextMenuHandler>(object: &mut _cef_context_menu_handler_t) {
        object.on_before_context_menu = Some(on_before_context_menu::<I>);
        object.run_context_menu = Some(run_context_menu::<I>);
        object.on_context_menu_command = Some(on_context_menu_command::<I>);
        object.on_context_menu_dismissed = Some(on_context_menu_dismissed::<I>);
        object.run_quick_menu = Some(run_quick_menu::<I>);
        object.on_quick_menu_command = Some(on_quick_menu_command::<I>);
        object.on_quick_menu_dismissed = Some(on_quick_menu_dismissed::<I>);
    }
    extern "C" fn on_before_context_menu<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        params: *mut _cef_context_menu_params_t,
        model: *mut _cef_menu_model_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_params, arg_model) =
            (self_, browser, frame, params, model);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_params = unsafe { arg_params.as_mut() }
            .map(|arg| ContextMenuParams(unsafe { RefGuard::from_raw(arg) }));
        let arg_params = arg_params.as_mut();
        let mut arg_model =
            unsafe { arg_model.as_mut() }.map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_model = arg_model.as_mut();
        ImplContextMenuHandler::on_before_context_menu(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_params,
            arg_model,
        )
    }
    extern "C" fn run_context_menu<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        params: *mut _cef_context_menu_params_t,
        model: *mut _cef_menu_model_t,
        callback: *mut _cef_run_context_menu_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_params, arg_model, arg_callback) =
            (self_, browser, frame, params, model, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_params = unsafe { arg_params.as_mut() }
            .map(|arg| ContextMenuParams(unsafe { RefGuard::from_raw(arg) }));
        let arg_params = arg_params.as_mut();
        let mut arg_model =
            unsafe { arg_model.as_mut() }.map(|arg| MenuModel(unsafe { RefGuard::from_raw(arg) }));
        let arg_model = arg_model.as_mut();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| RunContextMenuCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplContextMenuHandler::run_context_menu(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_params,
            arg_model,
            arg_callback,
        )
    }
    extern "C" fn on_context_menu_command<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        params: *mut _cef_context_menu_params_t,
        command_id: ::std::os::raw::c_int,
        event_flags: cef_event_flags_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_params, arg_command_id, arg_event_flags) =
            (self_, browser, frame, params, command_id, event_flags);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_params = unsafe { arg_params.as_mut() }
            .map(|arg| ContextMenuParams(unsafe { RefGuard::from_raw(arg) }));
        let arg_params = arg_params.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        let arg_event_flags = arg_event_flags.into_raw();
        ImplContextMenuHandler::on_context_menu_command(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_params,
            arg_command_id,
            arg_event_flags,
        )
    }
    extern "C" fn on_context_menu_dismissed<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_frame) = (self_, browser, frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        ImplContextMenuHandler::on_context_menu_dismissed(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
        )
    }
    extern "C" fn run_quick_menu<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        location: *const _cef_point_t,
        size: *const _cef_size_t,
        edit_state_flags: cef_quick_menu_edit_state_flags_t,
        callback: *mut _cef_run_quick_menu_callback_t,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_frame,
            arg_location,
            arg_size,
            arg_edit_state_flags,
            arg_callback,
        ) = (
            self_,
            browser,
            frame,
            location,
            size,
            edit_state_flags,
            callback,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_location = if arg_location.is_null() {
            None
        } else {
            Some(WrapParamRef::<Point, _>::from(arg_location))
        };
        let arg_location = arg_location.as_ref().map(|arg| arg.as_ref());
        let arg_size = if arg_size.is_null() {
            None
        } else {
            Some(WrapParamRef::<Size, _>::from(arg_size))
        };
        let arg_size = arg_size.as_ref().map(|arg| arg.as_ref());
        let arg_edit_state_flags = arg_edit_state_flags.into_raw();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| RunQuickMenuCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplContextMenuHandler::run_quick_menu(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_location,
            arg_size,
            arg_edit_state_flags,
            arg_callback,
        )
    }
    extern "C" fn on_quick_menu_command<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        command_id: ::std::os::raw::c_int,
        event_flags: cef_event_flags_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_command_id, arg_event_flags) =
            (self_, browser, frame, command_id, event_flags);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_command_id = arg_command_id.into_raw();
        let arg_event_flags = arg_event_flags.into_raw();
        ImplContextMenuHandler::on_quick_menu_command(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_command_id,
            arg_event_flags,
        )
    }
    extern "C" fn on_quick_menu_dismissed<I: ImplContextMenuHandler>(
        self_: *mut _cef_context_menu_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_frame) = (self_, browser, frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        ImplContextMenuHandler::on_quick_menu_dismissed(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
        )
    }
}
impl ImplContextMenuHandler for ContextMenuHandler {
    fn on_before_context_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        model: Option<&mut MenuModel>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_before_context_menu {
                let (arg_browser, arg_frame, arg_params, arg_model) =
                    (browser, frame, params, model);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_params = arg_params
                    .map(|arg| {
                        arg.add_ref();
                        ImplContextMenuParams::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_model = arg_model
                    .map(|arg| {
                        arg.add_ref();
                        ImplMenuModel::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame, arg_params, arg_model);
            }
        }
    }
    fn run_context_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        model: Option<&mut MenuModel>,
        callback: Option<&mut RunContextMenuCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .run_context_menu
                .map(|f| {
                    let (arg_browser, arg_frame, arg_params, arg_model, arg_callback) =
                        (browser, frame, params, model, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_params = arg_params
                        .map(|arg| {
                            arg.add_ref();
                            ImplContextMenuParams::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_model = arg_model
                        .map(|arg| {
                            arg.add_ref();
                            ImplMenuModel::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplRunContextMenuCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_params,
                        arg_model,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_context_menu_command(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        params: Option<&mut ContextMenuParams>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_context_menu_command
                .map(|f| {
                    let (arg_browser, arg_frame, arg_params, arg_command_id, arg_event_flags) =
                        (browser, frame, params, command_id, event_flags);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_params = arg_params
                        .map(|arg| {
                            arg.add_ref();
                            ImplContextMenuParams::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_event_flags = arg_event_flags.into_raw();
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_params,
                        arg_command_id,
                        arg_event_flags,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_context_menu_dismissed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {
        unsafe {
            if let Some(f) = self.0.on_context_menu_dismissed {
                let (arg_browser, arg_frame) = (browser, frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame);
            }
        }
    }
    fn run_quick_menu(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        location: Option<&Point>,
        size: Option<&Size>,
        edit_state_flags: QuickMenuEditStateFlags,
        callback: Option<&mut RunQuickMenuCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .run_quick_menu
                .map(|f| {
                    let (
                        arg_browser,
                        arg_frame,
                        arg_location,
                        arg_size,
                        arg_edit_state_flags,
                        arg_callback,
                    ) = (browser, frame, location, size, edit_state_flags, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_location = arg_location.cloned().map(|arg| arg.into());
                    let arg_location = arg_location
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_size = arg_size.cloned().map(|arg| arg.into());
                    let arg_size = arg_size
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_edit_state_flags = arg_edit_state_flags.into_raw();
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplRunQuickMenuCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_location,
                        arg_size,
                        arg_edit_state_flags,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_quick_menu_command(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        command_id: ::std::os::raw::c_int,
        event_flags: EventFlags,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_quick_menu_command
                .map(|f| {
                    let (arg_browser, arg_frame, arg_command_id, arg_event_flags) =
                        (browser, frame, command_id, event_flags);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_event_flags = arg_event_flags.into_raw();
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_command_id,
                        arg_event_flags,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_quick_menu_dismissed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {
        unsafe {
            if let Some(f) = self.0.on_quick_menu_dismissed {
                let (arg_browser, arg_frame) = (browser, frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_context_menu_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_context_menu_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ContextMenuHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_context_menu_handler_t> for &ContextMenuHandler {
    fn into_raw(self) -> *mut _cef_context_menu_handler_t {
        ImplContextMenuHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_context_menu_handler_t> for &mut ContextMenuHandler {
    fn into_raw(self) -> *mut _cef_context_menu_handler_t {
        ImplContextMenuHandler::get_raw(self)
    }
}
impl ConvertReturnValue<ContextMenuHandler> for *mut _cef_context_menu_handler_t {
    fn wrap_result(self) -> ContextMenuHandler {
        ContextMenuHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ContextMenuHandler> for *mut _cef_context_menu_handler_t {
    fn from(value: ContextMenuHandler) -> Self {
        let object = ImplContextMenuHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ContextMenuHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_context_menu_params_t`] for more documentation.
#[derive(Clone)]
pub struct ContextMenuParams(RefGuard<_cef_context_menu_params_t>);
pub trait ImplContextMenuParams: Clone + Sized + Rc {
    #[doc = "See [`_cef_context_menu_params_t::get_xcoord`] for more documentation."]
    fn xcoord(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::get_ycoord`] for more documentation."]
    fn ycoord(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::get_type_flags`] for more documentation."]
    fn type_flags(&self) -> ContextMenuTypeFlags;
    #[doc = "See [`_cef_context_menu_params_t::get_link_url`] for more documentation."]
    fn link_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_unfiltered_link_url`] for more documentation."]
    fn unfiltered_link_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_source_url`] for more documentation."]
    fn source_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::has_image_contents`] for more documentation."]
    fn has_image_contents(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::get_title_text`] for more documentation."]
    fn title_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_page_url`] for more documentation."]
    fn page_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_frame_url`] for more documentation."]
    fn frame_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_frame_charset`] for more documentation."]
    fn frame_charset(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_media_type`] for more documentation."]
    fn media_type(&self) -> ContextMenuMediaType;
    #[doc = "See [`_cef_context_menu_params_t::get_media_state_flags`] for more documentation."]
    fn media_state_flags(&self) -> ContextMenuMediaStateFlags;
    #[doc = "See [`_cef_context_menu_params_t::get_selection_text`] for more documentation."]
    fn selection_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_misspelled_word`] for more documentation."]
    fn misspelled_word(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_context_menu_params_t::get_dictionary_suggestions`] for more documentation."]
    fn dictionary_suggestions(
        &self,
        suggestions: Option<&mut CefStringList>,
    ) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::is_editable`] for more documentation."]
    fn is_editable(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::is_spell_check_enabled`] for more documentation."]
    fn is_spell_check_enabled(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_context_menu_params_t::get_edit_state_flags`] for more documentation."]
    fn edit_state_flags(&self) -> ContextMenuEditStateFlags;
    #[doc = "See [`_cef_context_menu_params_t::is_custom_menu`] for more documentation."]
    fn is_custom_menu(&self) -> ::std::os::raw::c_int;
    fn get_raw(&self) -> *mut _cef_context_menu_params_t;
}
impl ImplContextMenuParams for ContextMenuParams {
    fn xcoord(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_xcoord
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn ycoord(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_ycoord
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn type_flags(&self) -> ContextMenuTypeFlags {
        unsafe {
            self.0
                .get_type_flags
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn link_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_link_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn unfiltered_link_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_unfiltered_link_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn source_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_source_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn has_image_contents(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .has_image_contents
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn title_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_title_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn page_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_page_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn frame_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_frame_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn frame_charset(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_frame_charset
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn media_type(&self) -> ContextMenuMediaType {
        unsafe {
            self.0
                .get_media_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn media_state_flags(&self) -> ContextMenuMediaStateFlags {
        unsafe {
            self.0
                .get_media_state_flags
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn selection_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_selection_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn misspelled_word(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_misspelled_word
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn dictionary_suggestions(
        &self,
        suggestions: Option<&mut CefStringList>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_dictionary_suggestions
                .map(|f| {
                    let arg_suggestions = suggestions;
                    let arg_self_ = self.into_raw();
                    let arg_suggestions = arg_suggestions
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_suggestions);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_editable(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_editable
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_spell_check_enabled(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_spell_check_enabled
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn edit_state_flags(&self) -> ContextMenuEditStateFlags {
        unsafe {
            self.0
                .get_edit_state_flags
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_custom_menu(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_custom_menu
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_context_menu_params_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_context_menu_params_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ContextMenuParams {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_context_menu_params_t> for &ContextMenuParams {
    fn into_raw(self) -> *mut _cef_context_menu_params_t {
        ImplContextMenuParams::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_context_menu_params_t> for &mut ContextMenuParams {
    fn into_raw(self) -> *mut _cef_context_menu_params_t {
        ImplContextMenuParams::get_raw(self)
    }
}
impl ConvertReturnValue<ContextMenuParams> for *mut _cef_context_menu_params_t {
    fn wrap_result(self) -> ContextMenuParams {
        ContextMenuParams(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ContextMenuParams> for *mut _cef_context_menu_params_t {
    fn from(value: ContextMenuParams) -> Self {
        let object = ImplContextMenuParams::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ContextMenuParams {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_file_dialog_callback_t`] for more documentation.
#[derive(Clone)]
pub struct FileDialogCallback(RefGuard<_cef_file_dialog_callback_t>);
pub trait ImplFileDialogCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_file_dialog_callback_t::cont`] for more documentation."]
    fn cont(&self, file_paths: Option<&mut CefStringList>);
    #[doc = "See [`_cef_file_dialog_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_file_dialog_callback_t;
}
impl ImplFileDialogCallback for FileDialogCallback {
    fn cont(&self, file_paths: Option<&mut CefStringList>) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_file_paths = file_paths;
                let arg_self_ = self.into_raw();
                let arg_file_paths = arg_file_paths
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_file_paths);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_file_dialog_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_file_dialog_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for FileDialogCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_file_dialog_callback_t> for &FileDialogCallback {
    fn into_raw(self) -> *mut _cef_file_dialog_callback_t {
        ImplFileDialogCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_file_dialog_callback_t> for &mut FileDialogCallback {
    fn into_raw(self) -> *mut _cef_file_dialog_callback_t {
        ImplFileDialogCallback::get_raw(self)
    }
}
impl ConvertReturnValue<FileDialogCallback> for *mut _cef_file_dialog_callback_t {
    fn wrap_result(self) -> FileDialogCallback {
        FileDialogCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<FileDialogCallback> for *mut _cef_file_dialog_callback_t {
    fn from(value: FileDialogCallback) -> Self {
        let object = ImplFileDialogCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for FileDialogCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_dialog_handler_t`] for more documentation.
#[derive(Clone)]
pub struct DialogHandler(RefGuard<_cef_dialog_handler_t>);
impl DialogHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDialogHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDialogHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDialogHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_dialog_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDialogHandler: ImplDialogHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_dialog_handler_t, Self>);
}
pub trait ImplDialogHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_dialog_handler_t::on_file_dialog`] for more documentation."]
    fn on_file_dialog(
        &self,
        browser: Option<&mut Browser>,
        mode: FileDialogMode,
        title: Option<&CefString>,
        default_file_path: Option<&CefString>,
        accept_filters: Option<&mut CefStringList>,
        accept_extensions: Option<&mut CefStringList>,
        accept_descriptions: Option<&mut CefStringList>,
        callback: Option<&mut FileDialogCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_dialog_handler_t) {
        impl_cef_dialog_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_dialog_handler_t;
}
mod impl_cef_dialog_handler_t {
    use super::*;
    pub fn init_methods<I: ImplDialogHandler>(object: &mut _cef_dialog_handler_t) {
        object.on_file_dialog = Some(on_file_dialog::<I>);
    }
    extern "C" fn on_file_dialog<I: ImplDialogHandler>(
        self_: *mut _cef_dialog_handler_t,
        browser: *mut _cef_browser_t,
        mode: cef_file_dialog_mode_t,
        title: *const cef_string_t,
        default_file_path: *const cef_string_t,
        accept_filters: *mut _cef_string_list_t,
        accept_extensions: *mut _cef_string_list_t,
        accept_descriptions: *mut _cef_string_list_t,
        callback: *mut _cef_file_dialog_callback_t,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_mode,
            arg_title,
            arg_default_file_path,
            arg_accept_filters,
            arg_accept_extensions,
            arg_accept_descriptions,
            arg_callback,
        ) = (
            self_,
            browser,
            mode,
            title,
            default_file_path,
            accept_filters,
            accept_extensions,
            accept_descriptions,
            callback,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_mode = arg_mode.into_raw();
        let arg_title = if arg_title.is_null() {
            None
        } else {
            Some(arg_title.into())
        };
        let arg_title = arg_title.as_ref();
        let arg_default_file_path = if arg_default_file_path.is_null() {
            None
        } else {
            Some(arg_default_file_path.into())
        };
        let arg_default_file_path = arg_default_file_path.as_ref();
        let mut arg_accept_filters = if arg_accept_filters.is_null() {
            None
        } else {
            Some(arg_accept_filters.into())
        };
        let arg_accept_filters = arg_accept_filters.as_mut();
        let mut arg_accept_extensions = if arg_accept_extensions.is_null() {
            None
        } else {
            Some(arg_accept_extensions.into())
        };
        let arg_accept_extensions = arg_accept_extensions.as_mut();
        let mut arg_accept_descriptions = if arg_accept_descriptions.is_null() {
            None
        } else {
            Some(arg_accept_descriptions.into())
        };
        let arg_accept_descriptions = arg_accept_descriptions.as_mut();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| FileDialogCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplDialogHandler::on_file_dialog(
            &arg_self_.interface,
            arg_browser,
            arg_mode,
            arg_title,
            arg_default_file_path,
            arg_accept_filters,
            arg_accept_extensions,
            arg_accept_descriptions,
            arg_callback,
        )
    }
}
impl ImplDialogHandler for DialogHandler {
    fn on_file_dialog(
        &self,
        browser: Option<&mut Browser>,
        mode: FileDialogMode,
        title: Option<&CefString>,
        default_file_path: Option<&CefString>,
        accept_filters: Option<&mut CefStringList>,
        accept_extensions: Option<&mut CefStringList>,
        accept_descriptions: Option<&mut CefStringList>,
        callback: Option<&mut FileDialogCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_file_dialog
                .map(|f| {
                    let (
                        arg_browser,
                        arg_mode,
                        arg_title,
                        arg_default_file_path,
                        arg_accept_filters,
                        arg_accept_extensions,
                        arg_accept_descriptions,
                        arg_callback,
                    ) = (
                        browser,
                        mode,
                        title,
                        default_file_path,
                        accept_filters,
                        accept_extensions,
                        accept_descriptions,
                        callback,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_mode = arg_mode.into_raw();
                    let arg_title = arg_title
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_default_file_path = arg_default_file_path
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_accept_filters = arg_accept_filters
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let arg_accept_extensions = arg_accept_extensions
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let arg_accept_descriptions = arg_accept_descriptions
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplFileDialogCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_mode,
                        arg_title,
                        arg_default_file_path,
                        arg_accept_filters,
                        arg_accept_extensions,
                        arg_accept_descriptions,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_dialog_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_dialog_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DialogHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_dialog_handler_t> for &DialogHandler {
    fn into_raw(self) -> *mut _cef_dialog_handler_t {
        ImplDialogHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_dialog_handler_t> for &mut DialogHandler {
    fn into_raw(self) -> *mut _cef_dialog_handler_t {
        ImplDialogHandler::get_raw(self)
    }
}
impl ConvertReturnValue<DialogHandler> for *mut _cef_dialog_handler_t {
    fn wrap_result(self) -> DialogHandler {
        DialogHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DialogHandler> for *mut _cef_dialog_handler_t {
    fn from(value: DialogHandler) -> Self {
        let object = ImplDialogHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DialogHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_display_handler_t`] for more documentation.
#[derive(Clone)]
pub struct DisplayHandler(RefGuard<_cef_display_handler_t>);
impl DisplayHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDisplayHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDisplayHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDisplayHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_display_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDisplayHandler: ImplDisplayHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_display_handler_t, Self>);
}
pub trait ImplDisplayHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_display_handler_t::on_address_change`] for more documentation."]
    fn on_address_change(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        url: Option<&CefString>,
    ) {
    }
    #[doc = "See [`_cef_display_handler_t::on_title_change`] for more documentation."]
    fn on_title_change(&self, browser: Option<&mut Browser>, title: Option<&CefString>) {}
    #[doc = "See [`_cef_display_handler_t::on_favicon_urlchange`] for more documentation."]
    fn on_favicon_urlchange(
        &self,
        browser: Option<&mut Browser>,
        icon_urls: Option<&mut CefStringList>,
    ) {
    }
    #[doc = "See [`_cef_display_handler_t::on_fullscreen_mode_change`] for more documentation."]
    fn on_fullscreen_mode_change(
        &self,
        browser: Option<&mut Browser>,
        fullscreen: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_display_handler_t::on_tooltip`] for more documentation."]
    fn on_tooltip(
        &self,
        browser: Option<&mut Browser>,
        text: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_display_handler_t::on_status_message`] for more documentation."]
    fn on_status_message(&self, browser: Option<&mut Browser>, value: Option<&CefString>) {}
    #[doc = "See [`_cef_display_handler_t::on_console_message`] for more documentation."]
    fn on_console_message(
        &self,
        browser: Option<&mut Browser>,
        level: LogSeverity,
        message: Option<&CefString>,
        source: Option<&CefString>,
        line: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_display_handler_t::on_auto_resize`] for more documentation."]
    fn on_auto_resize(
        &self,
        browser: Option<&mut Browser>,
        new_size: Option<&Size>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_display_handler_t::on_loading_progress_change`] for more documentation."]
    fn on_loading_progress_change(&self, browser: Option<&mut Browser>, progress: f64) {}
    #[doc = "See [`_cef_display_handler_t::on_cursor_change`] for more documentation."]
    fn on_cursor_change(
        &self,
        browser: Option<&mut Browser>,
        cursor: ::std::os::raw::c_ulong,
        type_: CursorType,
        custom_cursor_info: Option<&CursorInfo>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_display_handler_t::on_media_access_change`] for more documentation."]
    fn on_media_access_change(
        &self,
        browser: Option<&mut Browser>,
        has_video_access: ::std::os::raw::c_int,
        has_audio_access: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_display_handler_t::on_contents_bounds_change`] for more documentation."]
    fn on_contents_bounds_change(
        &self,
        browser: Option<&mut Browser>,
        new_bounds: Option<&Rect>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_display_handler_t::get_root_window_screen_rect`] for more documentation."]
    fn root_window_screen_rect(
        &self,
        browser: Option<&mut Browser>,
        rect: Option<&mut Rect>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_display_handler_t) {
        impl_cef_display_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_display_handler_t;
}
mod impl_cef_display_handler_t {
    use super::*;
    pub fn init_methods<I: ImplDisplayHandler>(object: &mut _cef_display_handler_t) {
        object.on_address_change = Some(on_address_change::<I>);
        object.on_title_change = Some(on_title_change::<I>);
        object.on_favicon_urlchange = Some(on_favicon_urlchange::<I>);
        object.on_fullscreen_mode_change = Some(on_fullscreen_mode_change::<I>);
        object.on_tooltip = Some(on_tooltip::<I>);
        object.on_status_message = Some(on_status_message::<I>);
        object.on_console_message = Some(on_console_message::<I>);
        object.on_auto_resize = Some(on_auto_resize::<I>);
        object.on_loading_progress_change = Some(on_loading_progress_change::<I>);
        object.on_cursor_change = Some(on_cursor_change::<I>);
        object.on_media_access_change = Some(on_media_access_change::<I>);
        object.on_contents_bounds_change = Some(on_contents_bounds_change::<I>);
        object.get_root_window_screen_rect = Some(get_root_window_screen_rect::<I>);
    }
    extern "C" fn on_address_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        url: *const cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_url) = (self_, browser, frame, url);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_url = if arg_url.is_null() {
            None
        } else {
            Some(arg_url.into())
        };
        let arg_url = arg_url.as_ref();
        ImplDisplayHandler::on_address_change(&arg_self_.interface, arg_browser, arg_frame, arg_url)
    }
    extern "C" fn on_title_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        title: *const cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_title) = (self_, browser, title);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_title = if arg_title.is_null() {
            None
        } else {
            Some(arg_title.into())
        };
        let arg_title = arg_title.as_ref();
        ImplDisplayHandler::on_title_change(&arg_self_.interface, arg_browser, arg_title)
    }
    extern "C" fn on_favicon_urlchange<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        icon_urls: *mut _cef_string_list_t,
    ) {
        let (arg_self_, arg_browser, arg_icon_urls) = (self_, browser, icon_urls);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_icon_urls = if arg_icon_urls.is_null() {
            None
        } else {
            Some(arg_icon_urls.into())
        };
        let arg_icon_urls = arg_icon_urls.as_mut();
        ImplDisplayHandler::on_favicon_urlchange(&arg_self_.interface, arg_browser, arg_icon_urls)
    }
    extern "C" fn on_fullscreen_mode_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        fullscreen: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_fullscreen) = (self_, browser, fullscreen);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_fullscreen = arg_fullscreen.into_raw();
        ImplDisplayHandler::on_fullscreen_mode_change(
            &arg_self_.interface,
            arg_browser,
            arg_fullscreen,
        )
    }
    extern "C" fn on_tooltip<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        text: *mut cef_string_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_text) = (self_, browser, text);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_text = if arg_text.is_null() {
            None
        } else {
            Some(arg_text.into())
        };
        let arg_text = arg_text.as_mut();
        ImplDisplayHandler::on_tooltip(&arg_self_.interface, arg_browser, arg_text)
    }
    extern "C" fn on_status_message<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        value: *const cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_value) = (self_, browser, value);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_value = if arg_value.is_null() {
            None
        } else {
            Some(arg_value.into())
        };
        let arg_value = arg_value.as_ref();
        ImplDisplayHandler::on_status_message(&arg_self_.interface, arg_browser, arg_value)
    }
    extern "C" fn on_console_message<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        level: cef_log_severity_t,
        message: *const cef_string_t,
        source: *const cef_string_t,
        line: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_level, arg_message, arg_source, arg_line) =
            (self_, browser, level, message, source, line);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_level = arg_level.into_raw();
        let arg_message = if arg_message.is_null() {
            None
        } else {
            Some(arg_message.into())
        };
        let arg_message = arg_message.as_ref();
        let arg_source = if arg_source.is_null() {
            None
        } else {
            Some(arg_source.into())
        };
        let arg_source = arg_source.as_ref();
        let arg_line = arg_line.into_raw();
        ImplDisplayHandler::on_console_message(
            &arg_self_.interface,
            arg_browser,
            arg_level,
            arg_message,
            arg_source,
            arg_line,
        )
    }
    extern "C" fn on_auto_resize<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        new_size: *const _cef_size_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_new_size) = (self_, browser, new_size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_new_size = if arg_new_size.is_null() {
            None
        } else {
            Some(WrapParamRef::<Size, _>::from(arg_new_size))
        };
        let arg_new_size = arg_new_size.as_ref().map(|arg| arg.as_ref());
        ImplDisplayHandler::on_auto_resize(&arg_self_.interface, arg_browser, arg_new_size)
    }
    extern "C" fn on_loading_progress_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        progress: f64,
    ) {
        let (arg_self_, arg_browser, arg_progress) = (self_, browser, progress);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_progress = arg_progress.into_raw();
        ImplDisplayHandler::on_loading_progress_change(
            &arg_self_.interface,
            arg_browser,
            arg_progress,
        )
    }
    extern "C" fn on_cursor_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        cursor: ::std::os::raw::c_ulong,
        type_: cef_cursor_type_t,
        custom_cursor_info: *const _cef_cursor_info_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_cursor, arg_type_, arg_custom_cursor_info) =
            (self_, browser, cursor, type_, custom_cursor_info);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_cursor = arg_cursor.into_raw();
        let arg_type_ = arg_type_.into_raw();
        let arg_custom_cursor_info = if arg_custom_cursor_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<CursorInfo, _>::from(arg_custom_cursor_info))
        };
        let arg_custom_cursor_info = arg_custom_cursor_info.as_ref().map(|arg| arg.as_ref());
        ImplDisplayHandler::on_cursor_change(
            &arg_self_.interface,
            arg_browser,
            arg_cursor,
            arg_type_,
            arg_custom_cursor_info,
        )
    }
    extern "C" fn on_media_access_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        has_video_access: ::std::os::raw::c_int,
        has_audio_access: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_has_video_access, arg_has_audio_access) =
            (self_, browser, has_video_access, has_audio_access);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_has_video_access = arg_has_video_access.into_raw();
        let arg_has_audio_access = arg_has_audio_access.into_raw();
        ImplDisplayHandler::on_media_access_change(
            &arg_self_.interface,
            arg_browser,
            arg_has_video_access,
            arg_has_audio_access,
        )
    }
    extern "C" fn on_contents_bounds_change<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        new_bounds: *const _cef_rect_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_new_bounds) = (self_, browser, new_bounds);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_new_bounds = if arg_new_bounds.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_new_bounds))
        };
        let arg_new_bounds = arg_new_bounds.as_ref().map(|arg| arg.as_ref());
        ImplDisplayHandler::on_contents_bounds_change(
            &arg_self_.interface,
            arg_browser,
            arg_new_bounds,
        )
    }
    extern "C" fn get_root_window_screen_rect<I: ImplDisplayHandler>(
        self_: *mut _cef_display_handler_t,
        browser: *mut _cef_browser_t,
        rect: *mut _cef_rect_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_rect) = (self_, browser, rect);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_rect = if arg_rect.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_rect))
        };
        let arg_rect = arg_rect.as_mut().map(|arg| arg.as_mut());
        ImplDisplayHandler::root_window_screen_rect(&arg_self_.interface, arg_browser, arg_rect)
    }
}
impl ImplDisplayHandler for DisplayHandler {
    fn on_address_change(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        url: Option<&CefString>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_address_change {
                let (arg_browser, arg_frame, arg_url) = (browser, frame, url);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_frame, arg_url);
            }
        }
    }
    fn on_title_change(&self, browser: Option<&mut Browser>, title: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.on_title_change {
                let (arg_browser, arg_title) = (browser, title);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_title = arg_title
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_title);
            }
        }
    }
    fn on_favicon_urlchange(
        &self,
        browser: Option<&mut Browser>,
        icon_urls: Option<&mut CefStringList>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_favicon_urlchange {
                let (arg_browser, arg_icon_urls) = (browser, icon_urls);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_icon_urls = arg_icon_urls
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_icon_urls);
            }
        }
    }
    fn on_fullscreen_mode_change(
        &self,
        browser: Option<&mut Browser>,
        fullscreen: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_fullscreen_mode_change {
                let (arg_browser, arg_fullscreen) = (browser, fullscreen);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_fullscreen);
            }
        }
    }
    fn on_tooltip(
        &self,
        browser: Option<&mut Browser>,
        text: Option<&mut CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_tooltip
                .map(|f| {
                    let (arg_browser, arg_text) = (browser, text);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_text = arg_text
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_text);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_status_message(&self, browser: Option<&mut Browser>, value: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.on_status_message {
                let (arg_browser, arg_value) = (browser, value);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_value = arg_value
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_value);
            }
        }
    }
    fn on_console_message(
        &self,
        browser: Option<&mut Browser>,
        level: LogSeverity,
        message: Option<&CefString>,
        source: Option<&CefString>,
        line: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_console_message
                .map(|f| {
                    let (arg_browser, arg_level, arg_message, arg_source, arg_line) =
                        (browser, level, message, source, line);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_level = arg_level.into_raw();
                    let arg_message = arg_message
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_source = arg_source
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_level,
                        arg_message,
                        arg_source,
                        arg_line,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_auto_resize(
        &self,
        browser: Option<&mut Browser>,
        new_size: Option<&Size>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_auto_resize
                .map(|f| {
                    let (arg_browser, arg_new_size) = (browser, new_size);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_new_size = arg_new_size.cloned().map(|arg| arg.into());
                    let arg_new_size = arg_new_size
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_browser, arg_new_size);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_loading_progress_change(&self, browser: Option<&mut Browser>, progress: f64) {
        unsafe {
            if let Some(f) = self.0.on_loading_progress_change {
                let (arg_browser, arg_progress) = (browser, progress);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_progress);
            }
        }
    }
    fn on_cursor_change(
        &self,
        browser: Option<&mut Browser>,
        cursor: ::std::os::raw::c_ulong,
        type_: CursorType,
        custom_cursor_info: Option<&CursorInfo>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_cursor_change
                .map(|f| {
                    let (arg_browser, arg_cursor, arg_type_, arg_custom_cursor_info) =
                        (browser, cursor, type_, custom_cursor_info);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_type_ = arg_type_.into_raw();
                    let arg_custom_cursor_info =
                        arg_custom_cursor_info.cloned().map(|arg| arg.into());
                    let arg_custom_cursor_info = arg_custom_cursor_info
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_cursor,
                        arg_type_,
                        arg_custom_cursor_info,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_media_access_change(
        &self,
        browser: Option<&mut Browser>,
        has_video_access: ::std::os::raw::c_int,
        has_audio_access: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_media_access_change {
                let (arg_browser, arg_has_video_access, arg_has_audio_access) =
                    (browser, has_video_access, has_audio_access);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_browser,
                    arg_has_video_access,
                    arg_has_audio_access,
                );
            }
        }
    }
    fn on_contents_bounds_change(
        &self,
        browser: Option<&mut Browser>,
        new_bounds: Option<&Rect>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_contents_bounds_change
                .map(|f| {
                    let (arg_browser, arg_new_bounds) = (browser, new_bounds);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_new_bounds = arg_new_bounds.cloned().map(|arg| arg.into());
                    let arg_new_bounds = arg_new_bounds
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_browser, arg_new_bounds);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn root_window_screen_rect(
        &self,
        browser: Option<&mut Browser>,
        rect: Option<&mut Rect>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_root_window_screen_rect
                .map(|f| {
                    let (arg_browser, arg_rect) = (browser, rect);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let mut arg_rect = arg_rect.cloned().map(|arg| arg.into());
                    let arg_rect = arg_rect
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_rect);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_display_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_display_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DisplayHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_display_handler_t> for &DisplayHandler {
    fn into_raw(self) -> *mut _cef_display_handler_t {
        ImplDisplayHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_display_handler_t> for &mut DisplayHandler {
    fn into_raw(self) -> *mut _cef_display_handler_t {
        ImplDisplayHandler::get_raw(self)
    }
}
impl ConvertReturnValue<DisplayHandler> for *mut _cef_display_handler_t {
    fn wrap_result(self) -> DisplayHandler {
        DisplayHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DisplayHandler> for *mut _cef_display_handler_t {
    fn from(value: DisplayHandler) -> Self {
        let object = ImplDisplayHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DisplayHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_download_item_t`] for more documentation.
#[derive(Clone)]
pub struct DownloadItem(RefGuard<_cef_download_item_t>);
pub trait ImplDownloadItem: Clone + Sized + Rc {
    #[doc = "See [`_cef_download_item_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::is_in_progress`] for more documentation."]
    fn is_in_progress(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::is_complete`] for more documentation."]
    fn is_complete(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::is_canceled`] for more documentation."]
    fn is_canceled(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::is_interrupted`] for more documentation."]
    fn is_interrupted(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::get_interrupt_reason`] for more documentation."]
    fn interrupt_reason(&self) -> DownloadInterruptReason;
    #[doc = "See [`_cef_download_item_t::get_current_speed`] for more documentation."]
    fn current_speed(&self) -> i64;
    #[doc = "See [`_cef_download_item_t::get_percent_complete`] for more documentation."]
    fn percent_complete(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_download_item_t::get_total_bytes`] for more documentation."]
    fn total_bytes(&self) -> i64;
    #[doc = "See [`_cef_download_item_t::get_received_bytes`] for more documentation."]
    fn received_bytes(&self) -> i64;
    #[doc = "See [`_cef_download_item_t::get_start_time`] for more documentation."]
    fn start_time(&self) -> Basetime;
    #[doc = "See [`_cef_download_item_t::get_end_time`] for more documentation."]
    fn end_time(&self) -> Basetime;
    #[doc = "See [`_cef_download_item_t::get_full_path`] for more documentation."]
    fn full_path(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_download_item_t::get_id`] for more documentation."]
    fn id(&self) -> u32;
    #[doc = "See [`_cef_download_item_t::get_url`] for more documentation."]
    fn url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_download_item_t::get_original_url`] for more documentation."]
    fn original_url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_download_item_t::get_suggested_file_name`] for more documentation."]
    fn suggested_file_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_download_item_t::get_content_disposition`] for more documentation."]
    fn content_disposition(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_download_item_t::get_mime_type`] for more documentation."]
    fn mime_type(&self) -> CefStringUserfree;
    fn get_raw(&self) -> *mut _cef_download_item_t;
}
impl ImplDownloadItem for DownloadItem {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_in_progress(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_in_progress
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_complete(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_complete
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_canceled(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_canceled
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_interrupted(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_interrupted
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn interrupt_reason(&self) -> DownloadInterruptReason {
        unsafe {
            self.0
                .get_interrupt_reason
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn current_speed(&self) -> i64 {
        unsafe {
            self.0
                .get_current_speed
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn percent_complete(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_percent_complete
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn total_bytes(&self) -> i64 {
        unsafe {
            self.0
                .get_total_bytes
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn received_bytes(&self) -> i64 {
        unsafe {
            self.0
                .get_received_bytes
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn start_time(&self) -> Basetime {
        unsafe {
            self.0
                .get_start_time
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn end_time(&self) -> Basetime {
        unsafe {
            self.0
                .get_end_time
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn full_path(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_full_path
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn id(&self) -> u32 {
        unsafe {
            self.0
                .get_id
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn original_url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_original_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn suggested_file_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_suggested_file_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn content_disposition(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_content_disposition
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn mime_type(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_mime_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_download_item_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_download_item_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DownloadItem {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_download_item_t> for &DownloadItem {
    fn into_raw(self) -> *mut _cef_download_item_t {
        ImplDownloadItem::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_download_item_t> for &mut DownloadItem {
    fn into_raw(self) -> *mut _cef_download_item_t {
        ImplDownloadItem::get_raw(self)
    }
}
impl ConvertReturnValue<DownloadItem> for *mut _cef_download_item_t {
    fn wrap_result(self) -> DownloadItem {
        DownloadItem(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DownloadItem> for *mut _cef_download_item_t {
    fn from(value: DownloadItem) -> Self {
        let object = ImplDownloadItem::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DownloadItem {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_before_download_callback_t`] for more documentation.
#[derive(Clone)]
pub struct BeforeDownloadCallback(RefGuard<_cef_before_download_callback_t>);
pub trait ImplBeforeDownloadCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_before_download_callback_t::cont`] for more documentation."]
    fn cont(&self, download_path: Option<&CefString>, show_dialog: ::std::os::raw::c_int);
    fn get_raw(&self) -> *mut _cef_before_download_callback_t;
}
impl ImplBeforeDownloadCallback for BeforeDownloadCallback {
    fn cont(&self, download_path: Option<&CefString>, show_dialog: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.cont {
                let (arg_download_path, arg_show_dialog) = (download_path, show_dialog);
                let arg_self_ = self.into_raw();
                let arg_download_path = arg_download_path
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_download_path, arg_show_dialog);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_before_download_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_before_download_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for BeforeDownloadCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_before_download_callback_t> for &BeforeDownloadCallback {
    fn into_raw(self) -> *mut _cef_before_download_callback_t {
        ImplBeforeDownloadCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_before_download_callback_t> for &mut BeforeDownloadCallback {
    fn into_raw(self) -> *mut _cef_before_download_callback_t {
        ImplBeforeDownloadCallback::get_raw(self)
    }
}
impl ConvertReturnValue<BeforeDownloadCallback> for *mut _cef_before_download_callback_t {
    fn wrap_result(self) -> BeforeDownloadCallback {
        BeforeDownloadCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<BeforeDownloadCallback> for *mut _cef_before_download_callback_t {
    fn from(value: BeforeDownloadCallback) -> Self {
        let object = ImplBeforeDownloadCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for BeforeDownloadCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_download_item_callback_t`] for more documentation.
#[derive(Clone)]
pub struct DownloadItemCallback(RefGuard<_cef_download_item_callback_t>);
pub trait ImplDownloadItemCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_download_item_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    #[doc = "See [`_cef_download_item_callback_t::pause`] for more documentation."]
    fn pause(&self);
    #[doc = "See [`_cef_download_item_callback_t::resume`] for more documentation."]
    fn resume(&self);
    fn get_raw(&self) -> *mut _cef_download_item_callback_t;
}
impl ImplDownloadItemCallback for DownloadItemCallback {
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn pause(&self) {
        unsafe {
            if let Some(f) = self.0.pause {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn resume(&self) {
        unsafe {
            if let Some(f) = self.0.resume {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_download_item_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_download_item_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DownloadItemCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_download_item_callback_t> for &DownloadItemCallback {
    fn into_raw(self) -> *mut _cef_download_item_callback_t {
        ImplDownloadItemCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_download_item_callback_t> for &mut DownloadItemCallback {
    fn into_raw(self) -> *mut _cef_download_item_callback_t {
        ImplDownloadItemCallback::get_raw(self)
    }
}
impl ConvertReturnValue<DownloadItemCallback> for *mut _cef_download_item_callback_t {
    fn wrap_result(self) -> DownloadItemCallback {
        DownloadItemCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DownloadItemCallback> for *mut _cef_download_item_callback_t {
    fn from(value: DownloadItemCallback) -> Self {
        let object = ImplDownloadItemCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DownloadItemCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_download_handler_t`] for more documentation.
#[derive(Clone)]
pub struct DownloadHandler(RefGuard<_cef_download_handler_t>);
impl DownloadHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDownloadHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDownloadHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDownloadHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_download_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDownloadHandler: ImplDownloadHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_download_handler_t, Self>);
}
pub trait ImplDownloadHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_download_handler_t::can_download`] for more documentation."]
    fn can_download(
        &self,
        browser: Option<&mut Browser>,
        url: Option<&CefString>,
        request_method: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_download_handler_t::on_before_download`] for more documentation."]
    fn on_before_download(
        &self,
        browser: Option<&mut Browser>,
        download_item: Option<&mut DownloadItem>,
        suggested_name: Option<&CefString>,
        callback: Option<&mut BeforeDownloadCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_download_handler_t::on_download_updated`] for more documentation."]
    fn on_download_updated(
        &self,
        browser: Option<&mut Browser>,
        download_item: Option<&mut DownloadItem>,
        callback: Option<&mut DownloadItemCallback>,
    ) {
    }
    fn init_methods(object: &mut _cef_download_handler_t) {
        impl_cef_download_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_download_handler_t;
}
mod impl_cef_download_handler_t {
    use super::*;
    pub fn init_methods<I: ImplDownloadHandler>(object: &mut _cef_download_handler_t) {
        object.can_download = Some(can_download::<I>);
        object.on_before_download = Some(on_before_download::<I>);
        object.on_download_updated = Some(on_download_updated::<I>);
    }
    extern "C" fn can_download<I: ImplDownloadHandler>(
        self_: *mut _cef_download_handler_t,
        browser: *mut _cef_browser_t,
        url: *const cef_string_t,
        request_method: *const cef_string_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_url, arg_request_method) =
            (self_, browser, url, request_method);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_url = if arg_url.is_null() {
            None
        } else {
            Some(arg_url.into())
        };
        let arg_url = arg_url.as_ref();
        let arg_request_method = if arg_request_method.is_null() {
            None
        } else {
            Some(arg_request_method.into())
        };
        let arg_request_method = arg_request_method.as_ref();
        ImplDownloadHandler::can_download(
            &arg_self_.interface,
            arg_browser,
            arg_url,
            arg_request_method,
        )
    }
    extern "C" fn on_before_download<I: ImplDownloadHandler>(
        self_: *mut _cef_download_handler_t,
        browser: *mut _cef_browser_t,
        download_item: *mut _cef_download_item_t,
        suggested_name: *const cef_string_t,
        callback: *mut _cef_before_download_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_download_item, arg_suggested_name, arg_callback) =
            (self_, browser, download_item, suggested_name, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_download_item = unsafe { arg_download_item.as_mut() }
            .map(|arg| DownloadItem(unsafe { RefGuard::from_raw(arg) }));
        let arg_download_item = arg_download_item.as_mut();
        let arg_suggested_name = if arg_suggested_name.is_null() {
            None
        } else {
            Some(arg_suggested_name.into())
        };
        let arg_suggested_name = arg_suggested_name.as_ref();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| BeforeDownloadCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplDownloadHandler::on_before_download(
            &arg_self_.interface,
            arg_browser,
            arg_download_item,
            arg_suggested_name,
            arg_callback,
        )
    }
    extern "C" fn on_download_updated<I: ImplDownloadHandler>(
        self_: *mut _cef_download_handler_t,
        browser: *mut _cef_browser_t,
        download_item: *mut _cef_download_item_t,
        callback: *mut _cef_download_item_callback_t,
    ) {
        let (arg_self_, arg_browser, arg_download_item, arg_callback) =
            (self_, browser, download_item, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_download_item = unsafe { arg_download_item.as_mut() }
            .map(|arg| DownloadItem(unsafe { RefGuard::from_raw(arg) }));
        let arg_download_item = arg_download_item.as_mut();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| DownloadItemCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplDownloadHandler::on_download_updated(
            &arg_self_.interface,
            arg_browser,
            arg_download_item,
            arg_callback,
        )
    }
}
impl ImplDownloadHandler for DownloadHandler {
    fn can_download(
        &self,
        browser: Option<&mut Browser>,
        url: Option<&CefString>,
        request_method: Option<&CefString>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_download
                .map(|f| {
                    let (arg_browser, arg_url, arg_request_method) = (browser, url, request_method);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_url = arg_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_request_method = arg_request_method
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_browser, arg_url, arg_request_method);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_before_download(
        &self,
        browser: Option<&mut Browser>,
        download_item: Option<&mut DownloadItem>,
        suggested_name: Option<&CefString>,
        callback: Option<&mut BeforeDownloadCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_before_download
                .map(|f| {
                    let (arg_browser, arg_download_item, arg_suggested_name, arg_callback) =
                        (browser, download_item, suggested_name, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_download_item = arg_download_item
                        .map(|arg| {
                            arg.add_ref();
                            ImplDownloadItem::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_suggested_name = arg_suggested_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplBeforeDownloadCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_download_item,
                        arg_suggested_name,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_download_updated(
        &self,
        browser: Option<&mut Browser>,
        download_item: Option<&mut DownloadItem>,
        callback: Option<&mut DownloadItemCallback>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_download_updated {
                let (arg_browser, arg_download_item, arg_callback) =
                    (browser, download_item, callback);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_download_item = arg_download_item
                    .map(|arg| {
                        arg.add_ref();
                        ImplDownloadItem::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_callback = arg_callback
                    .map(|arg| {
                        arg.add_ref();
                        ImplDownloadItemCallback::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_download_item, arg_callback);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_download_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_download_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DownloadHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_download_handler_t> for &DownloadHandler {
    fn into_raw(self) -> *mut _cef_download_handler_t {
        ImplDownloadHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_download_handler_t> for &mut DownloadHandler {
    fn into_raw(self) -> *mut _cef_download_handler_t {
        ImplDownloadHandler::get_raw(self)
    }
}
impl ConvertReturnValue<DownloadHandler> for *mut _cef_download_handler_t {
    fn wrap_result(self) -> DownloadHandler {
        DownloadHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DownloadHandler> for *mut _cef_download_handler_t {
    fn from(value: DownloadHandler) -> Self {
        let object = ImplDownloadHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DownloadHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_drag_handler_t`] for more documentation.
#[derive(Clone)]
pub struct DragHandler(RefGuard<_cef_drag_handler_t>);
impl DragHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapDragHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplDragHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapDragHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_drag_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapDragHandler: ImplDragHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_drag_handler_t, Self>);
}
pub trait ImplDragHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_drag_handler_t::on_drag_enter`] for more documentation."]
    fn on_drag_enter(
        &self,
        browser: Option<&mut Browser>,
        drag_data: Option<&mut DragData>,
        mask: DragOperationsMask,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_drag_handler_t::on_draggable_regions_changed`] for more documentation."]
    fn on_draggable_regions_changed(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        regions_count: usize,
        regions: Option<&DraggableRegion>,
    ) {
    }
    fn init_methods(object: &mut _cef_drag_handler_t) {
        impl_cef_drag_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_drag_handler_t;
}
mod impl_cef_drag_handler_t {
    use super::*;
    pub fn init_methods<I: ImplDragHandler>(object: &mut _cef_drag_handler_t) {
        object.on_drag_enter = Some(on_drag_enter::<I>);
        object.on_draggable_regions_changed = Some(on_draggable_regions_changed::<I>);
    }
    extern "C" fn on_drag_enter<I: ImplDragHandler>(
        self_: *mut _cef_drag_handler_t,
        browser: *mut _cef_browser_t,
        drag_data: *mut _cef_drag_data_t,
        mask: cef_drag_operations_mask_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_drag_data, arg_mask) = (self_, browser, drag_data, mask);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_drag_data = unsafe { arg_drag_data.as_mut() }
            .map(|arg| DragData(unsafe { RefGuard::from_raw(arg) }));
        let arg_drag_data = arg_drag_data.as_mut();
        let arg_mask = arg_mask.into_raw();
        ImplDragHandler::on_drag_enter(&arg_self_.interface, arg_browser, arg_drag_data, arg_mask)
    }
    extern "C" fn on_draggable_regions_changed<I: ImplDragHandler>(
        self_: *mut _cef_drag_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        regions_count: usize,
        regions: *const _cef_draggable_region_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_regions_count, arg_regions) =
            (self_, browser, frame, regions_count, regions);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_regions_count = arg_regions_count.into_raw();
        let arg_regions = if arg_regions.is_null() {
            None
        } else {
            Some(WrapParamRef::<DraggableRegion, _>::from(arg_regions))
        };
        let arg_regions = arg_regions.as_ref().map(|arg| arg.as_ref());
        ImplDragHandler::on_draggable_regions_changed(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_regions_count,
            arg_regions,
        )
    }
}
impl ImplDragHandler for DragHandler {
    fn on_drag_enter(
        &self,
        browser: Option<&mut Browser>,
        drag_data: Option<&mut DragData>,
        mask: DragOperationsMask,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_drag_enter
                .map(|f| {
                    let (arg_browser, arg_drag_data, arg_mask) = (browser, drag_data, mask);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_drag_data = arg_drag_data
                        .map(|arg| {
                            arg.add_ref();
                            ImplDragData::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_mask = arg_mask.into_raw();
                    let result = f(arg_self_, arg_browser, arg_drag_data, arg_mask);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_draggable_regions_changed(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        regions_count: usize,
        regions: Option<&DraggableRegion>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_draggable_regions_changed {
                let (arg_browser, arg_frame, arg_regions_count, arg_regions) =
                    (browser, frame, regions_count, regions);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_regions = arg_regions.cloned().map(|arg| arg.into());
                let arg_regions = arg_regions
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_frame,
                    arg_regions_count,
                    arg_regions,
                );
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_drag_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_drag_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for DragHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_drag_handler_t> for &DragHandler {
    fn into_raw(self) -> *mut _cef_drag_handler_t {
        ImplDragHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_drag_handler_t> for &mut DragHandler {
    fn into_raw(self) -> *mut _cef_drag_handler_t {
        ImplDragHandler::get_raw(self)
    }
}
impl ConvertReturnValue<DragHandler> for *mut _cef_drag_handler_t {
    fn wrap_result(self) -> DragHandler {
        DragHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<DragHandler> for *mut _cef_drag_handler_t {
    fn from(value: DragHandler) -> Self {
        let object = ImplDragHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for DragHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_find_handler_t`] for more documentation.
#[derive(Clone)]
pub struct FindHandler(RefGuard<_cef_find_handler_t>);
impl FindHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapFindHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplFindHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapFindHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_find_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapFindHandler: ImplFindHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_find_handler_t, Self>);
}
pub trait ImplFindHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_find_handler_t::on_find_result`] for more documentation."]
    fn on_find_result(
        &self,
        browser: Option<&mut Browser>,
        identifier: ::std::os::raw::c_int,
        count: ::std::os::raw::c_int,
        selection_rect: Option<&Rect>,
        active_match_ordinal: ::std::os::raw::c_int,
        final_update: ::std::os::raw::c_int,
    ) {
    }
    fn init_methods(object: &mut _cef_find_handler_t) {
        impl_cef_find_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_find_handler_t;
}
mod impl_cef_find_handler_t {
    use super::*;
    pub fn init_methods<I: ImplFindHandler>(object: &mut _cef_find_handler_t) {
        object.on_find_result = Some(on_find_result::<I>);
    }
    extern "C" fn on_find_result<I: ImplFindHandler>(
        self_: *mut _cef_find_handler_t,
        browser: *mut _cef_browser_t,
        identifier: ::std::os::raw::c_int,
        count: ::std::os::raw::c_int,
        selection_rect: *const _cef_rect_t,
        active_match_ordinal: ::std::os::raw::c_int,
        final_update: ::std::os::raw::c_int,
    ) {
        let (
            arg_self_,
            arg_browser,
            arg_identifier,
            arg_count,
            arg_selection_rect,
            arg_active_match_ordinal,
            arg_final_update,
        ) = (
            self_,
            browser,
            identifier,
            count,
            selection_rect,
            active_match_ordinal,
            final_update,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_identifier = arg_identifier.into_raw();
        let arg_count = arg_count.into_raw();
        let arg_selection_rect = if arg_selection_rect.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_selection_rect))
        };
        let arg_selection_rect = arg_selection_rect.as_ref().map(|arg| arg.as_ref());
        let arg_active_match_ordinal = arg_active_match_ordinal.into_raw();
        let arg_final_update = arg_final_update.into_raw();
        ImplFindHandler::on_find_result(
            &arg_self_.interface,
            arg_browser,
            arg_identifier,
            arg_count,
            arg_selection_rect,
            arg_active_match_ordinal,
            arg_final_update,
        )
    }
}
impl ImplFindHandler for FindHandler {
    fn on_find_result(
        &self,
        browser: Option<&mut Browser>,
        identifier: ::std::os::raw::c_int,
        count: ::std::os::raw::c_int,
        selection_rect: Option<&Rect>,
        active_match_ordinal: ::std::os::raw::c_int,
        final_update: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_find_result {
                let (
                    arg_browser,
                    arg_identifier,
                    arg_count,
                    arg_selection_rect,
                    arg_active_match_ordinal,
                    arg_final_update,
                ) = (
                    browser,
                    identifier,
                    count,
                    selection_rect,
                    active_match_ordinal,
                    final_update,
                );
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_selection_rect = arg_selection_rect.cloned().map(|arg| arg.into());
                let arg_selection_rect = arg_selection_rect
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_identifier,
                    arg_count,
                    arg_selection_rect,
                    arg_active_match_ordinal,
                    arg_final_update,
                );
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_find_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_find_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for FindHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_find_handler_t> for &FindHandler {
    fn into_raw(self) -> *mut _cef_find_handler_t {
        ImplFindHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_find_handler_t> for &mut FindHandler {
    fn into_raw(self) -> *mut _cef_find_handler_t {
        ImplFindHandler::get_raw(self)
    }
}
impl ConvertReturnValue<FindHandler> for *mut _cef_find_handler_t {
    fn wrap_result(self) -> FindHandler {
        FindHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<FindHandler> for *mut _cef_find_handler_t {
    fn from(value: FindHandler) -> Self {
        let object = ImplFindHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for FindHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_focus_handler_t`] for more documentation.
#[derive(Clone)]
pub struct FocusHandler(RefGuard<_cef_focus_handler_t>);
impl FocusHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapFocusHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplFocusHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapFocusHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_focus_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapFocusHandler: ImplFocusHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_focus_handler_t, Self>);
}
pub trait ImplFocusHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_focus_handler_t::on_take_focus`] for more documentation."]
    fn on_take_focus(&self, browser: Option<&mut Browser>, next: ::std::os::raw::c_int) {}
    #[doc = "See [`_cef_focus_handler_t::on_set_focus`] for more documentation."]
    fn on_set_focus(
        &self,
        browser: Option<&mut Browser>,
        source: FocusSource,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_focus_handler_t::on_got_focus`] for more documentation."]
    fn on_got_focus(&self, browser: Option<&mut Browser>) {}
    fn init_methods(object: &mut _cef_focus_handler_t) {
        impl_cef_focus_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_focus_handler_t;
}
mod impl_cef_focus_handler_t {
    use super::*;
    pub fn init_methods<I: ImplFocusHandler>(object: &mut _cef_focus_handler_t) {
        object.on_take_focus = Some(on_take_focus::<I>);
        object.on_set_focus = Some(on_set_focus::<I>);
        object.on_got_focus = Some(on_got_focus::<I>);
    }
    extern "C" fn on_take_focus<I: ImplFocusHandler>(
        self_: *mut _cef_focus_handler_t,
        browser: *mut _cef_browser_t,
        next: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_next) = (self_, browser, next);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_next = arg_next.into_raw();
        ImplFocusHandler::on_take_focus(&arg_self_.interface, arg_browser, arg_next)
    }
    extern "C" fn on_set_focus<I: ImplFocusHandler>(
        self_: *mut _cef_focus_handler_t,
        browser: *mut _cef_browser_t,
        source: cef_focus_source_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_source) = (self_, browser, source);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_source = arg_source.into_raw();
        ImplFocusHandler::on_set_focus(&arg_self_.interface, arg_browser, arg_source)
    }
    extern "C" fn on_got_focus<I: ImplFocusHandler>(
        self_: *mut _cef_focus_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplFocusHandler::on_got_focus(&arg_self_.interface, arg_browser)
    }
}
impl ImplFocusHandler for FocusHandler {
    fn on_take_focus(&self, browser: Option<&mut Browser>, next: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.on_take_focus {
                let (arg_browser, arg_next) = (browser, next);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_next);
            }
        }
    }
    fn on_set_focus(
        &self,
        browser: Option<&mut Browser>,
        source: FocusSource,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_set_focus
                .map(|f| {
                    let (arg_browser, arg_source) = (browser, source);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_source = arg_source.into_raw();
                    let result = f(arg_self_, arg_browser, arg_source);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_got_focus(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_got_focus {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_focus_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_focus_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for FocusHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_focus_handler_t> for &FocusHandler {
    fn into_raw(self) -> *mut _cef_focus_handler_t {
        ImplFocusHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_focus_handler_t> for &mut FocusHandler {
    fn into_raw(self) -> *mut _cef_focus_handler_t {
        ImplFocusHandler::get_raw(self)
    }
}
impl ConvertReturnValue<FocusHandler> for *mut _cef_focus_handler_t {
    fn wrap_result(self) -> FocusHandler {
        FocusHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<FocusHandler> for *mut _cef_focus_handler_t {
    fn from(value: FocusHandler) -> Self {
        let object = ImplFocusHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for FocusHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_frame_handler_t`] for more documentation.
#[derive(Clone)]
pub struct FrameHandler(RefGuard<_cef_frame_handler_t>);
impl FrameHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapFrameHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplFrameHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapFrameHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_frame_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapFrameHandler: ImplFrameHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_frame_handler_t, Self>);
}
pub trait ImplFrameHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_frame_handler_t::on_frame_created`] for more documentation."]
    fn on_frame_created(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {}
    #[doc = "See [`_cef_frame_handler_t::on_frame_destroyed`] for more documentation."]
    fn on_frame_destroyed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {}
    #[doc = "See [`_cef_frame_handler_t::on_frame_attached`] for more documentation."]
    fn on_frame_attached(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        reattached: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_frame_handler_t::on_frame_detached`] for more documentation."]
    fn on_frame_detached(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {}
    #[doc = "See [`_cef_frame_handler_t::on_main_frame_changed`] for more documentation."]
    fn on_main_frame_changed(
        &self,
        browser: Option<&mut Browser>,
        old_frame: Option<&mut Frame>,
        new_frame: Option<&mut Frame>,
    ) {
    }
    fn init_methods(object: &mut _cef_frame_handler_t) {
        impl_cef_frame_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_frame_handler_t;
}
mod impl_cef_frame_handler_t {
    use super::*;
    pub fn init_methods<I: ImplFrameHandler>(object: &mut _cef_frame_handler_t) {
        object.on_frame_created = Some(on_frame_created::<I>);
        object.on_frame_destroyed = Some(on_frame_destroyed::<I>);
        object.on_frame_attached = Some(on_frame_attached::<I>);
        object.on_frame_detached = Some(on_frame_detached::<I>);
        object.on_main_frame_changed = Some(on_main_frame_changed::<I>);
    }
    extern "C" fn on_frame_created<I: ImplFrameHandler>(
        self_: *mut _cef_frame_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_frame) = (self_, browser, frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        ImplFrameHandler::on_frame_created(&arg_self_.interface, arg_browser, arg_frame)
    }
    extern "C" fn on_frame_destroyed<I: ImplFrameHandler>(
        self_: *mut _cef_frame_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_frame) = (self_, browser, frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        ImplFrameHandler::on_frame_destroyed(&arg_self_.interface, arg_browser, arg_frame)
    }
    extern "C" fn on_frame_attached<I: ImplFrameHandler>(
        self_: *mut _cef_frame_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        reattached: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_reattached) =
            (self_, browser, frame, reattached);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_reattached = arg_reattached.into_raw();
        ImplFrameHandler::on_frame_attached(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_reattached,
        )
    }
    extern "C" fn on_frame_detached<I: ImplFrameHandler>(
        self_: *mut _cef_frame_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_frame) = (self_, browser, frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        ImplFrameHandler::on_frame_detached(&arg_self_.interface, arg_browser, arg_frame)
    }
    extern "C" fn on_main_frame_changed<I: ImplFrameHandler>(
        self_: *mut _cef_frame_handler_t,
        browser: *mut _cef_browser_t,
        old_frame: *mut _cef_frame_t,
        new_frame: *mut _cef_frame_t,
    ) {
        let (arg_self_, arg_browser, arg_old_frame, arg_new_frame) =
            (self_, browser, old_frame, new_frame);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_old_frame =
            unsafe { arg_old_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_old_frame = arg_old_frame.as_mut();
        let mut arg_new_frame =
            unsafe { arg_new_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_new_frame = arg_new_frame.as_mut();
        ImplFrameHandler::on_main_frame_changed(
            &arg_self_.interface,
            arg_browser,
            arg_old_frame,
            arg_new_frame,
        )
    }
}
impl ImplFrameHandler for FrameHandler {
    fn on_frame_created(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {
        unsafe {
            if let Some(f) = self.0.on_frame_created {
                let (arg_browser, arg_frame) = (browser, frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame);
            }
        }
    }
    fn on_frame_destroyed(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {
        unsafe {
            if let Some(f) = self.0.on_frame_destroyed {
                let (arg_browser, arg_frame) = (browser, frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame);
            }
        }
    }
    fn on_frame_attached(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        reattached: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_frame_attached {
                let (arg_browser, arg_frame, arg_reattached) = (browser, frame, reattached);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame, arg_reattached);
            }
        }
    }
    fn on_frame_detached(&self, browser: Option<&mut Browser>, frame: Option<&mut Frame>) {
        unsafe {
            if let Some(f) = self.0.on_frame_detached {
                let (arg_browser, arg_frame) = (browser, frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame);
            }
        }
    }
    fn on_main_frame_changed(
        &self,
        browser: Option<&mut Browser>,
        old_frame: Option<&mut Frame>,
        new_frame: Option<&mut Frame>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_main_frame_changed {
                let (arg_browser, arg_old_frame, arg_new_frame) = (browser, old_frame, new_frame);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_old_frame = arg_old_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_new_frame = arg_new_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_old_frame, arg_new_frame);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_frame_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_frame_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for FrameHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_frame_handler_t> for &FrameHandler {
    fn into_raw(self) -> *mut _cef_frame_handler_t {
        ImplFrameHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_frame_handler_t> for &mut FrameHandler {
    fn into_raw(self) -> *mut _cef_frame_handler_t {
        ImplFrameHandler::get_raw(self)
    }
}
impl ConvertReturnValue<FrameHandler> for *mut _cef_frame_handler_t {
    fn wrap_result(self) -> FrameHandler {
        FrameHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<FrameHandler> for *mut _cef_frame_handler_t {
    fn from(value: FrameHandler) -> Self {
        let object = ImplFrameHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for FrameHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_jsdialog_callback_t`] for more documentation.
#[derive(Clone)]
pub struct JsdialogCallback(RefGuard<_cef_jsdialog_callback_t>);
pub trait ImplJsdialogCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_jsdialog_callback_t::cont`] for more documentation."]
    fn cont(&self, success: ::std::os::raw::c_int, user_input: Option<&CefString>);
    fn get_raw(&self) -> *mut _cef_jsdialog_callback_t;
}
impl ImplJsdialogCallback for JsdialogCallback {
    fn cont(&self, success: ::std::os::raw::c_int, user_input: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.cont {
                let (arg_success, arg_user_input) = (success, user_input);
                let arg_self_ = self.into_raw();
                let arg_user_input = arg_user_input
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_success, arg_user_input);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_jsdialog_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_jsdialog_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for JsdialogCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_jsdialog_callback_t> for &JsdialogCallback {
    fn into_raw(self) -> *mut _cef_jsdialog_callback_t {
        ImplJsdialogCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_jsdialog_callback_t> for &mut JsdialogCallback {
    fn into_raw(self) -> *mut _cef_jsdialog_callback_t {
        ImplJsdialogCallback::get_raw(self)
    }
}
impl ConvertReturnValue<JsdialogCallback> for *mut _cef_jsdialog_callback_t {
    fn wrap_result(self) -> JsdialogCallback {
        JsdialogCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<JsdialogCallback> for *mut _cef_jsdialog_callback_t {
    fn from(value: JsdialogCallback) -> Self {
        let object = ImplJsdialogCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for JsdialogCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_jsdialog_handler_t`] for more documentation.
#[derive(Clone)]
pub struct JsdialogHandler(RefGuard<_cef_jsdialog_handler_t>);
impl JsdialogHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapJsdialogHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplJsdialogHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapJsdialogHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_jsdialog_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapJsdialogHandler: ImplJsdialogHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_jsdialog_handler_t, Self>);
}
pub trait ImplJsdialogHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_jsdialog_handler_t::on_jsdialog`] for more documentation."]
    fn on_jsdialog(
        &self,
        browser: Option<&mut Browser>,
        origin_url: Option<&CefString>,
        dialog_type: JsdialogType,
        message_text: Option<&CefString>,
        default_prompt_text: Option<&CefString>,
        callback: Option<&mut JsdialogCallback>,
        suppress_message: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_jsdialog_handler_t::on_before_unload_dialog`] for more documentation."]
    fn on_before_unload_dialog(
        &self,
        browser: Option<&mut Browser>,
        message_text: Option<&CefString>,
        is_reload: ::std::os::raw::c_int,
        callback: Option<&mut JsdialogCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_jsdialog_handler_t::on_reset_dialog_state`] for more documentation."]
    fn on_reset_dialog_state(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_jsdialog_handler_t::on_dialog_closed`] for more documentation."]
    fn on_dialog_closed(&self, browser: Option<&mut Browser>) {}
    fn init_methods(object: &mut _cef_jsdialog_handler_t) {
        impl_cef_jsdialog_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_jsdialog_handler_t;
}
mod impl_cef_jsdialog_handler_t {
    use super::*;
    pub fn init_methods<I: ImplJsdialogHandler>(object: &mut _cef_jsdialog_handler_t) {
        object.on_jsdialog = Some(on_jsdialog::<I>);
        object.on_before_unload_dialog = Some(on_before_unload_dialog::<I>);
        object.on_reset_dialog_state = Some(on_reset_dialog_state::<I>);
        object.on_dialog_closed = Some(on_dialog_closed::<I>);
    }
    extern "C" fn on_jsdialog<I: ImplJsdialogHandler>(
        self_: *mut _cef_jsdialog_handler_t,
        browser: *mut _cef_browser_t,
        origin_url: *const cef_string_t,
        dialog_type: cef_jsdialog_type_t,
        message_text: *const cef_string_t,
        default_prompt_text: *const cef_string_t,
        callback: *mut _cef_jsdialog_callback_t,
        suppress_message: *mut ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_origin_url,
            arg_dialog_type,
            arg_message_text,
            arg_default_prompt_text,
            arg_callback,
            arg_suppress_message,
        ) = (
            self_,
            browser,
            origin_url,
            dialog_type,
            message_text,
            default_prompt_text,
            callback,
            suppress_message,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_origin_url = if arg_origin_url.is_null() {
            None
        } else {
            Some(arg_origin_url.into())
        };
        let arg_origin_url = arg_origin_url.as_ref();
        let arg_dialog_type = arg_dialog_type.into_raw();
        let arg_message_text = if arg_message_text.is_null() {
            None
        } else {
            Some(arg_message_text.into())
        };
        let arg_message_text = arg_message_text.as_ref();
        let arg_default_prompt_text = if arg_default_prompt_text.is_null() {
            None
        } else {
            Some(arg_default_prompt_text.into())
        };
        let arg_default_prompt_text = arg_default_prompt_text.as_ref();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| JsdialogCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        let mut arg_suppress_message = if arg_suppress_message.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_suppress_message,
            ))
        };
        let arg_suppress_message = arg_suppress_message.as_mut().map(|arg| arg.as_mut());
        ImplJsdialogHandler::on_jsdialog(
            &arg_self_.interface,
            arg_browser,
            arg_origin_url,
            arg_dialog_type,
            arg_message_text,
            arg_default_prompt_text,
            arg_callback,
            arg_suppress_message,
        )
    }
    extern "C" fn on_before_unload_dialog<I: ImplJsdialogHandler>(
        self_: *mut _cef_jsdialog_handler_t,
        browser: *mut _cef_browser_t,
        message_text: *const cef_string_t,
        is_reload: ::std::os::raw::c_int,
        callback: *mut _cef_jsdialog_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_message_text, arg_is_reload, arg_callback) =
            (self_, browser, message_text, is_reload, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_message_text = if arg_message_text.is_null() {
            None
        } else {
            Some(arg_message_text.into())
        };
        let arg_message_text = arg_message_text.as_ref();
        let arg_is_reload = arg_is_reload.into_raw();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| JsdialogCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplJsdialogHandler::on_before_unload_dialog(
            &arg_self_.interface,
            arg_browser,
            arg_message_text,
            arg_is_reload,
            arg_callback,
        )
    }
    extern "C" fn on_reset_dialog_state<I: ImplJsdialogHandler>(
        self_: *mut _cef_jsdialog_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplJsdialogHandler::on_reset_dialog_state(&arg_self_.interface, arg_browser)
    }
    extern "C" fn on_dialog_closed<I: ImplJsdialogHandler>(
        self_: *mut _cef_jsdialog_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplJsdialogHandler::on_dialog_closed(&arg_self_.interface, arg_browser)
    }
}
impl ImplJsdialogHandler for JsdialogHandler {
    fn on_jsdialog(
        &self,
        browser: Option<&mut Browser>,
        origin_url: Option<&CefString>,
        dialog_type: JsdialogType,
        message_text: Option<&CefString>,
        default_prompt_text: Option<&CefString>,
        callback: Option<&mut JsdialogCallback>,
        suppress_message: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_jsdialog
                .map(|f| {
                    let (
                        arg_browser,
                        arg_origin_url,
                        arg_dialog_type,
                        arg_message_text,
                        arg_default_prompt_text,
                        arg_callback,
                        arg_suppress_message,
                    ) = (
                        browser,
                        origin_url,
                        dialog_type,
                        message_text,
                        default_prompt_text,
                        callback,
                        suppress_message,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_origin_url = arg_origin_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_dialog_type = arg_dialog_type.into_raw();
                    let arg_message_text = arg_message_text
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_default_prompt_text = arg_default_prompt_text
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplJsdialogCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_suppress_message = arg_suppress_message
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_origin_url,
                        arg_dialog_type,
                        arg_message_text,
                        arg_default_prompt_text,
                        arg_callback,
                        arg_suppress_message,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_before_unload_dialog(
        &self,
        browser: Option<&mut Browser>,
        message_text: Option<&CefString>,
        is_reload: ::std::os::raw::c_int,
        callback: Option<&mut JsdialogCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_before_unload_dialog
                .map(|f| {
                    let (arg_browser, arg_message_text, arg_is_reload, arg_callback) =
                        (browser, message_text, is_reload, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_message_text = arg_message_text
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplJsdialogCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_message_text,
                        arg_is_reload,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_reset_dialog_state(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_reset_dialog_state {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn on_dialog_closed(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_dialog_closed {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_jsdialog_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_jsdialog_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for JsdialogHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_jsdialog_handler_t> for &JsdialogHandler {
    fn into_raw(self) -> *mut _cef_jsdialog_handler_t {
        ImplJsdialogHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_jsdialog_handler_t> for &mut JsdialogHandler {
    fn into_raw(self) -> *mut _cef_jsdialog_handler_t {
        ImplJsdialogHandler::get_raw(self)
    }
}
impl ConvertReturnValue<JsdialogHandler> for *mut _cef_jsdialog_handler_t {
    fn wrap_result(self) -> JsdialogHandler {
        JsdialogHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<JsdialogHandler> for *mut _cef_jsdialog_handler_t {
    fn from(value: JsdialogHandler) -> Self {
        let object = ImplJsdialogHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for JsdialogHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_keyboard_handler_t`] for more documentation.
#[derive(Clone)]
pub struct KeyboardHandler(RefGuard<_cef_keyboard_handler_t>);
impl KeyboardHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapKeyboardHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplKeyboardHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapKeyboardHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_keyboard_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapKeyboardHandler: ImplKeyboardHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_keyboard_handler_t, Self>);
}
pub trait ImplKeyboardHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_keyboard_handler_t::on_pre_key_event`] for more documentation."]
    fn on_pre_key_event(
        &self,
        browser: Option<&mut Browser>,
        event: Option<&KeyEvent>,
        os_event: Option<&mut XEvent>,
        is_keyboard_shortcut: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_keyboard_handler_t::on_key_event`] for more documentation."]
    fn on_key_event(
        &self,
        browser: Option<&mut Browser>,
        event: Option<&KeyEvent>,
        os_event: Option<&mut XEvent>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_keyboard_handler_t) {
        impl_cef_keyboard_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_keyboard_handler_t;
}
mod impl_cef_keyboard_handler_t {
    use super::*;
    pub fn init_methods<I: ImplKeyboardHandler>(object: &mut _cef_keyboard_handler_t) {
        object.on_pre_key_event = Some(on_pre_key_event::<I>);
        object.on_key_event = Some(on_key_event::<I>);
    }
    extern "C" fn on_pre_key_event<I: ImplKeyboardHandler>(
        self_: *mut _cef_keyboard_handler_t,
        browser: *mut _cef_browser_t,
        event: *const _cef_key_event_t,
        os_event: *mut XEvent,
        is_keyboard_shortcut: *mut ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_event, arg_os_event, arg_is_keyboard_shortcut) =
            (self_, browser, event, os_event, is_keyboard_shortcut);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_event = if arg_event.is_null() {
            None
        } else {
            Some(WrapParamRef::<KeyEvent, _>::from(arg_event))
        };
        let arg_event = arg_event.as_ref().map(|arg| arg.as_ref());
        let mut arg_os_event = if arg_os_event.is_null() {
            None
        } else {
            Some(WrapParamRef::<XEvent, _>::from(arg_os_event))
        };
        let arg_os_event = arg_os_event.as_mut().map(|arg| arg.as_mut());
        let mut arg_is_keyboard_shortcut = if arg_is_keyboard_shortcut.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_is_keyboard_shortcut,
            ))
        };
        let arg_is_keyboard_shortcut = arg_is_keyboard_shortcut.as_mut().map(|arg| arg.as_mut());
        ImplKeyboardHandler::on_pre_key_event(
            &arg_self_.interface,
            arg_browser,
            arg_event,
            arg_os_event,
            arg_is_keyboard_shortcut,
        )
    }
    extern "C" fn on_key_event<I: ImplKeyboardHandler>(
        self_: *mut _cef_keyboard_handler_t,
        browser: *mut _cef_browser_t,
        event: *const _cef_key_event_t,
        os_event: *mut XEvent,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_event, arg_os_event) = (self_, browser, event, os_event);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_event = if arg_event.is_null() {
            None
        } else {
            Some(WrapParamRef::<KeyEvent, _>::from(arg_event))
        };
        let arg_event = arg_event.as_ref().map(|arg| arg.as_ref());
        let mut arg_os_event = if arg_os_event.is_null() {
            None
        } else {
            Some(WrapParamRef::<XEvent, _>::from(arg_os_event))
        };
        let arg_os_event = arg_os_event.as_mut().map(|arg| arg.as_mut());
        ImplKeyboardHandler::on_key_event(
            &arg_self_.interface,
            arg_browser,
            arg_event,
            arg_os_event,
        )
    }
}
impl ImplKeyboardHandler for KeyboardHandler {
    fn on_pre_key_event(
        &self,
        browser: Option<&mut Browser>,
        event: Option<&KeyEvent>,
        os_event: Option<&mut XEvent>,
        is_keyboard_shortcut: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_pre_key_event
                .map(|f| {
                    let (arg_browser, arg_event, arg_os_event, arg_is_keyboard_shortcut) =
                        (browser, event, os_event, is_keyboard_shortcut);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_event = arg_event.cloned().map(|arg| arg.into());
                    let arg_event = arg_event
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_os_event = arg_os_event
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_is_keyboard_shortcut = arg_is_keyboard_shortcut
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_event,
                        arg_os_event,
                        arg_is_keyboard_shortcut,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_key_event(
        &self,
        browser: Option<&mut Browser>,
        event: Option<&KeyEvent>,
        os_event: Option<&mut XEvent>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_key_event
                .map(|f| {
                    let (arg_browser, arg_event, arg_os_event) = (browser, event, os_event);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_event = arg_event.cloned().map(|arg| arg.into());
                    let arg_event = arg_event
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let arg_os_event = arg_os_event
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_event, arg_os_event);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_keyboard_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_keyboard_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for KeyboardHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_keyboard_handler_t> for &KeyboardHandler {
    fn into_raw(self) -> *mut _cef_keyboard_handler_t {
        ImplKeyboardHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_keyboard_handler_t> for &mut KeyboardHandler {
    fn into_raw(self) -> *mut _cef_keyboard_handler_t {
        ImplKeyboardHandler::get_raw(self)
    }
}
impl ConvertReturnValue<KeyboardHandler> for *mut _cef_keyboard_handler_t {
    fn wrap_result(self) -> KeyboardHandler {
        KeyboardHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<KeyboardHandler> for *mut _cef_keyboard_handler_t {
    fn from(value: KeyboardHandler) -> Self {
        let object = ImplKeyboardHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for KeyboardHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_life_span_handler_t`] for more documentation.
#[derive(Clone)]
pub struct LifeSpanHandler(RefGuard<_cef_life_span_handler_t>);
impl LifeSpanHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapLifeSpanHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplLifeSpanHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapLifeSpanHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_life_span_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapLifeSpanHandler: ImplLifeSpanHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_life_span_handler_t, Self>);
}
pub trait ImplLifeSpanHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_life_span_handler_t::on_before_popup`] for more documentation."]
    fn on_before_popup(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        popup_id: ::std::os::raw::c_int,
        target_url: Option<&CefString>,
        target_frame_name: Option<&CefString>,
        target_disposition: WindowOpenDisposition,
        user_gesture: ::std::os::raw::c_int,
        popup_features: Option<&PopupFeatures>,
        window_info: Option<&mut WindowInfo>,
        client: Option<&mut Option<impl ImplClient>>,
        settings: Option<&mut BrowserSettings>,
        extra_info: Option<&mut Option<DictionaryValue>>,
        no_javascript_access: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_life_span_handler_t::on_before_popup_aborted`] for more documentation."]
    fn on_before_popup_aborted(
        &self,
        browser: Option<&mut Browser>,
        popup_id: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_life_span_handler_t::on_before_dev_tools_popup`] for more documentation."]
    fn on_before_dev_tools_popup(
        &self,
        browser: Option<&mut Browser>,
        window_info: Option<&mut WindowInfo>,
        client: Option<&mut Option<impl ImplClient>>,
        settings: Option<&mut BrowserSettings>,
        extra_info: Option<&mut Option<DictionaryValue>>,
        use_default_window: Option<&mut ::std::os::raw::c_int>,
    ) {
    }
    #[doc = "See [`_cef_life_span_handler_t::on_after_created`] for more documentation."]
    fn on_after_created(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_life_span_handler_t::do_close`] for more documentation."]
    fn do_close(&self, browser: Option<&mut Browser>) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_life_span_handler_t::on_before_close`] for more documentation."]
    fn on_before_close(&self, browser: Option<&mut Browser>) {}
    fn init_methods(object: &mut _cef_life_span_handler_t) {
        impl_cef_life_span_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_life_span_handler_t;
}
mod impl_cef_life_span_handler_t {
    use super::*;
    pub fn init_methods<I: ImplLifeSpanHandler>(object: &mut _cef_life_span_handler_t) {
        object.on_before_popup = Some(on_before_popup::<I>);
        object.on_before_popup_aborted = Some(on_before_popup_aborted::<I>);
        object.on_before_dev_tools_popup = Some(on_before_dev_tools_popup::<I>);
        object.on_after_created = Some(on_after_created::<I>);
        object.do_close = Some(do_close::<I>);
        object.on_before_close = Some(on_before_close::<I>);
    }
    extern "C" fn on_before_popup<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        popup_id: ::std::os::raw::c_int,
        target_url: *const cef_string_t,
        target_frame_name: *const cef_string_t,
        target_disposition: cef_window_open_disposition_t,
        user_gesture: ::std::os::raw::c_int,
        popup_features: *const _cef_popup_features_t,
        window_info: *mut _cef_window_info_t,
        client: *mut *mut _cef_client_t,
        settings: *mut _cef_browser_settings_t,
        extra_info: *mut *mut _cef_dictionary_value_t,
        no_javascript_access: *mut ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_frame,
            arg_popup_id,
            arg_target_url,
            arg_target_frame_name,
            arg_target_disposition,
            arg_user_gesture,
            arg_popup_features,
            arg_window_info,
            arg_client,
            arg_settings,
            arg_extra_info,
            arg_no_javascript_access,
        ) = (
            self_,
            browser,
            frame,
            popup_id,
            target_url,
            target_frame_name,
            target_disposition,
            user_gesture,
            popup_features,
            window_info,
            client,
            settings,
            extra_info,
            no_javascript_access,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_popup_id = arg_popup_id.into_raw();
        let arg_target_url = if arg_target_url.is_null() {
            None
        } else {
            Some(arg_target_url.into())
        };
        let arg_target_url = arg_target_url.as_ref();
        let arg_target_frame_name = if arg_target_frame_name.is_null() {
            None
        } else {
            Some(arg_target_frame_name.into())
        };
        let arg_target_frame_name = arg_target_frame_name.as_ref();
        let arg_target_disposition = arg_target_disposition.into_raw();
        let arg_user_gesture = arg_user_gesture.into_raw();
        let arg_popup_features = if arg_popup_features.is_null() {
            None
        } else {
            Some(WrapParamRef::<PopupFeatures, _>::from(arg_popup_features))
        };
        let arg_popup_features = arg_popup_features.as_ref().map(|arg| arg.as_ref());
        let mut arg_window_info = if arg_window_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<WindowInfo, _>::from(arg_window_info))
        };
        let arg_window_info = arg_window_info.as_mut().map(|arg| arg.as_mut());
        let mut arg_client = unsafe { arg_client.as_mut() }.and_then(|ptr| {
            if ptr.is_null() {
                None
            } else {
                Some(Client(unsafe { RefGuard::from_raw(*ptr) }))
            }
        });
        let arg_client = Some(&mut arg_client);
        let mut arg_settings = if arg_settings.is_null() {
            None
        } else {
            Some(WrapParamRef::<BrowserSettings, _>::from(arg_settings))
        };
        let arg_settings = arg_settings.as_mut().map(|arg| arg.as_mut());
        let mut arg_extra_info = unsafe { arg_extra_info.as_mut() }.and_then(|ptr| {
            if ptr.is_null() {
                None
            } else {
                Some(DictionaryValue(unsafe { RefGuard::from_raw(*ptr) }))
            }
        });
        let arg_extra_info = Some(&mut arg_extra_info);
        let mut arg_no_javascript_access = if arg_no_javascript_access.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_no_javascript_access,
            ))
        };
        let arg_no_javascript_access = arg_no_javascript_access.as_mut().map(|arg| arg.as_mut());
        ImplLifeSpanHandler::on_before_popup(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_popup_id,
            arg_target_url,
            arg_target_frame_name,
            arg_target_disposition,
            arg_user_gesture,
            arg_popup_features,
            arg_window_info,
            arg_client,
            arg_settings,
            arg_extra_info,
            arg_no_javascript_access,
        )
    }
    extern "C" fn on_before_popup_aborted<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
        popup_id: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_popup_id) = (self_, browser, popup_id);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_popup_id = arg_popup_id.into_raw();
        ImplLifeSpanHandler::on_before_popup_aborted(
            &arg_self_.interface,
            arg_browser,
            arg_popup_id,
        )
    }
    extern "C" fn on_before_dev_tools_popup<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
        window_info: *mut _cef_window_info_t,
        client: *mut *mut _cef_client_t,
        settings: *mut _cef_browser_settings_t,
        extra_info: *mut *mut _cef_dictionary_value_t,
        use_default_window: *mut ::std::os::raw::c_int,
    ) {
        let (
            arg_self_,
            arg_browser,
            arg_window_info,
            arg_client,
            arg_settings,
            arg_extra_info,
            arg_use_default_window,
        ) = (
            self_,
            browser,
            window_info,
            client,
            settings,
            extra_info,
            use_default_window,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_window_info = if arg_window_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<WindowInfo, _>::from(arg_window_info))
        };
        let arg_window_info = arg_window_info.as_mut().map(|arg| arg.as_mut());
        let mut arg_client = unsafe { arg_client.as_mut() }.and_then(|ptr| {
            if ptr.is_null() {
                None
            } else {
                Some(Client(unsafe { RefGuard::from_raw(*ptr) }))
            }
        });
        let arg_client = Some(&mut arg_client);
        let mut arg_settings = if arg_settings.is_null() {
            None
        } else {
            Some(WrapParamRef::<BrowserSettings, _>::from(arg_settings))
        };
        let arg_settings = arg_settings.as_mut().map(|arg| arg.as_mut());
        let mut arg_extra_info = unsafe { arg_extra_info.as_mut() }.and_then(|ptr| {
            if ptr.is_null() {
                None
            } else {
                Some(DictionaryValue(unsafe { RefGuard::from_raw(*ptr) }))
            }
        });
        let arg_extra_info = Some(&mut arg_extra_info);
        let mut arg_use_default_window = if arg_use_default_window.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_use_default_window,
            ))
        };
        let arg_use_default_window = arg_use_default_window.as_mut().map(|arg| arg.as_mut());
        ImplLifeSpanHandler::on_before_dev_tools_popup(
            &arg_self_.interface,
            arg_browser,
            arg_window_info,
            arg_client,
            arg_settings,
            arg_extra_info,
            arg_use_default_window,
        )
    }
    extern "C" fn on_after_created<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplLifeSpanHandler::on_after_created(&arg_self_.interface, arg_browser)
    }
    extern "C" fn do_close<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplLifeSpanHandler::do_close(&arg_self_.interface, arg_browser)
    }
    extern "C" fn on_before_close<I: ImplLifeSpanHandler>(
        self_: *mut _cef_life_span_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplLifeSpanHandler::on_before_close(&arg_self_.interface, arg_browser)
    }
}
impl ImplLifeSpanHandler for LifeSpanHandler {
    fn on_before_popup(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        popup_id: ::std::os::raw::c_int,
        target_url: Option<&CefString>,
        target_frame_name: Option<&CefString>,
        target_disposition: WindowOpenDisposition,
        user_gesture: ::std::os::raw::c_int,
        popup_features: Option<&PopupFeatures>,
        window_info: Option<&mut WindowInfo>,
        client: Option<&mut Option<impl ImplClient>>,
        settings: Option<&mut BrowserSettings>,
        extra_info: Option<&mut Option<DictionaryValue>>,
        no_javascript_access: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_before_popup
                .map(|f| {
                    let (
                        arg_browser,
                        arg_frame,
                        arg_popup_id,
                        arg_target_url,
                        arg_target_frame_name,
                        arg_target_disposition,
                        arg_user_gesture,
                        arg_popup_features,
                        arg_window_info,
                        arg_client,
                        arg_settings,
                        arg_extra_info,
                        arg_no_javascript_access,
                    ) = (
                        browser,
                        frame,
                        popup_id,
                        target_url,
                        target_frame_name,
                        target_disposition,
                        user_gesture,
                        popup_features,
                        window_info,
                        client,
                        settings,
                        extra_info,
                        no_javascript_access,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_target_url = arg_target_url
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_target_frame_name = arg_target_frame_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_target_disposition = arg_target_disposition.into_raw();
                    let arg_popup_features = arg_popup_features.cloned().map(|arg| arg.into());
                    let arg_popup_features = arg_popup_features
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let mut arg_window_info = arg_window_info.cloned().map(|arg| arg.into());
                    let arg_window_info = arg_window_info
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let mut ptr = std::ptr::null_mut();
                    let arg_client = arg_client
                        .map(|arg| {
                            if let Some(arg) = arg.as_mut() {
                                arg.add_ref();
                                ptr = arg.get_raw();
                            }
                            std::ptr::from_mut(&mut ptr)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let mut arg_settings = arg_settings.cloned().map(|arg| arg.into());
                    let arg_settings = arg_settings
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let mut ptr = std::ptr::null_mut();
                    let arg_extra_info = arg_extra_info
                        .map(|arg| {
                            if let Some(arg) = arg.as_mut() {
                                arg.add_ref();
                                ptr = arg.get_raw();
                            }
                            std::ptr::from_mut(&mut ptr)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_no_javascript_access = arg_no_javascript_access
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_popup_id,
                        arg_target_url,
                        arg_target_frame_name,
                        arg_target_disposition,
                        arg_user_gesture,
                        arg_popup_features,
                        arg_window_info,
                        arg_client,
                        arg_settings,
                        arg_extra_info,
                        arg_no_javascript_access,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_before_popup_aborted(
        &self,
        browser: Option<&mut Browser>,
        popup_id: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_before_popup_aborted {
                let (arg_browser, arg_popup_id) = (browser, popup_id);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_popup_id);
            }
        }
    }
    fn on_before_dev_tools_popup(
        &self,
        browser: Option<&mut Browser>,
        window_info: Option<&mut WindowInfo>,
        client: Option<&mut Option<impl ImplClient>>,
        settings: Option<&mut BrowserSettings>,
        extra_info: Option<&mut Option<DictionaryValue>>,
        use_default_window: Option<&mut ::std::os::raw::c_int>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_before_dev_tools_popup {
                let (
                    arg_browser,
                    arg_window_info,
                    arg_client,
                    arg_settings,
                    arg_extra_info,
                    arg_use_default_window,
                ) = (
                    browser,
                    window_info,
                    client,
                    settings,
                    extra_info,
                    use_default_window,
                );
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let mut arg_window_info = arg_window_info.cloned().map(|arg| arg.into());
                let arg_window_info = arg_window_info
                    .as_mut()
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                let mut ptr = std::ptr::null_mut();
                let arg_client = arg_client
                    .map(|arg| {
                        if let Some(arg) = arg.as_mut() {
                            arg.add_ref();
                            ptr = arg.get_raw();
                        }
                        std::ptr::from_mut(&mut ptr)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let mut arg_settings = arg_settings.cloned().map(|arg| arg.into());
                let arg_settings = arg_settings
                    .as_mut()
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                let mut ptr = std::ptr::null_mut();
                let arg_extra_info = arg_extra_info
                    .map(|arg| {
                        if let Some(arg) = arg.as_mut() {
                            arg.add_ref();
                            ptr = arg.get_raw();
                        }
                        std::ptr::from_mut(&mut ptr)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_use_default_window = arg_use_default_window
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_browser,
                    arg_window_info,
                    arg_client,
                    arg_settings,
                    arg_extra_info,
                    arg_use_default_window,
                );
            }
        }
    }
    fn on_after_created(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_after_created {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn do_close(&self, browser: Option<&mut Browser>) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .do_close
                .map(|f| {
                    let arg_browser = browser;
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_before_close(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_before_close {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_life_span_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_life_span_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for LifeSpanHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_life_span_handler_t> for &LifeSpanHandler {
    fn into_raw(self) -> *mut _cef_life_span_handler_t {
        ImplLifeSpanHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_life_span_handler_t> for &mut LifeSpanHandler {
    fn into_raw(self) -> *mut _cef_life_span_handler_t {
        ImplLifeSpanHandler::get_raw(self)
    }
}
impl ConvertReturnValue<LifeSpanHandler> for *mut _cef_life_span_handler_t {
    fn wrap_result(self) -> LifeSpanHandler {
        LifeSpanHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<LifeSpanHandler> for *mut _cef_life_span_handler_t {
    fn from(value: LifeSpanHandler) -> Self {
        let object = ImplLifeSpanHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for LifeSpanHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_load_handler_t`] for more documentation.
#[derive(Clone)]
pub struct LoadHandler(RefGuard<_cef_load_handler_t>);
impl LoadHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapLoadHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplLoadHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapLoadHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_load_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapLoadHandler: ImplLoadHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_load_handler_t, Self>);
}
pub trait ImplLoadHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_load_handler_t::on_loading_state_change`] for more documentation."]
    fn on_loading_state_change(
        &self,
        browser: Option<&mut Browser>,
        is_loading: ::std::os::raw::c_int,
        can_go_back: ::std::os::raw::c_int,
        can_go_forward: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_load_handler_t::on_load_start`] for more documentation."]
    fn on_load_start(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        transition_type: TransitionType,
    ) {
    }
    #[doc = "See [`_cef_load_handler_t::on_load_end`] for more documentation."]
    fn on_load_end(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        http_status_code: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_load_handler_t::on_load_error`] for more documentation."]
    fn on_load_error(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        error_code: Errorcode,
        error_text: Option<&CefString>,
        failed_url: Option<&CefString>,
    ) {
    }
    fn init_methods(object: &mut _cef_load_handler_t) {
        impl_cef_load_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_load_handler_t;
}
mod impl_cef_load_handler_t {
    use super::*;
    pub fn init_methods<I: ImplLoadHandler>(object: &mut _cef_load_handler_t) {
        object.on_loading_state_change = Some(on_loading_state_change::<I>);
        object.on_load_start = Some(on_load_start::<I>);
        object.on_load_end = Some(on_load_end::<I>);
        object.on_load_error = Some(on_load_error::<I>);
    }
    extern "C" fn on_loading_state_change<I: ImplLoadHandler>(
        self_: *mut _cef_load_handler_t,
        browser: *mut _cef_browser_t,
        is_loading: ::std::os::raw::c_int,
        can_go_back: ::std::os::raw::c_int,
        can_go_forward: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_is_loading, arg_can_go_back, arg_can_go_forward) =
            (self_, browser, is_loading, can_go_back, can_go_forward);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_is_loading = arg_is_loading.into_raw();
        let arg_can_go_back = arg_can_go_back.into_raw();
        let arg_can_go_forward = arg_can_go_forward.into_raw();
        ImplLoadHandler::on_loading_state_change(
            &arg_self_.interface,
            arg_browser,
            arg_is_loading,
            arg_can_go_back,
            arg_can_go_forward,
        )
    }
    extern "C" fn on_load_start<I: ImplLoadHandler>(
        self_: *mut _cef_load_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        transition_type: cef_transition_type_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_transition_type) =
            (self_, browser, frame, transition_type);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_transition_type = arg_transition_type.into_raw();
        ImplLoadHandler::on_load_start(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_transition_type,
        )
    }
    extern "C" fn on_load_end<I: ImplLoadHandler>(
        self_: *mut _cef_load_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        http_status_code: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_http_status_code) =
            (self_, browser, frame, http_status_code);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_http_status_code = arg_http_status_code.into_raw();
        ImplLoadHandler::on_load_end(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_http_status_code,
        )
    }
    extern "C" fn on_load_error<I: ImplLoadHandler>(
        self_: *mut _cef_load_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        error_code: cef_errorcode_t,
        error_text: *const cef_string_t,
        failed_url: *const cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_error_code, arg_error_text, arg_failed_url) =
            (self_, browser, frame, error_code, error_text, failed_url);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_error_code = arg_error_code.into_raw();
        let arg_error_text = if arg_error_text.is_null() {
            None
        } else {
            Some(arg_error_text.into())
        };
        let arg_error_text = arg_error_text.as_ref();
        let arg_failed_url = if arg_failed_url.is_null() {
            None
        } else {
            Some(arg_failed_url.into())
        };
        let arg_failed_url = arg_failed_url.as_ref();
        ImplLoadHandler::on_load_error(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_error_code,
            arg_error_text,
            arg_failed_url,
        )
    }
}
impl ImplLoadHandler for LoadHandler {
    fn on_loading_state_change(
        &self,
        browser: Option<&mut Browser>,
        is_loading: ::std::os::raw::c_int,
        can_go_back: ::std::os::raw::c_int,
        can_go_forward: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_loading_state_change {
                let (arg_browser, arg_is_loading, arg_can_go_back, arg_can_go_forward) =
                    (browser, is_loading, can_go_back, can_go_forward);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_browser,
                    arg_is_loading,
                    arg_can_go_back,
                    arg_can_go_forward,
                );
            }
        }
    }
    fn on_load_start(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        transition_type: TransitionType,
    ) {
        unsafe {
            if let Some(f) = self.0.on_load_start {
                let (arg_browser, arg_frame, arg_transition_type) =
                    (browser, frame, transition_type);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_transition_type = arg_transition_type.into_raw();
                f(arg_self_, arg_browser, arg_frame, arg_transition_type);
            }
        }
    }
    fn on_load_end(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        http_status_code: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_load_end {
                let (arg_browser, arg_frame, arg_http_status_code) =
                    (browser, frame, http_status_code);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_frame, arg_http_status_code);
            }
        }
    }
    fn on_load_error(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        error_code: Errorcode,
        error_text: Option<&CefString>,
        failed_url: Option<&CefString>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_load_error {
                let (arg_browser, arg_frame, arg_error_code, arg_error_text, arg_failed_url) =
                    (browser, frame, error_code, error_text, failed_url);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_error_code = arg_error_code.into_raw();
                let arg_error_text = arg_error_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_failed_url = arg_failed_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_frame,
                    arg_error_code,
                    arg_error_text,
                    arg_failed_url,
                );
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_load_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_load_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for LoadHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_load_handler_t> for &LoadHandler {
    fn into_raw(self) -> *mut _cef_load_handler_t {
        ImplLoadHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_load_handler_t> for &mut LoadHandler {
    fn into_raw(self) -> *mut _cef_load_handler_t {
        ImplLoadHandler::get_raw(self)
    }
}
impl ConvertReturnValue<LoadHandler> for *mut _cef_load_handler_t {
    fn wrap_result(self) -> LoadHandler {
        LoadHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<LoadHandler> for *mut _cef_load_handler_t {
    fn from(value: LoadHandler) -> Self {
        let object = ImplLoadHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for LoadHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_media_access_callback_t`] for more documentation.
#[derive(Clone)]
pub struct MediaAccessCallback(RefGuard<_cef_media_access_callback_t>);
pub trait ImplMediaAccessCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_media_access_callback_t::cont`] for more documentation."]
    fn cont(&self, allowed_permissions: u32);
    #[doc = "See [`_cef_media_access_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_media_access_callback_t;
}
impl ImplMediaAccessCallback for MediaAccessCallback {
    fn cont(&self, allowed_permissions: u32) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_allowed_permissions = allowed_permissions;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_allowed_permissions);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_media_access_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_media_access_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for MediaAccessCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_media_access_callback_t> for &MediaAccessCallback {
    fn into_raw(self) -> *mut _cef_media_access_callback_t {
        ImplMediaAccessCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_media_access_callback_t> for &mut MediaAccessCallback {
    fn into_raw(self) -> *mut _cef_media_access_callback_t {
        ImplMediaAccessCallback::get_raw(self)
    }
}
impl ConvertReturnValue<MediaAccessCallback> for *mut _cef_media_access_callback_t {
    fn wrap_result(self) -> MediaAccessCallback {
        MediaAccessCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<MediaAccessCallback> for *mut _cef_media_access_callback_t {
    fn from(value: MediaAccessCallback) -> Self {
        let object = ImplMediaAccessCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for MediaAccessCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_permission_prompt_callback_t`] for more documentation.
#[derive(Clone)]
pub struct PermissionPromptCallback(RefGuard<_cef_permission_prompt_callback_t>);
pub trait ImplPermissionPromptCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_permission_prompt_callback_t::cont`] for more documentation."]
    fn cont(&self, result: PermissionRequestResult);
    fn get_raw(&self) -> *mut _cef_permission_prompt_callback_t;
}
impl ImplPermissionPromptCallback for PermissionPromptCallback {
    fn cont(&self, result: PermissionRequestResult) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_result = result;
                let arg_self_ = self.into_raw();
                let arg_result = arg_result.into_raw();
                f(arg_self_, arg_result);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_permission_prompt_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_permission_prompt_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PermissionPromptCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_permission_prompt_callback_t> for &PermissionPromptCallback {
    fn into_raw(self) -> *mut _cef_permission_prompt_callback_t {
        ImplPermissionPromptCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_permission_prompt_callback_t> for &mut PermissionPromptCallback {
    fn into_raw(self) -> *mut _cef_permission_prompt_callback_t {
        ImplPermissionPromptCallback::get_raw(self)
    }
}
impl ConvertReturnValue<PermissionPromptCallback> for *mut _cef_permission_prompt_callback_t {
    fn wrap_result(self) -> PermissionPromptCallback {
        PermissionPromptCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PermissionPromptCallback> for *mut _cef_permission_prompt_callback_t {
    fn from(value: PermissionPromptCallback) -> Self {
        let object = ImplPermissionPromptCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PermissionPromptCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_permission_handler_t`] for more documentation.
#[derive(Clone)]
pub struct PermissionHandler(RefGuard<_cef_permission_handler_t>);
impl PermissionHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapPermissionHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplPermissionHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapPermissionHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_permission_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapPermissionHandler: ImplPermissionHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_permission_handler_t, Self>);
}
pub trait ImplPermissionHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_permission_handler_t::on_request_media_access_permission`] for more documentation."]
    fn on_request_media_access_permission(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        requesting_origin: Option<&CefString>,
        requested_permissions: u32,
        callback: Option<&mut MediaAccessCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_permission_handler_t::on_show_permission_prompt`] for more documentation."]
    fn on_show_permission_prompt(
        &self,
        browser: Option<&mut Browser>,
        prompt_id: u64,
        requesting_origin: Option<&CefString>,
        requested_permissions: u32,
        callback: Option<&mut PermissionPromptCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_permission_handler_t::on_dismiss_permission_prompt`] for more documentation."]
    fn on_dismiss_permission_prompt(
        &self,
        browser: Option<&mut Browser>,
        prompt_id: u64,
        result: PermissionRequestResult,
    ) {
    }
    fn init_methods(object: &mut _cef_permission_handler_t) {
        impl_cef_permission_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_permission_handler_t;
}
mod impl_cef_permission_handler_t {
    use super::*;
    pub fn init_methods<I: ImplPermissionHandler>(object: &mut _cef_permission_handler_t) {
        object.on_request_media_access_permission = Some(on_request_media_access_permission::<I>);
        object.on_show_permission_prompt = Some(on_show_permission_prompt::<I>);
        object.on_dismiss_permission_prompt = Some(on_dismiss_permission_prompt::<I>);
    }
    extern "C" fn on_request_media_access_permission<I: ImplPermissionHandler>(
        self_: *mut _cef_permission_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        requesting_origin: *const cef_string_t,
        requested_permissions: u32,
        callback: *mut _cef_media_access_callback_t,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_frame,
            arg_requesting_origin,
            arg_requested_permissions,
            arg_callback,
        ) = (
            self_,
            browser,
            frame,
            requesting_origin,
            requested_permissions,
            callback,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let arg_requesting_origin = if arg_requesting_origin.is_null() {
            None
        } else {
            Some(arg_requesting_origin.into())
        };
        let arg_requesting_origin = arg_requesting_origin.as_ref();
        let arg_requested_permissions = arg_requested_permissions.into_raw();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| MediaAccessCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplPermissionHandler::on_request_media_access_permission(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_requesting_origin,
            arg_requested_permissions,
            arg_callback,
        )
    }
    extern "C" fn on_show_permission_prompt<I: ImplPermissionHandler>(
        self_: *mut _cef_permission_handler_t,
        browser: *mut _cef_browser_t,
        prompt_id: u64,
        requesting_origin: *const cef_string_t,
        requested_permissions: u32,
        callback: *mut _cef_permission_prompt_callback_t,
    ) -> ::std::os::raw::c_int {
        let (
            arg_self_,
            arg_browser,
            arg_prompt_id,
            arg_requesting_origin,
            arg_requested_permissions,
            arg_callback,
        ) = (
            self_,
            browser,
            prompt_id,
            requesting_origin,
            requested_permissions,
            callback,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_prompt_id = arg_prompt_id.into_raw();
        let arg_requesting_origin = if arg_requesting_origin.is_null() {
            None
        } else {
            Some(arg_requesting_origin.into())
        };
        let arg_requesting_origin = arg_requesting_origin.as_ref();
        let arg_requested_permissions = arg_requested_permissions.into_raw();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| PermissionPromptCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplPermissionHandler::on_show_permission_prompt(
            &arg_self_.interface,
            arg_browser,
            arg_prompt_id,
            arg_requesting_origin,
            arg_requested_permissions,
            arg_callback,
        )
    }
    extern "C" fn on_dismiss_permission_prompt<I: ImplPermissionHandler>(
        self_: *mut _cef_permission_handler_t,
        browser: *mut _cef_browser_t,
        prompt_id: u64,
        result: cef_permission_request_result_t,
    ) {
        let (arg_self_, arg_browser, arg_prompt_id, arg_result) =
            (self_, browser, prompt_id, result);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_prompt_id = arg_prompt_id.into_raw();
        let arg_result = arg_result.into_raw();
        ImplPermissionHandler::on_dismiss_permission_prompt(
            &arg_self_.interface,
            arg_browser,
            arg_prompt_id,
            arg_result,
        )
    }
}
impl ImplPermissionHandler for PermissionHandler {
    fn on_request_media_access_permission(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        requesting_origin: Option<&CefString>,
        requested_permissions: u32,
        callback: Option<&mut MediaAccessCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_request_media_access_permission
                .map(|f| {
                    let (
                        arg_browser,
                        arg_frame,
                        arg_requesting_origin,
                        arg_requested_permissions,
                        arg_callback,
                    ) = (
                        browser,
                        frame,
                        requesting_origin,
                        requested_permissions,
                        callback,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_requesting_origin = arg_requesting_origin
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplMediaAccessCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_requesting_origin,
                        arg_requested_permissions,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_show_permission_prompt(
        &self,
        browser: Option<&mut Browser>,
        prompt_id: u64,
        requesting_origin: Option<&CefString>,
        requested_permissions: u32,
        callback: Option<&mut PermissionPromptCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_show_permission_prompt
                .map(|f| {
                    let (
                        arg_browser,
                        arg_prompt_id,
                        arg_requesting_origin,
                        arg_requested_permissions,
                        arg_callback,
                    ) = (
                        browser,
                        prompt_id,
                        requesting_origin,
                        requested_permissions,
                        callback,
                    );
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_requesting_origin = arg_requesting_origin
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplPermissionPromptCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_prompt_id,
                        arg_requesting_origin,
                        arg_requested_permissions,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_dismiss_permission_prompt(
        &self,
        browser: Option<&mut Browser>,
        prompt_id: u64,
        result: PermissionRequestResult,
    ) {
        unsafe {
            if let Some(f) = self.0.on_dismiss_permission_prompt {
                let (arg_browser, arg_prompt_id, arg_result) = (browser, prompt_id, result);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_result = arg_result.into_raw();
                f(arg_self_, arg_browser, arg_prompt_id, arg_result);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_permission_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_permission_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PermissionHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_permission_handler_t> for &PermissionHandler {
    fn into_raw(self) -> *mut _cef_permission_handler_t {
        ImplPermissionHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_permission_handler_t> for &mut PermissionHandler {
    fn into_raw(self) -> *mut _cef_permission_handler_t {
        ImplPermissionHandler::get_raw(self)
    }
}
impl ConvertReturnValue<PermissionHandler> for *mut _cef_permission_handler_t {
    fn wrap_result(self) -> PermissionHandler {
        PermissionHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PermissionHandler> for *mut _cef_permission_handler_t {
    fn from(value: PermissionHandler) -> Self {
        let object = ImplPermissionHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PermissionHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_print_settings_t`] for more documentation.
#[derive(Clone)]
pub struct PrintSettings(RefGuard<_cef_print_settings_t>);
pub trait ImplPrintSettings: Clone + Sized + Rc {
    #[doc = "See [`_cef_print_settings_t::is_valid`] for more documentation."]
    fn is_valid(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_orientation`] for more documentation."]
    fn set_orientation(&self, landscape: ::std::os::raw::c_int);
    #[doc = "See [`_cef_print_settings_t::is_landscape`] for more documentation."]
    fn is_landscape(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_printer_printable_area`] for more documentation."]
    fn set_printer_printable_area(
        &self,
        physical_size_device_units: Option<&Size>,
        printable_area_device_units: Option<&Rect>,
        landscape_needs_flip: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_print_settings_t::set_device_name`] for more documentation."]
    fn set_device_name(&self, name: Option<&CefString>);
    #[doc = "See [`_cef_print_settings_t::get_device_name`] for more documentation."]
    fn device_name(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_print_settings_t::set_dpi`] for more documentation."]
    fn set_dpi(&self, dpi: ::std::os::raw::c_int);
    #[doc = "See [`_cef_print_settings_t::get_dpi`] for more documentation."]
    fn dpi(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_page_ranges`] for more documentation."]
    fn set_page_ranges(&self, ranges_count: usize, ranges: Option<&Range>);
    #[doc = "See [`_cef_print_settings_t::get_page_ranges_count`] for more documentation."]
    fn page_ranges_count(&self) -> usize;
    #[doc = "See [`_cef_print_settings_t::get_page_ranges`] for more documentation."]
    fn page_ranges(&self, ranges_count: Option<&mut usize>, ranges: Option<&mut Range>);
    #[doc = "See [`_cef_print_settings_t::set_selection_only`] for more documentation."]
    fn set_selection_only(&self, selection_only: ::std::os::raw::c_int);
    #[doc = "See [`_cef_print_settings_t::is_selection_only`] for more documentation."]
    fn is_selection_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_collate`] for more documentation."]
    fn set_collate(&self, collate: ::std::os::raw::c_int);
    #[doc = "See [`_cef_print_settings_t::will_collate`] for more documentation."]
    fn will_collate(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_color_model`] for more documentation."]
    fn set_color_model(&self, model: ColorModel);
    #[doc = "See [`_cef_print_settings_t::get_color_model`] for more documentation."]
    fn color_model(&self) -> ColorModel;
    #[doc = "See [`_cef_print_settings_t::set_copies`] for more documentation."]
    fn set_copies(&self, copies: ::std::os::raw::c_int);
    #[doc = "See [`_cef_print_settings_t::get_copies`] for more documentation."]
    fn copies(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_print_settings_t::set_duplex_mode`] for more documentation."]
    fn set_duplex_mode(&self, mode: DuplexMode);
    #[doc = "See [`_cef_print_settings_t::get_duplex_mode`] for more documentation."]
    fn duplex_mode(&self) -> DuplexMode;
    fn get_raw(&self) -> *mut _cef_print_settings_t;
}
impl ImplPrintSettings for PrintSettings {
    fn is_valid(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_valid
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_orientation(&self, landscape: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_orientation {
                let arg_landscape = landscape;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_landscape);
            }
        }
    }
    fn is_landscape(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_landscape
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_printer_printable_area(
        &self,
        physical_size_device_units: Option<&Size>,
        printable_area_device_units: Option<&Rect>,
        landscape_needs_flip: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.set_printer_printable_area {
                let (
                    arg_physical_size_device_units,
                    arg_printable_area_device_units,
                    arg_landscape_needs_flip,
                ) = (
                    physical_size_device_units,
                    printable_area_device_units,
                    landscape_needs_flip,
                );
                let arg_self_ = self.into_raw();
                let arg_physical_size_device_units = arg_physical_size_device_units
                    .cloned()
                    .map(|arg| arg.into());
                let arg_physical_size_device_units = arg_physical_size_device_units
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_printable_area_device_units = arg_printable_area_device_units
                    .cloned()
                    .map(|arg| arg.into());
                let arg_printable_area_device_units = arg_printable_area_device_units
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_physical_size_device_units,
                    arg_printable_area_device_units,
                    arg_landscape_needs_flip,
                );
            }
        }
    }
    fn set_device_name(&self, name: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_device_name {
                let arg_name = name;
                let arg_self_ = self.into_raw();
                let arg_name = arg_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_name);
            }
        }
    }
    fn device_name(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_device_name
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_dpi(&self, dpi: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_dpi {
                let arg_dpi = dpi;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_dpi);
            }
        }
    }
    fn dpi(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_dpi
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_page_ranges(&self, ranges_count: usize, ranges: Option<&Range>) {
        unsafe {
            if let Some(f) = self.0.set_page_ranges {
                let (arg_ranges_count, arg_ranges) = (ranges_count, ranges);
                let arg_self_ = self.into_raw();
                let arg_ranges = arg_ranges.cloned().map(|arg| arg.into());
                let arg_ranges = arg_ranges
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_ranges_count, arg_ranges);
            }
        }
    }
    fn page_ranges_count(&self) -> usize {
        unsafe {
            self.0
                .get_page_ranges_count
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn page_ranges(&self, ranges_count: Option<&mut usize>, ranges: Option<&mut Range>) {
        unsafe {
            if let Some(f) = self.0.get_page_ranges {
                let (arg_ranges_count, arg_ranges) = (ranges_count, ranges);
                let arg_self_ = self.into_raw();
                let arg_ranges_count = arg_ranges_count
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                let mut arg_ranges = arg_ranges.cloned().map(|arg| arg.into());
                let arg_ranges = arg_ranges
                    .as_mut()
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_ranges_count, arg_ranges);
            }
        }
    }
    fn set_selection_only(&self, selection_only: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_selection_only {
                let arg_selection_only = selection_only;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_selection_only);
            }
        }
    }
    fn is_selection_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_selection_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_collate(&self, collate: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_collate {
                let arg_collate = collate;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_collate);
            }
        }
    }
    fn will_collate(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .will_collate
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_color_model(&self, model: ColorModel) {
        unsafe {
            if let Some(f) = self.0.set_color_model {
                let arg_model = model;
                let arg_self_ = self.into_raw();
                let arg_model = arg_model.into_raw();
                f(arg_self_, arg_model);
            }
        }
    }
    fn color_model(&self) -> ColorModel {
        unsafe {
            self.0
                .get_color_model
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_copies(&self, copies: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_copies {
                let arg_copies = copies;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_copies);
            }
        }
    }
    fn copies(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_copies
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_duplex_mode(&self, mode: DuplexMode) {
        unsafe {
            if let Some(f) = self.0.set_duplex_mode {
                let arg_mode = mode;
                let arg_self_ = self.into_raw();
                let arg_mode = arg_mode.into_raw();
                f(arg_self_, arg_mode);
            }
        }
    }
    fn duplex_mode(&self) -> DuplexMode {
        unsafe {
            self.0
                .get_duplex_mode
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_print_settings_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_print_settings_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PrintSettings {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_print_settings_t> for &PrintSettings {
    fn into_raw(self) -> *mut _cef_print_settings_t {
        ImplPrintSettings::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_print_settings_t> for &mut PrintSettings {
    fn into_raw(self) -> *mut _cef_print_settings_t {
        ImplPrintSettings::get_raw(self)
    }
}
impl ConvertReturnValue<PrintSettings> for *mut _cef_print_settings_t {
    fn wrap_result(self) -> PrintSettings {
        PrintSettings(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PrintSettings> for *mut _cef_print_settings_t {
    fn from(value: PrintSettings) -> Self {
        let object = ImplPrintSettings::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PrintSettings {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_print_dialog_callback_t`] for more documentation.
#[derive(Clone)]
pub struct PrintDialogCallback(RefGuard<_cef_print_dialog_callback_t>);
pub trait ImplPrintDialogCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_print_dialog_callback_t::cont`] for more documentation."]
    fn cont(&self, settings: Option<&mut PrintSettings>);
    #[doc = "See [`_cef_print_dialog_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_print_dialog_callback_t;
}
impl ImplPrintDialogCallback for PrintDialogCallback {
    fn cont(&self, settings: Option<&mut PrintSettings>) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_settings = settings;
                let arg_self_ = self.into_raw();
                let arg_settings = arg_settings
                    .map(|arg| {
                        arg.add_ref();
                        ImplPrintSettings::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_settings);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_print_dialog_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_print_dialog_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PrintDialogCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_print_dialog_callback_t> for &PrintDialogCallback {
    fn into_raw(self) -> *mut _cef_print_dialog_callback_t {
        ImplPrintDialogCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_print_dialog_callback_t> for &mut PrintDialogCallback {
    fn into_raw(self) -> *mut _cef_print_dialog_callback_t {
        ImplPrintDialogCallback::get_raw(self)
    }
}
impl ConvertReturnValue<PrintDialogCallback> for *mut _cef_print_dialog_callback_t {
    fn wrap_result(self) -> PrintDialogCallback {
        PrintDialogCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PrintDialogCallback> for *mut _cef_print_dialog_callback_t {
    fn from(value: PrintDialogCallback) -> Self {
        let object = ImplPrintDialogCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PrintDialogCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_print_job_callback_t`] for more documentation.
#[derive(Clone)]
pub struct PrintJobCallback(RefGuard<_cef_print_job_callback_t>);
pub trait ImplPrintJobCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_print_job_callback_t::cont`] for more documentation."]
    fn cont(&self);
    fn get_raw(&self) -> *mut _cef_print_job_callback_t;
}
impl ImplPrintJobCallback for PrintJobCallback {
    fn cont(&self) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_print_job_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_print_job_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PrintJobCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_print_job_callback_t> for &PrintJobCallback {
    fn into_raw(self) -> *mut _cef_print_job_callback_t {
        ImplPrintJobCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_print_job_callback_t> for &mut PrintJobCallback {
    fn into_raw(self) -> *mut _cef_print_job_callback_t {
        ImplPrintJobCallback::get_raw(self)
    }
}
impl ConvertReturnValue<PrintJobCallback> for *mut _cef_print_job_callback_t {
    fn wrap_result(self) -> PrintJobCallback {
        PrintJobCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PrintJobCallback> for *mut _cef_print_job_callback_t {
    fn from(value: PrintJobCallback) -> Self {
        let object = ImplPrintJobCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PrintJobCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_print_handler_t`] for more documentation.
#[derive(Clone)]
pub struct PrintHandler(RefGuard<_cef_print_handler_t>);
impl PrintHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapPrintHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplPrintHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapPrintHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_print_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapPrintHandler: ImplPrintHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_print_handler_t, Self>);
}
pub trait ImplPrintHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_print_handler_t::on_print_start`] for more documentation."]
    fn on_print_start(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_print_handler_t::on_print_settings`] for more documentation."]
    fn on_print_settings(
        &self,
        browser: Option<&mut Browser>,
        settings: Option<&mut PrintSettings>,
        get_defaults: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_print_handler_t::on_print_dialog`] for more documentation."]
    fn on_print_dialog(
        &self,
        browser: Option<&mut Browser>,
        has_selection: ::std::os::raw::c_int,
        callback: Option<&mut PrintDialogCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_print_handler_t::on_print_job`] for more documentation."]
    fn on_print_job(
        &self,
        browser: Option<&mut Browser>,
        document_name: Option<&CefString>,
        pdf_file_path: Option<&CefString>,
        callback: Option<&mut PrintJobCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_print_handler_t::on_print_reset`] for more documentation."]
    fn on_print_reset(&self, browser: Option<&mut Browser>) {}
    #[doc = "See [`_cef_print_handler_t::get_pdf_paper_size`] for more documentation."]
    fn pdf_paper_size(
        &self,
        browser: Option<&mut Browser>,
        device_units_per_inch: ::std::os::raw::c_int,
    ) -> Size {
        Default::default()
    }
    fn init_methods(object: &mut _cef_print_handler_t) {
        impl_cef_print_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_print_handler_t;
}
mod impl_cef_print_handler_t {
    use super::*;
    pub fn init_methods<I: ImplPrintHandler>(object: &mut _cef_print_handler_t) {
        object.on_print_start = Some(on_print_start::<I>);
        object.on_print_settings = Some(on_print_settings::<I>);
        object.on_print_dialog = Some(on_print_dialog::<I>);
        object.on_print_job = Some(on_print_job::<I>);
        object.on_print_reset = Some(on_print_reset::<I>);
        object.get_pdf_paper_size = Some(get_pdf_paper_size::<I>);
    }
    extern "C" fn on_print_start<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplPrintHandler::on_print_start(&arg_self_.interface, arg_browser)
    }
    extern "C" fn on_print_settings<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
        settings: *mut _cef_print_settings_t,
        get_defaults: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_settings, arg_get_defaults) =
            (self_, browser, settings, get_defaults);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_settings = unsafe { arg_settings.as_mut() }
            .map(|arg| PrintSettings(unsafe { RefGuard::from_raw(arg) }));
        let arg_settings = arg_settings.as_mut();
        let arg_get_defaults = arg_get_defaults.into_raw();
        ImplPrintHandler::on_print_settings(
            &arg_self_.interface,
            arg_browser,
            arg_settings,
            arg_get_defaults,
        )
    }
    extern "C" fn on_print_dialog<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
        has_selection: ::std::os::raw::c_int,
        callback: *mut _cef_print_dialog_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_has_selection, arg_callback) =
            (self_, browser, has_selection, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_has_selection = arg_has_selection.into_raw();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| PrintDialogCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplPrintHandler::on_print_dialog(
            &arg_self_.interface,
            arg_browser,
            arg_has_selection,
            arg_callback,
        )
    }
    extern "C" fn on_print_job<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
        document_name: *const cef_string_t,
        pdf_file_path: *const cef_string_t,
        callback: *mut _cef_print_job_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_document_name, arg_pdf_file_path, arg_callback) =
            (self_, browser, document_name, pdf_file_path, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_document_name = if arg_document_name.is_null() {
            None
        } else {
            Some(arg_document_name.into())
        };
        let arg_document_name = arg_document_name.as_ref();
        let arg_pdf_file_path = if arg_pdf_file_path.is_null() {
            None
        } else {
            Some(arg_pdf_file_path.into())
        };
        let arg_pdf_file_path = arg_pdf_file_path.as_ref();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| PrintJobCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplPrintHandler::on_print_job(
            &arg_self_.interface,
            arg_browser,
            arg_document_name,
            arg_pdf_file_path,
            arg_callback,
        )
    }
    extern "C" fn on_print_reset<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
    ) {
        let (arg_self_, arg_browser) = (self_, browser);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        ImplPrintHandler::on_print_reset(&arg_self_.interface, arg_browser)
    }
    extern "C" fn get_pdf_paper_size<I: ImplPrintHandler>(
        self_: *mut _cef_print_handler_t,
        browser: *mut _cef_browser_t,
        device_units_per_inch: ::std::os::raw::c_int,
    ) -> _cef_size_t {
        let (arg_self_, arg_browser, arg_device_units_per_inch) =
            (self_, browser, device_units_per_inch);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_device_units_per_inch = arg_device_units_per_inch.into_raw();
        let result = ImplPrintHandler::pdf_paper_size(
            &arg_self_.interface,
            arg_browser,
            arg_device_units_per_inch,
        );
        result.into()
    }
}
impl ImplPrintHandler for PrintHandler {
    fn on_print_start(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_print_start {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn on_print_settings(
        &self,
        browser: Option<&mut Browser>,
        settings: Option<&mut PrintSettings>,
        get_defaults: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_print_settings {
                let (arg_browser, arg_settings, arg_get_defaults) =
                    (browser, settings, get_defaults);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_settings = arg_settings
                    .map(|arg| {
                        arg.add_ref();
                        ImplPrintSettings::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_settings, arg_get_defaults);
            }
        }
    }
    fn on_print_dialog(
        &self,
        browser: Option<&mut Browser>,
        has_selection: ::std::os::raw::c_int,
        callback: Option<&mut PrintDialogCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_print_dialog
                .map(|f| {
                    let (arg_browser, arg_has_selection, arg_callback) =
                        (browser, has_selection, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplPrintDialogCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_has_selection, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_print_job(
        &self,
        browser: Option<&mut Browser>,
        document_name: Option<&CefString>,
        pdf_file_path: Option<&CefString>,
        callback: Option<&mut PrintJobCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_print_job
                .map(|f| {
                    let (arg_browser, arg_document_name, arg_pdf_file_path, arg_callback) =
                        (browser, document_name, pdf_file_path, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_document_name = arg_document_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_pdf_file_path = arg_pdf_file_path
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplPrintJobCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_document_name,
                        arg_pdf_file_path,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_print_reset(&self, browser: Option<&mut Browser>) {
        unsafe {
            if let Some(f) = self.0.on_print_reset {
                let arg_browser = browser;
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser);
            }
        }
    }
    fn pdf_paper_size(
        &self,
        browser: Option<&mut Browser>,
        device_units_per_inch: ::std::os::raw::c_int,
    ) -> Size {
        unsafe {
            self.0
                .get_pdf_paper_size
                .map(|f| {
                    let (arg_browser, arg_device_units_per_inch) = (browser, device_units_per_inch);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_device_units_per_inch);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_print_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_print_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for PrintHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_print_handler_t> for &PrintHandler {
    fn into_raw(self) -> *mut _cef_print_handler_t {
        ImplPrintHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_print_handler_t> for &mut PrintHandler {
    fn into_raw(self) -> *mut _cef_print_handler_t {
        ImplPrintHandler::get_raw(self)
    }
}
impl ConvertReturnValue<PrintHandler> for *mut _cef_print_handler_t {
    fn wrap_result(self) -> PrintHandler {
        PrintHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<PrintHandler> for *mut _cef_print_handler_t {
    fn from(value: PrintHandler) -> Self {
        let object = ImplPrintHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for PrintHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_accessibility_handler_t`] for more documentation.
#[derive(Clone)]
pub struct AccessibilityHandler(RefGuard<_cef_accessibility_handler_t>);
impl AccessibilityHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapAccessibilityHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplAccessibilityHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapAccessibilityHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_accessibility_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapAccessibilityHandler: ImplAccessibilityHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_accessibility_handler_t, Self>);
}
pub trait ImplAccessibilityHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_accessibility_handler_t::on_accessibility_tree_change`] for more documentation."]
    fn on_accessibility_tree_change(&self, value: Option<&mut Value>) {}
    #[doc = "See [`_cef_accessibility_handler_t::on_accessibility_location_change`] for more documentation."]
    fn on_accessibility_location_change(&self, value: Option<&mut Value>) {}
    fn init_methods(object: &mut _cef_accessibility_handler_t) {
        impl_cef_accessibility_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_accessibility_handler_t;
}
mod impl_cef_accessibility_handler_t {
    use super::*;
    pub fn init_methods<I: ImplAccessibilityHandler>(object: &mut _cef_accessibility_handler_t) {
        object.on_accessibility_tree_change = Some(on_accessibility_tree_change::<I>);
        object.on_accessibility_location_change = Some(on_accessibility_location_change::<I>);
    }
    extern "C" fn on_accessibility_tree_change<I: ImplAccessibilityHandler>(
        self_: *mut _cef_accessibility_handler_t,
        value: *mut _cef_value_t,
    ) {
        let (arg_self_, arg_value) = (self_, value);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_value =
            unsafe { arg_value.as_mut() }.map(|arg| Value(unsafe { RefGuard::from_raw(arg) }));
        let arg_value = arg_value.as_mut();
        ImplAccessibilityHandler::on_accessibility_tree_change(&arg_self_.interface, arg_value)
    }
    extern "C" fn on_accessibility_location_change<I: ImplAccessibilityHandler>(
        self_: *mut _cef_accessibility_handler_t,
        value: *mut _cef_value_t,
    ) {
        let (arg_self_, arg_value) = (self_, value);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_value =
            unsafe { arg_value.as_mut() }.map(|arg| Value(unsafe { RefGuard::from_raw(arg) }));
        let arg_value = arg_value.as_mut();
        ImplAccessibilityHandler::on_accessibility_location_change(&arg_self_.interface, arg_value)
    }
}
impl ImplAccessibilityHandler for AccessibilityHandler {
    fn on_accessibility_tree_change(&self, value: Option<&mut Value>) {
        unsafe {
            if let Some(f) = self.0.on_accessibility_tree_change {
                let arg_value = value;
                let arg_self_ = self.into_raw();
                let arg_value = arg_value
                    .map(|arg| {
                        arg.add_ref();
                        ImplValue::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_value);
            }
        }
    }
    fn on_accessibility_location_change(&self, value: Option<&mut Value>) {
        unsafe {
            if let Some(f) = self.0.on_accessibility_location_change {
                let arg_value = value;
                let arg_self_ = self.into_raw();
                let arg_value = arg_value
                    .map(|arg| {
                        arg.add_ref();
                        ImplValue::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_value);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_accessibility_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_accessibility_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for AccessibilityHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_accessibility_handler_t> for &AccessibilityHandler {
    fn into_raw(self) -> *mut _cef_accessibility_handler_t {
        ImplAccessibilityHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_accessibility_handler_t> for &mut AccessibilityHandler {
    fn into_raw(self) -> *mut _cef_accessibility_handler_t {
        ImplAccessibilityHandler::get_raw(self)
    }
}
impl ConvertReturnValue<AccessibilityHandler> for *mut _cef_accessibility_handler_t {
    fn wrap_result(self) -> AccessibilityHandler {
        AccessibilityHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<AccessibilityHandler> for *mut _cef_accessibility_handler_t {
    fn from(value: AccessibilityHandler) -> Self {
        let object = ImplAccessibilityHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for AccessibilityHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_render_handler_t`] for more documentation.
#[derive(Clone)]
pub struct RenderHandler(RefGuard<_cef_render_handler_t>);
impl RenderHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapRenderHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplRenderHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapRenderHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_render_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapRenderHandler: ImplRenderHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_render_handler_t, Self>);
}
pub trait ImplRenderHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_render_handler_t::get_accessibility_handler`] for more documentation."]
    fn accessibility_handler(&self) -> Option<AccessibilityHandler> {
        Default::default()
    }
    #[doc = "See [`_cef_render_handler_t::get_root_screen_rect`] for more documentation."]
    fn root_screen_rect(
        &self,
        browser: Option<&mut Browser>,
        rect: Option<&mut Rect>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_render_handler_t::get_view_rect`] for more documentation."]
    fn view_rect(&self, browser: Option<&mut Browser>, rect: Option<&mut Rect>) {}
    #[doc = "See [`_cef_render_handler_t::get_screen_point`] for more documentation."]
    fn screen_point(
        &self,
        browser: Option<&mut Browser>,
        view_x: ::std::os::raw::c_int,
        view_y: ::std::os::raw::c_int,
        screen_x: Option<&mut ::std::os::raw::c_int>,
        screen_y: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_render_handler_t::get_screen_info`] for more documentation."]
    fn screen_info(
        &self,
        browser: Option<&mut Browser>,
        screen_info: Option<&mut ScreenInfo>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_render_handler_t::on_popup_show`] for more documentation."]
    fn on_popup_show(&self, browser: Option<&mut Browser>, show: ::std::os::raw::c_int) {}
    #[doc = "See [`_cef_render_handler_t::on_popup_size`] for more documentation."]
    fn on_popup_size(&self, browser: Option<&mut Browser>, rect: Option<&Rect>) {}
    #[doc = "See [`_cef_render_handler_t::on_paint`] for more documentation."]
    fn on_paint(
        &self,
        browser: Option<&mut Browser>,
        type_: PaintElementType,
        dirty_rects_count: usize,
        dirty_rects: Option<&Rect>,
        buffer: *const u8,
        width: ::std::os::raw::c_int,
        height: ::std::os::raw::c_int,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::on_accelerated_paint`] for more documentation."]
    fn on_accelerated_paint(
        &self,
        browser: Option<&mut Browser>,
        type_: PaintElementType,
        dirty_rects_count: usize,
        dirty_rects: Option<&Rect>,
        info: Option<&AcceleratedPaintInfo>,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::get_touch_handle_size`] for more documentation."]
    fn touch_handle_size(
        &self,
        browser: Option<&mut Browser>,
        orientation: HorizontalAlignment,
        size: Option<&mut Size>,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::on_touch_handle_state_changed`] for more documentation."]
    fn on_touch_handle_state_changed(
        &self,
        browser: Option<&mut Browser>,
        state: Option<&TouchHandleState>,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::start_dragging`] for more documentation."]
    fn start_dragging(
        &self,
        browser: Option<&mut Browser>,
        drag_data: Option<&mut DragData>,
        allowed_ops: DragOperationsMask,
        x: ::std::os::raw::c_int,
        y: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_render_handler_t::update_drag_cursor`] for more documentation."]
    fn update_drag_cursor(&self, browser: Option<&mut Browser>, operation: DragOperationsMask) {}
    #[doc = "See [`_cef_render_handler_t::on_scroll_offset_changed`] for more documentation."]
    fn on_scroll_offset_changed(&self, browser: Option<&mut Browser>, x: f64, y: f64) {}
    #[doc = "See [`_cef_render_handler_t::on_ime_composition_range_changed`] for more documentation."]
    fn on_ime_composition_range_changed(
        &self,
        browser: Option<&mut Browser>,
        selected_range: Option<&Range>,
        character_bounds_count: usize,
        character_bounds: Option<&Rect>,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::on_text_selection_changed`] for more documentation."]
    fn on_text_selection_changed(
        &self,
        browser: Option<&mut Browser>,
        selected_text: Option<&CefString>,
        selected_range: Option<&Range>,
    ) {
    }
    #[doc = "See [`_cef_render_handler_t::on_virtual_keyboard_requested`] for more documentation."]
    fn on_virtual_keyboard_requested(
        &self,
        browser: Option<&mut Browser>,
        input_mode: TextInputMode,
    ) {
    }
    fn init_methods(object: &mut _cef_render_handler_t) {
        impl_cef_render_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_render_handler_t;
}
mod impl_cef_render_handler_t {
    use super::*;
    pub fn init_methods<I: ImplRenderHandler>(object: &mut _cef_render_handler_t) {
        object.get_accessibility_handler = Some(get_accessibility_handler::<I>);
        object.get_root_screen_rect = Some(get_root_screen_rect::<I>);
        object.get_view_rect = Some(get_view_rect::<I>);
        object.get_screen_point = Some(get_screen_point::<I>);
        object.get_screen_info = Some(get_screen_info::<I>);
        object.on_popup_show = Some(on_popup_show::<I>);
        object.on_popup_size = Some(on_popup_size::<I>);
        object.on_paint = Some(on_paint::<I>);
        object.on_accelerated_paint = Some(on_accelerated_paint::<I>);
        object.get_touch_handle_size = Some(get_touch_handle_size::<I>);
        object.on_touch_handle_state_changed = Some(on_touch_handle_state_changed::<I>);
        object.start_dragging = Some(start_dragging::<I>);
        object.update_drag_cursor = Some(update_drag_cursor::<I>);
        object.on_scroll_offset_changed = Some(on_scroll_offset_changed::<I>);
        object.on_ime_composition_range_changed = Some(on_ime_composition_range_changed::<I>);
        object.on_text_selection_changed = Some(on_text_selection_changed::<I>);
        object.on_virtual_keyboard_requested = Some(on_virtual_keyboard_requested::<I>);
    }
    extern "C" fn get_accessibility_handler<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
    ) -> *mut _cef_accessibility_handler_t {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let result = ImplRenderHandler::accessibility_handler(&arg_self_.interface);
        result
            .map(|result| result.into())
            .unwrap_or(std::ptr::null_mut())
    }
    extern "C" fn get_root_screen_rect<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        rect: *mut _cef_rect_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_rect) = (self_, browser, rect);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_rect = if arg_rect.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_rect))
        };
        let arg_rect = arg_rect.as_mut().map(|arg| arg.as_mut());
        ImplRenderHandler::root_screen_rect(&arg_self_.interface, arg_browser, arg_rect)
    }
    extern "C" fn get_view_rect<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        rect: *mut _cef_rect_t,
    ) {
        let (arg_self_, arg_browser, arg_rect) = (self_, browser, rect);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_rect = if arg_rect.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_rect))
        };
        let arg_rect = arg_rect.as_mut().map(|arg| arg.as_mut());
        ImplRenderHandler::view_rect(&arg_self_.interface, arg_browser, arg_rect)
    }
    extern "C" fn get_screen_point<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        view_x: ::std::os::raw::c_int,
        view_y: ::std::os::raw::c_int,
        screen_x: *mut ::std::os::raw::c_int,
        screen_y: *mut ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_view_x, arg_view_y, arg_screen_x, arg_screen_y) =
            (self_, browser, view_x, view_y, screen_x, screen_y);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_view_x = arg_view_x.into_raw();
        let arg_view_y = arg_view_y.into_raw();
        let mut arg_screen_x = if arg_screen_x.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(arg_screen_x))
        };
        let arg_screen_x = arg_screen_x.as_mut().map(|arg| arg.as_mut());
        let mut arg_screen_y = if arg_screen_y.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(arg_screen_y))
        };
        let arg_screen_y = arg_screen_y.as_mut().map(|arg| arg.as_mut());
        ImplRenderHandler::screen_point(
            &arg_self_.interface,
            arg_browser,
            arg_view_x,
            arg_view_y,
            arg_screen_x,
            arg_screen_y,
        )
    }
    extern "C" fn get_screen_info<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        screen_info: *mut _cef_screen_info_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_screen_info) = (self_, browser, screen_info);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_screen_info = if arg_screen_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<ScreenInfo, _>::from(arg_screen_info))
        };
        let arg_screen_info = arg_screen_info.as_mut().map(|arg| arg.as_mut());
        ImplRenderHandler::screen_info(&arg_self_.interface, arg_browser, arg_screen_info)
    }
    extern "C" fn on_popup_show<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        show: ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_show) = (self_, browser, show);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_show = arg_show.into_raw();
        ImplRenderHandler::on_popup_show(&arg_self_.interface, arg_browser, arg_show)
    }
    extern "C" fn on_popup_size<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        rect: *const _cef_rect_t,
    ) {
        let (arg_self_, arg_browser, arg_rect) = (self_, browser, rect);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_rect = if arg_rect.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_rect))
        };
        let arg_rect = arg_rect.as_ref().map(|arg| arg.as_ref());
        ImplRenderHandler::on_popup_size(&arg_self_.interface, arg_browser, arg_rect)
    }
    extern "C" fn on_paint<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        type_: cef_paint_element_type_t,
        dirty_rects_count: usize,
        dirty_rects: *const _cef_rect_t,
        buffer: *const ::std::os::raw::c_void,
        width: ::std::os::raw::c_int,
        height: ::std::os::raw::c_int,
    ) {
        let (
            arg_self_,
            arg_browser,
            arg_type_,
            arg_dirty_rects_count,
            arg_dirty_rects,
            arg_buffer,
            arg_width,
            arg_height,
        ) = (
            self_,
            browser,
            type_,
            dirty_rects_count,
            dirty_rects,
            buffer,
            width,
            height,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_type_ = arg_type_.into_raw();
        let arg_dirty_rects_count = arg_dirty_rects_count.into_raw();
        let arg_dirty_rects = if arg_dirty_rects.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_dirty_rects))
        };
        let arg_dirty_rects = arg_dirty_rects.as_ref().map(|arg| arg.as_ref());
        let arg_buffer = arg_buffer.cast();
        let arg_width = arg_width.into_raw();
        let arg_height = arg_height.into_raw();
        ImplRenderHandler::on_paint(
            &arg_self_.interface,
            arg_browser,
            arg_type_,
            arg_dirty_rects_count,
            arg_dirty_rects,
            arg_buffer,
            arg_width,
            arg_height,
        )
    }
    extern "C" fn on_accelerated_paint<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        type_: cef_paint_element_type_t,
        dirty_rects_count: usize,
        dirty_rects: *const _cef_rect_t,
        info: *const _cef_accelerated_paint_info_t,
    ) {
        let (arg_self_, arg_browser, arg_type_, arg_dirty_rects_count, arg_dirty_rects, arg_info) =
            (self_, browser, type_, dirty_rects_count, dirty_rects, info);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_type_ = arg_type_.into_raw();
        let arg_dirty_rects_count = arg_dirty_rects_count.into_raw();
        let arg_dirty_rects = if arg_dirty_rects.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_dirty_rects))
        };
        let arg_dirty_rects = arg_dirty_rects.as_ref().map(|arg| arg.as_ref());
        let arg_info = if arg_info.is_null() {
            None
        } else {
            Some(WrapParamRef::<AcceleratedPaintInfo, _>::from(arg_info))
        };
        let arg_info = arg_info.as_ref().map(|arg| arg.as_ref());
        ImplRenderHandler::on_accelerated_paint(
            &arg_self_.interface,
            arg_browser,
            arg_type_,
            arg_dirty_rects_count,
            arg_dirty_rects,
            arg_info,
        )
    }
    extern "C" fn get_touch_handle_size<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        orientation: cef_horizontal_alignment_t,
        size: *mut _cef_size_t,
    ) {
        let (arg_self_, arg_browser, arg_orientation, arg_size) =
            (self_, browser, orientation, size);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_orientation = arg_orientation.into_raw();
        let mut arg_size = if arg_size.is_null() {
            None
        } else {
            Some(WrapParamRef::<Size, _>::from(arg_size))
        };
        let arg_size = arg_size.as_mut().map(|arg| arg.as_mut());
        ImplRenderHandler::touch_handle_size(
            &arg_self_.interface,
            arg_browser,
            arg_orientation,
            arg_size,
        )
    }
    extern "C" fn on_touch_handle_state_changed<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        state: *const _cef_touch_handle_state_t,
    ) {
        let (arg_self_, arg_browser, arg_state) = (self_, browser, state);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_state = if arg_state.is_null() {
            None
        } else {
            Some(WrapParamRef::<TouchHandleState, _>::from(arg_state))
        };
        let arg_state = arg_state.as_ref().map(|arg| arg.as_ref());
        ImplRenderHandler::on_touch_handle_state_changed(
            &arg_self_.interface,
            arg_browser,
            arg_state,
        )
    }
    extern "C" fn start_dragging<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        drag_data: *mut _cef_drag_data_t,
        allowed_ops: cef_drag_operations_mask_t,
        x: ::std::os::raw::c_int,
        y: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_drag_data, arg_allowed_ops, arg_x, arg_y) =
            (self_, browser, drag_data, allowed_ops, x, y);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_drag_data = unsafe { arg_drag_data.as_mut() }
            .map(|arg| DragData(unsafe { RefGuard::from_raw(arg) }));
        let arg_drag_data = arg_drag_data.as_mut();
        let arg_allowed_ops = arg_allowed_ops.into_raw();
        let arg_x = arg_x.into_raw();
        let arg_y = arg_y.into_raw();
        ImplRenderHandler::start_dragging(
            &arg_self_.interface,
            arg_browser,
            arg_drag_data,
            arg_allowed_ops,
            arg_x,
            arg_y,
        )
    }
    extern "C" fn update_drag_cursor<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        operation: cef_drag_operations_mask_t,
    ) {
        let (arg_self_, arg_browser, arg_operation) = (self_, browser, operation);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_operation = arg_operation.into_raw();
        ImplRenderHandler::update_drag_cursor(&arg_self_.interface, arg_browser, arg_operation)
    }
    extern "C" fn on_scroll_offset_changed<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        x: f64,
        y: f64,
    ) {
        let (arg_self_, arg_browser, arg_x, arg_y) = (self_, browser, x, y);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_x = arg_x.into_raw();
        let arg_y = arg_y.into_raw();
        ImplRenderHandler::on_scroll_offset_changed(&arg_self_.interface, arg_browser, arg_x, arg_y)
    }
    extern "C" fn on_ime_composition_range_changed<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        selected_range: *const _cef_range_t,
        character_bounds_count: usize,
        character_bounds: *const _cef_rect_t,
    ) {
        let (
            arg_self_,
            arg_browser,
            arg_selected_range,
            arg_character_bounds_count,
            arg_character_bounds,
        ) = (
            self_,
            browser,
            selected_range,
            character_bounds_count,
            character_bounds,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_selected_range = if arg_selected_range.is_null() {
            None
        } else {
            Some(WrapParamRef::<Range, _>::from(arg_selected_range))
        };
        let arg_selected_range = arg_selected_range.as_ref().map(|arg| arg.as_ref());
        let arg_character_bounds_count = arg_character_bounds_count.into_raw();
        let arg_character_bounds = if arg_character_bounds.is_null() {
            None
        } else {
            Some(WrapParamRef::<Rect, _>::from(arg_character_bounds))
        };
        let arg_character_bounds = arg_character_bounds.as_ref().map(|arg| arg.as_ref());
        ImplRenderHandler::on_ime_composition_range_changed(
            &arg_self_.interface,
            arg_browser,
            arg_selected_range,
            arg_character_bounds_count,
            arg_character_bounds,
        )
    }
    extern "C" fn on_text_selection_changed<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        selected_text: *const cef_string_t,
        selected_range: *const _cef_range_t,
    ) {
        let (arg_self_, arg_browser, arg_selected_text, arg_selected_range) =
            (self_, browser, selected_text, selected_range);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_selected_text = if arg_selected_text.is_null() {
            None
        } else {
            Some(arg_selected_text.into())
        };
        let arg_selected_text = arg_selected_text.as_ref();
        let arg_selected_range = if arg_selected_range.is_null() {
            None
        } else {
            Some(WrapParamRef::<Range, _>::from(arg_selected_range))
        };
        let arg_selected_range = arg_selected_range.as_ref().map(|arg| arg.as_ref());
        ImplRenderHandler::on_text_selection_changed(
            &arg_self_.interface,
            arg_browser,
            arg_selected_text,
            arg_selected_range,
        )
    }
    extern "C" fn on_virtual_keyboard_requested<I: ImplRenderHandler>(
        self_: *mut _cef_render_handler_t,
        browser: *mut _cef_browser_t,
        input_mode: cef_text_input_mode_t,
    ) {
        let (arg_self_, arg_browser, arg_input_mode) = (self_, browser, input_mode);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let arg_input_mode = arg_input_mode.into_raw();
        ImplRenderHandler::on_virtual_keyboard_requested(
            &arg_self_.interface,
            arg_browser,
            arg_input_mode,
        )
    }
}
impl ImplRenderHandler for RenderHandler {
    fn accessibility_handler(&self) -> Option<AccessibilityHandler> {
        unsafe {
            self.0
                .get_accessibility_handler
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn root_screen_rect(
        &self,
        browser: Option<&mut Browser>,
        rect: Option<&mut Rect>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_root_screen_rect
                .map(|f| {
                    let (arg_browser, arg_rect) = (browser, rect);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let mut arg_rect = arg_rect.cloned().map(|arg| arg.into());
                    let arg_rect = arg_rect
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_rect);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn view_rect(&self, browser: Option<&mut Browser>, rect: Option<&mut Rect>) {
        unsafe {
            if let Some(f) = self.0.get_view_rect {
                let (arg_browser, arg_rect) = (browser, rect);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let mut arg_rect = arg_rect.cloned().map(|arg| arg.into());
                let arg_rect = arg_rect
                    .as_mut()
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_rect);
            }
        }
    }
    fn screen_point(
        &self,
        browser: Option<&mut Browser>,
        view_x: ::std::os::raw::c_int,
        view_y: ::std::os::raw::c_int,
        screen_x: Option<&mut ::std::os::raw::c_int>,
        screen_y: Option<&mut ::std::os::raw::c_int>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_screen_point
                .map(|f| {
                    let (arg_browser, arg_view_x, arg_view_y, arg_screen_x, arg_screen_y) =
                        (browser, view_x, view_y, screen_x, screen_y);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_screen_x = arg_screen_x
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_screen_y = arg_screen_y
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_view_x,
                        arg_view_y,
                        arg_screen_x,
                        arg_screen_y,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn screen_info(
        &self,
        browser: Option<&mut Browser>,
        screen_info: Option<&mut ScreenInfo>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_screen_info
                .map(|f| {
                    let (arg_browser, arg_screen_info) = (browser, screen_info);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let mut arg_screen_info = arg_screen_info.cloned().map(|arg| arg.into());
                    let arg_screen_info = arg_screen_info
                        .as_mut()
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_screen_info);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn on_popup_show(&self, browser: Option<&mut Browser>, show: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.on_popup_show {
                let (arg_browser, arg_show) = (browser, show);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_show);
            }
        }
    }
    fn on_popup_size(&self, browser: Option<&mut Browser>, rect: Option<&Rect>) {
        unsafe {
            if let Some(f) = self.0.on_popup_size {
                let (arg_browser, arg_rect) = (browser, rect);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_rect = arg_rect.cloned().map(|arg| arg.into());
                let arg_rect = arg_rect
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_rect);
            }
        }
    }
    fn on_paint(
        &self,
        browser: Option<&mut Browser>,
        type_: PaintElementType,
        dirty_rects_count: usize,
        dirty_rects: Option<&Rect>,
        buffer: *const u8,
        width: ::std::os::raw::c_int,
        height: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.on_paint {
                let (
                    arg_browser,
                    arg_type_,
                    arg_dirty_rects_count,
                    arg_dirty_rects,
                    arg_buffer,
                    arg_width,
                    arg_height,
                ) = (
                    browser,
                    type_,
                    dirty_rects_count,
                    dirty_rects,
                    buffer,
                    width,
                    height,
                );
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_type_ = arg_type_.into_raw();
                let arg_dirty_rects = arg_dirty_rects.cloned().map(|arg| arg.into());
                let arg_dirty_rects = arg_dirty_rects
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_buffer = arg_buffer.cast();
                f(
                    arg_self_,
                    arg_browser,
                    arg_type_,
                    arg_dirty_rects_count,
                    arg_dirty_rects,
                    arg_buffer,
                    arg_width,
                    arg_height,
                );
            }
        }
    }
    fn on_accelerated_paint(
        &self,
        browser: Option<&mut Browser>,
        type_: PaintElementType,
        dirty_rects_count: usize,
        dirty_rects: Option<&Rect>,
        info: Option<&AcceleratedPaintInfo>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_accelerated_paint {
                let (arg_browser, arg_type_, arg_dirty_rects_count, arg_dirty_rects, arg_info) =
                    (browser, type_, dirty_rects_count, dirty_rects, info);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_type_ = arg_type_.into_raw();
                let arg_dirty_rects = arg_dirty_rects.cloned().map(|arg| arg.into());
                let arg_dirty_rects = arg_dirty_rects
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_info = arg_info.cloned().map(|arg| arg.into());
                let arg_info = arg_info
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_type_,
                    arg_dirty_rects_count,
                    arg_dirty_rects,
                    arg_info,
                );
            }
        }
    }
    fn touch_handle_size(
        &self,
        browser: Option<&mut Browser>,
        orientation: HorizontalAlignment,
        size: Option<&mut Size>,
    ) {
        unsafe {
            if let Some(f) = self.0.get_touch_handle_size {
                let (arg_browser, arg_orientation, arg_size) = (browser, orientation, size);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_orientation = arg_orientation.into_raw();
                let mut arg_size = arg_size.cloned().map(|arg| arg.into());
                let arg_size = arg_size
                    .as_mut()
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_orientation, arg_size);
            }
        }
    }
    fn on_touch_handle_state_changed(
        &self,
        browser: Option<&mut Browser>,
        state: Option<&TouchHandleState>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_touch_handle_state_changed {
                let (arg_browser, arg_state) = (browser, state);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_state = arg_state.cloned().map(|arg| arg.into());
                let arg_state = arg_state
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_browser, arg_state);
            }
        }
    }
    fn start_dragging(
        &self,
        browser: Option<&mut Browser>,
        drag_data: Option<&mut DragData>,
        allowed_ops: DragOperationsMask,
        x: ::std::os::raw::c_int,
        y: ::std::os::raw::c_int,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .start_dragging
                .map(|f| {
                    let (arg_browser, arg_drag_data, arg_allowed_ops, arg_x, arg_y) =
                        (browser, drag_data, allowed_ops, x, y);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_drag_data = arg_drag_data
                        .map(|arg| {
                            arg.add_ref();
                            ImplDragData::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_allowed_ops = arg_allowed_ops.into_raw();
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_drag_data,
                        arg_allowed_ops,
                        arg_x,
                        arg_y,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn update_drag_cursor(&self, browser: Option<&mut Browser>, operation: DragOperationsMask) {
        unsafe {
            if let Some(f) = self.0.update_drag_cursor {
                let (arg_browser, arg_operation) = (browser, operation);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_operation = arg_operation.into_raw();
                f(arg_self_, arg_browser, arg_operation);
            }
        }
    }
    fn on_scroll_offset_changed(&self, browser: Option<&mut Browser>, x: f64, y: f64) {
        unsafe {
            if let Some(f) = self.0.on_scroll_offset_changed {
                let (arg_browser, arg_x, arg_y) = (browser, x, y);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_browser, arg_x, arg_y);
            }
        }
    }
    fn on_ime_composition_range_changed(
        &self,
        browser: Option<&mut Browser>,
        selected_range: Option<&Range>,
        character_bounds_count: usize,
        character_bounds: Option<&Rect>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_ime_composition_range_changed {
                let (
                    arg_browser,
                    arg_selected_range,
                    arg_character_bounds_count,
                    arg_character_bounds,
                ) = (
                    browser,
                    selected_range,
                    character_bounds_count,
                    character_bounds,
                );
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_selected_range = arg_selected_range.cloned().map(|arg| arg.into());
                let arg_selected_range = arg_selected_range
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                let arg_character_bounds = arg_character_bounds.cloned().map(|arg| arg.into());
                let arg_character_bounds = arg_character_bounds
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_selected_range,
                    arg_character_bounds_count,
                    arg_character_bounds,
                );
            }
        }
    }
    fn on_text_selection_changed(
        &self,
        browser: Option<&mut Browser>,
        selected_text: Option<&CefString>,
        selected_range: Option<&Range>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_text_selection_changed {
                let (arg_browser, arg_selected_text, arg_selected_range) =
                    (browser, selected_text, selected_range);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_selected_text = arg_selected_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_selected_range = arg_selected_range.cloned().map(|arg| arg.into());
                let arg_selected_range = arg_selected_range
                    .as_ref()
                    .map(std::ptr::from_ref)
                    .unwrap_or(std::ptr::null());
                f(
                    arg_self_,
                    arg_browser,
                    arg_selected_text,
                    arg_selected_range,
                );
            }
        }
    }
    fn on_virtual_keyboard_requested(
        &self,
        browser: Option<&mut Browser>,
        input_mode: TextInputMode,
    ) {
        unsafe {
            if let Some(f) = self.0.on_virtual_keyboard_requested {
                let (arg_browser, arg_input_mode) = (browser, input_mode);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_input_mode = arg_input_mode.into_raw();
                f(arg_self_, arg_browser, arg_input_mode);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_render_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_render_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for RenderHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_render_handler_t> for &RenderHandler {
    fn into_raw(self) -> *mut _cef_render_handler_t {
        ImplRenderHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_render_handler_t> for &mut RenderHandler {
    fn into_raw(self) -> *mut _cef_render_handler_t {
        ImplRenderHandler::get_raw(self)
    }
}
impl ConvertReturnValue<RenderHandler> for *mut _cef_render_handler_t {
    fn wrap_result(self) -> RenderHandler {
        RenderHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<RenderHandler> for *mut _cef_render_handler_t {
    fn from(value: RenderHandler) -> Self {
        let object = ImplRenderHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for RenderHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_auth_callback_t`] for more documentation.
#[derive(Clone)]
pub struct AuthCallback(RefGuard<_cef_auth_callback_t>);
pub trait ImplAuthCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_auth_callback_t::cont`] for more documentation."]
    fn cont(&self, username: Option<&CefString>, password: Option<&CefString>);
    #[doc = "See [`_cef_auth_callback_t::cancel`] for more documentation."]
    fn cancel(&self);
    fn get_raw(&self) -> *mut _cef_auth_callback_t;
}
impl ImplAuthCallback for AuthCallback {
    fn cont(&self, username: Option<&CefString>, password: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.cont {
                let (arg_username, arg_password) = (username, password);
                let arg_self_ = self.into_raw();
                let arg_username = arg_username
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_password = arg_password
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_username, arg_password);
            }
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_auth_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_auth_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for AuthCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_auth_callback_t> for &AuthCallback {
    fn into_raw(self) -> *mut _cef_auth_callback_t {
        ImplAuthCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_auth_callback_t> for &mut AuthCallback {
    fn into_raw(self) -> *mut _cef_auth_callback_t {
        ImplAuthCallback::get_raw(self)
    }
}
impl ConvertReturnValue<AuthCallback> for *mut _cef_auth_callback_t {
    fn wrap_result(self) -> AuthCallback {
        AuthCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<AuthCallback> for *mut _cef_auth_callback_t {
    fn from(value: AuthCallback) -> Self {
        let object = ImplAuthCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for AuthCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_response_t`] for more documentation.
#[derive(Clone)]
pub struct Response(RefGuard<_cef_response_t>);
pub trait ImplResponse: Clone + Sized + Rc {
    #[doc = "See [`_cef_response_t::is_read_only`] for more documentation."]
    fn is_read_only(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_response_t::get_error`] for more documentation."]
    fn error(&self) -> Errorcode;
    #[doc = "See [`_cef_response_t::set_error`] for more documentation."]
    fn set_error(&self, error: Errorcode);
    #[doc = "See [`_cef_response_t::get_status`] for more documentation."]
    fn status(&self) -> ::std::os::raw::c_int;
    #[doc = "See [`_cef_response_t::set_status`] for more documentation."]
    fn set_status(&self, status: ::std::os::raw::c_int);
    #[doc = "See [`_cef_response_t::get_status_text`] for more documentation."]
    fn status_text(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_response_t::set_status_text`] for more documentation."]
    fn set_status_text(&self, status_text: Option<&CefString>);
    #[doc = "See [`_cef_response_t::get_mime_type`] for more documentation."]
    fn mime_type(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_response_t::set_mime_type`] for more documentation."]
    fn set_mime_type(&self, mime_type: Option<&CefString>);
    #[doc = "See [`_cef_response_t::get_charset`] for more documentation."]
    fn charset(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_response_t::set_charset`] for more documentation."]
    fn set_charset(&self, charset: Option<&CefString>);
    #[doc = "See [`_cef_response_t::get_header_by_name`] for more documentation."]
    fn header_by_name(&self, name: Option<&CefString>) -> CefStringUserfree;
    #[doc = "See [`_cef_response_t::set_header_by_name`] for more documentation."]
    fn set_header_by_name(
        &self,
        name: Option<&CefString>,
        value: Option<&CefString>,
        overwrite: ::std::os::raw::c_int,
    );
    #[doc = "See [`_cef_response_t::get_header_map`] for more documentation."]
    fn header_map(&self, header_map: Option<&mut CefStringMultimap>);
    #[doc = "See [`_cef_response_t::set_header_map`] for more documentation."]
    fn set_header_map(&self, header_map: Option<&mut CefStringMultimap>);
    #[doc = "See [`_cef_response_t::get_url`] for more documentation."]
    fn url(&self) -> CefStringUserfree;
    #[doc = "See [`_cef_response_t::set_url`] for more documentation."]
    fn set_url(&self, url: Option<&CefString>);
    fn get_raw(&self) -> *mut _cef_response_t;
}
impl ImplResponse for Response {
    fn is_read_only(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .is_read_only
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn error(&self) -> Errorcode {
        unsafe {
            self.0
                .get_error
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_error(&self, error: Errorcode) {
        unsafe {
            if let Some(f) = self.0.set_error {
                let arg_error = error;
                let arg_self_ = self.into_raw();
                let arg_error = arg_error.into_raw();
                f(arg_self_, arg_error);
            }
        }
    }
    fn status(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .get_status
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_status(&self, status: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.set_status {
                let arg_status = status;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_status);
            }
        }
    }
    fn status_text(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_status_text
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_status_text(&self, status_text: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_status_text {
                let arg_status_text = status_text;
                let arg_self_ = self.into_raw();
                let arg_status_text = arg_status_text
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_status_text);
            }
        }
    }
    fn mime_type(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_mime_type
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_mime_type(&self, mime_type: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_mime_type {
                let arg_mime_type = mime_type;
                let arg_self_ = self.into_raw();
                let arg_mime_type = arg_mime_type
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_mime_type);
            }
        }
    }
    fn charset(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_charset
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_charset(&self, charset: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_charset {
                let arg_charset = charset;
                let arg_self_ = self.into_raw();
                let arg_charset = arg_charset
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_charset);
            }
        }
    }
    fn header_by_name(&self, name: Option<&CefString>) -> CefStringUserfree {
        unsafe {
            self.0
                .get_header_by_name
                .map(|f| {
                    let arg_name = name;
                    let arg_self_ = self.into_raw();
                    let arg_name = arg_name
                        .map(|arg| arg.into_raw())
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_name);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_header_by_name(
        &self,
        name: Option<&CefString>,
        value: Option<&CefString>,
        overwrite: ::std::os::raw::c_int,
    ) {
        unsafe {
            if let Some(f) = self.0.set_header_by_name {
                let (arg_name, arg_value, arg_overwrite) = (name, value, overwrite);
                let arg_self_ = self.into_raw();
                let arg_name = arg_name
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                let arg_value = arg_value
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_name, arg_value, arg_overwrite);
            }
        }
    }
    fn header_map(&self, header_map: Option<&mut CefStringMultimap>) {
        unsafe {
            if let Some(f) = self.0.get_header_map {
                let arg_header_map = header_map;
                let arg_self_ = self.into_raw();
                let arg_header_map = arg_header_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_header_map);
            }
        }
    }
    fn set_header_map(&self, header_map: Option<&mut CefStringMultimap>) {
        unsafe {
            if let Some(f) = self.0.set_header_map {
                let arg_header_map = header_map;
                let arg_self_ = self.into_raw();
                let arg_header_map = arg_header_map
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_header_map);
            }
        }
    }
    fn url(&self) -> CefStringUserfree {
        unsafe {
            self.0
                .get_url
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn set_url(&self, url: Option<&CefString>) {
        unsafe {
            if let Some(f) = self.0.set_url {
                let arg_url = url;
                let arg_self_ = self.into_raw();
                let arg_url = arg_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null());
                f(arg_self_, arg_url);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_response_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_response_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Response {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_response_t> for &Response {
    fn into_raw(self) -> *mut _cef_response_t {
        ImplResponse::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_response_t> for &mut Response {
    fn into_raw(self) -> *mut _cef_response_t {
        ImplResponse::get_raw(self)
    }
}
impl ConvertReturnValue<Response> for *mut _cef_response_t {
    fn wrap_result(self) -> Response {
        Response(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Response> for *mut _cef_response_t {
    fn from(value: Response) -> Self {
        let object = ImplResponse::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Response {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_resource_skip_callback_t`] for more documentation.
#[derive(Clone)]
pub struct ResourceSkipCallback(RefGuard<_cef_resource_skip_callback_t>);
pub trait ImplResourceSkipCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_resource_skip_callback_t::cont`] for more documentation."]
    fn cont(&self, bytes_skipped: i64);
    fn get_raw(&self) -> *mut _cef_resource_skip_callback_t;
}
impl ImplResourceSkipCallback for ResourceSkipCallback {
    fn cont(&self, bytes_skipped: i64) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_bytes_skipped = bytes_skipped;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_bytes_skipped);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_resource_skip_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_resource_skip_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResourceSkipCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_resource_skip_callback_t> for &ResourceSkipCallback {
    fn into_raw(self) -> *mut _cef_resource_skip_callback_t {
        ImplResourceSkipCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_resource_skip_callback_t> for &mut ResourceSkipCallback {
    fn into_raw(self) -> *mut _cef_resource_skip_callback_t {
        ImplResourceSkipCallback::get_raw(self)
    }
}
impl ConvertReturnValue<ResourceSkipCallback> for *mut _cef_resource_skip_callback_t {
    fn wrap_result(self) -> ResourceSkipCallback {
        ResourceSkipCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResourceSkipCallback> for *mut _cef_resource_skip_callback_t {
    fn from(value: ResourceSkipCallback) -> Self {
        let object = ImplResourceSkipCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResourceSkipCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_resource_read_callback_t`] for more documentation.
#[derive(Clone)]
pub struct ResourceReadCallback(RefGuard<_cef_resource_read_callback_t>);
pub trait ImplResourceReadCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_resource_read_callback_t::cont`] for more documentation."]
    fn cont(&self, bytes_read: ::std::os::raw::c_int);
    fn get_raw(&self) -> *mut _cef_resource_read_callback_t;
}
impl ImplResourceReadCallback for ResourceReadCallback {
    fn cont(&self, bytes_read: ::std::os::raw::c_int) {
        unsafe {
            if let Some(f) = self.0.cont {
                let arg_bytes_read = bytes_read;
                let arg_self_ = self.into_raw();
                f(arg_self_, arg_bytes_read);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_resource_read_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_resource_read_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResourceReadCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_resource_read_callback_t> for &ResourceReadCallback {
    fn into_raw(self) -> *mut _cef_resource_read_callback_t {
        ImplResourceReadCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_resource_read_callback_t> for &mut ResourceReadCallback {
    fn into_raw(self) -> *mut _cef_resource_read_callback_t {
        ImplResourceReadCallback::get_raw(self)
    }
}
impl ConvertReturnValue<ResourceReadCallback> for *mut _cef_resource_read_callback_t {
    fn wrap_result(self) -> ResourceReadCallback {
        ResourceReadCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResourceReadCallback> for *mut _cef_resource_read_callback_t {
    fn from(value: ResourceReadCallback) -> Self {
        let object = ImplResourceReadCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResourceReadCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_resource_handler_t`] for more documentation.
#[derive(Clone)]
pub struct ResourceHandler(RefGuard<_cef_resource_handler_t>);
impl ResourceHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapResourceHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplResourceHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapResourceHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_resource_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapResourceHandler: ImplResourceHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_resource_handler_t, Self>);
}
pub trait ImplResourceHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_resource_handler_t::open`] for more documentation."]
    fn open(
        &self,
        request: Option<&mut Request>,
        handle_request: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_handler_t::process_request`] for more documentation."]
    fn process_request(
        &self,
        request: Option<&mut Request>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_handler_t::get_response_headers`] for more documentation."]
    fn response_headers(
        &self,
        response: Option<&mut Response>,
        response_length: Option<&mut i64>,
        redirect_url: Option<&mut CefString>,
    ) {
    }
    #[doc = "See [`_cef_resource_handler_t::skip`] for more documentation."]
    fn skip(
        &self,
        bytes_to_skip: i64,
        bytes_skipped: Option<&mut i64>,
        callback: Option<&mut ResourceSkipCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_handler_t::read`] for more documentation."]
    fn read(
        &self,
        data_out: *mut u8,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut ResourceReadCallback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_handler_t::read_response`] for more documentation."]
    fn read_response(
        &self,
        data_out: *mut u8,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_handler_t::cancel`] for more documentation."]
    fn cancel(&self) {}
    fn init_methods(object: &mut _cef_resource_handler_t) {
        impl_cef_resource_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_resource_handler_t;
}
mod impl_cef_resource_handler_t {
    use super::*;
    pub fn init_methods<I: ImplResourceHandler>(object: &mut _cef_resource_handler_t) {
        object.open = Some(open::<I>);
        object.process_request = Some(process_request::<I>);
        object.get_response_headers = Some(get_response_headers::<I>);
        object.skip = Some(skip::<I>);
        object.read = Some(read::<I>);
        object.read_response = Some(read_response::<I>);
        object.cancel = Some(cancel::<I>);
    }
    extern "C" fn open<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        request: *mut _cef_request_t,
        handle_request: *mut ::std::os::raw::c_int,
        callback: *mut _cef_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_request, arg_handle_request, arg_callback) =
            (self_, request, handle_request, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_handle_request = if arg_handle_request.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_handle_request,
            ))
        };
        let arg_handle_request = arg_handle_request.as_mut().map(|arg| arg.as_mut());
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| Callback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplResourceHandler::open(
            &arg_self_.interface,
            arg_request,
            arg_handle_request,
            arg_callback,
        )
    }
    extern "C" fn process_request<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        request: *mut _cef_request_t,
        callback: *mut _cef_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_request, arg_callback) = (self_, request, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| Callback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplResourceHandler::process_request(&arg_self_.interface, arg_request, arg_callback)
    }
    extern "C" fn get_response_headers<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        response: *mut _cef_response_t,
        response_length: *mut i64,
        redirect_url: *mut cef_string_t,
    ) {
        let (arg_self_, arg_response, arg_response_length, arg_redirect_url) =
            (self_, response, response_length, redirect_url);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        let mut arg_response_length = if arg_response_length.is_null() {
            None
        } else {
            Some(WrapParamRef::<i64, _>::from(arg_response_length))
        };
        let arg_response_length = arg_response_length.as_mut().map(|arg| arg.as_mut());
        let mut arg_redirect_url = if arg_redirect_url.is_null() {
            None
        } else {
            Some(arg_redirect_url.into())
        };
        let arg_redirect_url = arg_redirect_url.as_mut();
        ImplResourceHandler::response_headers(
            &arg_self_.interface,
            arg_response,
            arg_response_length,
            arg_redirect_url,
        )
    }
    extern "C" fn skip<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        bytes_to_skip: i64,
        bytes_skipped: *mut i64,
        callback: *mut _cef_resource_skip_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_bytes_to_skip, arg_bytes_skipped, arg_callback) =
            (self_, bytes_to_skip, bytes_skipped, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_bytes_to_skip = arg_bytes_to_skip.into_raw();
        let mut arg_bytes_skipped = if arg_bytes_skipped.is_null() {
            None
        } else {
            Some(WrapParamRef::<i64, _>::from(arg_bytes_skipped))
        };
        let arg_bytes_skipped = arg_bytes_skipped.as_mut().map(|arg| arg.as_mut());
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| ResourceSkipCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplResourceHandler::skip(
            &arg_self_.interface,
            arg_bytes_to_skip,
            arg_bytes_skipped,
            arg_callback,
        )
    }
    extern "C" fn read<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        data_out: *mut ::std::os::raw::c_void,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: *mut ::std::os::raw::c_int,
        callback: *mut _cef_resource_read_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_data_out, arg_bytes_to_read, arg_bytes_read, arg_callback) =
            (self_, data_out, bytes_to_read, bytes_read, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_data_out = arg_data_out.cast();
        let arg_bytes_to_read = arg_bytes_to_read.into_raw();
        let mut arg_bytes_read = if arg_bytes_read.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_bytes_read,
            ))
        };
        let arg_bytes_read = arg_bytes_read.as_mut().map(|arg| arg.as_mut());
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| ResourceReadCallback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplResourceHandler::read(
            &arg_self_.interface,
            arg_data_out,
            arg_bytes_to_read,
            arg_bytes_read,
            arg_callback,
        )
    }
    extern "C" fn read_response<I: ImplResourceHandler>(
        self_: *mut _cef_resource_handler_t,
        data_out: *mut ::std::os::raw::c_void,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: *mut ::std::os::raw::c_int,
        callback: *mut _cef_callback_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_data_out, arg_bytes_to_read, arg_bytes_read, arg_callback) =
            (self_, data_out, bytes_to_read, bytes_read, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let arg_data_out = arg_data_out.cast();
        let arg_bytes_to_read = arg_bytes_to_read.into_raw();
        let mut arg_bytes_read = if arg_bytes_read.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_bytes_read,
            ))
        };
        let arg_bytes_read = arg_bytes_read.as_mut().map(|arg| arg.as_mut());
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| Callback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        ImplResourceHandler::read_response(
            &arg_self_.interface,
            arg_data_out,
            arg_bytes_to_read,
            arg_bytes_read,
            arg_callback,
        )
    }
    extern "C" fn cancel<I: ImplResourceHandler>(self_: *mut _cef_resource_handler_t) {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplResourceHandler::cancel(&arg_self_.interface)
    }
}
impl ImplResourceHandler for ResourceHandler {
    fn open(
        &self,
        request: Option<&mut Request>,
        handle_request: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .open
                .map(|f| {
                    let (arg_request, arg_handle_request, arg_callback) =
                        (request, handle_request, callback);
                    let arg_self_ = self.into_raw();
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_handle_request = arg_handle_request
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_request, arg_handle_request, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn process_request(
        &self,
        request: Option<&mut Request>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .process_request
                .map(|f| {
                    let (arg_request, arg_callback) = (request, callback);
                    let arg_self_ = self.into_raw();
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_request, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn response_headers(
        &self,
        response: Option<&mut Response>,
        response_length: Option<&mut i64>,
        redirect_url: Option<&mut CefString>,
    ) {
        unsafe {
            if let Some(f) = self.0.get_response_headers {
                let (arg_response, arg_response_length, arg_redirect_url) =
                    (response, response_length, redirect_url);
                let arg_self_ = self.into_raw();
                let arg_response = arg_response
                    .map(|arg| {
                        arg.add_ref();
                        ImplResponse::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_response_length = arg_response_length
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                let arg_redirect_url = arg_redirect_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_response,
                    arg_response_length,
                    arg_redirect_url,
                );
            }
        }
    }
    fn skip(
        &self,
        bytes_to_skip: i64,
        bytes_skipped: Option<&mut i64>,
        callback: Option<&mut ResourceSkipCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .skip
                .map(|f| {
                    let (arg_bytes_to_skip, arg_bytes_skipped, arg_callback) =
                        (bytes_to_skip, bytes_skipped, callback);
                    let arg_self_ = self.into_raw();
                    let arg_bytes_skipped = arg_bytes_skipped
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplResourceSkipCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_bytes_to_skip,
                        arg_bytes_skipped,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn read(
        &self,
        data_out: *mut u8,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut ResourceReadCallback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .read
                .map(|f| {
                    let (arg_data_out, arg_bytes_to_read, arg_bytes_read, arg_callback) =
                        (data_out, bytes_to_read, bytes_read, callback);
                    let arg_self_ = self.into_raw();
                    let arg_data_out = arg_data_out.cast();
                    let arg_bytes_read = arg_bytes_read
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplResourceReadCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_data_out,
                        arg_bytes_to_read,
                        arg_bytes_read,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn read_response(
        &self,
        data_out: *mut u8,
        bytes_to_read: ::std::os::raw::c_int,
        bytes_read: Option<&mut ::std::os::raw::c_int>,
        callback: Option<&mut Callback>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .read_response
                .map(|f| {
                    let (arg_data_out, arg_bytes_to_read, arg_bytes_read, arg_callback) =
                        (data_out, bytes_to_read, bytes_read, callback);
                    let arg_self_ = self.into_raw();
                    let arg_data_out = arg_data_out.cast();
                    let arg_bytes_read = arg_bytes_read
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_data_out,
                        arg_bytes_to_read,
                        arg_bytes_read,
                        arg_callback,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn cancel(&self) {
        unsafe {
            if let Some(f) = self.0.cancel {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_resource_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_resource_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResourceHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_resource_handler_t> for &ResourceHandler {
    fn into_raw(self) -> *mut _cef_resource_handler_t {
        ImplResourceHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_resource_handler_t> for &mut ResourceHandler {
    fn into_raw(self) -> *mut _cef_resource_handler_t {
        ImplResourceHandler::get_raw(self)
    }
}
impl ConvertReturnValue<ResourceHandler> for *mut _cef_resource_handler_t {
    fn wrap_result(self) -> ResourceHandler {
        ResourceHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResourceHandler> for *mut _cef_resource_handler_t {
    fn from(value: ResourceHandler) -> Self {
        let object = ImplResourceHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResourceHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_response_filter_t`] for more documentation.
#[derive(Clone)]
pub struct ResponseFilter(RefGuard<_cef_response_filter_t>);
impl ResponseFilter {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapResponseFilter,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplResponseFilter>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapResponseFilter>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_response_filter_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapResponseFilter: ImplResponseFilter {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_response_filter_t, Self>);
}
pub trait ImplResponseFilter: Clone + Sized + Rc {
    #[doc = "See [`_cef_response_filter_t::init_filter`] for more documentation."]
    fn init_filter(&self) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_response_filter_t::filter`] for more documentation."]
    fn filter(
        &self,
        data_in: Option<&mut Vec<u8>>,
        data_in_read: Option<&mut usize>,
        data_out: Option<&mut Vec<u8>>,
        data_out_written: Option<&mut usize>,
    ) -> ResponseFilterStatus {
        Default::default()
    }
    fn init_methods(object: &mut _cef_response_filter_t) {
        impl_cef_response_filter_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_response_filter_t;
}
mod impl_cef_response_filter_t {
    use super::*;
    pub fn init_methods<I: ImplResponseFilter>(object: &mut _cef_response_filter_t) {
        object.init_filter = Some(init_filter::<I>);
        object.filter = Some(filter::<I>);
    }
    extern "C" fn init_filter<I: ImplResponseFilter>(
        self_: *mut _cef_response_filter_t,
    ) -> ::std::os::raw::c_int {
        let arg_self_ = self_;
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        ImplResponseFilter::init_filter(&arg_self_.interface)
    }
    extern "C" fn filter<I: ImplResponseFilter>(
        self_: *mut _cef_response_filter_t,
        data_in: *mut ::std::os::raw::c_void,
        data_in_size: usize,
        data_in_read: *mut usize,
        data_out: *mut ::std::os::raw::c_void,
        data_out_size: usize,
        data_out_written: *mut usize,
    ) -> cef_response_filter_status_t {
        let (
            arg_self_,
            arg_data_in,
            arg_data_in_size,
            arg_data_in_read,
            arg_data_out,
            arg_data_out_size,
            arg_data_out_written,
        ) = (
            self_,
            data_in,
            data_in_size,
            data_in_read,
            data_out,
            data_out_size,
            data_out_written,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let out_data_in = (!arg_data_in.is_null() && arg_data_in_size > 0).then(|| unsafe {
            std::slice::from_raw_parts_mut(arg_data_in.cast(), arg_data_in_size)
        });
        let mut vec_data_in = out_data_in.as_ref().map(|arg| arg.to_vec());
        let arg_data_in = vec_data_in.as_mut();
        let mut arg_data_in_read = if arg_data_in_read.is_null() {
            None
        } else {
            Some(WrapParamRef::<usize, _>::from(arg_data_in_read))
        };
        let arg_data_in_read = arg_data_in_read.as_mut().map(|arg| arg.as_mut());
        let out_data_out = (!arg_data_out.is_null() && arg_data_out_size > 0).then(|| unsafe {
            std::slice::from_raw_parts_mut(arg_data_out.cast(), arg_data_out_size)
        });
        let mut vec_data_out = out_data_out.as_ref().map(|arg| arg.to_vec());
        let arg_data_out = vec_data_out.as_mut();
        let mut arg_data_out_written = if arg_data_out_written.is_null() {
            None
        } else {
            Some(WrapParamRef::<usize, _>::from(arg_data_out_written))
        };
        let arg_data_out_written = arg_data_out_written.as_mut().map(|arg| arg.as_mut());
        let result = ImplResponseFilter::filter(
            &arg_self_.interface,
            arg_data_in,
            arg_data_in_read,
            arg_data_out,
            arg_data_out_written,
        );
        if let (Some(out_data_in), Some(vec_data_in)) = (out_data_in, vec_data_in.as_mut()) {
            let size = vec_data_in.len().min(out_data_in.len());
            out_data_in[..size].copy_from_slice(&vec_data_in[..size]);
        }
        if let (Some(out_data_out), Some(vec_data_out)) = (out_data_out, vec_data_out.as_mut()) {
            let size = vec_data_out.len().min(out_data_out.len());
            out_data_out[..size].copy_from_slice(&vec_data_out[..size]);
        }
        result.into()
    }
}
impl ImplResponseFilter for ResponseFilter {
    fn init_filter(&self) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .init_filter
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn filter(
        &self,
        data_in: Option<&mut Vec<u8>>,
        data_in_read: Option<&mut usize>,
        data_out: Option<&mut Vec<u8>>,
        data_out_written: Option<&mut usize>,
    ) -> ResponseFilterStatus {
        unsafe {
            self.0
                .filter
                .map(|f| {
                    let (arg_data_in, arg_data_in_read, arg_data_out, arg_data_out_written) =
                        (data_in, data_in_read, data_out, data_out_written);
                    let arg_self_ = self.into_raw();
                    let arg_data_in_size = arg_data_in
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let mut out_data_in = arg_data_in;
                    let arg_data_in = out_data_in
                        .as_mut()
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_mut_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_data_in_read = arg_data_in_read
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let arg_data_out_size = arg_data_out
                        .as_ref()
                        .map(|arg| arg.len())
                        .unwrap_or_default();
                    let mut out_data_out = arg_data_out;
                    let arg_data_out = out_data_out
                        .as_mut()
                        .and_then(|arg| {
                            if arg.is_empty() {
                                None
                            } else {
                                Some(arg.as_mut_ptr().cast())
                            }
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_data_out_written = arg_data_out_written
                        .map(std::ptr::from_mut)
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(
                        arg_self_,
                        arg_data_in,
                        arg_data_in_size,
                        arg_data_in_read,
                        arg_data_out,
                        arg_data_out_size,
                        arg_data_out_written,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_response_filter_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_response_filter_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResponseFilter {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_response_filter_t> for &ResponseFilter {
    fn into_raw(self) -> *mut _cef_response_filter_t {
        ImplResponseFilter::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_response_filter_t> for &mut ResponseFilter {
    fn into_raw(self) -> *mut _cef_response_filter_t {
        ImplResponseFilter::get_raw(self)
    }
}
impl ConvertReturnValue<ResponseFilter> for *mut _cef_response_filter_t {
    fn wrap_result(self) -> ResponseFilter {
        ResponseFilter(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResponseFilter> for *mut _cef_response_filter_t {
    fn from(value: ResponseFilter) -> Self {
        let object = ImplResponseFilter::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResponseFilter {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_resource_request_handler_t`] for more documentation.
#[derive(Clone)]
pub struct ResourceRequestHandler(RefGuard<_cef_resource_request_handler_t>);
impl ResourceRequestHandler {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapResourceRequestHandler,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplResourceRequestHandler>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapResourceRequestHandler>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_resource_request_handler_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapResourceRequestHandler: ImplResourceRequestHandler {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_resource_request_handler_t, Self>);
}
pub trait ImplResourceRequestHandler: Clone + Sized + Rc {
    #[doc = "See [`_cef_resource_request_handler_t::get_cookie_access_filter`] for more documentation."]
    fn cookie_access_filter(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
    ) -> Option<CookieAccessFilter> {
        Default::default()
    }
    #[doc = "See [`_cef_resource_request_handler_t::on_before_resource_load`] for more documentation."]
    fn on_before_resource_load(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        callback: Option<&mut Callback>,
    ) -> ReturnValue {
        Default::default()
    }
    #[doc = "See [`_cef_resource_request_handler_t::get_resource_handler`] for more documentation."]
    fn resource_handler(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
    ) -> Option<ResourceHandler> {
        Default::default()
    }
    #[doc = "See [`_cef_resource_request_handler_t::on_resource_redirect`] for more documentation."]
    fn on_resource_redirect(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        new_url: Option<&mut CefString>,
    ) {
    }
    #[doc = "See [`_cef_resource_request_handler_t::on_resource_response`] for more documentation."]
    fn on_resource_response(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_resource_request_handler_t::get_resource_response_filter`] for more documentation."]
    fn resource_response_filter(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
    ) -> Option<ResponseFilter> {
        Default::default()
    }
    #[doc = "See [`_cef_resource_request_handler_t::on_resource_load_complete`] for more documentation."]
    fn on_resource_load_complete(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        status: UrlrequestStatus,
        received_content_length: i64,
    ) {
    }
    #[doc = "See [`_cef_resource_request_handler_t::on_protocol_execution`] for more documentation."]
    fn on_protocol_execution(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        allow_os_execution: Option<&mut ::std::os::raw::c_int>,
    ) {
    }
    fn init_methods(object: &mut _cef_resource_request_handler_t) {
        impl_cef_resource_request_handler_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_resource_request_handler_t;
}
mod impl_cef_resource_request_handler_t {
    use super::*;
    pub fn init_methods<I: ImplResourceRequestHandler>(
        object: &mut _cef_resource_request_handler_t,
    ) {
        object.get_cookie_access_filter = Some(get_cookie_access_filter::<I>);
        object.on_before_resource_load = Some(on_before_resource_load::<I>);
        object.get_resource_handler = Some(get_resource_handler::<I>);
        object.on_resource_redirect = Some(on_resource_redirect::<I>);
        object.on_resource_response = Some(on_resource_response::<I>);
        object.get_resource_response_filter = Some(get_resource_response_filter::<I>);
        object.on_resource_load_complete = Some(on_resource_load_complete::<I>);
        object.on_protocol_execution = Some(on_protocol_execution::<I>);
    }
    extern "C" fn get_cookie_access_filter<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
    ) -> *mut _cef_cookie_access_filter_t {
        let (arg_self_, arg_browser, arg_frame, arg_request) = (self_, browser, frame, request);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let result = ImplResourceRequestHandler::cookie_access_filter(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
        );
        result
            .map(|result| result.into())
            .unwrap_or(std::ptr::null_mut())
    }
    extern "C" fn on_before_resource_load<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        callback: *mut _cef_callback_t,
    ) -> cef_return_value_t {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_callback) =
            (self_, browser, frame, request, callback);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_callback = unsafe { arg_callback.as_mut() }
            .map(|arg| Callback(unsafe { RefGuard::from_raw(arg) }));
        let arg_callback = arg_callback.as_mut();
        let result = ImplResourceRequestHandler::on_before_resource_load(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_callback,
        );
        result.into()
    }
    extern "C" fn get_resource_handler<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
    ) -> *mut _cef_resource_handler_t {
        let (arg_self_, arg_browser, arg_frame, arg_request) = (self_, browser, frame, request);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let result = ImplResourceRequestHandler::resource_handler(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
        );
        result
            .map(|result| result.into())
            .unwrap_or(std::ptr::null_mut())
    }
    extern "C" fn on_resource_redirect<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        response: *mut _cef_response_t,
        new_url: *mut cef_string_t,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_response, arg_new_url) =
            (self_, browser, frame, request, response, new_url);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        let mut arg_new_url = if arg_new_url.is_null() {
            None
        } else {
            Some(arg_new_url.into())
        };
        let arg_new_url = arg_new_url.as_mut();
        ImplResourceRequestHandler::on_resource_redirect(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
            arg_new_url,
        )
    }
    extern "C" fn on_resource_response<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        response: *mut _cef_response_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_response) =
            (self_, browser, frame, request, response);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        ImplResourceRequestHandler::on_resource_response(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
        )
    }
    extern "C" fn get_resource_response_filter<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        response: *mut _cef_response_t,
    ) -> *mut _cef_response_filter_t {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_response) =
            (self_, browser, frame, request, response);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        let result = ImplResourceRequestHandler::resource_response_filter(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
        );
        result
            .map(|result| result.into())
            .unwrap_or(std::ptr::null_mut())
    }
    extern "C" fn on_resource_load_complete<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        response: *mut _cef_response_t,
        status: cef_urlrequest_status_t,
        received_content_length: i64,
    ) {
        let (
            arg_self_,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
            arg_status,
            arg_received_content_length,
        ) = (
            self_,
            browser,
            frame,
            request,
            response,
            status,
            received_content_length,
        );
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        let arg_status = arg_status.into_raw();
        let arg_received_content_length = arg_received_content_length.into_raw();
        ImplResourceRequestHandler::on_resource_load_complete(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
            arg_status,
            arg_received_content_length,
        )
    }
    extern "C" fn on_protocol_execution<I: ImplResourceRequestHandler>(
        self_: *mut _cef_resource_request_handler_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        allow_os_execution: *mut ::std::os::raw::c_int,
    ) {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_allow_os_execution) =
            (self_, browser, frame, request, allow_os_execution);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_allow_os_execution = if arg_allow_os_execution.is_null() {
            None
        } else {
            Some(WrapParamRef::<::std::os::raw::c_int, _>::from(
                arg_allow_os_execution,
            ))
        };
        let arg_allow_os_execution = arg_allow_os_execution.as_mut().map(|arg| arg.as_mut());
        ImplResourceRequestHandler::on_protocol_execution(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_allow_os_execution,
        )
    }
}
impl ImplResourceRequestHandler for ResourceRequestHandler {
    fn cookie_access_filter(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
    ) -> Option<CookieAccessFilter> {
        unsafe {
            self.0
                .get_cookie_access_filter
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request) = (browser, frame, request);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn on_before_resource_load(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        callback: Option<&mut Callback>,
    ) -> ReturnValue {
        unsafe {
            self.0
                .on_before_resource_load
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request, arg_callback) =
                        (browser, frame, request, callback);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_callback = arg_callback
                        .map(|arg| {
                            arg.add_ref();
                            ImplCallback::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request, arg_callback);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn resource_handler(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
    ) -> Option<ResourceHandler> {
        unsafe {
            self.0
                .get_resource_handler
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request) = (browser, frame, request);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn on_resource_redirect(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        new_url: Option<&mut CefString>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_resource_redirect {
                let (arg_browser, arg_frame, arg_request, arg_response, arg_new_url) =
                    (browser, frame, request, response, new_url);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_request = arg_request
                    .map(|arg| {
                        arg.add_ref();
                        ImplRequest::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_response = arg_response
                    .map(|arg| {
                        arg.add_ref();
                        ImplResponse::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_new_url = arg_new_url
                    .map(|arg| arg.into_raw())
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_browser,
                    arg_frame,
                    arg_request,
                    arg_response,
                    arg_new_url,
                );
            }
        }
    }
    fn on_resource_response(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .on_resource_response
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request, arg_response) =
                        (browser, frame, request, response);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_response = arg_response
                        .map(|arg| {
                            arg.add_ref();
                            ImplResponse::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request, arg_response);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn resource_response_filter(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
    ) -> Option<ResponseFilter> {
        unsafe {
            self.0
                .get_resource_response_filter
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request, arg_response) =
                        (browser, frame, request, response);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_response = arg_response
                        .map(|arg| {
                            arg.add_ref();
                            ImplResponse::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request, arg_response);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn on_resource_load_complete(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        status: UrlrequestStatus,
        received_content_length: i64,
    ) {
        unsafe {
            if let Some(f) = self.0.on_resource_load_complete {
                let (
                    arg_browser,
                    arg_frame,
                    arg_request,
                    arg_response,
                    arg_status,
                    arg_received_content_length,
                ) = (
                    browser,
                    frame,
                    request,
                    response,
                    status,
                    received_content_length,
                );
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_request = arg_request
                    .map(|arg| {
                        arg.add_ref();
                        ImplRequest::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_response = arg_response
                    .map(|arg| {
                        arg.add_ref();
                        ImplResponse::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_status = arg_status.into_raw();
                f(
                    arg_self_,
                    arg_browser,
                    arg_frame,
                    arg_request,
                    arg_response,
                    arg_status,
                    arg_received_content_length,
                );
            }
        }
    }
    fn on_protocol_execution(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        allow_os_execution: Option<&mut ::std::os::raw::c_int>,
    ) {
        unsafe {
            if let Some(f) = self.0.on_protocol_execution {
                let (arg_browser, arg_frame, arg_request, arg_allow_os_execution) =
                    (browser, frame, request, allow_os_execution);
                let arg_self_ = self.into_raw();
                let arg_browser = arg_browser
                    .map(|arg| {
                        arg.add_ref();
                        ImplBrowser::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_frame = arg_frame
                    .map(|arg| {
                        arg.add_ref();
                        ImplFrame::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_request = arg_request
                    .map(|arg| {
                        arg.add_ref();
                        ImplRequest::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                let arg_allow_os_execution = arg_allow_os_execution
                    .map(std::ptr::from_mut)
                    .unwrap_or(std::ptr::null_mut());
                f(
                    arg_self_,
                    arg_browser,
                    arg_frame,
                    arg_request,
                    arg_allow_os_execution,
                );
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_resource_request_handler_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_resource_request_handler_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for ResourceRequestHandler {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_resource_request_handler_t> for &ResourceRequestHandler {
    fn into_raw(self) -> *mut _cef_resource_request_handler_t {
        ImplResourceRequestHandler::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_resource_request_handler_t> for &mut ResourceRequestHandler {
    fn into_raw(self) -> *mut _cef_resource_request_handler_t {
        ImplResourceRequestHandler::get_raw(self)
    }
}
impl ConvertReturnValue<ResourceRequestHandler> for *mut _cef_resource_request_handler_t {
    fn wrap_result(self) -> ResourceRequestHandler {
        ResourceRequestHandler(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<ResourceRequestHandler> for *mut _cef_resource_request_handler_t {
    fn from(value: ResourceRequestHandler) -> Self {
        let object = ImplResourceRequestHandler::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for ResourceRequestHandler {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_cookie_access_filter_t`] for more documentation.
#[derive(Clone)]
pub struct CookieAccessFilter(RefGuard<_cef_cookie_access_filter_t>);
impl CookieAccessFilter {
    pub fn new<T>(interface: T) -> Self
    where
        T: WrapCookieAccessFilter,
    {
        unsafe {
            let mut cef_object = std::mem::zeroed();
            <T as ImplCookieAccessFilter>::init_methods(&mut cef_object);
            let object = RcImpl::new(cef_object, interface);
            <T as WrapCookieAccessFilter>::wrap_rc(&mut (*object).interface, object);
            let object: *mut _cef_cookie_access_filter_t = object.cast();
            object.wrap_result()
        }
    }
}
pub trait WrapCookieAccessFilter: ImplCookieAccessFilter {
    fn wrap_rc(&mut self, object: *mut RcImpl<_cef_cookie_access_filter_t, Self>);
}
pub trait ImplCookieAccessFilter: Clone + Sized + Rc {
    #[doc = "See [`_cef_cookie_access_filter_t::can_send_cookie`] for more documentation."]
    fn can_send_cookie(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        cookie: Option<&Cookie>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    #[doc = "See [`_cef_cookie_access_filter_t::can_save_cookie`] for more documentation."]
    fn can_save_cookie(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        cookie: Option<&Cookie>,
    ) -> ::std::os::raw::c_int {
        Default::default()
    }
    fn init_methods(object: &mut _cef_cookie_access_filter_t) {
        impl_cef_cookie_access_filter_t::init_methods::<Self>(object);
    }
    fn get_raw(&self) -> *mut _cef_cookie_access_filter_t;
}
mod impl_cef_cookie_access_filter_t {
    use super::*;
    pub fn init_methods<I: ImplCookieAccessFilter>(object: &mut _cef_cookie_access_filter_t) {
        object.can_send_cookie = Some(can_send_cookie::<I>);
        object.can_save_cookie = Some(can_save_cookie::<I>);
    }
    extern "C" fn can_send_cookie<I: ImplCookieAccessFilter>(
        self_: *mut _cef_cookie_access_filter_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        cookie: *const _cef_cookie_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_cookie) =
            (self_, browser, frame, request, cookie);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let arg_cookie = if arg_cookie.is_null() {
            None
        } else {
            Some(WrapParamRef::<Cookie, _>::from(arg_cookie))
        };
        let arg_cookie = arg_cookie.as_ref().map(|arg| arg.as_ref());
        ImplCookieAccessFilter::can_send_cookie(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_cookie,
        )
    }
    extern "C" fn can_save_cookie<I: ImplCookieAccessFilter>(
        self_: *mut _cef_cookie_access_filter_t,
        browser: *mut _cef_browser_t,
        frame: *mut _cef_frame_t,
        request: *mut _cef_request_t,
        response: *mut _cef_response_t,
        cookie: *const _cef_cookie_t,
    ) -> ::std::os::raw::c_int {
        let (arg_self_, arg_browser, arg_frame, arg_request, arg_response, arg_cookie) =
            (self_, browser, frame, request, response, cookie);
        let arg_self_: &RcImpl<_, I> = RcImpl::get(arg_self_);
        let mut arg_browser =
            unsafe { arg_browser.as_mut() }.map(|arg| Browser(unsafe { RefGuard::from_raw(arg) }));
        let arg_browser = arg_browser.as_mut();
        let mut arg_frame =
            unsafe { arg_frame.as_mut() }.map(|arg| Frame(unsafe { RefGuard::from_raw(arg) }));
        let arg_frame = arg_frame.as_mut();
        let mut arg_request =
            unsafe { arg_request.as_mut() }.map(|arg| Request(unsafe { RefGuard::from_raw(arg) }));
        let arg_request = arg_request.as_mut();
        let mut arg_response = unsafe { arg_response.as_mut() }
            .map(|arg| Response(unsafe { RefGuard::from_raw(arg) }));
        let arg_response = arg_response.as_mut();
        let arg_cookie = if arg_cookie.is_null() {
            None
        } else {
            Some(WrapParamRef::<Cookie, _>::from(arg_cookie))
        };
        let arg_cookie = arg_cookie.as_ref().map(|arg| arg.as_ref());
        ImplCookieAccessFilter::can_save_cookie(
            &arg_self_.interface,
            arg_browser,
            arg_frame,
            arg_request,
            arg_response,
            arg_cookie,
        )
    }
}
impl ImplCookieAccessFilter for CookieAccessFilter {
    fn can_send_cookie(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        cookie: Option<&Cookie>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_send_cookie
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request, arg_cookie) =
                        (browser, frame, request, cookie);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_cookie = arg_cookie.cloned().map(|arg| arg.into());
                    let arg_cookie = arg_cookie
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let result = f(arg_self_, arg_browser, arg_frame, arg_request, arg_cookie);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn can_save_cookie(
        &self,
        browser: Option<&mut Browser>,
        frame: Option<&mut Frame>,
        request: Option<&mut Request>,
        response: Option<&mut Response>,
        cookie: Option<&Cookie>,
    ) -> ::std::os::raw::c_int {
        unsafe {
            self.0
                .can_save_cookie
                .map(|f| {
                    let (arg_browser, arg_frame, arg_request, arg_response, arg_cookie) =
                        (browser, frame, request, response, cookie);
                    let arg_self_ = self.into_raw();
                    let arg_browser = arg_browser
                        .map(|arg| {
                            arg.add_ref();
                            ImplBrowser::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_frame = arg_frame
                        .map(|arg| {
                            arg.add_ref();
                            ImplFrame::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_request = arg_request
                        .map(|arg| {
                            arg.add_ref();
                            ImplRequest::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_response = arg_response
                        .map(|arg| {
                            arg.add_ref();
                            ImplResponse::get_raw(arg)
                        })
                        .unwrap_or(std::ptr::null_mut());
                    let arg_cookie = arg_cookie.cloned().map(|arg| arg.into());
                    let arg_cookie = arg_cookie
                        .as_ref()
                        .map(std::ptr::from_ref)
                        .unwrap_or(std::ptr::null());
                    let result = f(
                        arg_self_,
                        arg_browser,
                        arg_frame,
                        arg_request,
                        arg_response,
                        arg_cookie,
                    );
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_cookie_access_filter_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_cookie_access_filter_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for CookieAccessFilter {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_cookie_access_filter_t> for &CookieAccessFilter {
    fn into_raw(self) -> *mut _cef_cookie_access_filter_t {
        ImplCookieAccessFilter::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_cookie_access_filter_t> for &mut CookieAccessFilter {
    fn into_raw(self) -> *mut _cef_cookie_access_filter_t {
        ImplCookieAccessFilter::get_raw(self)
    }
}
impl ConvertReturnValue<CookieAccessFilter> for *mut _cef_cookie_access_filter_t {
    fn wrap_result(self) -> CookieAccessFilter {
        CookieAccessFilter(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<CookieAccessFilter> for *mut _cef_cookie_access_filter_t {
    fn from(value: CookieAccessFilter) -> Self {
        let object = ImplCookieAccessFilter::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for CookieAccessFilter {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_sslinfo_t`] for more documentation.
#[derive(Clone)]
pub struct Sslinfo(RefGuard<_cef_sslinfo_t>);
pub trait ImplSslinfo: Clone + Sized + Rc {
    #[doc = "See [`_cef_sslinfo_t::get_cert_status`] for more documentation."]
    fn cert_status(&self) -> CertStatus;
    #[doc = "See [`_cef_sslinfo_t::get_x509_certificate`] for more documentation."]
    fn x509_certificate(&self) -> Option<X509Certificate>;
    fn get_raw(&self) -> *mut _cef_sslinfo_t;
}
impl ImplSslinfo for Sslinfo {
    fn cert_status(&self) -> CertStatus {
        unsafe {
            self.0
                .get_cert_status
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    result.wrap_result()
                })
                .unwrap_or_default()
        }
    }
    fn x509_certificate(&self) -> Option<X509Certificate> {
        unsafe {
            self.0
                .get_x509_certificate
                .map(|f| {
                    let arg_self_ = self.into_raw();
                    let result = f(arg_self_);
                    if result.is_null() {
                        None
                    } else {
                        Some(result.wrap_result())
                    }
                })
                .unwrap_or_default()
        }
    }
    fn get_raw(&self) -> *mut _cef_sslinfo_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_sslinfo_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for Sslinfo {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_sslinfo_t> for &Sslinfo {
    fn into_raw(self) -> *mut _cef_sslinfo_t {
        ImplSslinfo::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_sslinfo_t> for &mut Sslinfo {
    fn into_raw(self) -> *mut _cef_sslinfo_t {
        ImplSslinfo::get_raw(self)
    }
}
impl ConvertReturnValue<Sslinfo> for *mut _cef_sslinfo_t {
    fn wrap_result(self) -> Sslinfo {
        Sslinfo(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<Sslinfo> for *mut _cef_sslinfo_t {
    fn from(value: Sslinfo) -> Self {
        let object = ImplSslinfo::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for Sslinfo {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_unresponsive_process_callback_t`] for more documentation.
#[derive(Clone)]
pub struct UnresponsiveProcessCallback(RefGuard<_cef_unresponsive_process_callback_t>);
pub trait ImplUnresponsiveProcessCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_unresponsive_process_callback_t::wait`] for more documentation."]
    fn wait(&self);
    #[doc = "See [`_cef_unresponsive_process_callback_t::terminate`] for more documentation."]
    fn terminate(&self);
    fn get_raw(&self) -> *mut _cef_unresponsive_process_callback_t;
}
impl ImplUnresponsiveProcessCallback for UnresponsiveProcessCallback {
    fn wait(&self) {
        unsafe {
            if let Some(f) = self.0.wait {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn terminate(&self) {
        unsafe {
            if let Some(f) = self.0.terminate {
                let arg_self_ = self.into_raw();
                f(arg_self_);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_unresponsive_process_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_unresponsive_process_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for UnresponsiveProcessCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_unresponsive_process_callback_t> for &UnresponsiveProcessCallback {
    fn into_raw(self) -> *mut _cef_unresponsive_process_callback_t {
        ImplUnresponsiveProcessCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_unresponsive_process_callback_t> for &mut UnresponsiveProcessCallback {
    fn into_raw(self) -> *mut _cef_unresponsive_process_callback_t {
        ImplUnresponsiveProcessCallback::get_raw(self)
    }
}
impl ConvertReturnValue<UnresponsiveProcessCallback> for *mut _cef_unresponsive_process_callback_t {
    fn wrap_result(self) -> UnresponsiveProcessCallback {
        UnresponsiveProcessCallback(unsafe { RefGuard::from_raw(self) })
    }
}
impl From<UnresponsiveProcessCallback> for *mut _cef_unresponsive_process_callback_t {
    fn from(value: UnresponsiveProcessCallback) -> Self {
        let object = ImplUnresponsiveProcessCallback::get_raw(&value);
        std::mem::forget(value);
        object
    }
}
impl Default for UnresponsiveProcessCallback {
    fn default() -> Self {
        unsafe { std::mem::zeroed() }
    }
}

/// See [`_cef_select_client_certificate_callback_t`] for more documentation.
#[derive(Clone)]
pub struct SelectClientCertificateCallback(RefGuard<_cef_select_client_certificate_callback_t>);
pub trait ImplSelectClientCertificateCallback: Clone + Sized + Rc {
    #[doc = "See [`_cef_select_client_certificate_callback_t::select`] for more documentation."]
    fn select(&self, cert: Option<&mut X509Certificate>);
    fn get_raw(&self) -> *mut _cef_select_client_certificate_callback_t;
}
impl ImplSelectClientCertificateCallback for SelectClientCertificateCallback {
    fn select(&self, cert: Option<&mut X509Certificate>) {
        unsafe {
            if let Some(f) = self.0.select {
                let arg_cert = cert;
                let arg_self_ = self.into_raw();
                let arg_cert = arg_cert
                    .map(|arg| {
                        arg.add_ref();
                        ImplX509Certificate::get_raw(arg)
                    })
                    .unwrap_or(std::ptr::null_mut());
                f(arg_self_, arg_cert);
            }
        }
    }
    fn get_raw(&self) -> *mut _cef_select_client_certificate_callback_t {
        unsafe { RefGuard::into_raw(&self.0) }
    }
}
impl Rc for _cef_select_client_certificate_callback_t {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.base.as_base()
    }
}
impl Rc for SelectClientCertificateCallback {
    fn as_base(&self) -> &_cef_base_ref_counted_t {
        self.0.as_base()
    }
}
impl ConvertParam<*mut _cef_select_client_certificate_callback_t>
    for &SelectClientCertificateCallback
{
    fn into_raw(self) -> *mut _cef_select_client_certificate_callback_t {
        ImplSelectClientCertificateCallback::get_raw(self)
    }
}
impl ConvertParam<*mut _cef_select_client_certificate_callback_t>
    for &mut SelectClientCertificateCallback
{
    fn into_raw(self) -> *mut _cef_select_client_certificate_callback_t {
        ImplSelectClientCertificateCallback::get_raw(self)
    }
}
impl ConvertReturnValue<SelectClientCertificateCallback>
    for *mut _cef_select_client_certificate_callback_t
{
    fn wrap_result(self) -> SelectClientCertificateCallback {
        SelectClientCertificateCallback(unsafe { RefGuard::from_raw(self) })