#include "quill/Backend.h"
#include "quill/Frontend.h"
#include "quill/LogMacros.h"
#include "quill/Logger.h"
#include "quill/sinks/ConsoleSink.h"

#include "quill/bundled/fmt/ostream.h"
#include "quill/std/Array.h"
#include "quill/std/Chrono.h"

#include <iostream>
#include <string>
#include <utility>

/**
 * Trivial logging example to console
 * Note: You can also pass STL types by including the relevant header files from quill/std/
 */

int main()
{
  quill::BackendOptions backend_options;
  quill::Backend::start(backend_options);

  // Frontend
  auto console_sink = quill::Frontend::create_or_get_sink<quill::ConsoleSink>("sink_id_1");

  quill::Logger* logger = quill::Frontend::create_or_get_logger("root", std::move(console_sink));

  // Change the LogLevel to print everything
  logger->set_log_level(quill::LogLevel::TraceL3);

  // A log message with number 123
  int a = 123;
  std::string l = "log";
  LOG_INFO(logger, "A {} message with number {}", l, a);

  // libfmt formatting language is supported 3.14e+00
  double pi = 3.141592653589793;
  LOG_INFO(logger, "libfmt formatting language is supported {:.2e}", pi);

  // Logging STD types is supported [1, 2, 3]
  std::array<int, 3> arr = {1, 2, 3};
  LOG_INFO(logger, "Logging STD types is supported {}", arr);

  // Logging STD types is supported [arr: [1, 2, 3]]
  LOGV_INFO(logger, "Logging STD types is supported", arr);

  // A message with two variables [a: 123, b: 3.17]
  double b = 3.17;
  LOGV_NOTICE(logger, "A message with two variables", a, b);

  for (uint32_t i = 0; i < 40; ++i)
  {
    // Will only log the message once per second
    LOG_INFO_LIMIT(std::chrono::seconds{1}, logger, "A {} message with number {}", l, a);
    LOGV_INFO_LIMIT(std::chrono::seconds{1}, logger, "A message with two variables", a, b);

    if (i % 10 == 0)
    {
      std::this_thread::sleep_for(std::chrono::milliseconds{(i / 10) * 500});
    }
  }

  for (uint32_t i = 0; i < 20; ++i)
  {
    // Will only log the message once per N occurrences second
    LOG_INFO_LIMIT_EVERY_N(10, logger, "Another {} message with occurrence {}", l, i);
    LOGV_INFO_LIMIT_EVERY_N(10, logger, "Another message with two variables", a, i);
  }
}
