from pathlib import Path
from PyQt6.QtWidgets import (
    QDialog,
    QTabWidget,
    QTextBrowser,
    QVBoxLayout,
    QHBoxLayout,
    QLabel,
    QPushButton,
    QFrame,
)
from PyQt6.QtGui import QPixmap, QFont, QIcon
from PyQt6.QtCore import Qt

from .app_info import APP_NAME, APP_VERSION, HOMEPAGE, REPO_URL, LICENSE_PATH


class AboutDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("About QuickBib")
        self.resize(600, 370)

        # Main layout
        vbox = QVBoxLayout()
        vbox.setContentsMargins(12, 12, 12, 12)
        vbox.setSpacing(12)
        self.setLayout(vbox)

        # Header: icon + title
        header = QHBoxLayout()
        header.setSpacing(12)
        vbox.addLayout(header)

        icon_label = QLabel()
        icon_label.setFixedSize(64, 64)
        icon_label.setScaledContents(True)
        try:
            theme_icon = QIcon.fromTheme("io.github.archisman_panigrahi.QuickBib")
            if not theme_icon.isNull():
                pix = theme_icon.pixmap(64, 64)
            else:
                asset_path = Path(__file__).parent.parent / "assets" / "icon" / "64x64" / "io.github.archisman_panigrahi.QuickBib.png"
                if asset_path.exists():
                    pix = QPixmap(str(asset_path))
                    if pix.isNull():
                        pix = QPixmap()
                else:
                    pix = QPixmap()
        except Exception:
            pix = QPixmap()
        icon_label.setPixmap(pix)
        icon_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        header.addWidget(icon_label)

        title_layout = QVBoxLayout()
        title_layout.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
        title_label = QLabel(f"{APP_NAME}")
        title_font = QFont()
        title_font.setPointSize(16)
        title_font.setBold(True)
        title_label.setFont(title_font)
        title_label.setAlignment(Qt.AlignmentFlag.AlignLeft)
        title_layout.addWidget(title_label)

        subtitle = QLabel(f"Version {APP_VERSION}")
        subtitle.setAlignment(Qt.AlignmentFlag.AlignLeft)
        title_layout.addWidget(subtitle)

        header.addLayout(title_layout)

        frame = QFrame()
        frame.setFrameShape(QFrame.Shape.StyledPanel)
        frame.setObjectName("aboutFrame")
        frame_layout = QVBoxLayout()
        frame_layout.setContentsMargins(8, 8, 8, 8)
        frame.setLayout(frame_layout)
        vbox.addWidget(frame)

        tabs = QTabWidget()
        frame_layout.addWidget(tabs)

        about_text = QTextBrowser()
        about_html = f"""
        <p>{APP_NAME} fetches BibTeX entries from DOIs, arXiv IDs, and known journal URLs.
        It is a simple utility to quickly convert identifiers into usable BibTeX records.</p>
        <p> QuickBib uses <a href="https://github.com/archisman-panigrahi/doi2bib3">doi2bib3</a>
        as its backend for DOI to BibTeX conversion.</p>
        <p>
          <b>Homepage:</b> <a href="{HOMEPAGE}">{HOMEPAGE}</a><br>
          <b>Source Code:</b> <a href="{REPO_URL}">{REPO_URL}</a>
        </p>
        <p><b>License:</b> Released under the <a href="https://www.gnu.org/licenses/gpl-3.0.en.html#license-text">GNU General Public License Version 3</a>. Full text available on the <i>License</i> tab.</p>
        """
        about_text.setHtml(about_html)
        about_text.setOpenExternalLinks(True)
        tabs.addTab(about_text, "About")

        authors_text = QTextBrowser()
        authors_html = f"""
        <h3>Authors & Contributors</h3>
        <ul>
          <li><a href="https://github.com/archisman-panigrahi/">Archisman Panigrahi</a></li>
        </ul>
        <p> This project was inspired by <a href="https://github.com/bibcure/doi2bib">doi2bib</a>, whose development unfortunately stopped.</p>
        <p> This project would not have been possible without the help from GitHub copilot.</p>
        <p>Bug reports and pull requests are welcome on the <a href="{REPO_URL}">project's GitHub page</a>.</p>
        """
        authors_text.setHtml(authors_html)
        authors_text.setOpenExternalLinks(True)
        tabs.addTab(authors_text, "Authors")

        license_text = QTextBrowser()
        if LICENSE_PATH.exists():
            try:
                license_content = LICENSE_PATH.read_text(encoding="utf-8")
                license_text.setPlainText(license_content)
            except Exception:
                license_text.setHtml("<p>Unable to read LICENSE file.</p>")
        else:
            license_text.setHtml(f"<p>License file not found in repository. See <a href=\"{REPO_URL}\">project page</a>.</p>")
        tabs.addTab(license_text, "License")

        btn_hbox = QHBoxLayout()
        btn_hbox.addStretch()

        dedication = QLabel("<em>Dedicated to all my friends</em>")
        dedication.setAlignment(Qt.AlignmentFlag.AlignCenter)
        dedication.setMinimumWidth(240)
        btn_hbox.addWidget(dedication)

        btn_hbox.addStretch()
        close_btn = QPushButton("\u2715 Close")
        close_btn.clicked.connect(self.reject)
        close_btn.setFixedHeight(28)
        btn_hbox.addWidget(close_btn)
        vbox.addLayout(btn_hbox)
