/*
 * Copyright © 2025 Alain M. (https://github.com/alainm23/planify)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 *
 * Authored by: Alain M. <alainmh23@gmail.com>
 */
 
namespace Chrono {
    /**
     * English casual time expressions parser
     * 
     * Supports: morning, afternoon, evening, night, midnight, midday, noon
     * With optional "this" prefix
     */
    public class ENCasualTimeParser : Object {
        private Regex casual_time_regex;
        
        public ENCasualTimeParser () {
            try {
                casual_time_regex = new Regex (
                    "(?:this)?\\s{0,3}(morning|afternoon|evening|night|midnight|midday|noon)(?=\\W|$)",
                    RegexCompileFlags.CASELESS
                );
            } catch (Error e) {
                warning ("Error creating casual time regex: %s", e.message);
            }
        }
        
        public ParseResult? parse (string text) {
            try {
                MatchInfo match;
                if (!casual_time_regex.match (text, 0, out match)) {
                    return null;
                }
                
                string keyword = match.fetch (1).down ();
                var now = new DateTime.now_local ();
                int hour;
                
                switch (keyword) {
                    case "morning":
                        hour = 9;
                        break;
                    case "afternoon":
                        hour = 14;
                        break;
                    case "evening":
                        hour = 18;
                        break;
                    case "night":
                        hour = 20;
                        break;
                    case "midnight":
                        hour = 0;
                        break;
                    case "midday":
                    case "noon":
                        hour = 12;
                        break;
                    default:
                        return null;
                }
                
                var result = new ParseResult ();
                result.date = new DateTime.local (
                    now.get_year (),
                    now.get_month (),
                    now.get_day_of_month (),
                    hour, 0, 0
                );
                
                int start_pos, end_pos;
                match.fetch_pos (0, out start_pos, out end_pos);
                result.start_index = start_pos;
                result.end_index = end_pos;
                result.matched_text = match.fetch (0);
                
                return result;
            } catch (Error e) {
                return null;
            }
        }
    }
}
