public class Printer : Gtk.PrintOperation {

    private unowned GLib.ListStore? model = null;
    private unowned GLib.Settings? settings = null;
    int lines_per_page = 0;

    public Printer (GLib.ListStore model, GLib.Settings settings) {
        this.model = model;
        this.settings = settings;
    }

    public override void begin_print (Gtk.PrintContext context) {
        debug ("begin_print");

        var layout = context.create_pango_layout ();
        var font = settings.get_string ("print-font-description");
        var desc = Pango.FontDescription.from_string (font);
        layout.set_font_description (desc);

        layout.set_width ((int) context.get_width () * Pango.SCALE);
        layout.set_height ((int) context.get_height () * Pango.SCALE);
        layout.set_alignment (Pango.Alignment.LEFT);
        layout.set_ellipsize (Pango.EllipsizeMode.END);

        layout.set_text ("X", 1);

        Pango.Rectangle ink_rect, logical_rect;
        Pango.LayoutLine line = layout.get_line (0);
        line.get_extents (out ink_rect, out logical_rect);
        var line_height = logical_rect.height / Pango.SCALE;

        /* find n_pages */
        uint size = model.get_n_items ();
        this.lines_per_page = (int) context.get_height () / line_height;
        int n_pages = (int) size / this.lines_per_page;
        /* do not forget trailing lines  */
        n_pages += (size % this.lines_per_page) > 0 ? 1 : 0;

        debug ("Setting number of pages to %d", n_pages);
        set_n_pages (n_pages);
    }

    public override void draw_page (Gtk.PrintContext context, int page_nr) {
        debug ("draw_page");

        var cairo = context.get_cairo_context ();
        cairo.set_source_rgb (0, 0, 0);
        cairo.set_line_width (1);

        var layout = context.create_pango_layout ();
        var font = settings.get_string ("print-font-description");
        var desc = Pango.FontDescription.from_string (font);
        layout.set_font_description (desc);

        layout.set_width ((int) (context.get_width () * Pango.SCALE));
        /* do not set heigth to allow ellipsize */
        layout.set_alignment (Pango.Alignment.LEFT);
        layout.set_ellipsize (Pango.EllipsizeMode.END);

        string text = "";

        /* page_nr starts with 0 */
        uint cur_index = page_nr * this.lines_per_page;

        Item[] items = {};
        for (uint i = 0; i < model.get_n_items (); i++) {
            Item item = (Item) model.get_item (i);
            items += item;
        }

        Pango.AttrList? attrs = new Pango.AttrList ();
        bool strike = settings.get_boolean ("strikethrough-checked");
        uint c = 0;

        while (cur_index < this.lines_per_page * (page_nr + 1) && cur_index < items.length) {
            uint beg;
            uint end;

            string item = "";
            var it = items[cur_index];

            if (it.checked) {
                beg = c + "☑\t".length;
                end = beg + it.text.length;

                Pango.Attribute attr = Pango.attr_strikethrough_new (strike);
                attr.start_index = beg;
                attr.end_index = end;
                attrs.insert (attr.copy ());

                item = "☑\t" + it.text;
                c = end;
            } else {
                beg = c + "☐\t".length;
                end = beg + it.text.length;

                item = "☐\t" + it.text;
                c = end;
            }

            text += item + "\n";
            c += 1; /* current char index */

            cur_index += 1; /* current line index */

            if (it.text.contains (settings.get_string ("important-word"))) {
                Gdk.RGBA color = Gdk.RGBA ();
                color.parse (settings.get_string ("important-color"));
                Pango.Attribute attr = Pango.attr_foreground_new ((uint16) (color.red * 0xffff), (uint16) (color.green * 0xffff), (uint16) (color.blue * 0xffff));
                attr.start_index = beg;
                attr.end_index = end;
                attrs.insert (attr.copy ());
            }
        }

        layout.set_attributes (attrs);
        layout.set_text (text, text.length);
        Pango.cairo_show_layout (cairo, layout);
    }
}
// vim:sw=4:ts=4:et

