public class Myosotis : Gtk.Application {
    Myosotis () {
        Object (
            application_id: APP_ID,
            flags: ApplicationFlags.HANDLES_OPEN
            );

        set_version (VERSION);
        set_option_context_parameter_string (_("[FILE]"));
        set_option_context_summary (_("FILE: text file to import as checklist"));
        this.builder = new Gtk.Builder ();
        this.model = new GLib.ListStore (typeof (Item));
        this.history = new UndoRedo ();
    }

    private void load_css (Gtk.Widget widget) {
        var display = Gdk.Display.get_default ();
        var provider = new Gtk.CssProvider ();
        var priority = 0;


        Gdk.RGBA color;
        widget.get_style_context ().lookup_color ("theme_selected_bg_color", out color);

        provider.load_from_string ("text>selection { background-color: " + color.to_string() + "; }");
        Gtk.StyleContext.add_provider_for_display (display, provider, priority);
    }

    protected override void activate () {
        if (get_windows ().length () > 0) {
            debug ("already present");
            get_windows ().first ().data.present ();
            return;
        }

        try {
            this.builder.add_from_resource ("/rocks/noise/myosotis/myosotis.ui");
            this.builder.add_from_resource ("/rocks/noise/myosotis/menu.ui");
            this.builder.set_translation_domain (APP_BIN);

            var window = this.builder.get_object ("ApplicationWindow") as Gtk.ApplicationWindow;
            window.set_title ("Myosotis");

            var size = Myosotis.settings.get_value ("window-size");
            if (size.n_children () == 2) {
                int width = (int) size.get_child_value (0);
                int height = (int) size.get_child_value (1);
                window.set_default_size (width, height);
            } else {
                window.set_default_size (480, 640);
            }

            var picture = this.builder.get_object ("Picture") as Gtk.Image;
            picture.set_from_file (PREFIX + "/share/icons/hicolor/scalable/apps/" + APP_BIN + ".svg");

            var burger = this.builder.get_object ("BurgerMenuButton") as Gtk.MenuButton;

            var menu = this.builder.get_object ("menu") as GLib.MenuModel;
            burger.set_menu_model (menu);

            var listbox = this.builder.get_object("ListBox") as Gtk.ListBox;
            listbox.activate_on_single_click = true;
            listbox.selection_mode = Gtk.SelectionMode.NONE;
            listbox.row_activated.connect ((row) => { disable_edit (listbox); ((CheckableRow)row).set_editable (true); });
            listbox.bind_model (this.model, this.listbox_create_widget_func);

            /* entry Escape press handler */
            var key_controller = new Gtk.EventControllerKey ();
            key_controller.key_pressed.connect (this.entry_key_press);
            var entry = this.builder.get_object ("MainEntry") as Gtk.Entry;
            entry.add_controller (key_controller);

            entry.state_flags_changed.connect ((flags) => {
                if ((flags == Gtk.StateFlags.DIR_RTL+Gtk.StateFlags.PRELIGHT+Gtk.StateFlags.ACTIVE) ||
                    (flags == Gtk.StateFlags.DIR_LTR+Gtk.StateFlags.PRELIGHT+Gtk.StateFlags.ACTIVE)) {
                    disable_edit (listbox);
                }
            });
            entry.activate.connect (this.entry_activate);
            entry.grab_focus ();

            load_css (entry);

            /* mainbox Escape press handler */
            key_controller = new Gtk.EventControllerKey ();
            key_controller.key_pressed.connect (this.mainbox_key_press);
            var vbox = this.builder.get_object ("MainBox") as Gtk.Box;
            vbox.add_controller (key_controller);

            /* listbox row drag-n-drop handler */
            var drag_controller = new Gtk.DragSource ();
            drag_controller.actions = Gdk.DragAction.MOVE;
            drag_controller.prepare.connect (this.drag_prepare);
            drag_controller.drag_begin.connect (this.drag_begin);
            drag_controller.drag_end.connect (this.drag_end);
            drag_controller.drag_cancel.connect ((source, drag, reason) => { return false; });
            listbox.add_controller(drag_controller);

            var drop_controller = new Gtk.DropTarget (typeof (int), Gdk.DragAction.MOVE);
            drop_controller.drop.connect (this.drop);
            listbox.add_controller (drop_controller);

            /* listbox string drop handler */
            var url_drop = new Gtk.DropTarget (typeof (string), Gdk.DragAction.COPY|Gdk.DragAction.MOVE);
            url_drop.drop.connect ((target, val, x, y) => {
                var url =  val.get_string ();
                var re = /^(http|ftp|smb|nfs|sftp).*$/;
                if (!re.match(url)) {
                    return false;
                }

                File file = File.new_for_uri (url);
                open ({ file }, "");
                return true;
            });
            listbox.add_controller (url_drop);

            /* listbox File drop handler */
            var file_drop = new Gtk.DropTarget (typeof (File), Gdk.DragAction.COPY);
            file_drop.drop.connect ((target, val, x, y) => {
                var file = val.get_object () as GLib.File;

                return parse_file (file);
            });
            listbox.add_controller (file_drop);

            var button = this.builder.get_object ("AddButton") as Gtk.Button;
            button.clicked.connect (this.button_add);

            window.close_request.connect ((window) => { this.save_quit(); return true; } );
            window.set_application (this);
            this.add_window (window);
            window.present ();


            /* load items */
            this.backend = new UserDataBackend ();
            var list_name = ini.get_list_name ();
            Item[] items = backend.use (list_name);
            foreach (var item in items) {
                model.append (item);
            }

            window.set_title ("Myosotis / " + list_name);
            history.push (model);
        } catch (Error e) {
            error ("Could not load Glade UI: %s", e.message);
        }
    }

    protected override void startup () {
        base.startup ();

        var quit_action = new SimpleAction ("quit", null);
        quit_action.activate.connect (this.save_quit);
        add_action (quit_action);

        var about_action = new SimpleAction ("about", null);
        about_action.activate.connect (this.about);
        add_action (about_action);

        var choose_action = new SimpleAction ("choose", null);
        choose_action.activate.connect (this.choose);
        add_action (choose_action);

        var prefs_action = new SimpleAction ("prefs", null);
        prefs_action.activate.connect (this.prefs);
        add_action (prefs_action);

        var md_action = new SimpleAction ("markdown", null);
        md_action.activate.connect (this.markdown);
        add_action (md_action);

        var latex_action = new SimpleAction ("latex", null);
        latex_action.activate.connect (this.latex);
        add_action (latex_action);

        var mail_action = new SimpleAction ("mail", null);
        mail_action.activate.connect (this.mail);
        add_action (mail_action);

        var print_action = new SimpleAction ("print", null);
        print_action.activate.connect (this.print);
        add_action (print_action);

        var copy_action = new SimpleAction ("copy", null);
        copy_action.activate.connect (this.copy);
        add_action (copy_action);

        var sort_action = new SimpleAction ("sort", null);
        sort_action.activate.connect (this.sort_checked);
        add_action (sort_action);

        var remove_action = new SimpleAction ("remove", null);
        remove_action.activate.connect (this.remove_checked);
        add_action (remove_action);

        var undo_action = new SimpleAction ("undo", null);
        undo_action.activate.connect (this.undo);
        add_action (undo_action);

        var redo_action = new SimpleAction ("redo", null);
        redo_action.activate.connect (this.redo);
        add_action (redo_action);

        set_accels_for_action ("app.quit", {"<Ctrl>Q"});
        set_accels_for_action ("app.undo", {"<Ctrl>Z"});
        set_accels_for_action ("app.redo", {"<Ctrl><Shift>Z"});
    }

    protected override void open (File[] files, string hint) {
        foreach (var file in files) {
            debug ("open %s", file.get_uri ());
            parse_file (file);
        }

        activate ();
    }

    private void disable_edit (Gtk.ListBox listbox) {
        for (var i = 0; i < model.get_n_items (); i++) {
            var crow = (CheckableRow) listbox.get_row_at_index (i);
            crow.set_editable (false);
        }
    }

    private void undo (SimpleAction act, Variant? param) {
        info ("undo");
        history.undo (model);
    }

    private void redo (SimpleAction act, Variant? param) {
        info ("redo");
        history.redo (model);
    }

    private void about (SimpleAction act, Variant? param) {
        const string[] authors = {"Benoît Rouits", null};
        Gtk.AboutDialog dialog = new Gtk.AboutDialog ();
        dialog.set_transient_for (this.get_active_window ());
        dialog.set_modal (true);
        dialog.set_program_name ("Myosotis");
        dialog.set_logo_icon_name (APP_BIN);
        dialog.set_comments (_("Vala/Gtk Check list"));
        dialog.set_copyright ("© 2025 Benoît Rouits");
        dialog.set_version (VERSION);
        dialog.set_license ("Copyright © 2025 Benoît Rouits\n\nThis program is free software: you can redistribute it and/or modify\nit under the terms of the GNU General Public License as published by\nthe Free Software Foundation, either version 3 of the License, or\n(at your option) any later version.\n\nThis program is distributed in the hope that it will be useful,\nbut WITHOUT ANY WARRANTY; without even the implied warranty of\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\nGNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License\nalong with this program.  If not, see <http://www.gnu.org/licenses/>.\n\nSPDX-License-Identifier: GPL-3.0-or-later");
        dialog.set_authors (authors);
        dialog.set_website ("https://git.noise.rocks/ben/myosotis");
        dialog.set_website_label (_("Myosotis website"));
        dialog.present ();
    }

    private void choose (SimpleAction act, Variant? param) {
        save ();
        var chooser = new ChooseWindow (this.get_active_window ());
        chooser.use.connect ((str) => {
            model.remove_all ();
            foreach (var item in backend.use (str)) {
                model.append (item);
            }
            chooser.destroy ();
            ini.set_list_name (str);
            this.get_active_window().set_title ("Myosotis / " + str);
            this.history.clear ();
        });

        chooser.present ();
    }

    private void prefs (SimpleAction act, Variant? param) {
        var dialog = new PrefsWindow (this.get_active_window ());
        dialog.present ();
    }

    private void sort_checked (SimpleAction act, Variant? param) {
        model.sort (Item.cmp);
        history.push (model);
    }

    private void remove_checked (SimpleAction act, Variant? param) {
        for (uint i = 0; i < model.get_n_items (); i++) {
            Item item = (Item) model.get_item (i);
            if (item.checked) {
                model.remove (i);
                i--;
            }
        }

        history.push (model);
    }

    private int model_index_of (Gtk.ListBox listbox, CheckableRow row) {
        /* find index of the row (FIXME: not clever) */
        var index = -1;
        for (int i = 0; i < model.get_n_items (); i++) {
            if (listbox.get_row_at_index (i) == row) {
                index = i;
                break;
            }
        }

        return index;
    }

    private bool drop (Gtk.DropTarget target, GLib.Value val, double x, double y) {
        debug ("drop");
        var index = val.get_int ();

        var listbox = (Gtk.ListBox) target.get_widget ();
        var picked = listbox.pick (x, y, Gtk.PickFlags.DEFAULT);

        if (picked.get_type () != typeof (Gtk.Box))
            return false;

        var target_row = picked.get_ancestor (typeof (CheckableRow)) as CheckableRow;

        int target_index = model_index_of (listbox, target_row);

        var item = model.get_item (index);
        model.remove (index);
        model.insert (target_index, item);

        history.push (model);
        return true;
    }

    private Gdk.ContentProvider? drag_prepare (Gtk.DragSource source, double x, double y) {
        //debug ("drag prepare");
        var listbox = source.get_widget () as Gtk.ListBox;
        var picked = listbox.pick (x, y, Gtk.PickFlags.DEFAULT);

        /* start drag with the CheckableRow's label */
        if (picked.get_type () != typeof (Gtk.Box))
            return null;

        var row = picked.get_ancestor (typeof (CheckableRow)) as CheckableRow?;

        if (row == null)
            return null;

        int index = model_index_of (listbox, row);

        listbox.set_data<Gtk.Text>("dragged-item", row.label);

        return new Gdk.ContentProvider.for_value (index);
    }

    private void drag_begin (Gtk.DragSource source, Gdk.Drag drag) {
        debug ("drag begin");
        var listbox = source.get_widget ();
        var label = listbox.get_data<Gtk.Text>("dragged-item");

        Gdk.Paintable paintable = new Gtk.WidgetPaintable (label);
        source.set_icon (paintable, 0, 0);
        label.opacity = 0.5;
    }

    private void drag_end (Gtk.DragSource source, Gdk.Drag drag, bool delete_data) {
        var listbox = source.get_widget ();
        var label = listbox.get_data<Gtk.Text> ("dragged-item");

        label.opacity = 1;

        listbox.set_data ("dragged-item", null);
    }

    private bool parse_file (File file) {
        string ctype = Util.mime_of (file);

        if (!ctype.contains ("text/"))
            return false;

        //model.remove_all ();

        bool md = ctype.contains ("/markdown");
        bool html = ctype.contains ("/html");
        bool latex = ctype.contains ("/x-tex");

        try {
            string line;
            var file_input_stream = file.read ();
            var f_dis = new DataInputStream (file_input_stream);

            while ((line = f_dis.read_line (null)) != null) {
                Item? item = null;
                if (md) {
                    /* find check lists in md */
                    item = Item.parse_md (line);
                } else if (html) {
                    item = Item.parse_html (line);
                } else if (latex) {
                    item = Item.parse_latex (line);
                } else {
                    /* every line is an item */
                    item = new Item (false, Item.Flags.NOFLAG, line);
                }

                if (item != null)
                    model.append (item);
            }

            history.push (model);
        } catch (Error e) {
            if (e is IOError.NOT_FOUND) {
                info ("%s", e.message);
            } else {
                warning ("%s", e.message);
            }

            return false;
        }

        return true;
    }

    private async void markdown (SimpleAction act, Variant? param) {
        Gtk.FileDialog dialog = new Gtk.FileDialog ();
        dialog.set_initial_name (ini.get_list_name () + ".md");
        try {
            File file = yield dialog.save (get_active_window (), null);
            var file_fos = file.replace (null, false, FileCreateFlags.NONE);
            var file_dos = new DataOutputStream (file_fos);

            bool strike = settings.get_boolean("strikethrough-checked");
            for (int i = 0; i < model.get_n_items (); i++) {
                Item item = (Item) model.get_item (i);
                string line = item.checked ? "- [x] " : "- [ ] ";
                line += strike && item.checked ? "~~" + item.text + "~~" : item.text;
                file_dos.put_string (line + "\n");
            }
        } catch (Error e) {
            warning ("Error: %s\n", e.message);
        }
    }

    private async void latex (SimpleAction act, Variant? param) {
        Gtk.FileDialog dialog = new Gtk.FileDialog ();
        dialog.set_initial_name (ini.get_list_name () + ".tex");
        try {
            File file = yield dialog.save (get_active_window (), null);
            string latex_begin = """\documentclass{article}
\usepackage{enumitem,amssymb,ulem,xcolor}
\newlist{todolist}{itemize}{2}
\setlist[todolist]{label=$\square$}
\usepackage{pifont}
\newcommand{\cmark}{\ding{51}}
\newcommand{\xmark}{\ding{55}}
\newcommand{\checked}{\rlap{$\square$}{\raisebox{2pt}{\large\hspace{1pt}\cmark}}\hspace{-2.5pt}}
\newcommand{\wontfix}{\rlap{$\square$}{\large\hspace{1pt}\xmark}}

\begin{document}
\begin{todolist}
""";
            string latex_end = "\\end{todolist}\n\\end{document}";

            var file_fos = file.replace (null, false, FileCreateFlags.NONE);
            var file_dos = new DataOutputStream (file_fos);

            file_dos.put_string (latex_begin);

            Gdk.RGBA color = Gdk.RGBA ();
            color.parse (settings.get_string ("important-color"));
            bool strike = settings.get_boolean("strikethrough-checked");

            for (int i = 0; i < model.get_n_items (); i++) {
                Item item = (Item) model.get_item (i);

                bool important = item.text.contains (settings.get_string ("important-word"));
                string itm = item.checked ? "\\item[\\checked] " : "\\item ";
                string colorize = "\\textcolor[rgb]{" + color.red.to_string() + "," + color.green.to_string () + "," + color.blue.to_string () + "}";
                string text = strike && item.checked ? "\\sout{" + Util.escape_latex (item.text) + "}" : Util.escape_latex (item.text);
                string line = itm + (important ? colorize + "{%s}".printf (text) : text);
                file_dos.put_string (line + "\n");
            }

            file_dos.put_string (latex_end);
        } catch (Error e) {
            warning ("Error: %s\n", e.message);
        }
    }

    private void mail (SimpleAction act, Variant? param) {
        var body = "";
        for (int i = 0; i < model.get_n_items (); i++) {
            Item item = (Item) model.get_item (i);
            string line = (item.checked ? "☑\t" : "☐\t") + item.text;
            body += line + "\n";
        }

        try {
            GLib.AppInfo.launch_default_for_uri ("mailto:?subject=%s&body=%s".printf (Uri.escape_string ("Myosotis / " + ini.get_list_name ()), Uri.escape_string (body)), null);
        } catch (Error e) {
            warning (e.message);
        }
    }

    private void copy (SimpleAction act, Variant? param) {
        var txt = "";
        for (int i = 0; i < model.get_n_items (); i++) {
            Item item = (Item) model.get_item (i);
            string line = (item.checked ? "☑\t" : "☐\t") + item.text;
            txt += line + "\n";
        }

        var clip = Gdk.Display.get_default ().get_clipboard ();
        clip.set_text (txt);
    }

    private void print (SimpleAction act, Variant? param) {
        Printer printer = new Printer (this.model, this.settings);
        try {
            var res = printer.run (Gtk.PrintOperationAction.PRINT_DIALOG, this.get_active_window ());
            debug ("print res: %d\n", res);
        } catch (Error e) {
            warning (e.message);
        }
    }

    private bool entry_key_press (uint keyval, uint keycode, Gdk.ModifierType state) {
        var entry = this.builder.get_object ("MainEntry") as Gtk.Entry;

        if (keyval == Gdk.Key.Escape && entry.text != "") {
            entry.text = "";
            return true;
        }

        return false;
    }

    private bool mainbox_key_press (uint keyval, uint keycode, Gdk.ModifierType state) {
        if (keyval == Gdk.Key.Escape)  {
            save_quit ();
            return true;
        }

        return false;
    }

    private void append_row (string text) {
        Item item = new Item (false, Item.Flags.NOFLAG, text);
        model.append (item);
        history.push (model);
    }

    Gtk.Widget listbox_create_widget_func (Object item) {
        CheckableRow row = new CheckableRow ((Item) item);
        row.remove.connect (() => {
            uint pos;
            bool found = model.find (item, out pos);
            if (found) {
                model.remove (pos);
                history.push (model);
            }

            var entry = this.builder.get_object ("MainEntry") as Gtk.Entry;
            entry.grab_focus ();
        });
        row.check.connect (() => { ((Item) item).checked =  true; history.push (model); });
        row.uncheck.connect (() => { ((Item) item).checked = false; history.push (model); });
        row.modified.connect ((text) => { ((Item) item).text = text; });
        row.validated.connect (() => {
            history.push (model);
            row.set_editable (false);
            var entry = this.builder.get_object ("MainEntry") as Gtk.Entry;
            entry.grab_focus ();
        });

        return row;
    }

    private void entry_activate (Gtk.Entry entry) {
        if (entry.text != "") {
            append_row (entry.text);
            entry.text = "";
        }
    }

    private void button_add (Gtk.Button button) {
        var entry = this.builder.get_object ("MainEntry") as Gtk.Entry;
        if (entry.text != "") {
            append_row (entry.text);
            entry.text = "";
        }
    }

    private void save () {
        Item[] items = {};
        for (uint i = 0; i < model.get_n_items (); i++) {
            Item item = (Item) model.get_item (i);
            items += item;
        }

        backend.save (items);
    }

    private void save_quit () {
        /* save window geometry */
        var width = get_active_window ().get_width ();
        var height = get_active_window ().get_height ();

        Myosotis.settings.set_value ("window-size", new int[] { width, height });

        /* set 'first time' use as false */
        Myosotis.settings.set_boolean ("first-time", false);

        /* update last list name */
        ini.sync ();

        save ();
        quit ();
    }

    public static int main (string[] args) {
        Intl.setlocale (GLib.LocaleCategory.ALL, "");
        GLib.Environment.set_variable("LC_NUMERIC", "C", true);
        //Intl.setlocale (GLib.LocaleCategory.NUMERIC, "C");
        Intl.bindtextdomain (APP_BIN, PREFIX + "/share/locale");
        Intl.textdomain (APP_BIN);

        return new Myosotis ().run (args);
    }

    private Backend backend;

    public Gtk.Builder builder;
    public GLib.ListStore model;
    public static GLib.Settings settings = new GLib.Settings (APP_ID);
    public static Ini ini = new Ini ();

    private UndoRedo history;
}
// vim:sw=4:ts=4:et
