public class Item : GLib.Object {
    public enum Flags { NOFLAG } /* for future use */

    public Item (bool checked, int flags, string text) {
        this.checked = checked;
        this.flags = flags;
        this.text = text;
    }

    public Item.from_string (string line) {
        var item_line = line.split ("\t", 3);
        if (item_line.length == 3) {
            this.checked = bool.parse (item_line[0]);
            this.flags = int.parse(item_line[1]);
            this.text = item_line[2];
        } else {
            this.checked = false;
            this.flags = Item.Flags.NOFLAG;
            this.text = line;
        }
    }

    public static Item? parse_md (string line) {
        Item? item = null;
        int idx;
        bool checked = line.contains("[x] ");
        bool unchecked = line.contains ("[ ] ");
        bool simple = line.index_of ("- ") == 0 || line.index_of("* ") == 0;

        /* order is important */
        if (checked) {
            idx = line.index_of ("[x] ") + 4;
            item = new Item (true, Flags.NOFLAG, Util.strip_md_markup(line.substring (idx)));
        } else if (unchecked) {
            idx = line.index_of ("[ ] ") + 4;
            item = new Item (false, Flags.NOFLAG, Util.strip_md_markup(line.substring (idx)));
        } else if (simple) {
            idx = 2; /* len of "- " or "* " */
            item = new Item (false, Flags.NOFLAG, Util.strip_md_markup(line.substring (idx)));
        } else {
            debug ("Markdown line ignored");
        }

        return item;
    }

    public static Item? parse_html (string line) {
        Item? item = null;
        int idx;
        bool simple = (line.contains ("<li ") || line.contains("<li>")) && line.contains("</li>");
        bool checkbox = simple && line.contains ("<input type=\"checkbox\"");
        bool checked = checkbox && line.contains("checked=\"\">");

        /* order is important */
        if (checked) {
            idx = line.index_of ("checked=\"\">") + 11;
            string capture = line.substring (idx, line.index_of ("</li>") - idx);
            item = new Item (true, Flags.NOFLAG, Util.unescape_html(Util.strip_html_markup(capture)));
        } else if (checkbox) {
            idx = line.index_of ("<input type=\"checkbox\"");
            string input = line.substring (idx);
            idx = input.index_of ("\">") + 2; /* first "> encourtened after '<input type="checkbox"' */
            string capture = input.substring (idx, input.index_of ("</li>") - idx);
            item = new Item (false, Flags.NOFLAG, Util.unescape_html(Util.strip_html_markup(capture)));
        } else if (simple) {
            idx = line.index_of (">") + 1; /* first "> encourtened */
            string capture = line.substring (idx, line.index_of ("</li>") - idx);
            item = new Item (false, Flags.NOFLAG, Util.unescape_html(Util.strip_html_markup(capture)));
        } else {
            debug ("HTML line ignored");
        }

        return item;
    }

    public static Item? parse_latex (string line) {
        Item? item = null;
        int idx;
        bool simple = line.contains ("\\item ") || line.contains ("\\item[");
        bool checked = simple && line.contains ("[\\checked]");
        if (checked) {
            idx = line.index_of ("[\\checked]") + 10;
            string capture = line.substring (idx).chug();
            item = new Item (true, Flags.NOFLAG, Util.unescape_latex(Util.strip_latex_markup(capture)));
        } else if (simple) {
            idx = line.index_of ("\\item") + 5;
            string capture = line.substring (idx).chug();
            item = new Item (false, Flags.NOFLAG, Util.unescape_latex(Util.strip_latex_markup(capture)));
        } else {
            debug ("LaTeX line ignored");
        }

        return item;
    }

    public Item copy () {
        return new Item (this.checked, this.flags, this.text);
    }

    public string to_string () {
        return checked.to_string ()  + "\t" + flags.to_string () + " \t" + text.chomp ();
    }

    public bool checked;
    public int flags;
    public string text;
    public static GLib.Settings settings = new GLib.Settings (APP_ID);

    public static int cmp (GLib.Object a, GLib.Object b) {
        if (((Item) a).checked == ((Item) b).checked) {
            return 0;
        }

        if (settings.get_boolean ("sort-checked-upward"))
            return ((Item) a).checked ? -1 : 1;
        else
            return ((Item) a).checked ? 1 : -1;
    }
}

