/***

    Copyright (C) 2025 Benoît Rouits

    This file is part of Myosotis.

    Myosotis is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Myosotis is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Myosotis.  If not, see <http://www.gnu.org/licenses/>.

***/


public class ChooseWindow : Gtk.Window {
    private Gtk.Entry create_entry;
    private GLib.ListStore model;
    private Backend backend;

    public ChooseWindow (Gtk.Window parent) {
        Object ();

        this.set_transient_for (parent);
        this.set_modal (true);

        backend = new UserDataBackend ();
	model = new GLib.ListStore (typeof (ChooseItem));

        foreach (var title in backend.list ()) {
            model.append (new ChooseItem (title));
        }

        setup_ui (parent);
    }

    private void setup_ui (Gtk.Window parent) {
        this.set_title (_("Choose a list"));

	this.create_entry = new Gtk.Entry ();
	create_entry.set_placeholder_text (_("Create a new list"));
	create_entry.activate.connect (handle_create);
	var create_button = new Gtk.Button.from_icon_name ("add");
	create_button.clicked.connect ((button) => { handle_create (this.create_entry); });

	var hbox = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 6);
	hbox.append (create_entry);
	hbox.append (create_button);

        var vbox = new Gtk.Box (Gtk.Orientation.VERTICAL, 6);
        vbox.spacing = 6;
        vbox.margin_end = vbox.margin_start = vbox.margin_top = vbox.margin_bottom = 6;
        vbox.append (hbox);

	var listbox = new Gtk.ListBox ();
        listbox.activate_on_single_click = true;
        listbox.selection_mode = Gtk.SelectionMode.NONE;
        listbox.row_activated.connect ((row) => {
            for (int idx = 0; idx < model.get_n_items (); idx++) {
                if (listbox.get_row_at_index (idx) == row) {
                    var item = model.get_item (idx);
                    use (((ChooseItem) item).text);
                    return;
                }
            }
        });
        listbox.bind_model (this.model, this.listbox_create_widget_func);
        vbox.append (listbox);

        this.set_child (vbox);
    }

    void handle_create (Gtk.Entry entry) {
        string msg = null;

        if (entry.text.contains ("/")) {
            msg = _("Cannot have \"/\" in the list title");
        } else if (entry.text == "") {
            msg = _("Cannot create a list with empty title");
        }

        if (msg != null) {
            var dialog = new Gtk.AlertDialog(msg);
            dialog.show (this);
            return;
        }

        use (entry.text);
    }

    Gtk.Widget listbox_create_widget_func (Object item) {
        Gtk.ListBoxRow row = new Gtk.ListBoxRow ();
        Gtk.Box hbox = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 6);

        Gtk.Label label = new Gtk.Label (((ChooseItem) item).text);
        label.hexpand = true;
        hbox.append (label);

        Gtk.Button but = new Gtk.Button.from_icon_name ("remove");
        but.clicked.connect ((but) => { backend.remove (((ChooseItem) item).text); uint idx; model.find (item, out idx); model.remove (idx); } );
        hbox.append (but);
        row.set_child (hbox);
        return row;
    }


    public signal void use (string title);
}

public class ChooseItem: GLib.Object {
    public string text;

    public ChooseItem (string text) {
        this.text = text;
    }
}
