using Gtk;

public class CheckableRow: ListBoxRow {
    public CheckableRow (Item item) {
        hbox = new Box (Orientation.HORIZONTAL, 6);
        hbox.margin_start = 6;
        hbox.margin_end = 6;
        set_child (hbox);

        check_button = new CheckButton ();
        hbox.append (check_button);

        label = new Text ();
        label.editable = false;
        label.sensitive = false;
        label.truncate_multiline = true;

        label.changed.connect ((edit) => {
            set_important (edit.text.contains (settings.get_string ("important-word")));
            modified (edit.text);
        });
        label.activate.connect (() => { validated (); });

        label.hexpand = true;
        label.set_text (item.text);
        hbox.append (label);

        remove_button = new Button.from_icon_name ("list-remove");
        hbox.append (remove_button);

        check_button.active = item.checked;

        set_strike (item.checked && settings.get_boolean ("strikethrough-checked"));
        set_important (item.text.contains (settings.get_string ("important-word")));

        settings.changed["strikethrough-checked"].connect (() => { set_strike (is_checked () && settings.get_boolean ("strikethrough-checked")); });
        settings.changed["important-word"].connect (() => { set_important (label.text.contains (settings.get_string ("important-word"))); });
        settings.changed["important-color"].connect (() => { set_important (label.text.contains (settings.get_string ("important-word"))); });

        remove_button.clicked.connect (remove_button_clicked);
        check_button.toggled.connect (check_button_toggled);
    }

    public void set_editable (bool enable) {
        label.focusable = label.editable = enable;

        if (enable) {
            label.sensitive = true;
            label.grab_focus_without_selecting ();
            label.set_position (label.text.length);
        } else {
            /* avoid "focus-out" warning */
            label.set_state_flags (Gtk.StateFlags.INSENSITIVE, true);
        }
    }

    public void set_important (bool enable) {
        var attrs = label.get_attributes ();
        if (enable) {
            if (attrs != null) {
                Gdk.RGBA color = Gdk.RGBA ();
                color.parse (settings.get_string ("important-color"));
                Pango.Attribute att = Pango.attr_foreground_new ((uint16) (color.red * 0xffff), (uint16) (color.green * 0xffff), (uint16) (color.blue * 0xffff));
                attrs.change (att.copy ());
                label.set_attributes (attrs);
            } else {
                var neue = new Pango.AttrList ();
                Gdk.RGBA color = Gdk.RGBA ();
                color.parse (settings.get_string ("important-color"));
                Pango.Attribute att = Pango.attr_foreground_new ((uint16) (color.red * 0xffff), (uint16) (color.green * 0xffff), (uint16) (color.blue * 0xffff));
                neue.insert (att.copy ());
                label.set_attributes (neue);
            }
        } else {
            /* we cannot reset old color in case of importance switch
             * so, we 'just remove' the foregroud attribute */
            if (attrs != null) {
                var neue = new Pango.AttrList ();
                foreach (unowned var attr in attrs.get_attributes ()) {
                    if (attr.klass.type == Pango.AttrType.FOREGROUND)
                        continue;

                    neue.insert (attr.copy ());
                }
                label.set_attributes (neue);
            }
        }
    }

    public void set_strike (bool enable) {
        var attrs = label.get_attributes ();
        if (attrs != null) {
            Pango.Attribute attr = Pango.attr_strikethrough_new (enable);
            attrs.change (attr.copy ());
            label.set_attributes (attrs);
        } else {
            Pango.AttrList neue = new Pango.AttrList ();
            Pango.Attribute attr = Pango.attr_strikethrough_new (enable);
            neue.insert (attr.copy ());
            label.set_attributes (neue);
        }
    }

    public bool is_checked () {
        return check_button.active;
    }

    private void check_button_toggled (CheckButton button) {
        set_strike (button.active && settings.get_boolean ("strikethrough-checked"));
        if (button.active) {
            check ();
        } else {
            uncheck ();
        }
    }

    private void remove_button_clicked (Button button) {
        remove ();
    }

    public signal void remove ();
    public signal void check ();
    public signal void uncheck ();
    public signal void validated ();
    public signal void modified (string text);

    public Text label;
    public CheckButton check_button;
    private Box hbox;
    private Button remove_button;
    private static GLib.Settings settings = new GLib.Settings (APP_ID);
}
