/*
 * Logserver
 * Copyright (C) 2017-2025 Joel Reardon
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef __MOCK_LOG_LINES__H__
#define __MOCK_LOG_LINES__H__

#include <list>
#include <string>

using namespace std;

class MockLogLines : public ILogLines {
public:
	MockLogLines() : _eof(false) {}

	virtual ~MockLogLines() {
		CHECK(_lines.empty());
	}

        virtual size_t add_line(const string& line) {
		unique_lock<mutex> ul(_m);
		CHECK(_lines.size());
		CHECK(_lines.front() == line);
		_lines.pop_front();
		return 0;
	}

        // adds a Line object to log lines
        virtual size_t add_line(Line* line) {
		unique_lock<mutex> ul(_m);
		return add_line(line->get());
	}

	// adds Line objects to log lines
        virtual void add_lines(list<Line*>* lines) {
		for (Line* line : *lines) {
			add_line(line->get());
			delete line;
		}
		lines->clear();
	}

        // sets eof if input reader is done
        virtual bool eof() const {
		unique_lock<mutex> ul(_m);
		return _eof;
	}

        // unset eof, e.g., if the file being tailed is updated
        virtual void set_eof(bool value) {
		unique_lock<mutex> ul(_m);
		_eof = value;
		if (_eof) CHECK(!_lines.size());
	}

	virtual void expect(const list<string>& lines) {
		unique_lock<mutex> ul(_m);
		_lines = lines;
	}

	virtual void expect(const string& line) {
		unique_lock<mutex> ul(_m);
		_lines.push_back(line);
	}

	virtual bool no_expectations() {
		unique_lock<mutex> ul(_m);
		return _lines.empty();
	}

protected:
	list<string> _lines;
	bool _eof;
	mutable mutex _m;
};


#endif  // __MOCK_LOG_LINES__H__
