
#!/bin/bash
# 目标语言列表
LANGUAGES=("zh_CN" "zh_TW" "zh_HK" "bo_CN" "ug_CN" "kk_KZ" "ky_KG" "mn_MN" "de_DE" "es_ES" "fr_FR")

# 检查必要的命令是否可用
check_commands() {
    if ! command -v lupdate &> /dev/null; then
        echo "错误: 未找到 lupdate 命令, 请确保 Qt Linguist 工具已安装"
        exit 1
    fi
}

# 生成 .ts 文件（保留已有翻译）
generate_ts_files() {
    local count=0
    for lang in "${LANGUAGES[@]}"; do
        local ts_file="translations/${lang}.ts"
        echo "正在处理: ${ts_file}"
        
        # 如果文件已存在，使用 -no-obsolete 保留已有翻译
        if [ -f "$ts_file" ]; then
            echo "  → 更新已有文件（保留翻译）..."
            lupdate -recursive . -no-obsolete -ts "$ts_file"
        else
            echo "  → 创建新文件..."
            lupdate -recursive . -ts "$ts_file"
        fi
        
        if [ -f "$ts_file" ]; then
            echo "  ✓ 成功: ${ts_file}"
            ((count++))
        else
            echo "  ✗ 失败: ${ts_file}"
        fi
        echo ""
    done
    echo "共处理 ${count} 个 .ts 文件"
    return $count
}

# 生成汇总报告
generate_report() {
    echo "=== 翻译文件生成报告 ==="
    echo "生成时间: $(date)"
    echo "目标语言: ${#LANGUAGES[@]} 种"
    echo ""
    
    echo ".ts 文件状态:"
    for lang in "${LANGUAGES[@]}"; do
        local ts_file="translations/${lang}.ts"
        if [ -f "$ts_file" ]; then
            local line_count=$(wc -l < "$ts_file" 2>/dev/null || echo "0")
            echo "  ✓ ${ts_file} (${line_count} 行)"
        else
            echo "  ✗ ${ts_file} (缺失)"
        fi
    done
    echo ""
}

# 主函数
main() {
    echo "开始生成翻译文件..."
    echo "目标语言: ${LANGUAGES[*]}"
    echo ""
    
    check_commands
    create_translation_dir
    generate_ts_files
    compile_qm_files
    echo ""
    generate_report
    
    echo ""
    echo "=== 后续操作建议 ==="
    echo "1. 使用 Qt Linguist 编辑 .ts 文件:"
    echo "   linguist translations/zh_CN.ts &"
    echo ""
    echo "2. 重新编译更新后的 .ts 文件:"
    echo "   lrelease translations/*.ts"
    echo ""
}

# 运行主函数
main