// SPDX-FileCopyrightText: 2023 KylinSoft Co., Ltd.
//
// SPDX-License-Identifier: Expat

#define _POSIX_C_SOURCE 200809L
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <wlr/backend/drm.h>
#include <wlr/backend/headless.h>
#include <wlr/render/wlr_renderer.h>
#include <wlr/types/wlr_output_layout.h>
#include <wlr/util/region.h>

#include <kywc/log.h>

#include "backend/drm.h"
#include "backend/fbdev.h"
#include "effect/output_transform.h"
#include "output_p.h"
#include "render/profile.h"
#include "util/debug.h"
#include "util/quirks.h"
#include "xwayland.h"

#define FALLBACK_OUTPUT "FALLBACK"
#define COLORTEMP_CLAMP(val) ((val) < 1000 ? 6500 : ((val) > 25000 ? 25000 : (val)))
#define BRIGHTNESS_CLAMP(val) ((val) > 100 ? 100 : (val))

struct output_manager *output_manager = NULL;

struct output *output_from_kywc_output(struct kywc_output *kywc_output)
{
    struct output *output = wl_container_of(kywc_output, output, base);
    return output;
}

struct output *output_from_wlr_output(struct wlr_output *wlr_output)
{
    return wlr_output->data;
}

static void handle_server_destroy(struct wl_listener *listener, void *data)
{
    wl_list_remove(&output_manager->server_destroy.link);
    wl_list_remove(&output_manager->server_suspend.link);
    wl_list_remove(&output_manager->server_resume.link);
    wl_list_remove(&output_manager->configured.link);

    assert(wl_list_empty(&output_manager->events.new_output.listener_list));
    assert(wl_list_empty(&output_manager->events.new_enabled_output.listener_list));
    assert(wl_list_empty(&output_manager->events.primary_output.listener_list));
    assert(wl_list_empty(&output_manager->events.configured.listener_list));
    assert(wl_list_empty(&output_manager->events.layout_damage.listener_list));
    assert(wl_list_empty(&output_manager->events.max_scale.listener_list));

    free(output_manager);
    output_manager = NULL;
}

static void handle_server_suspend(struct wl_listener *listener, void *data)
{
    kywc_log(KYWC_DEBUG, "Handle server D-Bus suspend");
    output_manager_power_outputs(false);
}

static void handle_server_resume(struct wl_listener *listener, void *data)
{
    kywc_log(KYWC_DEBUG, "Handle server D-Bus resume");
    output_manager_power_outputs(true);
}

static const char *output_get_edid(struct wlr_output *wlr_output)
{
    size_t len = 0;
    const void *edid_raw = wlr_output_get_edid(wlr_output, &len);
    if (!edid_raw) {
        return NULL;
    }

    const char *edid = kywc_identifier_base64_generate(edid_raw, len);
    return edid;
}

static void output_get_prop(struct output *output, struct kywc_output_prop *prop)
{
    struct wlr_output *wlr_output = output->wlr_output;
    static char *unknown = " ";

    if (wlr_output_is_drm(wlr_output)) {
        prop->backend = KYWC_OUTPUT_BACKEND_DRM;
    } else if (wlr_output_is_headless(wlr_output)) {
        prop->backend = KYWC_OUTPUT_BACKEND_HEADLESS;
    } else if (wlr_output_is_fbdev(wlr_output)) {
        prop->backend = KYWC_OUTPUT_BACKEND_FBDEV;
    } else {
        prop->backend = KYWC_OUTPUT_BACKEND_NONE;
    }

    prop->phys_width = wlr_output->phys_width;
    prop->phys_height = wlr_output->phys_height;
    prop->make = wlr_output->make ? wlr_output->make : wlr_output->name;
    prop->model = wlr_output->model ? wlr_output->model : unknown;
    prop->serial = wlr_output->serial; // May be NULL
    prop->desc = wlr_output->description ? wlr_output->description : wlr_output->name;
    prop->edid = output_get_edid(wlr_output);

    prop->capabilities = KYWC_OUTPUT_CAPABILITY_POWER;
    if (prop->backend == KYWC_OUTPUT_BACKEND_DRM) {
        prop->capabilities |= KYWC_OUTPUT_CAPABILITY_BRIGHTNESS |
                              KYWC_OUTPUT_CAPABILITY_COLOR_TEMP |
                              KYWC_OUTPUT_CAPABILITY_COLOR_FILTER;
    }
    if (wlr_output_get_vrr_capable(wlr_output)) {
        prop->capabilities |= KYWC_OUTPUT_CAPABILITY_VRR;
    }
    if (wlr_output_get_rgb_range(wlr_output, &output->base.state.rgb_range)) {
        prop->capabilities |= KYWC_OUTPUT_CAPABILITY_RGB_RANGE;
    }
    if (wlr_output_get_overscan(wlr_output, &output->base.state.overscan)) {
        prop->capabilities |= KYWC_OUTPUT_CAPABILITY_OVERSCAN;
    }
    if (wlr_output_get_scaling_mode(wlr_output, &output->base.state.scaling_mode)) {
        prop->capabilities |= KYWC_OUTPUT_CAPABILITY_SCALING_MODE;
    }

    /* fix zero mode in some backend, like wayland */
    if (wl_list_empty(&wlr_output->modes)) {
        struct wlr_output_mode mode = {
            .width = wlr_output->width,
            .height = wlr_output->height,
            .refresh = wlr_output->refresh,
            .preferred = true,
        };
        output_add_mode(output, &mode, false);
    } else {
        struct wlr_output_mode *mode;
        wl_list_for_each(mode, &wlr_output->modes, link) {
            output_add_mode(output, mode, false);
        }
    }
}

static void output_get_state(struct output *output, struct kywc_output_state *state)
{
    struct wlr_output *wlr_output = output->wlr_output;

    state->enabled = wlr_output->enabled;
    state->power = wlr_output->enabled;
    state->width = wlr_output->width;
    state->height = wlr_output->height;
    state->refresh = wlr_output->refresh;
    state->transform = wlr_output->transform;
    state->scale = wlr_output->scale;

    struct wlr_output_layout_output *layout_output;
    layout_output = wlr_output_layout_get(output_manager->server->layout, wlr_output);
    if (layout_output) {
        state->lx = layout_output->x;
        state->ly = layout_output->y;
    } else {
        state->lx = state->ly = -1;
    }

    state->brightness = output->brightness;
    state->color_temp = output->color_temp;
    state->color_filter = output->color_filter;
    state->color_feature = output->color_feature;
    state->vrr_policy = output->vrr_policy;
    wlr_output_get_rgb_range(wlr_output, &state->rgb_range);
    wlr_output_get_overscan(wlr_output, &state->overscan);
    wlr_output_get_scaling_mode(wlr_output, &state->scaling_mode);
}

void fallback_output_set_state(bool enabled, struct output *mirror)
{
    struct kywc_output *kywc_output = output_manager->fallback_output;
    /* no fallback output or still be disabled */
    if (!kywc_output || (!kywc_output->state.enabled && !enabled)) {
        return;
    }

    struct output *output = output_from_kywc_output(kywc_output);
    fallback_output_fill_state(output, enabled, mirror);
    kywc_output_set_state(kywc_output, &output->pending_state);
}

static void output_init_quirks(struct output *output)
{
    if (!wlr_output_is_drm(output->wlr_output)) {
        return;
    }

    int drm_fd = drm_backend_get_non_master_fd(output->wlr_output->backend);
    output->quirks = quirks_by_backend(drm_fd);

    /*  using software curosr, depending on the quirks mask */
    if (output->quirks & QUIRKS_MASK_SOFTWARE_CURSOR) {
        wlr_output_lock_software_cursors(output->wlr_output, true);
    }

    close(drm_fd);
}

static void output_uuid_generate(struct kywc_output *kywc_output)
{
    char description[128];
    snprintf(description, sizeof(description), "%s %s%s%s (%s)", kywc_output->prop.make,
             kywc_output->prop.model, kywc_output->prop.serial ? " " : "",
             kywc_output->prop.serial ? kywc_output->prop.serial : "", kywc_output->name);
    kywc_output->uuid = kywc_identifier_md5_generate_uuid((void *)description, strlen(description));
}

void output_log_state(enum kywc_log_level level, struct output *output, const char *desc,
                      const struct kywc_output_state *state)
{
    if (kywc_log_get_level() < level || !state) {
        return;
    }

    kywc_log(level,
             "%s(%s): mode (%d x %d @ %d) scale %f pos (%d, %d) transform %d %s brightness %d "
             "colortemp %d colorfilter %d %s",
             output->base.name, desc ? desc : "", state->width, state->height, state->refresh,
             state->scale, state->lx, state->ly, state->transform,
             state->enabled ? "enabled" : "disabled", state->brightness, state->color_temp,
             state->color_filter,
             output_manager->pending_primary == &output->base ? "primary" : "");
}

static void output_log_prop(enum kywc_log_level level, struct output *output, const char *desc,
                            const struct kywc_output_prop *prop)
{
    if (kywc_log_get_level() < level || !prop) {
        return;
    }

    static char *backend[] = { "none", "drm", "headless", "fbdev" };

    kywc_log(level, "%s(%s): %s backend, [%s]: [%s, %s, %s] (%d x %d)", output->base.name,
             desc ? desc : "", backend[prop->backend], prop->desc, prop->make, prop->model,
             prop->serial ? prop->serial : "", prop->phys_width, prop->phys_height);
}

/* the actual enabled output except for fallback */
bool output_manager_has_enabled_outputs(void)
{
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (&output->base == output_manager->fallback_output) {
            continue;
        }
        if (output->base.state.enabled) {
            return true;
        }
    }
    return false;
}

bool output_manager_has_actual_outputs(void)
{
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (&output->base != output_manager->fallback_output) {
            return true;
        }
    }
    return false;
}

static struct output *output_create(const char *name, struct wlr_output *wlr_output)
{
    struct output *output = calloc(1, sizeof(*output));
    if (!output) {
        return NULL;
    }

    output->wlr_output = wlr_output;
    wlr_output->data = output;

    struct kywc_output *kywc_output = &output->base;
    kywc_output->name = name;

    wl_signal_init(&kywc_output->events.on);
    wl_signal_init(&kywc_output->events.off);
    wl_signal_init(&kywc_output->events.scale);
    wl_signal_init(&kywc_output->events.transform);
    wl_signal_init(&kywc_output->events.mode);
    wl_signal_init(&kywc_output->events.position);
    wl_signal_init(&kywc_output->events.power);
    wl_signal_init(&kywc_output->events.brightness);
    wl_signal_init(&kywc_output->events.color_temp);
    wl_signal_init(&kywc_output->events.color_filter);
    wl_signal_init(&kywc_output->events.destroy);

    wl_signal_init(&output->events.disable);
    wl_signal_init(&output->events.geometry);
    wl_signal_init(&output->events.usable_area);
    wl_signal_init(&output->events.update_usable_area);
    wl_signal_init(&output->events.update_late_usable_area);

    wl_list_init(&output->configure_link);
    output->manager = output_manager;
    wl_list_insert(&output_manager->outputs, &output->link);

    /* add modes for eDP/LVDS panel */
    output_add_custom_modes(output);

    /* get props */
    wl_list_init(&kywc_output->prop.modes);
    output_get_prop(output, &kywc_output->prop);
    output_log_prop(KYWC_DEBUG, output, "prop", &kywc_output->prop);

    if (kywc_output->prop.backend == KYWC_OUTPUT_BACKEND_HEADLESS &&
        strcmp(name, FALLBACK_OUTPUT) == 0) {
        output_manager->fallback_output = kywc_output;
    }

    output_init_quirks(output);
    output_get_state(output, &kywc_output->state);

    output_uuid_generate(kywc_output);
    kywc_log(KYWC_INFO, "New output %s: %s", kywc_output->name, kywc_output->uuid);

    /* configure the output */
    bool found = output_read_config(output, &output->pending_state);
    found = output_manager->has_layout_manager ? found : false;
    if (!found) {
        output_fill_preferred_state(output);
    }
    if (!output_manager->has_layout_manager || output_manager->fallback_output == kywc_output) {
        output_manager_add_pending_state(output, &output->pending_state);
    } else {
        output_manager_get_layout_configs(output_manager);
    }
    output_manager_configure_outputs();

    output->initialized = true;
    wl_signal_emit_mutable(&output_manager->events.new_output, kywc_output);
    if (kywc_output->state.enabled) {
        wl_signal_emit_mutable(&output_manager->events.new_enabled_output, kywc_output);
    }

    return output;
}

static void handle_output_present(struct wl_listener *listener, void *data)
{
    KY_PROFILE_ZONE(zone, __func__);

    struct output *output = wl_container_of(listener, output, present);
    output->scene_commit = false;

    KY_PROFILE_RENDER_COLLECT(output->wlr_output->renderer);

    if (output->has_pending) {
        output_manager_add_pending_state(output, &output->pending_state);
        /* has_pending is reset in handle_output_frame */
        if (!output_manager_configure_outputs()) {
            output->has_pending = false;
        }
    }

    KY_PROFILE_ZONE_END(zone);
}

static void handle_output_frame(struct wl_listener *listener, void *data)
{
    if (!output_manager->server->active) {
        return;
    }

    struct output *output = wl_container_of(listener, output, frame);
    if (output_manager->primary_output == &output->base) {
        KY_PROFILE_FRAME();
    }
    KY_PROFILE_FRAME_NAME(output->wlr_output->name);
    KY_PROFILE_ZONE(zone, __func__);

    /* skip rendering if has_pending, otherwise drm may report busy */
    if (output->has_pending) {
        output->wlr_output->frame_pending = true;
        output->has_pending = false;
        KY_PROFILE_ZONE_END(zone);
        return;
    }

    output->scene_commit = ky_scene_output_commit(output->scene_output, NULL);

    struct timespec now = { 0 };
    clock_gettime(CLOCK_MONOTONIC, &now);
    ky_scene_output_send_frame_done(output->scene_output, &now);

    KY_PROFILE_ZONE_END(zone);
}

static void handle_output_commit(struct wl_listener *listener, void *data)
{
    KY_PROFILE_ZONE(zone, __func__);

    struct output *output = wl_container_of(listener, output, commit);
    struct wlr_output_event_commit *event = data;
    bool has_damage = event->state->committed & WLR_OUTPUT_STATE_DAMAGE &&
                      pixman_region32_not_empty(&output->scene_output->frame_damage);
    if (has_damage) {
        wl_signal_emit_mutable(&output_manager->events.layout_damage,
                               &output->scene_output->frame_damage);
    }

    KY_PROFILE_ZONE_END(zone);
}

static void output_manager_update_max_scale(struct kywc_output *kywc_output)
{
    float scale = 1.0;
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        struct kywc_output *kywc_output = &output->base;
        if (kywc_output->destroying || !kywc_output->state.enabled) {
            continue;
        }
        if (kywc_output->state.scale > scale) {
            scale = kywc_output->state.scale;
        }
    }

    if (scale != output_manager->max_scale) {
        output_manager->max_scale = scale;
        wl_signal_emit_mutable(&output_manager->events.max_scale, kywc_output);
    }
}

static void output_destroy(struct output *output)
{
    struct kywc_output *kywc_output = &output->base;

    kywc_output->destroying = true;
    output_manager_update_max_scale(kywc_output);

    /* output may be disabled before */
    if (output->scene_output) {
        ky_scene_output_destroy(output->scene_output);
        wlr_output_layout_remove(output_manager->server->layout, output->wlr_output);
        output->scene_output = NULL;
    }

    wl_signal_emit_mutable(&output->events.disable, NULL);
    wl_signal_emit_mutable(&kywc_output->events.destroy, NULL);
    assert(wl_list_empty(&output->events.disable.listener_list));
    assert(wl_list_empty(&output->events.geometry.listener_list));
    assert(wl_list_empty(&output->events.usable_area.listener_list));
    assert(wl_list_empty(&output->events.update_usable_area.listener_list));
    assert(wl_list_empty(&output->events.update_late_usable_area.listener_list));
    assert(wl_list_empty(&kywc_output->events.on.listener_list));
    assert(wl_list_empty(&kywc_output->events.off.listener_list));
    assert(wl_list_empty(&kywc_output->events.scale.listener_list));
    assert(wl_list_empty(&kywc_output->events.transform.listener_list));
    assert(wl_list_empty(&kywc_output->events.mode.listener_list));
    assert(wl_list_empty(&kywc_output->events.position.listener_list));
    assert(wl_list_empty(&kywc_output->events.power.listener_list));
    assert(wl_list_empty(&kywc_output->events.brightness.listener_list));
    assert(wl_list_empty(&kywc_output->events.color_temp.listener_list));
    assert(wl_list_empty(&kywc_output->events.destroy.listener_list));

    wl_list_remove(&output->link);
    wl_list_remove(&output->configure_link);

    /* get layouts configure outputs */
    output_manager_get_layout_configs(output_manager);

    /* fix primary output that is being destroyed */
    if (output_manager->primary_output == kywc_output) {
        output_manager->primary_output = NULL;
        /* set primary to a enabled output */
        if (wl_list_empty(&output_manager->configure_outputs) &&
            !wl_list_empty(&output_manager->outputs)) {
            struct output *_output;
            wl_list_for_each(_output, &output_manager->outputs, link) {
                if (_output->base.state.enabled) {
                    output_manager_set_pending_primary(_output);
                    break;
                }
            }
        }
    }

    output_manager_configure_outputs();

    if (!output_manager->server->terminate && output_manager->fallback_output &&
        !output_manager_has_enabled_outputs()) {
        fallback_output_set_state(true, output);
        kywc_output_set_primary(output_manager->fallback_output);
        output_manager_emit_configured(CONFIGURE_TYPE_UPDATE);
    }

    struct kywc_output_mode *mode, *tmp_mode;
    wl_list_for_each_safe(mode, tmp_mode, &kywc_output->prop.modes, link) {
        wl_list_remove(&mode->link);
        free(mode);
    }

    free((void *)kywc_output->prop.edid);
    free((void *)kywc_output->uuid);
    free(output);
}

static void handle_output_destroy(struct wl_listener *listener, void *data)
{
    struct output *output = wl_container_of(listener, output, destroy);

    wl_list_remove(&output->destroy.link);
    wl_list_remove(&output->frame.link);
    wl_list_remove(&output->present.link);
    wl_list_remove(&output->commit.link);

    output_destroy(output);
}

static void handle_new_output(struct wl_listener *listener, void *data)
{
    struct server *server = output_manager->server;
    struct wlr_output *wlr_output = data;

    if (wlr_output->non_desktop) {
        kywc_log(KYWC_WARN, "Not configuring non-desktop output");
        return;
    }

    if (!wlr_output_init_render(wlr_output, server->allocator, server->renderer)) {
        kywc_log(KYWC_ERROR, "Unable to init output renderer");
        return;
    }

    struct output *output = output_create(wlr_output->name, wlr_output);
    if (!output) {
        return;
    }

    output->present.notify = handle_output_present;
    wl_signal_add(&wlr_output->events.present, &output->present);
    output->frame.notify = handle_output_frame;
    wl_signal_add(&wlr_output->events.frame, &output->frame);
    output->destroy.notify = handle_output_destroy;
    wl_signal_add(&wlr_output->events.destroy, &output->destroy);
    output->commit.notify = handle_output_commit;
    wl_signal_add(&wlr_output->events.commit, &output->commit);
}

void output_manager_power_outputs(bool power)
{
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (!output->base.state.enabled || output->base.state.power == power) {
            continue;
        }
        struct kywc_output_state state = output->base.state;
        state.power = power;
        kywc_output_set_state(&output->base, &state);
        output_manager_emit_configured(CONFIGURE_TYPE_NONE);
    }
}

float output_manager_get_max_scale(void)
{
    return output_manager->max_scale;
}

struct kywc_output *output_manager_get_fallback(void)
{
    return output_manager->fallback_output;
}

static void handle_backend_destroy(struct wl_listener *listener, void *data)
{
    wl_list_remove(&output_manager->backend_destroy.link);
    wl_list_remove(&output_manager->new_output.link);
}

struct output_manager *output_manager_create(struct server *server)
{
    output_manager = calloc(1, sizeof(*output_manager));
    if (!output_manager) {
        return NULL;
    }

    output_manager->server = server;
    output_manager->max_scale = 1.0;
    wl_list_init(&output_manager->outputs);
    wl_list_init(&output_manager->configure_outputs);
    wl_list_init(&output_manager->configured.link);
    wl_signal_init(&output_manager->events.new_output);
    wl_signal_init(&output_manager->events.new_enabled_output);
    wl_signal_init(&output_manager->events.primary_output);
    wl_signal_init(&output_manager->events.configured);
    wl_signal_init(&output_manager->events.layout_damage);
    wl_signal_init(&output_manager->events.max_scale);

    output_manager->backend_destroy.notify = handle_backend_destroy;
    wl_signal_add(&server->backend->events.destroy, &output_manager->backend_destroy);

    output_manager->server_destroy.notify = handle_server_destroy;
    server_add_destroy_listener(server, &output_manager->server_destroy);

    output_manager->server_suspend.notify = handle_server_suspend;
    wl_signal_add(&server->events.suspend, &output_manager->server_suspend);

    output_manager->server_resume.notify = handle_server_resume;
    wl_signal_add(&server->events.resume, &output_manager->server_resume);

    xdg_output_manager_v1_create(server);
    output_manager->new_output.notify = handle_new_output;
    wl_signal_add(&server->backend->events.new_output, &output_manager->new_output);

    struct wlr_output *wlr_output = wlr_headless_add_output(server->headless_backend, 1920, 1080);
    wlr_output_set_name(wlr_output, FALLBACK_OUTPUT);

    char *env = getenv("KYWC_USE_LAYOUT_MANAGER");
    output_manager->has_layout_manager = env && strcmp(env, "1") == 0;
    output_manager_config_init(output_manager);

    ky_output_manager_create(server);
    kde_output_management_create(server);
    wlr_output_management_create(server);
    ukui_output_management_create(server);

    return output_manager;
}

uint32_t output_manager_for_each_output(output_iterator_func_t iterator, bool enabled, void *data)
{
    int index = 0;
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (output->base.state.enabled == enabled) {
            if (iterator(&output->base, index++, data)) {
                break;
            }
        }
    }
    return index;
}

void kywc_output_set_primary(struct kywc_output *kywc_output)
{
    if (output_manager->primary_output == kywc_output) {
        return;
    }

    kywc_log(KYWC_INFO, "Primary output is changed to %s",
             kywc_output ? kywc_output->name : "none");
    output_manager->primary_output = kywc_output;
    wl_signal_emit_mutable(&output_manager->events.primary_output, kywc_output);
}

struct kywc_output *kywc_output_get_primary(void)
{
    return output_manager->primary_output;
}

void kywc_output_add_new_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.new_output, listener);
}

void kywc_output_add_primary_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.primary_output, listener);
}

void output_manager_add_configured_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.configured, listener);
}

void output_manager_emit_configured(enum configure_type type)
{
    struct configure_event event = { .type = type };
    wl_signal_emit_mutable(&output_manager->events.configured, &event);
}

void output_manager_add_layout_damage_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.layout_damage, listener);
}

void output_manager_add_max_scale_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.max_scale, listener);
}

void output_manager_add_new_enabled_listener(struct wl_listener *listener)
{
    wl_signal_add(&output_manager->events.new_enabled_output, listener);
}

struct output *output_from_resource(struct wl_resource *resource)
{
    struct wlr_output *wlr_output = wl_resource_get_user_data(resource);
    return output_from_wlr_output(wlr_output);
}

static void output_find_best_mode(struct wlr_output *wlr_output, int32_t width, int32_t height,
                                  int32_t refresh, struct wlr_output_mode **best)
{
    /* find best mode */
    struct wlr_output_mode *m;
    wl_list_for_each(m, &wlr_output->modes, link) {
        if (width != m->width || height != m->height) {
            continue;
        }
        if (refresh == m->refresh) {
            *best = m;
            break;
        }
        if (!*best || m->refresh > (*best)->refresh) {
            *best = m;
        }
    }
}

static void output_ensure_mode(struct wlr_output *wlr_output, struct wlr_output_state *wlr_state,
                               struct wlr_output_mode *mode)
{
    if (wlr_output_test_state(wlr_output, wlr_state)) {
        return;
    }

    kywc_log(KYWC_ERROR, "Mode rejected, falling back to another mode");
    struct wlr_output_mode *m;
    wl_list_for_each(m, &wlr_output->modes, link) {
        if (mode && mode == m) {
            continue;
        }

        wlr_output_state_set_mode(wlr_state, m);
        if (wlr_output_test_state(wlr_output, wlr_state)) {
            break;
        }
    }
}

static bool output_mode_changed(struct output *output, const struct kywc_output_state *state)
{
    struct kywc_output_state *current_state = &output->base.state;
    return current_state->width != state->width || current_state->height != state->height ||
           current_state->refresh != state->refresh;
}

static bool output_color_changed(struct output *output, const struct kywc_output_state *state)
{
    struct kywc_output_state *current_state = &output->base.state;
    return !output->initialized || current_state->power != state->power ||
           current_state->color_temp != state->color_temp ||
           current_state->brightness != state->brightness ||
           current_state->color_filter != state->color_filter;
}

static bool output_color_feature_changed(struct output *output,
                                         const struct kywc_output_state *state)
{
    struct kywc_output_state *current_state = &output->base.state;
    return !output->initialized || current_state->power != state->power ||
           current_state->color_feature != state->color_feature;
}

static bool output_state_changed(struct output *output, const struct kywc_output_state *state)
{
    struct kywc_output_state *current_state = &output->base.state;
    return current_state->enabled != state->enabled || current_state->power != state->power;
}

static bool output_params_changed(struct output *output, const struct kywc_output_state *state)
{
    struct kywc_output_state *current_state = &output->base.state;

    return current_state->transform != state->transform ||
           current_state->vrr_policy != state->vrr_policy || current_state->scale != state->scale ||
           current_state->rgb_range != state->rgb_range ||
           current_state->overscan != state->overscan ||
           current_state->scaling_mode != state->scaling_mode ||
           current_state->scale != state->scale;
}

bool output_use_hardware_color(struct output *output)
{
    return output->hardware_color;
}

bool output_state_attempt_vrr(struct output *output, struct wlr_output_state *state,
                              bool fullscreen)
{
    if (!(output->base.prop.capabilities & KYWC_OUTPUT_CAPABILITY_VRR)) {
        return false;
    }

    if (output->vrr_policy == KYWC_OUTPUT_VRR_POLICY_NEVER ||
        (output->vrr_policy == KYWC_OUTPUT_VRR_POLICY_AUTO && !fullscreen)) {
        if (output->wlr_output->adaptive_sync_status) {
            wlr_output_state_set_adaptive_sync_enabled(state, false);
        }
        return true;
    }

    if (!output->wlr_output->adaptive_sync_status) {
        wlr_output_state_set_adaptive_sync_enabled(state, true);
    }

    return true;
}

bool output_state_attempt_tearing(struct output *output, struct wlr_output_state *state,
                                  bool is_tearing)
{
    if (!is_tearing) {
        return false;
    }

    state->tearing_page_flip = true;
    if (!wlr_output_test_state(output->wlr_output, state)) {
        state->tearing_page_flip = false;
        return false;
    }

    return true;
}

static bool output_set_state(struct output *output, struct kywc_output_state *state)
{
    struct wlr_output *wlr_output = output->wlr_output;
    struct server *server = output_manager->server;

    bool state_changed = output_state_changed(output, state);
    bool mode_changed = output_mode_changed(output, state);
    bool params_changed = output_params_changed(output, state);
    bool enabled = state->enabled && state->power;

    bool changed = !output->initialized || state_changed || mode_changed || params_changed;
    if (changed) {
        struct wlr_output_state wlr_state;
        wlr_output_state_init(&wlr_state);
        if (!output->initialized || state_changed) {
            wlr_output_state_set_enabled(&wlr_state, enabled);
        }

        if (enabled) {
            if (mode_changed) {
                struct wlr_output_mode *best = NULL;
                if (state->width <= 0 || state->height <= 0) {
                    kywc_log(KYWC_INFO, "Set preferred mode as no config found");
                    best = wlr_output_preferred_mode(wlr_output);
                } else {
                    output_find_best_mode(wlr_output, state->width, state->height, state->refresh,
                                          &best);
                }

                if (best) {
                    wlr_output_state_set_mode(&wlr_state, best);
                } else {
                    wlr_output_state_set_custom_mode(&wlr_state, state->width, state->height,
                                                     state->refresh);
                }
                output_ensure_mode(wlr_output, &wlr_state, best);
            } else if (output->wlr_output->enabled && wlr_output_is_drm(wlr_output) &&
                       output->wlr_output->width == state->width &&
                       output->wlr_output->height == state->height && output->scene_commit &&
                       output->initialized) {
                output->has_pending = true;
                kywc_log(KYWC_WARN, "Drm output commit need waiting for pageflip");
                return true;
            }

            wlr_output_set_color_changed(wlr_output, enabled);
            wlr_output_state_set_transform(&wlr_state, state->transform);
            wlr_output_state_set_scale(&wlr_state, state->scale);
            wlr_output_set_rgb_range(wlr_output, state->rgb_range);
            wlr_output_set_overscan(wlr_output, state->overscan);
            wlr_output_set_scaling_mode(wlr_output, state->scaling_mode);
        }

        if (output->scene_commit) {
            output->has_pending = false;
        }
        if (!wlr_output_commit_state(wlr_output, &wlr_state)) {
            kywc_log(KYWC_ERROR, "Failed to commit output: %s", wlr_output->name);
            wlr_output_state_finish(&wlr_state);
            return false;
        }
        wlr_output_state_finish(&wlr_state);
    }

    bool color_feature_changed = output_color_feature_changed(output, state);
    bool color_changed = color_feature_changed || output_color_changed(output, state);
    if (enabled && color_changed) {
        wlr_output_set_color_changed(wlr_output, color_changed);
        bool hardware_color = wlr_output_is_hardware_color_allowed(output->wlr_output, state);
        if (color_feature_changed || output->hardware_color != hardware_color || !hardware_color) {
            if (output->scene_output) {
                ky_scene_output_damage_whole(output->scene_output);
            }
        } else {
            output_schedule_frame(output->wlr_output);
        }
        output->hardware_color = hardware_color;
    }

    output->color_temp = state->color_temp;
    output->brightness = state->brightness;
    output->color_feature = state->color_feature;
    output->color_filter = state->color_filter;
    output->vrr_policy = state->vrr_policy;

    /* if output disabled, skip output_layout_add */
    if (!state->enabled) {
        ky_scene_output_destroy(output->scene_output);
        wlr_output_layout_remove(server->layout, wlr_output);
        output->scene_output = NULL;
        return true;
    }

    struct wlr_output_layout_output *loutput = NULL;
    /* force to reconfigure even if have loutput */
    if (state->lx == -1 || state->ly == -1) {
        loutput = wlr_output_layout_add_auto(server->layout, wlr_output);
        if (loutput) {
            loutput->auto_configured = false;
        }
    } else {
        loutput = wlr_output_layout_get(server->layout, wlr_output);
        if (!loutput || loutput->x != state->lx || loutput->y != state->ly) {
            loutput = wlr_output_layout_add(server->layout, wlr_output, state->lx, state->ly);
        }
    }

    if (!output->scene_output && loutput) {
        output->scene_output = ky_scene_output_create(server->scene, wlr_output);
        ky_scene_output_layout_add_output(server->scene_layout, loutput, output->scene_output);
    }

    return true;
}

static void output_update_geometry(struct output *output, struct kywc_box *box)
{
    struct kywc_output_state *state = &output->base.state;

    if (state->transform % 2 == 0) {
        box->width = state->width;
        box->height = state->height;
    } else {
        box->width = state->height;
        box->height = state->width;
    }

    box->x = state->lx;
    box->y = state->ly;
    box->width = ceil(box->width / state->scale);
    box->height = ceil(box->height / state->scale);
}

static void output_do_update_usable_area(struct output *output, struct kywc_box *usable)
{
    *usable = output->geometry;
    wl_signal_emit_mutable(&output->events.update_usable_area, usable);
    wl_signal_emit_mutable(&output->events.update_late_usable_area, usable);
}

static void output_emit_usable_area(struct output *output)
{
    kywc_log(KYWC_DEBUG, "Output %s usable area is (%d, %d) %d x %d", output->base.name,
             output->usable_area.x, output->usable_area.y, output->usable_area.width,
             output->usable_area.height);

    xwayland_update_workarea();

    wl_signal_emit_mutable(&output->events.usable_area, NULL);
}

void output_update_usable_area(struct kywc_output *kywc_output)
{
    /* no need to update usable area when disabled or destroyed */
    if (!kywc_output || kywc_output->destroying || !kywc_output->state.enabled) {
        return;
    }

    struct output *output = output_from_kywc_output(kywc_output);
    struct kywc_box usable_area;

    output_do_update_usable_area(output, &usable_area);
    if (kywc_box_equal(&output->usable_area, &usable_area)) {
        return;
    }

    output->usable_area = usable_area;
    output_emit_usable_area(output);
}

bool kywc_output_set_state(struct kywc_output *kywc_output, struct kywc_output_state *state)
{
    struct output *output = output_from_kywc_output(kywc_output);
    if (!output_set_state(output, state)) {
        return false;
    }

    struct kywc_output_state *current = &kywc_output->state;
    struct kywc_output_state old = kywc_output->state;
    output_get_state(output, current);

    // fix current.enabled for dpms power
    current->enabled = state->enabled;

    /* update geometry and usable area before all signals */
    bool geometry_changed = false;
    bool usable_area_changed = false;

    if (current->enabled) {
        struct kywc_box geo = output->geometry;
        output_update_geometry(output, &output->geometry);
        geometry_changed = !kywc_box_equal(&geo, &output->geometry);
        /* only update usable area when geometry changed */
        if (!old.enabled || geometry_changed) {
            output_do_update_usable_area(output, &geo);
            if (!kywc_box_equal(&geo, &output->usable_area)) {
                output->usable_area = geo;
                usable_area_changed = true;
            }
        }
    }

    /* sort output manager outputs */
    if (geometry_changed || current->width != old.width || current->height != old.height ||
        current->transform != old.transform || current->scale != old.scale ||
        current->lx != old.lx || current->ly != old.ly) {
        output_manager_sort_outputs();
    }

    if (current->enabled != old.enabled || current->scale != old.scale) {
        output_manager_update_max_scale(kywc_output);
    }

    output_write_config(output);

    /* early return when not initialized as new_output and new_enabled_output not emitted */
    if (!output->initialized) {
        return true;
    }

    /* check state changes */
    if (current->enabled != old.enabled) {
        if (!current->enabled) {
            wl_signal_emit_mutable(&kywc_output->events.off, NULL);
            wl_signal_emit_mutable(&output->events.disable, NULL);
            return true;
        } else {
            wl_signal_emit_mutable(&kywc_output->events.on, NULL);
            wl_signal_emit_mutable(&output_manager->events.new_enabled_output, kywc_output);
        }
    }

    if (geometry_changed) {
        kywc_log(KYWC_DEBUG, "Output %s geometry is (%d, %d) %d x %d", output->base.name,
                 output->geometry.x, output->geometry.y, output->geometry.width,
                 output->geometry.height);
        wl_signal_emit_mutable(&output->events.geometry, NULL);
    }

    if (usable_area_changed) {
        output_emit_usable_area(output);
    }

    if (kywc_output->prop.capabilities & KYWC_OUTPUT_CAPABILITY_POWER &&
        current->power != old.power) {
        wl_signal_emit_mutable(&kywc_output->events.power, NULL);
    }

    if (kywc_output->prop.capabilities & KYWC_OUTPUT_CAPABILITY_BRIGHTNESS &&
        current->brightness != old.brightness) {
        wl_signal_emit_mutable(&kywc_output->events.brightness, NULL);
    }

    if (kywc_output->prop.capabilities & KYWC_OUTPUT_CAPABILITY_COLOR_TEMP &&
        current->color_temp != old.color_temp) {
        wl_signal_emit_mutable(&kywc_output->events.color_temp, NULL);
    }

    if (kywc_output->prop.capabilities & KYWC_OUTPUT_CAPABILITY_COLOR_FILTER &&
        current->color_filter != old.color_filter) {
        wl_signal_emit_mutable(&kywc_output->events.color_filter, NULL);
    }

    if (current->width != old.width || current->height != old.height ||
        current->refresh != old.refresh) {
        wl_signal_emit_mutable(&kywc_output->events.mode, NULL);
    }

    if (current->transform != old.transform) {
        output_add_transform_effect(kywc_output, &old, current);
        wl_signal_emit_mutable(&kywc_output->events.transform, NULL);
    }

    if (current->scale != old.scale) {
        wl_signal_emit_mutable(&kywc_output->events.scale, NULL);
    }

    if (current->lx != old.lx || current->ly != old.ly) {
        wl_signal_emit_mutable(&kywc_output->events.position, NULL);
    }

    return true;
}

struct kywc_output *kywc_output_by_name(const char *name)
{
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (strcmp(name, output->base.name) == 0) {
            return &output->base;
        }
    }

    return NULL;
}

struct kywc_output *kywc_output_by_uuid(const char *uuid)
{
    struct output *output;
    wl_list_for_each(output, &output_manager->outputs, link) {
        if (strcmp(uuid, output->base.uuid) == 0) {
            return &output->base;
        }
    }

    return NULL;
}

void kywc_output_effective_geometry(struct kywc_output *kywc_output, struct kywc_box *box)
{
    struct output *output = output_from_kywc_output(kywc_output);

    *box = output->geometry;
}

bool kywc_output_contains_point(struct kywc_output *kywc_output, int x, int y)
{
    struct output *output = output_from_kywc_output(kywc_output);

    return kywc_box_contains_point(&output->geometry, x, y);
}

void output_add_update_usable_area_listener(struct kywc_output *kywc_output,
                                            struct wl_listener *listener, bool late)
{
    struct output *output = output_from_kywc_output(kywc_output);

    if (late) {
        wl_signal_add(&output->events.update_late_usable_area, listener);
    } else {
        wl_signal_add(&output->events.update_usable_area, listener);
    }
}

void output_schedule_frame(struct wlr_output *wlr_output)
{
    if (output_manager->server->active) {
        wlr_output_schedule_frame(wlr_output);
    }
}

bool output_set_color_feature(struct kywc_output *kywc_output,
                              enum kywc_output_color_feature color_feature)
{
    if (!kywc_output->state.enabled) {
        return false;
    }
    struct kywc_output_state state = kywc_output->state;
    state.color_feature = color_feature;
    kywc_output_set_state(kywc_output, &state);
    return true;
}

bool output_set_color_filter(struct kywc_output *kywc_output,
                             enum kywc_output_color_filter color_filter)
{
    if (!kywc_output->state.enabled) {
        return false;
    }

    struct kywc_output_state state = kywc_output->state;
    state.color_filter = color_filter;
    kywc_output_set_state(kywc_output, &state);
    return true;
}

bool output_set_color_temp(struct kywc_output *kywc_output, uint32_t color_temp)
{
    if (!kywc_output->state.enabled) {
        return false;
    }

    struct kywc_output_state state = kywc_output->state;
    state.color_temp = COLORTEMP_CLAMP(color_temp);
    kywc_output_set_state(kywc_output, &state);
    return true;
}

bool output_set_brightness(struct kywc_output *kywc_output, uint32_t brightness)
{
    if (!kywc_output->state.enabled) {
        return false;
    }

    struct kywc_output_state state = kywc_output->state;
    state.brightness = BRIGHTNESS_CLAMP(brightness);
    kywc_output_set_state(kywc_output, &state);
    return true;
}
