// SPDX-FileCopyrightText: 2025 KylinSoft Co., Ltd.
//
// SPDX-License-Identifier: Expat

#include <assert.h>

#include "output_p.h"
#include "util/macros.h"

static struct kywc_output_mode *output_preferred_mode(struct kywc_output *kywc_output)
{
    struct kywc_output_mode *mode;
    wl_list_for_each(mode, &kywc_output->prop.modes, link) {
        if (mode->preferred) {
            return mode;
        }
    }
    // No preferred mode, choose the first one
    return wl_container_of(kywc_output->prop.modes.prev, mode, link);
}

static float output_preferred_scale(struct kywc_output *kywc_output, int width, int height)
{
    if (kywc_output->prop.phys_width == 0 || kywc_output->prop.phys_height == 0) {
        return 1.0;
    }

    float phys_width = kywc_output->prop.phys_width / 25.4;
    float phys_height = kywc_output->prop.phys_height / 25.4;
    float phys_inch = sqrtf(pow(phys_width, 2) + pow(phys_height, 2));
    float pixels = sqrtf(pow(width, 2) + pow(height, 2));
    float ppi = pixels / phys_inch;
    float scale = 1.0;

    if (ppi > 140.0) {
        float range = ppi - 140.0;
        int step = range / 40.0;
        scale += step * 0.25;
        if (range >= step * 40) {
            scale += 0.25;
        }
    }

    kywc_log(KYWC_DEBUG, "Output %s(%d x %d %f inch %f ppi) scale = %f", kywc_output->name, width,
             height, phys_inch, ppi, scale);

    return scale;
}

void output_fill_preferred_state(struct output *output)
{
    struct kywc_output *kywc_output = &output->base;
    struct kywc_output_state *pending = &output->pending_state;
    *pending = output->base.state;

    pending->enabled = pending->power = true;
    /* use the preferred mode */
    struct kywc_output_mode *mode = output_preferred_mode(kywc_output);
    pending->width = mode->width;
    pending->height = mode->height;
    pending->refresh = mode->refresh;
    /* use the preferred scale */
    pending->scale = output_preferred_scale(kywc_output, pending->width, pending->height);

    /* fallback output position is (0, 0) */
    if (output_manager->fallback_output == kywc_output) {
        pending->lx = pending->ly = 0;
    } else { /* use auto position from layout */
        pending->lx = pending->ly = -1;
    }

    pending->transform = WL_OUTPUT_TRANSFORM_NORMAL;
    pending->brightness = pending->brightness == 0 ? 100 : CLAMP(pending->brightness, 10, 100);
    pending->color_temp = pending->color_temp == 0 ? 6500 : CLAMP(pending->color_temp, 1000, 25000);
}

void fallback_output_fill_state(struct output *output, bool enabled, struct output *mirror)
{
    /* early return if disable the fallback output */
    if (!enabled) {
        output->pending_state.enabled = false;
        return;
    }

    if (!mirror) {
        output_fill_preferred_state(output);
        return;
    }

    struct kywc_output_state *state = &output->pending_state;
    *state = mirror->base.state;
    /* fixup the state */
    state->enabled = state->power = true;
    state->lx = state->ly = 0;
    /* try to add this mode to fallback output */
    struct wlr_output_mode mode = {
        .width = state->width,
        .height = state->height,
        .refresh = state->refresh,
    };
    output_add_mode(output, &mode, true);
}

void output_manager_set_pending_primary(struct output *output)
{
    struct kywc_output *kywc_output = &output->base;
    output_manager->pending_primary = kywc_output;
}

void output_manager_add_pending_state(struct output *output, struct kywc_output_state *state)
{
    assert(!output->base.destroying);

    if (state != &output->pending_state) {
        output->pending_state = *state;
    }

    wl_list_remove(&output->configure_link);
    wl_list_insert(&output_manager->configure_outputs, &output->configure_link);

    output_log_state(KYWC_INFO, output, "pending_state", state);
}

static void clear_pending_configures(void)
{
    struct output *output, *tmp;
    wl_list_for_each_safe(output, tmp, &output_manager->configure_outputs, configure_link) {
        wl_list_remove(&output->configure_link);
        wl_list_init(&output->configure_link);
    }
    assert(wl_list_empty(&output_manager->configure_outputs));

    output_manager->pending_primary = NULL;
}

bool output_manager_configure_outputs(void)
{
    if (wl_list_empty(&output_manager->configure_outputs)) {
        return true;
    }

    if (output_manager->server->terminate) {
        clear_pending_configures();
        return false;
    }

    /* 1. check configs */
    if (!output_manager->pending_primary && output_manager->primary_output &&
        output_manager->primary_output != output_manager->fallback_output) {
        output_manager->pending_primary = output_manager->primary_output;
        kywc_log(KYWC_DEBUG, "Keep primary output to %s", output_manager->primary_output->name);
    }

    struct kywc_output *kywc_output = NULL;
    struct output *output = NULL, *pending_output = NULL;
    bool need_fix_primary_output = !output_manager->pending_primary;

    /* primary output may be disabled, fixup it */
    wl_list_for_each(output, &output_manager->configure_outputs, configure_link) {
        /* It's going to disable the primary output and no new primary config */
        kywc_output = &output->base;
        if (kywc_output == output_manager->pending_primary && !output->pending_state.enabled) {
            need_fix_primary_output = true;
            break;
        }
    }

    /* if all outputs will be disabled in config, find others not in config */
    bool have_enabled_output = false, have_zero_coord = false;
    bool have_actual_output = output_manager_has_actual_outputs();

    wl_list_for_each(output, &output_manager->outputs, link) {
        kywc_output = &output->base;
        if (have_actual_output && kywc_output == output_manager->fallback_output) {
            continue;
        }

        if (!wl_list_empty(&output->configure_link)) {
            pending_output = output;
            have_enabled_output |= output->pending_state.enabled;
            have_zero_coord |= output->pending_state.enabled && output->pending_state.lx == 0 &&
                               output->pending_state.ly == 0;
        } else {
            have_enabled_output |= kywc_output->state.enabled;
            have_zero_coord |= kywc_output->state.enabled && kywc_output->state.lx == 0 &&
                               kywc_output->state.ly == 0;
        }

        if (have_actual_output && have_enabled_output &&
            output_manager->pending_primary == output_manager->fallback_output) {
            need_fix_primary_output = true;
        }

        /* fixup primary output */
        if (have_enabled_output && need_fix_primary_output) {
            kywc_log(KYWC_DEBUG, "Fixup primary output to %s", output->base.name);
            need_fix_primary_output = false;
            output_manager_set_pending_primary(output);
        }

        if (have_enabled_output && have_zero_coord) {
            break;
        }
    }

    /* have one actual output and coord is not zero */
    if (!have_zero_coord && pending_output) {
        int output_num = wl_list_length(&output_manager->outputs);
        if (output_num == 1 ||
            (output_num == 2 && have_actual_output && output_manager->fallback_output)) {
            kywc_log(KYWC_DEBUG, "Fixup output %s coord to zero", pending_output->base.name);
            pending_output->pending_state.lx = pending_output->pending_state.ly = 0;
        }
    }

    if (!have_enabled_output) {
        kywc_log(KYWC_WARN, "All outputs will be disabled, reject this configuration");
        clear_pending_configures();
        return false;
    }

    /* 2. config outputs */
    /* call kywc_output_set_state in all pending outputs, enable first and disable outputs later
     */
    bool ret = false;

    wl_list_for_each(output, &output_manager->configure_outputs, configure_link) {
        if (!output->pending_state.enabled) {
            continue;
        }
        kywc_output = &output->base;
        if (!kywc_output->state.enabled) {
            continue;
        }
        if (!kywc_output_set_state(kywc_output, &output->pending_state)) {
            goto error;
        }
    }

    wl_list_for_each(output, &output_manager->configure_outputs, configure_link) {
        if (!output->pending_state.enabled) {
            continue;
        }
        kywc_output = &output->base;
        if (kywc_output->state.enabled) {
            continue;
        }
        if (!kywc_output_set_state(kywc_output, &output->pending_state)) {
            goto error;
        }
    }

    kywc_output_set_primary(output_manager->pending_primary);

    wl_list_for_each(output, &output_manager->configure_outputs, configure_link) {
        if (output->pending_state.enabled) {
            continue;
        }
        kywc_output = &output->base;
        if (!kywc_output->state.enabled) {
            continue;
        }
        if (!kywc_output_set_state(kywc_output, &output->pending_state)) {
            goto error;
        }
    }

    /* disable the fallback when an actual enabled output exists */
    if (output_manager->fallback_output && output_manager_has_enabled_outputs()) {
        fallback_output_set_state(false, NULL);
    }

    wl_list_for_each(output, &output_manager->configure_outputs, configure_link) {
        output_log_state(KYWC_INFO, output, "current", &output->base.state);
    }

    ret = true;
error:
    output_manager_emit_configured(CONFIGURE_TYPE_UPDATE);
    clear_pending_configures();
    return ret;
}
