#! /bin/sh
# SPDX-License-Identifier: MIT
# Copyright (c) 2024 Linaro Ltd
# Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
# Author: Dmitry Baryshkov

set -e

GITHUB_REPO="https://github.com/WOA-Project/Qualcomm-Reference-Drivers"
OUTDIR=${OUTDIR:-.}

do_prepare() {
	FWTEMPDIR=$(mktemp -d firmware-woa.XXXXXX --tmpdir )
	trap "rm -rf ${FWTEMPDIR}" EXIT

	UNPACKDIR="${FWTEMPDIR}/unpack"
	mkdir -p "${UNPACKDIR}"

	if [ -z "${DLDIR}" ]
	then
		DLDIR="${FWTEMPDIR}/dl"
		mkdir -p "${DLDIR}"
	fi

	WORKDIR="${FWTEMPDIR}/package"
	rm -rf "${WORKDIR}"
	mkdir -p "${WORKDIR}"

	FW_OUT="${WORKDIR}/qcom/${LIB_FW_DIR}"
	mkdir -p "${FW_OUT}"

	DEBDIR="${WORKDIR}/debian"
	mkdir -p "${DEBDIR}"

	cat > "${DEBDIR}/rules" << EOF
#!/usr/bin/make -f

%:
	dh \$@

# comaptibility with old and new Debian repos.
override_dh_installdeb:
	dh_installdeb
	if command -v dh_movetousr >/dev/null; then dh_movetousr; fi
EOF

	cat > "${DEBDIR}/control" << EOF
Source: ${FW_PKG_NAME}
Section: non-free-firmware/kernel
Priority: optional
Maintainer: Dmitry Baryshkov <dmitry.baryshkov@oss.qualcomm.com>
Standards-Version: 4.6.2
Build-Depends: debhelper-compat (= 12)

Package: ${FW_PKG_NAME}
Architecture: all
Depends: firmware-qcom-soc, firmware-atheros, \${misc:Depends}
Description: Binary firmware for the ${FW_DEVNAME}
 This package contains the binary firmware for
 the ${FW_DEVNAME}.
 .
 The firmware has been repackaged from WoA cab archives.
EOF

	cat > "${DEBDIR}/changelog" << EOF
${FW_PKG_NAME} (${FW_VER}-1) unstable; urgency=medium

  * Initial upload.

 -- Dmitry Baryshkov <dmitry.baryshkov@oss.qualcomm.com>  Mon, 1 Jan 2024 00:00:00 +0000
EOF

	cat > "${DEBDIR}/${FW_PKG_NAME}.install" << EOF
qcom /lib/firmware/
EOF

	if [ -n "${DSP_DIR}" ]
	then
		DSP_PKG_NAME=hexagon-dsp-binaries-${FW_PKG_NAME##firmware-}
		cat >> "${DEBDIR}/control" << EOF

Package: ${DSP_PKG_NAME}
Architecture: all
Depends: ${FW_PKG_NAME} (= ${FW_VER}-1), \${misc:Depends}
Description: Binary Hexagon DSP libraries for the ${FW_DEVNAME}
 This package contains the Hexagon DSP libraries for
 the ${FW_DEVNAME}.
 .
 The libraries have been repackaged from WoA cab archives.
EOF

		cat > "${DEBDIR}/${DSP_PKG_NAME}.install" << EOF
hexagon-dsp /usr/share
EOF
		DSP_OUT="${WORKDIR}/hexagon-dsp/${DSP_DIR}"
		mkdir -p "${DSP_OUT}"
	fi
}

do_shacheck() {
	[ ! -r $2 ] && return 1

	[ `sha256sum $2 | cut -d' ' -f 1` = "$1" ]
}

do_dl() {
	while [ -n "$1" ]
	do
		sha="$1"
		cab="$2"
		dlcab="${DLDIR}/${FW_VER}/${cab}"

		if ! do_shacheck $sha "${dlcab}"
		then
			rm -f "${dlcab}"
			mkdir -p $(dirname ${dlcab})
			wget "${GITHUB_REPO}/raw/master/${FW_SUBDIR}/${FW_VER}/${cab}" -O "${dlcab}"
		fi

		if ! do_shacheck $sha "${dlcab}"
		then
			echo "Checksum failure for ${cab}"
			exit 1
		fi

		cabextract -q -d "${UNPACKDIR}" "${dlcab}"

		shift
		shift
	done
}

do_install() {
	for file in $@
	do
		if [ "$file" = "wlanmdsp.mbn" -a -e "${UNPACKDIR}/WLANMDSP.MBN" ]
		then
			mv "${UNPACKDIR}/WLANMDSP.MBN" "${FW_OUT}"
		else
			mv "${UNPACKDIR}"/${file} "${FW_OUT}"
		fi
	done
}

do_install_jsn() {
	for file in $@
	do
		cp "${SHAREDIR}/jsn/${file}" "${FW_OUT}"
	done
}

do_divert() {
	[ ! -r ${DEBDIR}/${FW_PKG_NAME}.preinst ] && { echo '#!/bin/sh' ; echo "" ; } > ${DEBDIR}/${FW_PKG_NAME}.preinst
	[ ! -r ${DEBDIR}/${FW_PKG_NAME}.postrm ] && { echo '#!/bin/sh' ; echo "" ; } > ${DEBDIR}/${FW_PKG_NAME}.postrm

	cat >> ${DEBDIR}/${FW_PKG_NAME}.preinst << EOF
dpkg-divert --package ${FW_PKG_NAME} --rename --divert /lib/firmware/$1.usr --add /usr/lib/firmware/$1
dpkg-divert --package ${FW_PKG_NAME} --rename --divert /lib/firmware/$1.lib --add /lib/firmware/$1
EOF

	cat >> ${DEBDIR}/${FW_PKG_NAME}.postrm << EOF
dpkg-divert --package ${FW_PKG_NAME} --rename --remove /lib/firmware/$1
dpkg-divert --package ${FW_PKG_NAME} --rename --remove /usr/lib/firmware/$1
EOF
}

do_ignore() {
	mkdir -p "${WORKDIR}/ignored"
	for file in $@
	do
		mv "${UNPACKDIR}/"${file} "${WORKDIR}/ignored"
	done
}

do_dsp() {
	DSP="$1"
	shift
	mkdir -p "${DSP_OUT}/dsp/${DSP}"

	for file in $@
	do
		mv "${UNPACKDIR}/"${file} "${DSP_OUT}/dsp/${DSP}"
	done
}

do_listremain() {
	rm -f ${UNPACKDIR}/*.dll ${UNPACKDIR}/*.qdb ${UNPACKDIR}/*.cat ${UNPACKDIR}/*.inf ${UNPACKDIR}/*.sys
	rm -f ${UNPACKDIR}/map*txt ${UNPACKDIR}/*/map*txt
	rmdir --ignore-fail-on-non-empty `find ${UNPACKDIR} -type d -depth`
	if [ -d ${UNPACKDIR} ]
	then
		echo Remaining Files:
		ls -1R ${UNPACKDIR}
		echo
		exit 1
	else
		echo No remaining Files
	fi
}

do_build() {
	[ -n "${LISTREMAIN}" ] && do_listremain

	cd ${WORKDIR}
	dpkg-buildpackage -b --no-sign
	cd -

	mkdir -p ${OUTDIR}
	mv ${FWTEMPDIR}/*.* ${OUTDIR}
}
